/*
 * Copyright (c) 2008-2015, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.cache.impl.client;

import com.hazelcast.cache.impl.CacheOperationProvider;
import com.hazelcast.cache.impl.CachePortableHook;
import com.hazelcast.cache.impl.operation.CacheGetAndRemoveOperation;
import com.hazelcast.config.InMemoryFormat;
import com.hazelcast.nio.ObjectDataInput;
import com.hazelcast.nio.ObjectDataOutput;
import com.hazelcast.nio.serialization.Data;
import com.hazelcast.nio.serialization.PortableReader;
import com.hazelcast.nio.serialization.PortableWriter;
import com.hazelcast.security.permission.ActionConstants;
import com.hazelcast.security.permission.CachePermission;
import com.hazelcast.spi.Operation;

import java.io.IOException;
import java.security.Permission;

/**
 * This client request  specifically calls {@link CacheGetAndRemoveOperation} on the server side.
 *
 * @see com.hazelcast.cache.impl.operation.CacheGetAndRemoveOperation
 */
public class CacheGetAndRemoveRequest
        extends AbstractCacheRequest {

    private Data key;
    private int completionId = -1;

    public CacheGetAndRemoveRequest() {
    }

    public CacheGetAndRemoveRequest(String name, Data key, InMemoryFormat inMemoryFormat) {
        super(name, inMemoryFormat);
        this.key = key;
    }

    @Override
    public int getClassId() {
        return CachePortableHook.GET_AND_REMOVE;
    }

    @Override
    protected Object getKey() {
        return key;
    }

    @Override
    protected Operation prepareOperation() {
        CacheOperationProvider operationProvider = getOperationProvider();
        return operationProvider.createGetAndRemoveOperation(key, completionId);
    }

    @Override
    public void write(PortableWriter writer)
            throws IOException {
        super.write(writer);
        writer.writeInt("c", completionId);
        final ObjectDataOutput out = writer.getRawDataOutput();
        out.writeData(key);
    }

    @Override
    public void read(PortableReader reader)
            throws IOException {
        super.read(reader);
        completionId = reader.readInt("c");
        final ObjectDataInput in = reader.getRawDataInput();
        key = in.readData();
    }

    @Override
    public void setCompletionId(Integer completionId) {
        this.completionId = completionId != null ? completionId : -1;
    }

    @Override
    public Permission getRequiredPermission() {
        return new CachePermission(name, ActionConstants.ACTION_READ, ActionConstants.ACTION_REMOVE);
    }

    @Override
    public Object[] getParameters() {
        return new Object[]{key};
    }

    @Override
    public String getMethodName() {
        return "getAndRemove";
    }

    @Override
    public String getDistributedObjectName() {
        return name;
    }

}
