/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package com.hazelcast.org.apache.hc.client5.http.impl.classic;

import java.io.IOException;
import java.net.Socket;

import com.hazelcast.org.apache.hc.client5.http.AuthenticationStrategy;
import com.hazelcast.org.apache.hc.client5.http.HttpRoute;
import com.hazelcast.org.apache.hc.client5.http.RouteInfo.LayerType;
import com.hazelcast.org.apache.hc.client5.http.RouteInfo.TunnelType;
import com.hazelcast.org.apache.hc.client5.http.auth.AuthExchange;
import com.hazelcast.org.apache.hc.client5.http.auth.AuthSchemeFactory;
import com.hazelcast.org.apache.hc.client5.http.auth.AuthScope;
import com.hazelcast.org.apache.hc.client5.http.auth.ChallengeType;
import com.hazelcast.org.apache.hc.client5.http.auth.Credentials;
import com.hazelcast.org.apache.hc.client5.http.auth.StandardAuthScheme;
import com.hazelcast.org.apache.hc.client5.http.config.RequestConfig;
import com.hazelcast.org.apache.hc.client5.http.impl.DefaultAuthenticationStrategy;
import com.hazelcast.org.apache.hc.client5.http.impl.DefaultClientConnectionReuseStrategy;
import com.hazelcast.org.apache.hc.client5.http.impl.TunnelRefusedException;
import com.hazelcast.org.apache.hc.client5.http.impl.auth.BasicCredentialsProvider;
import com.hazelcast.org.apache.hc.client5.http.impl.auth.BasicSchemeFactory;
import com.hazelcast.org.apache.hc.client5.http.impl.auth.DigestSchemeFactory;
import com.hazelcast.org.apache.hc.client5.http.impl.auth.HttpAuthenticator;
import com.hazelcast.org.apache.hc.client5.http.impl.auth.KerberosSchemeFactory;
import com.hazelcast.org.apache.hc.client5.http.impl.auth.NTLMSchemeFactory;
import com.hazelcast.org.apache.hc.client5.http.impl.auth.SPNegoSchemeFactory;
import com.hazelcast.org.apache.hc.client5.http.impl.io.ManagedHttpClientConnectionFactory;
import com.hazelcast.org.apache.hc.client5.http.io.ManagedHttpClientConnection;
import com.hazelcast.org.apache.hc.client5.http.protocol.HttpClientContext;
import com.hazelcast.org.apache.hc.client5.http.protocol.RequestClientConnControl;
import com.hazelcast.org.apache.hc.core5.http.ClassicHttpRequest;
import com.hazelcast.org.apache.hc.core5.http.ClassicHttpResponse;
import com.hazelcast.org.apache.hc.core5.http.ConnectionReuseStrategy;
import com.hazelcast.org.apache.hc.core5.http.HttpEntity;
import com.hazelcast.org.apache.hc.core5.http.HttpException;
import com.hazelcast.org.apache.hc.core5.http.HttpHeaders;
import com.hazelcast.org.apache.hc.core5.http.HttpHost;
import com.hazelcast.org.apache.hc.core5.http.Method;
import com.hazelcast.org.apache.hc.core5.http.config.CharCodingConfig;
import com.hazelcast.org.apache.hc.core5.http.config.Http1Config;
import com.hazelcast.org.apache.hc.core5.http.config.Lookup;
import com.hazelcast.org.apache.hc.core5.http.config.RegistryBuilder;
import com.hazelcast.org.apache.hc.core5.http.impl.io.HttpRequestExecutor;
import com.hazelcast.org.apache.hc.core5.http.io.HttpConnectionFactory;
import com.hazelcast.org.apache.hc.core5.http.io.entity.EntityUtils;
import com.hazelcast.org.apache.hc.core5.http.message.BasicClassicHttpRequest;
import com.hazelcast.org.apache.hc.core5.http.message.StatusLine;
import com.hazelcast.org.apache.hc.core5.http.protocol.BasicHttpContext;
import com.hazelcast.org.apache.hc.core5.http.protocol.DefaultHttpProcessor;
import com.hazelcast.org.apache.hc.core5.http.protocol.HttpContext;
import com.hazelcast.org.apache.hc.core5.http.protocol.HttpCoreContext;
import com.hazelcast.org.apache.hc.core5.http.protocol.HttpProcessor;
import com.hazelcast.org.apache.hc.core5.http.protocol.RequestTargetHost;
import com.hazelcast.org.apache.hc.core5.http.protocol.RequestUserAgent;
import com.hazelcast.org.apache.hc.core5.util.Args;

/**
 * ProxyClient can be used to establish a tunnel via an HTTP/1.1 proxy.
 */
public class ProxyClient {

    private final HttpConnectionFactory<ManagedHttpClientConnection> connFactory;
    private final RequestConfig requestConfig;
    private final HttpProcessor httpProcessor;
    private final HttpRequestExecutor requestExec;
    private final AuthenticationStrategy proxyAuthStrategy;
    private final HttpAuthenticator authenticator;
    private final AuthExchange proxyAuthExchange;
    private final Lookup<AuthSchemeFactory> authSchemeRegistry;
    private final ConnectionReuseStrategy reuseStrategy;

    /**
     * @since 5.0
     */
    public ProxyClient(
            final HttpConnectionFactory<ManagedHttpClientConnection> connFactory,
            final Http1Config h1Config,
            final CharCodingConfig charCodingConfig,
            final RequestConfig requestConfig) {
        super();
        this.connFactory = connFactory != null
                ? connFactory
                : ManagedHttpClientConnectionFactory.builder()
                .http1Config(h1Config)
                .charCodingConfig(charCodingConfig)
                .build();
        this.requestConfig = requestConfig != null ? requestConfig : RequestConfig.DEFAULT;
        this.httpProcessor = new DefaultHttpProcessor(
                new RequestTargetHost(), new RequestClientConnControl(), new RequestUserAgent());
        this.requestExec = new HttpRequestExecutor();
        this.proxyAuthStrategy = new DefaultAuthenticationStrategy();
        this.authenticator = new HttpAuthenticator();
        this.proxyAuthExchange = new AuthExchange();
        this.authSchemeRegistry = RegistryBuilder.<AuthSchemeFactory>create()
                .register(StandardAuthScheme.BASIC, BasicSchemeFactory.INSTANCE)
                .register(StandardAuthScheme.DIGEST, DigestSchemeFactory.INSTANCE)
                .register(StandardAuthScheme.NTLM, NTLMSchemeFactory.INSTANCE)
                .register(StandardAuthScheme.SPNEGO, SPNegoSchemeFactory.DEFAULT)
                .register(StandardAuthScheme.KERBEROS, KerberosSchemeFactory.DEFAULT)
                .build();
        this.reuseStrategy = DefaultClientConnectionReuseStrategy.INSTANCE;
    }

    /**
     * @since 4.3
     */
    public ProxyClient(final RequestConfig requestConfig) {
        this(null, null, null, requestConfig);
    }

    public ProxyClient() {
        this(null, null, null, null);
    }

    public Socket tunnel(
            final HttpHost proxy,
            final HttpHost target,
            final Credentials credentials) throws IOException, HttpException {
        Args.notNull(proxy, "Proxy host");
        Args.notNull(target, "Target host");
        Args.notNull(credentials, "Credentials");
        HttpHost host = target;
        if (host.getPort() <= 0) {
            host = new HttpHost(host.getSchemeName(), host.getHostName(), 80);
        }
        final HttpRoute route = new HttpRoute(
                host,
                null,
                proxy, false, TunnelType.TUNNELLED, LayerType.PLAIN);

        final ManagedHttpClientConnection conn = this.connFactory.createConnection(null);
        final HttpContext context = new BasicHttpContext();
        ClassicHttpResponse response;

        final ClassicHttpRequest connect = new BasicClassicHttpRequest(Method.CONNECT, host.toHostString());

        final BasicCredentialsProvider credsProvider = new BasicCredentialsProvider();
        credsProvider.setCredentials(new AuthScope(proxy), credentials);

        // Populate the execution context
        context.setAttribute(HttpCoreContext.HTTP_REQUEST, connect);
        context.setAttribute(HttpClientContext.HTTP_ROUTE, route);
        context.setAttribute(HttpClientContext.CREDS_PROVIDER, credsProvider);
        context.setAttribute(HttpClientContext.AUTHSCHEME_REGISTRY, this.authSchemeRegistry);
        context.setAttribute(HttpClientContext.REQUEST_CONFIG, this.requestConfig);

        this.requestExec.preProcess(connect, this.httpProcessor, context);

        for (;;) {
            if (!conn.isOpen()) {
                final Socket socket = new Socket(proxy.getHostName(), proxy.getPort());
                conn.bind(socket);
            }

            this.authenticator.addAuthResponse(proxy, ChallengeType.PROXY, connect, this.proxyAuthExchange, context);

            response = this.requestExec.execute(connect, conn, context);

            final int status = response.getCode();
            if (status < 200) {
                throw new HttpException("Unexpected response to CONNECT request: " + response);
            }
            if (this.authenticator.isChallenged(proxy, ChallengeType.PROXY, response, this.proxyAuthExchange, context)) {
                if (this.authenticator.updateAuthState(proxy, ChallengeType.PROXY, response,
                        this.proxyAuthStrategy, this.proxyAuthExchange, context)) {
                    // Retry request
                    if (this.reuseStrategy.keepAlive(connect, response, context)) {
                        // Consume response content
                        final HttpEntity entity = response.getEntity();
                        EntityUtils.consume(entity);
                    } else {
                        conn.close();
                    }
                    // discard previous auth header
                    connect.removeHeaders(HttpHeaders.PROXY_AUTHORIZATION);
                } else {
                    break;
                }
            } else {
                break;
            }
        }

        final int status = response.getCode();

        if (status > 299) {

            // Buffer response content
            final HttpEntity entity = response.getEntity();
            final String responseMessage = entity != null ? EntityUtils.toString(entity) : null;
            conn.close();
            throw new TunnelRefusedException("CONNECT refused by proxy: " + new StatusLine(response), responseMessage);
        }
        return conn.getSocket();
    }

}
