/*
 * Copyright (c) 2008-2020, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.sql.impl.calcite.schema;

import com.hazelcast.sql.impl.QueryUtils;
import com.hazelcast.sql.impl.calcite.SqlToQueryType;
import com.hazelcast.sql.impl.schema.SqlCatalog;
import com.hazelcast.sql.impl.schema.Table;
import com.hazelcast.sql.impl.schema.TableField;
import com.hazelcast.sql.impl.type.QueryDataType;
import com.hazelcast.sql.impl.type.QueryDataTypeFamily;
import com.hazelcast.org.apache.calcite.rel.type.RelDataType;
import com.hazelcast.org.apache.calcite.rel.type.RelDataTypeFactory;
import com.hazelcast.org.apache.calcite.schema.Schema;
import com.hazelcast.org.apache.calcite.schema.Statistic;
import com.hazelcast.org.apache.calcite.sql.type.SqlTypeName;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Utility methods for schema resolution.
 */
public final class HazelcastSchemaUtils {
    private HazelcastSchemaUtils() {
        // No-op.
    }

    /**
     * Creates the top-level catalog containing the given child schema.
     *
     * @param schema Schema.
     * @return Catalog.
     */
    public static HazelcastSchema createCatalog(Schema schema) {
        return new HazelcastSchema(
            Collections.singletonMap(QueryUtils.CATALOG, schema),
            Collections.emptyMap()
        );
    }

    /**
     * Construct a schema from the given table resolvers.
     * <p>
     * Currently we assume that all tables are resolved upfront by querying a table resolver. It works well for predefined
     * objects such as IMap and ReplicatedMap as well as external tables created by Jet. This approach will not work well
     * should we need a relaxed/dynamic object resolution at some point in future.
     *
     * @return Top-level schema.
     */
    public static HazelcastSchema createRootSchema(SqlCatalog catalog) {
        // Create schemas.
        Map<String, Schema> schemaMap = new HashMap<>();

        for (Map.Entry<String, Map<String, Table>> currentSchemaEntry : catalog.getSchemas().entrySet()) {
            String schemaName = currentSchemaEntry.getKey();

            Map<String, com.hazelcast.org.apache.calcite.schema.Table> schemaTables = new HashMap<>();

            for (Map.Entry<String, Table> tableEntry : currentSchemaEntry.getValue().entrySet()) {
                String tableName = tableEntry.getKey();
                Table table = tableEntry.getValue();

                HazelcastTable convertedTable = new HazelcastTable(
                    table,
                    createTableStatistic(table)
                );

                schemaTables.put(tableName, convertedTable);
            }

            HazelcastSchema currentSchema = new HazelcastSchema(Collections.emptyMap(), schemaTables);

            schemaMap.put(schemaName, currentSchema);
        }

        HazelcastSchema rootSchema = new HazelcastSchema(schemaMap, Collections.emptyMap());

        return createCatalog(rootSchema);
    }

    /**
     * Create Calcite {@link Statistic} object for the given table.
     * <p>
     * As neither IMDG core, nor Jet has dependency on the SQL module, we cannot get that object from the outside. Instead,
     * it should be created in the SQL module through {@code instanceof} checks (or similar).
     *
     * @param table Target table.
     * @return Statistics for the table.
     */
    private static Statistic createTableStatistic(Table table) {
        return new HazelcastTableStatistic(table.getStatistics().getRowCount());
    }

    /**
     * Converts a {@link TableField} to {@link RelDataType}.
     */
    public static RelDataType convert(TableField field, RelDataTypeFactory typeFactory) {
        QueryDataType fieldType = field.getType();
        QueryDataTypeFamily fieldTypeFamily = fieldType.getTypeFamily();

        SqlTypeName sqlTypeName = SqlToQueryType.map(fieldTypeFamily);

        if (sqlTypeName == null) {
            throw new IllegalStateException("Unexpected type family: " + fieldTypeFamily);
        }

        RelDataType relType = typeFactory.createSqlType(sqlTypeName);
        return typeFactory.createTypeWithNullability(relType, true);
    }
}
