/*
 * Copyright (c) 2008-2020, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.client.impl.protocol.task.topic;

import com.hazelcast.client.impl.protocol.ClientMessage;
import com.hazelcast.client.impl.protocol.codec.TopicPublishAllCodec;
import com.hazelcast.client.impl.protocol.task.AbstractPartitionMessageTask;
import com.hazelcast.instance.impl.Node;
import com.hazelcast.internal.cluster.Versions;
import com.hazelcast.internal.nio.Connection;
import com.hazelcast.internal.serialization.Data;
import com.hazelcast.security.permission.ActionConstants;
import com.hazelcast.security.permission.TopicPermission;
import com.hazelcast.spi.impl.operationservice.Operation;
import com.hazelcast.topic.impl.PublishAllOperation;
import com.hazelcast.topic.impl.TopicService;
import com.hazelcast.version.Version;

import java.security.Permission;
import java.util.List;

public class TopicPublishAllMessageTask
        extends AbstractPartitionMessageTask<TopicPublishAllCodec.RequestParameters> {

    public TopicPublishAllMessageTask(ClientMessage clientMessage, Node node, Connection connection) {
        super(clientMessage, node, connection);
    }

    @Override
    protected Operation prepareOperation() {
        // RU_COMPAT_4_0
        Version clusterVersion = nodeEngine.getClusterService().getClusterVersion();
        if (!clusterVersion.isGreaterOrEqual(Versions.V4_1)) {
            throw new UnsupportedOperationException(
                    "Publish all is unavailable at cluster version " + clusterVersion);
        }
        return new PublishAllOperation(parameters.name, items());
    }

    @Override
    protected TopicPublishAllCodec.RequestParameters decodeClientMessage(ClientMessage clientMessage) {
        return TopicPublishAllCodec.decodeRequest(clientMessage);
    }

    @Override
    protected ClientMessage encodeResponse(Object response) {
        return TopicPublishAllCodec.encodeResponse();
    }

    @Override
    public String getServiceName() {
        return TopicService.SERVICE_NAME;
    }

    @Override
    public Permission getRequiredPermission() {
        return new TopicPermission(parameters.name, ActionConstants.ACTION_PUBLISH);
    }

    @Override
    public String getDistributedObjectName() {
        return parameters.name;
    }

    @Override
    public String getMethodName() {
        return "publishAll";
    }

    @Override
    public Object[] getParameters() {
        return new Object[]{parameters.messages};
    }

    private Data[] items() {
        List<Data> messageList = parameters.messages;
        Data[] array = new Data[messageList.size()];
        return messageList.toArray(array);
    }
}
