/*
 * Copyright (c) 2008-2020, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.sql.impl.extract;

import com.hazelcast.internal.serialization.Data;
import com.hazelcast.internal.serialization.InternalSerializationService;
import com.hazelcast.nio.serialization.Portable;
import com.hazelcast.query.impl.getters.Extractors;
import com.hazelcast.sql.impl.type.QueryDataType;

public class GenericQueryTarget implements QueryTarget, GenericTargetAccessor {

    private final InternalSerializationService serializationService;
    private final Extractors extractors;
    private final boolean key;

    private Object rawTarget;
    private Object target;

    public GenericQueryTarget(InternalSerializationService serializationService, Extractors extractors, boolean key) {
        this.serializationService = serializationService;
        this.extractors = extractors;
        this.key = key;
    }

    @Override
    public void setTarget(Object target) {
        this.rawTarget = target;
        this.target = null;
    }

    @Override
    public QueryExtractor createExtractor(String path, QueryDataType type) {
        if (path == null) {
            return new GenericTargetExtractor(key, this, type);
        } else {
            return new GenericFieldExtractor(key, this, type, extractors, path);
        }
    }

    @Override
    public Object getTargetForFieldAccess() {
        if (target == null) {
            // General rule: Portable must be Data, other objects must be deserialized.
            if (rawTarget instanceof Data) {
                Data rawTarget0 = (Data) rawTarget;

                if (rawTarget0.isPortable()) {
                    target = rawTarget;
                } else {
                    // Deserialize non-Portable.
                    target = serializationService.toObject(rawTarget);
                }
            } else {
                if (rawTarget instanceof Portable) {
                    // Serialize Portable to Data.
                    target = serializationService.toData(rawTarget);
                } else {
                    target = rawTarget;
                }
            }
        }

        return target;
    }

    @Override
    public Object getTargetDeserialized() {
        if (!(rawTarget instanceof Data)) {
            // Raw target is already deserialized, use it
            return rawTarget;
        }

        // Try using field target if possible
        if (target != null && !(target instanceof Data)) {
            return target;
        }

        // Raw target is Data, field target is not initialized, deserialize
        Data rawTarget0 = (Data) rawTarget;

        Object result = serializationService.toObject(rawTarget0);

        // Check if the deserialized result could be useful for subsequent field access
        boolean cacheDeserialized = target == null && !rawTarget0.isPortable() && !rawTarget0.isJson();

        if (cacheDeserialized) {
            target = result;
        }

        // Done
        return result;
    }

    public boolean isKey() {
        return key;
    }
}
