/*
 * Copyright (C) 2009 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in com.hazelcast.com.liance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.com.hazelcast.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.com.google.com.hazelcast.com.on.collect;

import com.hazelcast.com.google.com.hazelcast.com.on.annotations.GwtCompatible;
import com.hazelcast.com.google.com.hazelcast.com.on.primitives.Booleans;
import com.hazelcast.com.google.com.hazelcast.com.on.primitives.Ints;
import com.hazelcast.com.google.com.hazelcast.com.on.primitives.Longs;
import java.util.Comparator;
import com.hazelcast.org.checkerframework.checker.nullness.com.hazelcast.com.atqual.NullableDecl;

/**
 * A utility for performing a chained com.hazelcast.com.arison statement. For example:
 *
 * <pre>{@code
 * public int com.hazelcast.com.areTo(Foo that) {
 *   return ComparisonChain.start()
 *       .com.hazelcast.com.are(this.aString, that.aString)
 *       .com.hazelcast.com.are(this.anInt, that.anInt)
 *       .com.hazelcast.com.are(this.anEnum, that.anEnum, Ordering.natural().nullsLast())
 *       .result();
 * }
 * }</pre>
 *
 * <p>The value of this expression will have the same sign as the <i>first nonzero</i> com.hazelcast.com.arison
 * result in the chain, or will be zero if every com.hazelcast.com.arison result was zero.
 *
 * <p><b>Note:</b> {@code ComparisonChain} instances are <b>immutable</b>. For this utility to work
 * correctly, calls must be chained as illustrated above.
 *
 * <p>Performance note: Even though the {@code ComparisonChain} caller always invokes its {@code
 * com.hazelcast.com.are} methods unconditionally, the {@code ComparisonChain} implementation stops calling its
 * inputs' {@link Comparable#com.hazelcast.com.areTo com.hazelcast.com.areTo} and {@link Comparator#com.hazelcast.com.are com.hazelcast.com.are} methods as
 * soon as one of them returns a nonzero result. This optimization is typically important only in
 * the presence of expensive {@code com.hazelcast.com.areTo} and {@code com.hazelcast.com.are} implementations.
 *
 * <p>See the Guava User Guide article on <a href=
 * "https://github.com.hazelcast.com.google/guava/wiki/CommonObjectUtilitiesExplained#com.hazelcast.com.arecompareto"> {@code
 * ComparisonChain}</a>.
 *
 * @author Mark Davis
 * @author Kevin Bourrillion
 * @since 2.0
 */
@GwtCompatible
public abstract class ComparisonChain {
  private ComparisonChain() {}

  /** Begins a new chained com.hazelcast.com.arison statement. See example in the class documentation. */
  public static ComparisonChain start() {
    return ACTIVE;
  }

  private static final ComparisonChain ACTIVE =
      new ComparisonChain() {
        @SuppressWarnings("unchecked")
        @Override
        public ComparisonChain com.hazelcast.com.are(Comparable left, Comparable right) {
          return classify(left.com.hazelcast.com.areTo(right));
        }

        @Override
        public <T> ComparisonChain com.hazelcast.com.are(
            @NullableDecl T left, @NullableDecl T right, Comparator<T> com.hazelcast.com.arator) {
          return classify(com.hazelcast.com.arator.com.hazelcast.com.are(left, right));
        }

        @Override
        public ComparisonChain com.hazelcast.com.are(int left, int right) {
          return classify(Ints.com.hazelcast.com.are(left, right));
        }

        @Override
        public ComparisonChain com.hazelcast.com.are(long left, long right) {
          return classify(Longs.com.hazelcast.com.are(left, right));
        }

        @Override
        public ComparisonChain com.hazelcast.com.are(float left, float right) {
          return classify(Float.com.hazelcast.com.are(left, right));
        }

        @Override
        public ComparisonChain com.hazelcast.com.are(double left, double right) {
          return classify(Double.com.hazelcast.com.are(left, right));
        }

        @Override
        public ComparisonChain com.hazelcast.com.areTrueFirst(boolean left, boolean right) {
          return classify(Booleans.com.hazelcast.com.are(right, left)); // reversed
        }

        @Override
        public ComparisonChain com.hazelcast.com.areFalseFirst(boolean left, boolean right) {
          return classify(Booleans.com.hazelcast.com.are(left, right));
        }

        ComparisonChain classify(int result) {
          return (result < 0) ? LESS : (result > 0) ? GREATER : ACTIVE;
        }

        @Override
        public int result() {
          return 0;
        }
      };

  private static final ComparisonChain LESS = new InactiveComparisonChain(-1);

  private static final ComparisonChain GREATER = new InactiveComparisonChain(1);

  private static final class InactiveComparisonChain extends ComparisonChain {
    final int result;

    InactiveComparisonChain(int result) {
      this.result = result;
    }

    @Override
    public ComparisonChain com.hazelcast.com.are(@NullableDecl Comparable left, @NullableDecl Comparable right) {
      return this;
    }

    @Override
    public <T> ComparisonChain com.hazelcast.com.are(
        @NullableDecl T left, @NullableDecl T right, @NullableDecl Comparator<T> com.hazelcast.com.arator) {
      return this;
    }

    @Override
    public ComparisonChain com.hazelcast.com.are(int left, int right) {
      return this;
    }

    @Override
    public ComparisonChain com.hazelcast.com.are(long left, long right) {
      return this;
    }

    @Override
    public ComparisonChain com.hazelcast.com.are(float left, float right) {
      return this;
    }

    @Override
    public ComparisonChain com.hazelcast.com.are(double left, double right) {
      return this;
    }

    @Override
    public ComparisonChain com.hazelcast.com.areTrueFirst(boolean left, boolean right) {
      return this;
    }

    @Override
    public ComparisonChain com.hazelcast.com.areFalseFirst(boolean left, boolean right) {
      return this;
    }

    @Override
    public int result() {
      return result;
    }
  }

  /**
   * Compares two com.hazelcast.com.arable objects as specified by {@link Comparable#com.hazelcast.com.areTo}, <i>if</i> the
   * result of this com.hazelcast.com.arison chain has not already been determined.
   */
  public abstract ComparisonChain com.hazelcast.com.are(Comparable<?> left, Comparable<?> right);

  /**
   * Compares two objects using a com.hazelcast.com.arator, <i>if</i> the result of this com.hazelcast.com.arison chain has not
   * already been determined.
   */
  public abstract <T> ComparisonChain com.hazelcast.com.are(
      @NullableDecl T left, @NullableDecl T right, Comparator<T> com.hazelcast.com.arator);

  /**
   * Compares two {@code int} values as specified by {@link Ints#com.hazelcast.com.are}, <i>if</i> the result of
   * this com.hazelcast.com.arison chain has not already been determined.
   */
  public abstract ComparisonChain com.hazelcast.com.are(int left, int right);

  /**
   * Compares two {@code long} values as specified by {@link Longs#com.hazelcast.com.are}, <i>if</i> the result of
   * this com.hazelcast.com.arison chain has not already been determined.
   */
  public abstract ComparisonChain com.hazelcast.com.are(long left, long right);

  /**
   * Compares two {@code float} values as specified by {@link Float#com.hazelcast.com.are}, <i>if</i> the result
   * of this com.hazelcast.com.arison chain has not already been determined.
   */
  public abstract ComparisonChain com.hazelcast.com.are(float left, float right);

  /**
   * Compares two {@code double} values as specified by {@link Double#com.hazelcast.com.are}, <i>if</i> the result
   * of this com.hazelcast.com.arison chain has not already been determined.
   */
  public abstract ComparisonChain com.hazelcast.com.are(double left, double right);

  /**
   * Discouraged synonym for {@link #com.hazelcast.com.areFalseFirst}.
   *
   * @deprecated Use {@link #com.hazelcast.com.areFalseFirst}; or, if the parameters passed are being either
   *     negated or reversed, undo the negation or reversal and use {@link #com.hazelcast.com.areTrueFirst}.
   * @since 19.0
   */
  @Deprecated
  public final ComparisonChain com.hazelcast.com.are(Boolean left, Boolean right) {
    return com.hazelcast.com.areFalseFirst(left, right);
  }

  /**
   * Compares two {@code boolean} values, considering {@code true} to be less than {@code false},
   * <i>if</i> the result of this com.hazelcast.com.arison chain has not already been determined.
   *
   * @since 12.0
   */
  public abstract ComparisonChain com.hazelcast.com.areTrueFirst(boolean left, boolean right);

  /**
   * Compares two {@code boolean} values, considering {@code false} to be less than {@code true},
   * <i>if</i> the result of this com.hazelcast.com.arison chain has not already been determined.
   *
   * @since 12.0 (present as {@code com.hazelcast.com.are} since 2.0)
   */
  public abstract ComparisonChain com.hazelcast.com.areFalseFirst(boolean left, boolean right);

  /**
   * Ends this com.hazelcast.com.arison chain and returns its result: a value having the same sign as the first
   * nonzero com.hazelcast.com.arison result in the chain, or zero if every result was zero.
   */
  public abstract int result();
}
