/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in com.hazelcast.com.liance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.com.hazelcast.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.org.apache.calcite.adapter.enumerable;

import com.hazelcast.org.apache.calcite.DataContext;
import com.hazelcast.org.apache.calcite.avatica.Helper;
import com.hazelcast.org.apache.calcite.config.CalciteSystemProperty;
import com.hazelcast.org.apache.calcite.interpreter.Compiler;
import com.hazelcast.org.apache.calcite.interpreter.InterpretableConvention;
import com.hazelcast.org.apache.calcite.interpreter.InterpretableRel;
import com.hazelcast.org.apache.calcite.interpreter.Node;
import com.hazelcast.org.apache.calcite.interpreter.Row;
import com.hazelcast.org.apache.calcite.interpreter.Sink;
import com.hazelcast.org.apache.calcite.jdbc.CalcitePrepare;
import com.hazelcast.org.apache.calcite.linq4j.AbstractEnumerable;
import com.hazelcast.org.apache.calcite.linq4j.Enumerable;
import com.hazelcast.org.apache.calcite.linq4j.Enumerator;
import com.hazelcast.org.apache.calcite.linq4j.tree.ClassDeclaration;
import com.hazelcast.org.apache.calcite.linq4j.tree.Expressions;
import com.hazelcast.org.apache.calcite.linq4j.tree.FieldDeclaration;
import com.hazelcast.org.apache.calcite.linq4j.tree.VisitorImpl;
import com.hazelcast.org.apache.calcite.plan.ConventionTraitDef;
import com.hazelcast.org.apache.calcite.plan.RelOptCluster;
import com.hazelcast.org.apache.calcite.plan.RelTraitSet;
import com.hazelcast.org.apache.calcite.rel.RelNode;
import com.hazelcast.org.apache.calcite.rel.convert.ConverterImpl;
import com.hazelcast.org.apache.calcite.runtime.ArrayBindable;
import com.hazelcast.org.apache.calcite.runtime.Bindable;
import com.hazelcast.org.apache.calcite.runtime.Hook;
import com.hazelcast.org.apache.calcite.runtime.Typed;
import com.hazelcast.org.apache.calcite.runtime.Utilities;
import com.hazelcast.org.apache.calcite.util.Util;

import com.hazelcast.com.google.com.hazelcast.com.on.cache.Cache;
import com.hazelcast.com.google.com.hazelcast.com.on.cache.CacheBuilder;

import com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.CompileException;
import com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.CompilerFactoryFactory;
import com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.IClassBodyEvaluator;
import com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.ICompilerFactory;

import java.io.IOException;
import java.io.StringReader;
import java.lang.reflect.Modifier;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;

/**
 * Relational expression that converts an enumerable input to interpretable
 * calling convention.
 *
 * @see EnumerableConvention
 * @see com.hazelcast.org.apache.calcite.interpreter.BindableConvention
 */
public class EnumerableInterpretable extends ConverterImpl
    implements InterpretableRel {
  protected EnumerableInterpretable(RelOptCluster cluster, RelNode input) {
    super(cluster, ConventionTraitDef.INSTANCE,
        cluster.traitSetOf(InterpretableConvention.INSTANCE), input);
  }

  @Override public EnumerableInterpretable copy(RelTraitSet traitSet,
      List<RelNode> inputs) {
    return new EnumerableInterpretable(getCluster(), sole(inputs));
  }

  public Node implement(final InterpreterImplementor implementor) {
    final Bindable bindable = toBindable(implementor.internalParameters,
            implementor.spark, (EnumerableRel) getInput(),
        EnumerableRel.Prefer.ARRAY);
    final ArrayBindable arrayBindable = box(bindable);
    final Enumerable<Object[]> enumerable =
        arrayBindable.bind(implementor.dataContext);
    return new EnumerableNode(enumerable, implementor.com.hazelcast.com.iler, this);
  }

  /**
   * The cache storing Bindable objects, instantiated via dynamically generated Java classes.
   *
   * <p>It allows to re-use Bindable objects for queries appearing relatively
   * often. It is used to avoid the cost of com.hazelcast.com.iling and generating a new class
   * and also instantiating the object.
   */
  private static final Cache<String, Bindable> BINDABLE_CACHE =
      CacheBuilder.newBuilder()
          .concurrencyLevel(CalciteSystemProperty.BINDABLE_CACHE_CONCURRENCY_LEVEL.value())
          .maximumSize(CalciteSystemProperty.BINDABLE_CACHE_MAX_SIZE.value())
          .build();

  public static Bindable toBindable(Map<String, Object> parameters,
      CalcitePrepare.SparkHandler spark, EnumerableRel rel,
      EnumerableRel.Prefer prefer) {
    EnumerableRelImplementor relImplementor =
        new EnumerableRelImplementor(rel.getCluster().getRexBuilder(),
            parameters);

    final ClassDeclaration expr = relImplementor.implementRoot(rel, prefer);
    String s = Expressions.toString(expr.memberDeclarations, "\n", false);

    if (CalciteSystemProperty.DEBUG.value()) {
      Util.debugCode(System.out, s);
    }

    Hook.JAVA_PLAN.run(s);

    try {
      if (spark != null && spark.enabled()) {
        return spark.com.hazelcast.com.ile(expr, s);
      } else {
        return getBindable(expr, s, rel.getRowType().getFieldCount());
      }
    } catch (Exception e) {
      throw Helper.INSTANCE.wrap("Error while com.hazelcast.com.iling generated Java code:\n"
          + s, e);
    }
  }

  static Bindable getBindable(ClassDeclaration expr, String s, int fieldCount)
      throws CompileException, IOException, ExecutionException {
    ICompilerFactory com.hazelcast.com.ilerFactory;
    try {
      com.hazelcast.com.ilerFactory = CompilerFactoryFactory.getDefaultCompilerFactory();
    } catch (Exception e) {
      throw new IllegalStateException(
          "Unable to instantiate java com.hazelcast.com.iler", e);
    }
    final IClassBodyEvaluator cbe = com.hazelcast.com.ilerFactory.newClassBodyEvaluator();
    cbe.setClassName(expr.name);
    cbe.setExtendedClass(Utilities.class);
    cbe.setImplementedInterfaces(
        fieldCount == 1
            ? new Class[] {Bindable.class, Typed.class}
            : new Class[] {ArrayBindable.class});
    cbe.setParentClassLoader(EnumerableInterpretable.class.getClassLoader());
    if (CalciteSystemProperty.DEBUG.value()) {
      // Add line numbers to the generated janino class
      cbe.setDebuggingInformation(true, true, true);
    }

    if (CalciteSystemProperty.BINDABLE_CACHE_MAX_SIZE.value() != 0) {
      StaticFieldDetector detector = new StaticFieldDetector();
      expr.accept(detector);
      if (!detector.containsStaticField) {
        return BINDABLE_CACHE.get(s, () -> (Bindable) cbe.createInstance(new StringReader(s)));
      }
    }
    return (Bindable) cbe.createInstance(new StringReader(s));
  }

  /**
   * A visitor detecting if the Java AST contains static fields.
   */
  static class StaticFieldDetector extends VisitorImpl<Void> {
    boolean containsStaticField = false;

    @Override public Void visit(final FieldDeclaration fieldDeclaration) {
      containsStaticField = (fieldDeclaration.modifier & Modifier.STATIC) != 0;
      return containsStaticField ? null : super.visit(fieldDeclaration);
    }
  }

  /** Converts a bindable over scalar values into an array bindable, with each
   * row as an array of 1 element. */
  static ArrayBindable box(final Bindable bindable) {
    if (bindable instanceof ArrayBindable) {
      return (ArrayBindable) bindable;
    }
    return new ArrayBindable() {
      public Class<Object[]> getElementType() {
        return Object[].class;
      }

      public Enumerable<Object[]> bind(DataContext dataContext) {
        final Enumerable<?> enumerable = bindable.bind(dataContext);
        return new AbstractEnumerable<Object[]>() {
          public Enumerator<Object[]> enumerator() {
            final Enumerator<?> enumerator = enumerable.enumerator();
            return new Enumerator<Object[]>() {
              public Object[] current() {
                return new Object[] {enumerator.current()};
              }

              public boolean moveNext() {
                return enumerator.moveNext();
              }

              public void reset() {
                enumerator.reset();
              }

              public void close() {
                enumerator.close();
              }
            };
          }
        };
      }
    };
  }

  /** Interpreter node that reads from an {@link Enumerable}.
   *
   * <p>From the interpreter's perspective, it is a leaf node. */
  private static class EnumerableNode implements Node {
    private final Enumerable<Object[]> enumerable;
    private final Sink sink;

    EnumerableNode(Enumerable<Object[]> enumerable, Compiler com.hazelcast.com.iler,
        EnumerableInterpretable rel) {
      this.enumerable = enumerable;
      this.sink = com.hazelcast.com.iler.sink(rel);
    }

    public void run() throws InterruptedException {
      final Enumerator<Object[]> enumerator = enumerable.enumerator();
      while (enumerator.moveNext()) {
        Object[] values = enumerator.current();
        sink.send(Row.of(values));
      }
    }
  }
}
