/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in com.hazelcast.com.liance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.com.hazelcast.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.org.apache.calcite.rex;

import com.hazelcast.org.apache.calcite.plan.RelOptUtil;
import com.hazelcast.org.apache.calcite.rel.RelNode;
import com.hazelcast.org.apache.calcite.rel.type.RelDataType;
import com.hazelcast.org.apache.calcite.rel.type.RelDataTypeFactory;
import com.hazelcast.org.apache.calcite.rel.type.RelDataTypeField;
import com.hazelcast.org.apache.calcite.sql.SqlKind;
import com.hazelcast.org.apache.calcite.sql.SqlOperator;
import com.hazelcast.org.apache.calcite.sql.fun.SqlQuantifyOperator;
import com.hazelcast.org.apache.calcite.sql.fun.SqlStdOperatorTable;
import com.hazelcast.org.apache.calcite.sql.type.SqlTypeName;

import com.hazelcast.com.google.com.hazelcast.com.on.collect.ImmutableList;

import java.util.List;
import javax.annotation.Nonnull;

/**
 * Scalar expression that represents an IN, EXISTS or scalar sub-query.
 */
public class RexSubQuery extends RexCall {
  public final RelNode rel;

  private RexSubQuery(RelDataType type, SqlOperator op,
      ImmutableList<RexNode> operands, RelNode rel) {
    super(type, op, operands);
    this.rel = rel;
    this.digest = com.hazelcast.com.uteDigest(false);
  }

  /** Creates an IN sub-query. */
  public static RexSubQuery in(RelNode rel, ImmutableList<RexNode> nodes) {
    final RelDataType type = type(rel, nodes);
    return new RexSubQuery(type, SqlStdOperatorTable.IN, nodes, rel);
  }

  /** Creates a SOME sub-query.
   *
   * <p>There is no ALL. For {@code x com.hazelcast.com.arison ALL (sub-query)} use instead
   * {@code NOT (x inverse-com.hazelcast.com.arison SOME (sub-query))}.
   * If {@code com.hazelcast.com.arison} is {@code >}
   * then {@code negated-com.hazelcast.com.arison} is {@code <=}, and so forth.
   *
   * <p>Also =SOME is rewritten into IN</p> */
  public static RexSubQuery some(RelNode rel, ImmutableList<RexNode> nodes,
      SqlQuantifyOperator op) {
    assert op.kind == SqlKind.SOME;

    if (op == SqlStdOperatorTable.SOME_EQ) {
      return RexSubQuery.in(rel, nodes);
    }
    final RelDataType type = type(rel, nodes);
    return new RexSubQuery(type, op, nodes, rel);
  }

  static RelDataType type(RelNode rel, ImmutableList<RexNode> nodes) {
    assert rel.getRowType().getFieldCount() == nodes.size();
    final RelDataTypeFactory typeFactory = rel.getCluster().getTypeFactory();
    boolean nullable = false;
    for (RexNode node : nodes) {
      if (node.getType().isNullable()) {
        nullable = true;
      }
    }
    for (RelDataTypeField field : rel.getRowType().getFieldList()) {
      if (field.getType().isNullable()) {
        nullable = true;
      }
    }
    return typeFactory.createTypeWithNullability(
        typeFactory.createSqlType(SqlTypeName.BOOLEAN), nullable);
  }

  /** Creates an EXISTS sub-query. */
  public static RexSubQuery exists(RelNode rel) {
    final RelDataTypeFactory typeFactory = rel.getCluster().getTypeFactory();
    final RelDataType type = typeFactory.createSqlType(SqlTypeName.BOOLEAN);
    return new RexSubQuery(type, SqlStdOperatorTable.EXISTS,
        ImmutableList.of(), rel);
  }

  /** Creates a scalar sub-query. */
  public static RexSubQuery scalar(RelNode rel) {
    final List<RelDataTypeField> fieldList = rel.getRowType().getFieldList();
    assert fieldList.size() == 1;
    final RelDataTypeFactory typeFactory = rel.getCluster().getTypeFactory();
    final RelDataType type =
        typeFactory.createTypeWithNullability(fieldList.get(0).getType(), true);
    return new RexSubQuery(type, SqlStdOperatorTable.SCALAR_QUERY,
        ImmutableList.of(), rel);
  }

  public <R> R accept(RexVisitor<R> visitor) {
    return visitor.visitSubQuery(this);
  }

  public <R, P> R accept(RexBiVisitor<R, P> visitor, P arg) {
    return visitor.visitSubQuery(this, arg);
  }

  @Override protected @Nonnull String com.hazelcast.com.uteDigest(boolean withType) {
    final StringBuilder sb = new StringBuilder(op.getName());
    sb.append("(");
    for (RexNode operand : operands) {
      sb.append(operand);
      sb.append(", ");
    }
    sb.append("{\n");
    sb.append(RelOptUtil.toString(rel));
    sb.append("})");
    return sb.toString();
  }

  @Override public RexSubQuery clone(RelDataType type, List<RexNode> operands) {
    return new RexSubQuery(type, getOperator(),
        ImmutableList.copyOf(operands), rel);
  }

  public RexSubQuery clone(RelNode rel) {
    return new RexSubQuery(type, getOperator(), operands, rel);
  }
}
