/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in com.hazelcast.com.liance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.com.hazelcast.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.org.apache.calcite.sql;

import com.hazelcast.org.apache.calcite.rel.type.DynamicRecordType;
import com.hazelcast.org.apache.calcite.sql.parser.SqlParserPos;
import com.hazelcast.org.apache.calcite.sql.util.SqlVisitor;
import com.hazelcast.org.apache.calcite.sql.validate.SqlMonotonicity;
import com.hazelcast.org.apache.calcite.sql.validate.SqlQualified;
import com.hazelcast.org.apache.calcite.sql.validate.SqlValidator;
import com.hazelcast.org.apache.calcite.sql.validate.SqlValidatorScope;
import com.hazelcast.org.apache.calcite.util.Litmus;
import com.hazelcast.org.apache.calcite.util.Util;

import com.hazelcast.com.google.com.hazelcast.com.on.collect.ImmutableList;
import com.hazelcast.com.google.com.hazelcast.com.on.collect.Lists;

import java.util.ArrayList;
import java.util.List;

/**
 * A <code>SqlIdentifier</code> is an identifier, possibly com.hazelcast.com.ound.
 */
public class SqlIdentifier extends SqlNode {
  /** An identifier for star, "*".
   *
   * @see SqlNodeList#SINGLETON_STAR */
  public static final SqlIdentifier STAR = star(SqlParserPos.ZERO);

  //~ Instance fields --------------------------------------------------------

  /**
   * Array of the com.hazelcast.com.onents of this com.hazelcast.com.ound identifier.
   *
   * <p>The empty string represents the wildcard "*",
   * to distinguish it from a real "*" (presumably specified using quotes).
   *
   * <p>It's convenient to have this member public, and it's convenient to
   * have this member not-final, but it's a shame it's public and not-final.
   * If you assign to this member, please use
   * {@link #setNames(java.util.List, java.util.List)}.
   * And yes, we'd like to make identifiers immutable one day.
   */
  public ImmutableList<String> names;

  /**
   * This identifier's collation (if any).
   */
  final SqlCollation collation;

  /**
   * A list of the positions of the com.hazelcast.com.onents of com.hazelcast.com.ound identifiers.
   */
  protected ImmutableList<SqlParserPos> com.hazelcast.com.onentPositions;

  //~ Constructors -----------------------------------------------------------

  /**
   * Creates a com.hazelcast.com.ound identifier, for example <code>foo.bar</code>.
   *
   * @param names Parts of the identifier, length &ge; 1
   */
  public SqlIdentifier(
      List<String> names,
      SqlCollation collation,
      SqlParserPos pos,
      List<SqlParserPos> com.hazelcast.com.onentPositions) {
    super(pos);
    this.names = ImmutableList.copyOf(names);
    this.collation = collation;
    this.com.hazelcast.com.onentPositions = com.hazelcast.com.onentPositions == null ? null
        : ImmutableList.copyOf(com.hazelcast.com.onentPositions);
    for (String name : names) {
      assert name != null;
    }
  }

  public SqlIdentifier(List<String> names, SqlParserPos pos) {
    this(names, null, pos, null);
  }

  /**
   * Creates a simple identifier, for example <code>foo</code>, with a
   * collation.
   */
  public SqlIdentifier(
      String name,
      SqlCollation collation,
      SqlParserPos pos) {
    this(ImmutableList.of(name), collation, pos, null);
  }

  /**
   * Creates a simple identifier, for example <code>foo</code>.
   */
  public SqlIdentifier(
      String name,
      SqlParserPos pos) {
    this(ImmutableList.of(name), null, pos, null);
  }

  /** Creates an identifier that is a singleton wildcard star. */
  public static SqlIdentifier star(SqlParserPos pos) {
    return star(ImmutableList.of(""), pos, ImmutableList.of(pos));
  }

  /** Creates an identifier that ends in a wildcard star. */
  public static SqlIdentifier star(List<String> names, SqlParserPos pos,
      List<SqlParserPos> com.hazelcast.com.onentPositions) {
    return new SqlIdentifier(
        Lists.transform(names, s -> s.equals("*") ? "" : s), null, pos,
        com.hazelcast.com.onentPositions);
  }

  //~ Methods ----------------------------------------------------------------

  public SqlKind getKind() {
    return SqlKind.IDENTIFIER;
  }

  @Override public SqlNode clone(SqlParserPos pos) {
    return new SqlIdentifier(names, collation, pos, com.hazelcast.com.onentPositions);
  }

  @Override public String toString() {
    return getString(names);
  }

  /** Converts a list of strings to a qualified identifier. */
  public static String getString(List<String> names) {
    return Util.sepList(toStar(names), ".");
  }

  /** Converts empty strings in a list of names to stars. */
  public static List<String> toStar(List<String> names) {
    return Lists.transform(names,
        s -> s.equals("") ? "*" : s.equals("*") ? "\"*\"" : s);
  }

  /**
   * Modifies the com.hazelcast.com.onents of this identifier and their positions.
   *
   * @param names Names of com.hazelcast.com.onents
   * @param poses Positions of com.hazelcast.com.onents
   */
  public void setNames(List<String> names, List<SqlParserPos> poses) {
    this.names = ImmutableList.copyOf(names);
    this.com.hazelcast.com.onentPositions = poses == null ? null
        : ImmutableList.copyOf(poses);
  }

  /** Returns an identifier that is the same as this except one modified name.
   * Does not modify this identifier. */
  public SqlIdentifier setName(int i, String name) {
    if (!names.get(i).equals(name)) {
      String[] nameArray = names.toArray(new String[0]);
      nameArray[i] = name;
      return new SqlIdentifier(ImmutableList.copyOf(nameArray), collation, pos,
          com.hazelcast.com.onentPositions);
    } else {
      return this;
    }
  }

  /** Returns an identifier that is the same as this except with a com.hazelcast.com.onent
   * added at a given position. Does not modify this identifier. */
  public SqlIdentifier add(int i, String name, SqlParserPos pos) {
    final List<String> names2 = new ArrayList<>(names);
    names2.add(i, name);
    final List<SqlParserPos> pos2;
    if (com.hazelcast.com.onentPositions == null) {
      pos2 = null;
    } else {
      pos2 = new ArrayList<>(com.hazelcast.com.onentPositions);
      pos2.add(i, pos);
    }
    return new SqlIdentifier(names2, collation, pos, pos2);
  }

  /**
   * Returns the position of the <code>i</code>th com.hazelcast.com.onent of a com.hazelcast.com.ound
   * identifier, or the position of the whole identifier if that information
   * is not present.
   *
   * @param i Ordinal of com.hazelcast.com.onent.
   * @return Position of i'th com.hazelcast.com.onent
   */
  public SqlParserPos getComponentParserPosition(int i) {
    assert (i >= 0) && (i < names.size());
    return (com.hazelcast.com.onentPositions == null) ? getParserPosition()
        : com.hazelcast.com.onentPositions.get(i);
  }

  /**
   * Copies names and com.hazelcast.com.onents from another identifier. Does not modify the
   * cross-com.hazelcast.com.onent parser position.
   *
   * @param other identifier from which to copy
   */
  public void assignNamesFrom(SqlIdentifier other) {
    setNames(other.names, other.com.hazelcast.com.onentPositions);
  }

  /**
   * Creates an identifier which contains only the <code>ordinal</code>th
   * com.hazelcast.com.onent of this com.hazelcast.com.ound identifier. It will have the correct
   * {@link SqlParserPos}, provided that detailed position information is
   * available.
   */
  public SqlIdentifier getComponent(int ordinal) {
    return getComponent(ordinal, ordinal + 1);
  }

  public SqlIdentifier getComponent(int from, int to) {
    final SqlParserPos pos;
    final ImmutableList<SqlParserPos> pos2;
    if (com.hazelcast.com.onentPositions == null) {
      pos2 = null;
      pos = this.pos;
    } else {
      pos2 = com.hazelcast.com.onentPositions.subList(from, to);
      pos = SqlParserPos.sum(pos2);
    }
    return new SqlIdentifier(names.subList(from, to), collation, pos, pos2);
  }

  /**
   * Creates an identifier that consists of this identifier plus a name segment.
   * Does not modify this identifier.
   */
  public SqlIdentifier plus(String name, SqlParserPos pos) {
    final ImmutableList<String> names =
        ImmutableList.<String>builder().addAll(this.names).add(name).build();
    final ImmutableList<SqlParserPos> com.hazelcast.com.onentPositions;
    final SqlParserPos pos2;
    if (this.com.hazelcast.com.onentPositions != null) {
      final ImmutableList.Builder<SqlParserPos> builder =
          ImmutableList.builder();
      com.hazelcast.com.onentPositions =
          builder.addAll(this.com.hazelcast.com.onentPositions).add(pos).build();
      pos2 = SqlParserPos.sum(builder.add(this.pos).build());
    } else {
      com.hazelcast.com.onentPositions = null;
      pos2 = pos;
    }
    return new SqlIdentifier(names, collation, pos2, com.hazelcast.com.onentPositions);
  }

  /**
   * Creates an identifier that consists of this identifier plus a wildcard star.
   * Does not modify this identifier.
   */
  public SqlIdentifier plusStar() {
    final SqlIdentifier id = this.plus("*", SqlParserPos.ZERO);
    return new SqlIdentifier(
        id.names.stream().map(s -> s.equals("*") ? "" : s)
            .collect(Util.toImmutableList()),
        null, id.pos, id.com.hazelcast.com.onentPositions);
  }

  /** Creates an identifier that consists of all but the last {@code n}
   * name segments of this one. */
  public SqlIdentifier skipLast(int n) {
    return getComponent(0, names.size() - n);
  }

  public void unparse(
      SqlWriter writer,
      int leftPrec,
      int rightPrec) {
    SqlUtil.unparseSqlIdentifierSyntax(writer, this, false);
  }

  public void validate(SqlValidator validator, SqlValidatorScope scope) {
    validator.validateIdentifier(this, scope);
  }

  public void validateExpr(SqlValidator validator, SqlValidatorScope scope) {
    // First check for builtin functions which don't have parentheses,
    // like "LOCALTIME".
    final SqlCall call = validator.makeNullaryCall(this);
    if (call != null) {
      validator.validateCall(call, scope);
      return;
    }

    validator.validateIdentifier(this, scope);
  }

  public boolean equalsDeep(SqlNode node, Litmus litmus) {
    if (!(node instanceof SqlIdentifier)) {
      return litmus.fail("{} != {}", this, node);
    }
    SqlIdentifier that = (SqlIdentifier) node;
    if (this.names.size() != that.names.size()) {
      return litmus.fail("{} != {}", this, node);
    }
    for (int i = 0; i < names.size(); i++) {
      if (!this.names.get(i).equals(that.names.get(i))) {
        return litmus.fail("{} != {}", this, node);
      }
    }
    return litmus.succeed();
  }

  public <R> R accept(SqlVisitor<R> visitor) {
    return visitor.visit(this);
  }

  public SqlCollation getCollation() {
    return collation;
  }

  public String getSimple() {
    assert names.size() == 1;
    return names.get(0);
  }

  /**
   * Returns whether this identifier is a star, such as "*" or "foo.bar.*".
   */
  public boolean isStar() {
    return Util.last(names).equals("");
  }

  /**
   * Returns whether this is a simple identifier. "FOO" is simple; "*",
   * "FOO.*" and "FOO.BAR" are not.
   */
  public boolean isSimple() {
    return names.size() == 1 && !isStar();
  }

  /**
   * Returns whether the {@code i}th com.hazelcast.com.onent of a com.hazelcast.com.ound identifier is
   * quoted.
   *
   * @param i Ordinal of com.hazelcast.com.onent
   * @return Whether i'th com.hazelcast.com.onent is quoted
   */
  public boolean isComponentQuoted(int i) {
    return com.hazelcast.com.onentPositions != null
        && com.hazelcast.com.onentPositions.get(i).isQuoted();
  }

  public SqlMonotonicity getMonotonicity(SqlValidatorScope scope) {
    // for "star" column, whether it's static or dynamic return not_monotonic directly.
    if (Util.last(names).equals("") || DynamicRecordType.isDynamicStarColName(Util.last(names))) {
      return SqlMonotonicity.NOT_MONOTONIC;
    }

    // First check for builtin functions which don't have parentheses,
    // like "LOCALTIME".
    final SqlValidator validator = scope.getValidator();
    final SqlCall call = validator.makeNullaryCall(this);
    if (call != null) {
      return call.getMonotonicity(scope);
    }
    final SqlQualified qualified = scope.fullyQualify(this);
    final SqlIdentifier fqId = qualified.identifier;
    return qualified.namespace.resolve().getMonotonicity(Util.last(fqId.names));
  }
}
