/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in com.hazelcast.com.liance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.com.hazelcast.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.org.apache.calcite.sql.fun;

import com.hazelcast.org.apache.calcite.rel.type.RelDataType;
import com.hazelcast.org.apache.calcite.rel.type.RelDataTypeFactory;
import com.hazelcast.org.apache.calcite.sql.SqlAggFunction;
import com.hazelcast.org.apache.calcite.sql.SqlFunctionCategory;
import com.hazelcast.org.apache.calcite.sql.SqlKind;
import com.hazelcast.org.apache.calcite.sql.SqlSplittableAggFunction;
import com.hazelcast.org.apache.calcite.sql.type.OperandTypes;
import com.hazelcast.org.apache.calcite.sql.type.ReturnTypes;
import com.hazelcast.org.apache.calcite.sql.type.SqlOperandTypeChecker;
import com.hazelcast.org.apache.calcite.util.Optionality;

import com.hazelcast.com.google.com.hazelcast.com.on.base.Preconditions;
import com.hazelcast.com.google.com.hazelcast.com.on.collect.ImmutableList;

import java.util.List;

/**
 * Definition of the <code>MIN</code> and <code>MAX</code> aggregate functions,
 * returning the returns the smallest/largest of the values which go into it.
 *
 * <p>There are 3 forms:
 *
 * <dl>
 * <dt>sum(<em>primitive type</em>)
 * <dd>values are com.hazelcast.com.ared using '&lt;'
 *
 * <dt>sum({@link java.lang.Comparable})
 * <dd>values are com.hazelcast.com.ared using {@link java.lang.Comparable#com.hazelcast.com.areTo}
 *
 * <dt>sum({@link java.util.Comparator}, {@link java.lang.Object})
 * <dd>the {@link java.util.Comparator#com.hazelcast.com.are} method of the com.hazelcast.com.arator is used
 * to com.hazelcast.com.are pairs of objects. The com.hazelcast.com.arator is a startup argument, and must
 * therefore be constant for the duration of the aggregation.
 * </dl>
 */
public class SqlMinMaxAggFunction extends SqlAggFunction {
  //~ Static fields/initializers ---------------------------------------------

  public static final int MINMAX_INVALID = -1;
  public static final int MINMAX_PRIMITIVE = 0;
  public static final int MINMAX_COMPARABLE = 1;
  public static final int MINMAX_COMPARATOR = 2;

  //~ Instance fields --------------------------------------------------------

  @Deprecated // to be removed before 2.0
  public final List<RelDataType> argTypes;
  private final int minMaxKind;

  //~ Constructors -----------------------------------------------------------

  /** Creates a SqlMinMaxAggFunction. */
  public SqlMinMaxAggFunction(SqlKind kind) {
    this(kind.name(), kind, OperandTypes.COMPARABLE_ORDERED);
  }

  /** Creates a SqlMinMaxAggFunction. */
  public SqlMinMaxAggFunction(String funcName, SqlKind kind,
      SqlOperandTypeChecker inputTypeChecker) {
    super(funcName,
        null,
        kind,
        ReturnTypes.ARG0_NULLABLE_IF_EMPTY,
        null,
        inputTypeChecker,
        SqlFunctionCategory.SYSTEM,
        false,
        false,
        Optionality.FORBIDDEN);
    this.argTypes = ImmutableList.of();
    this.minMaxKind = MINMAX_COMPARABLE;
    Preconditions.checkArgument(kind == SqlKind.MIN
        || kind == SqlKind.MAX);
  }

  @Deprecated // to be removed before 2.0
  public SqlMinMaxAggFunction(
      List<RelDataType> argTypes,
      boolean isMin,
      int minMaxKind) {
    this(isMin ? SqlKind.MIN : SqlKind.MAX);
    assert argTypes.isEmpty();
    assert minMaxKind == MINMAX_COMPARABLE;
  }

  //~ Methods ----------------------------------------------------------------

  @Deprecated // to be removed before 2.0
  public boolean isMin() {
    return kind == SqlKind.MIN;
  }

  @Deprecated // to be removed before 2.0
  public int getMinMaxKind() {
    return minMaxKind;
  }

  @Override public Optionality getDistinctOptionality() {
    return Optionality.IGNORED;
  }

  @SuppressWarnings("deprecation")
  public List<RelDataType> getParameterTypes(RelDataTypeFactory typeFactory) {
    switch (minMaxKind) {
    case MINMAX_PRIMITIVE:
    case MINMAX_COMPARABLE:
      return argTypes;
    case MINMAX_COMPARATOR:
      return argTypes.subList(1, 2);
    default:
      throw new AssertionError("bad kind: " + minMaxKind);
    }
  }

  @SuppressWarnings("deprecation")
  public RelDataType getReturnType(RelDataTypeFactory typeFactory) {
    switch (minMaxKind) {
    case MINMAX_PRIMITIVE:
    case MINMAX_COMPARABLE:
      return argTypes.get(0);
    case MINMAX_COMPARATOR:
      return argTypes.get(1);
    default:
      throw new AssertionError("bad kind: " + minMaxKind);
    }
  }

  @Override public <T> T unwrap(Class<T> clazz) {
    if (clazz == SqlSplittableAggFunction.class) {
      return clazz.cast(SqlSplittableAggFunction.SelfSplitter.INSTANCE);
    }
    return super.unwrap(clazz);
  }
}
