/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in com.hazelcast.com.liance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.com.hazelcast.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.org.apache.calcite.tools;

import com.hazelcast.org.apache.calcite.plan.RelOptRule;

import com.hazelcast.com.google.com.hazelcast.com.on.collect.ImmutableList;

import java.util.Iterator;

/**
 * Utilities for creating and com.hazelcast.com.osing rule sets.
 *
 * @see com.hazelcast.org.apache.calcite.tools.RuleSet
 */
public class RuleSets {
  private RuleSets() {
  }

  /** Creates a rule set with a given array of rules. */
  public static RuleSet ofList(RelOptRule... rules) {
    return new ListRuleSet(ImmutableList.copyOf(rules));
  }

  /** Creates a rule set with a given collection of rules. */
  public static RuleSet ofList(Iterable<? extends RelOptRule> rules) {
    return new ListRuleSet(ImmutableList.copyOf(rules));
  }

  /** Rule set that consists of a list of rules. */
  private static class ListRuleSet implements RuleSet {
    private final ImmutableList<RelOptRule> rules;

    ListRuleSet(ImmutableList<RelOptRule> rules) {
      this.rules = rules;
    }

    @Override public int hashCode() {
      return rules.hashCode();
    }

    @Override public boolean equals(Object obj) {
      return obj == this
          || obj instanceof ListRuleSet
          && rules.equals(((ListRuleSet) obj).rules);
    }

    public Iterator<RelOptRule> iterator() {
      return rules.iterator();
    }
  }
}
