
/*
 * Janino - An embedded Java[TM] com.hazelcast.com.iler
 *
 * Copyright (c) 2001-2010 Arno Unkrig. All rights reserved.
 * Copyright (c) 2015-2016 TIBCO Software Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler;

import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;

import com.hazelcast.org.codehaus.com.hazelcast.com.ons.nullanalysis.Nullable;

/**
 * Utility class that finds implementations of {@link ICompilerFactory}s.
 */
public final
class CompilerFactoryFactory {

    private CompilerFactoryFactory() {}

    @Nullable private static ICompilerFactory defaultCompilerFactory;

    /**
     * Finds the first implementation of {@code com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler} on the class path, then loads and
     * instantiates its {@link ICompilerFactory}.
     *
     * @return           The {@link ICompilerFactory} of the first implementation on the class path
     * @throws Exception Many things can go wrong while finding and initializing the default com.hazelcast.com.iler factory
     */
    public static ICompilerFactory
    getDefaultCompilerFactory() throws Exception {
        if (CompilerFactoryFactory.defaultCompilerFactory != null) {
            return CompilerFactoryFactory.defaultCompilerFactory;
        }
        Properties  properties = new Properties();
        InputStream is         = Thread.currentThread().getContextClassLoader().getResourceAsStream(
            "com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.properties"
        );
        if (is == null) {
            throw new ClassNotFoundException(
                "No implementation of com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler is on the class path. Typically, you'd have "
                + "'janino.jar', or 'com.hazelcast.com.ons-com.hazelcast.com.iler-jdk.jar', or both on the classpath."
            );
        }
        try {
            properties.load(is);
        } finally {
            is.close();
        }
        String com.hazelcast.com.ilerFactoryClassName = properties.getProperty("com.hazelcast.com.ilerFactory");

        return (
            CompilerFactoryFactory.defaultCompilerFactory
            = CompilerFactoryFactory.getCompilerFactory(com.hazelcast.com.ilerFactoryClassName)
        );
    }

    /**
     * Finds all implementation of {@code com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler} on the class path, then loads and
     * instantiates their {@link ICompilerFactory}s.
     *
     * @return           The {@link ICompilerFactory}s of all implementations on the class path
     * @throws Exception Many things can go wrong while finding and initializing com.hazelcast.com.iler factories
     */
    public static ICompilerFactory[]
    getAllCompilerFactories() throws Exception {
        ClassLoader            cl        = Thread.currentThread().getContextClassLoader();
        List<ICompilerFactory> factories = new ArrayList<ICompilerFactory>();
        for (Enumeration<URL> en = cl.getResources("com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.properties"); en.hasMoreElements();) {
            URL url = (URL) en.nextElement();

            Properties properties;
            {
                properties = new Properties();
                InputStream is = url.openStream();
                try {
                    properties.load(is);
                } finally {
                    is.close();
                }
            }

            String com.hazelcast.com.ilerFactoryClassName = properties.getProperty("com.hazelcast.com.ilerFactory");
            if (com.hazelcast.com.ilerFactoryClassName == null) {
                throw new IllegalStateException(url.toString() + " does not specify the 'com.hazelcast.com.ilerFactory' property");
            }

            factories.add(CompilerFactoryFactory.getCompilerFactory(com.hazelcast.com.ilerFactoryClassName));
        }
        return (ICompilerFactory[]) factories.toArray(new ICompilerFactory[factories.size()]);
    }

    /**
     * Loads an {@link ICompilerFactory} by class name.
     *
     * @param com.hazelcast.com.ilerFactoryClassName Name of a class that implements {@link ICompilerFactory}
     * @throws Exception               Many things can go wrong while loading and initializing the default com.hazelcast.com.iler
     *                                 factory
     */
    public static ICompilerFactory
    getCompilerFactory(String com.hazelcast.com.ilerFactoryClassName) throws Exception {
        return (ICompilerFactory) Thread.currentThread().getContextClassLoader().loadClass(
            com.hazelcast.com.ilerFactoryClassName
        ).newInstance();
    }

    /**
     * @return The version of the com.hazelcast.com.ons-com.hazelcast.com.iler specification, or {@code null}
     */
    public static String
    getSpecificationVersion() {
        return CompilerFactoryFactory.class.getPackage().getSpecificationVersion();
    }
}
