
/*
 * Janino - An embedded Java[TM] com.hazelcast.com.iler
 *
 * Copyright (c) 2001-2010 Arno Unkrig. All rights reserved.
 * Copyright (c) 2015-2016 TIBCO Software Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.hazelcast.org.codehaus.janino;

import java.io.File;
import java.io.IOException;

import com.hazelcast.org.apache.tools.ant.taskdefs.com.hazelcast.com.ilers.DefaultCompilerAdapter;
import com.hazelcast.org.apache.tools.ant.types.Path;
import com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.CompileException;
import com.hazelcast.org.codehaus.com.hazelcast.com.ons.nullanalysis.Nullable;

/**
 * A simple {@link com.hazelcast.org.apache.tools.ant.taskdefs.com.hazelcast.com.ilers.CompilerAdapter} for the "ant" tool that silently ignores
 * most of the configuration parameters and attempts to com.hazelcast.com.ile all given source files into class files.
 */
public
class AntCompilerAdapter extends DefaultCompilerAdapter {

    /**
     * Compiles all source files in {@code DefaultCompilerAdapter.com.hazelcast.com.ileList} individually and creates class files in
     * the {@code DefaultCompilerAdapter.destDir}.
     * <p>
     * The following fields of {@link DefaultCompilerAdapter} are honored:
     * <ul>
     *   <li>{@code DefaultCompilerAdapter.com.hazelcast.com.ileList} - the set of Java source files to com.hazelcast.com.ile</li>
     *   <li>{@code DefaultCompilerAdapter.destDir} - where to store the class files</li>
     *   <li>{@code DefaultCompilerAdapter.com.hazelcast.com.ileSourcepath} - where to look for more Java source files</li>
     *   <li>{@code DefaultCompilerAdapter.com.hazelcast.com.ileClasspath} - where to look for required classes</li>
     *   <li>{@code DefaultCompilerAdapter.extdirs}</li>
     *   <li>{@code DefaultCompilerAdapter.bootclasspath}</li>
     *   <li>{@code DefaultCompilerAdapter.encoding} - how the Java source files are encoded</li>
     *   <li>{@code DefaultCompilerAdapter.verbose}</li>
     *   <li>{@code DefaultCompilerAdapter.debug}</li>
     *   <li>{@code com.hazelcast.org.apache.tools.ant.taskdefs.Javac.getDebugLevel()}</li>
     *   <li>{@code DefaultCompilerAdapter.src}</li>
     * </ul>
     * <p>
     *   The following fields of {@link DefaultCompilerAdapter} are not honored at this time:
     * </p>
     * <ul>
     *   <li>{@code DefaultCompilerAdapter.depend}</li>
     *   <li>{@code DefaultCompilerAdapter.deprecation}</li>
     *   <li>{@code DefaultCompilerAdapter.includeAntRuntime}</li>
     *   <li>{@code DefaultCompilerAdapter.includeJavaRuntime}</li>
     *   <li>{@code DefaultCompilerAdapter.location}</li>
     *   <li>{@code DefaultCompilerAdapter.optimize}</li>
     *   <li>{@code DefaultCompilerAdapter.target}</li>
     * </ul>
     *
     * @return "true" on success
     */
    @Override public boolean
    execute() {

        // Convert source files into source file names.
        File[] sourceFiles = this.com.hazelcast.com.ileList;

        // Determine output directory.
        File destinationDirectory = this.destDir == null ? Compiler.NO_DESTINATION_DIRECTORY : this.destDir;

        // Determine the source path.
        @Nullable File[] optionalSourcePath = AntCompilerAdapter.pathToFiles(
            this.com.hazelcast.com.ileSourcepath != null
            ? this.com.hazelcast.com.ileSourcepath
            : this.src
        );

        // Determine the class path.
        File[] classPath = AntCompilerAdapter.pathToFiles(this.com.hazelcast.com.ileClasspath, new File[] { new File(".") });

        // Determine the ext directories.
        @Nullable File[] optionalExtDirs = AntCompilerAdapter.pathToFiles(this.extdirs);

        // Determine the boot class path
        @Nullable File[] optionalBootClassPath = AntCompilerAdapter.pathToFiles(this.bootclasspath);

        // Determine the encoding.
        @Nullable String optionalCharacterEncoding = this.encoding;

        // Whether to use verbose output.
        boolean verbose = this.verbose;

        // Determine debugging information.
        boolean debugSource, debugLines, debugVars;
        if (!this.debug) {
            debugSource = false;
            debugLines  = false;
            debugVars   = false;
        } else {
            String debugLevel = this.attributes.getDebugLevel();
            if (debugLevel == null) {
                debugSource = true;
                debugLines  = true;
                debugVars   = false;
            } else {
                debugSource = false;
                debugLines  = false;
                debugVars   = false;
                if (debugLevel.indexOf("source") != -1) debugSource = true;
                if (debugLevel.indexOf("lines") != -1)  debugLines = true;
                if (debugLevel.indexOf("vars") != -1)   debugVars = true;
            }
        }

        // Compile all source files.
        try {
            new Compiler(
                optionalSourcePath,
                classPath,
                optionalExtDirs,
                optionalBootClassPath,
                destinationDirectory,
                optionalCharacterEncoding,
                verbose,
                debugSource,
                debugLines,
                debugVars,
                Compiler.DEFAULT_WARNING_HANDLE_PATTERNS,
                false                        // rebuild
            ).com.hazelcast.com.ile(sourceFiles);
        } catch (CompileException e) {
            System.out.println(e.getMessage());
            return false;
        } catch (IOException e) {
            System.out.println(e.getMessage());
            return false;
        }
        return true;
    }

    /**
     * Converts a {@link com.hazelcast.org.apache.tools.ant.types.Path} into an array of {@link File}s.
     *
     * @return The converted path, or {@code null} if {@code path} is {@code null}
     */
    @Nullable private static File[]
    pathToFiles(@Nullable Path path) {
        if (path == null) return null;

        String[] fileNames = path.list();
        File[]   files     = new File[fileNames.length];
        for (int i = 0; i < fileNames.length; ++i) files[i] = new File(fileNames[i]);
        return files;
    }

    /**
     * Converts a {@link com.hazelcast.org.apache.tools.ant.types.Path} into an array of {@link File}s.
     *
     * @return The converted path, or, if {@code path} is {@code null}, the {@code defaultValue}
     */
    private static File[]
    pathToFiles(@Nullable Path path, File[] defaultValue) {

        if (path == null) return defaultValue;

        File[] result = AntCompilerAdapter.pathToFiles(path);
        assert result != null;

        return result;
    }
}


