
/*
 * Janino - An embedded Java[TM] com.hazelcast.com.iler
 *
 * Copyright (c) 2001-2013 Arno Unkrig. All rights reserved.
 * Copyright (c) 2015-2016 TIBCO Software Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.hazelcast.org.codehaus.janino;

import java.io.ByteArrayInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.CompileException;
import com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.ErrorHandler;
import com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.Location;
import com.hazelcast.org.codehaus.com.hazelcast.com.ons.com.hazelcast.com.iler.WarningHandler;
import com.hazelcast.org.codehaus.com.hazelcast.com.ons.nullanalysis.Nullable;
import com.hazelcast.org.codehaus.janino.CodeContext.Inserter;
import com.hazelcast.org.codehaus.janino.CodeContext.Offset;
import com.hazelcast.org.codehaus.janino.IClass.IAnnotation;
import com.hazelcast.org.codehaus.janino.IClass.IConstructor;
import com.hazelcast.org.codehaus.janino.IClass.IField;
import com.hazelcast.org.codehaus.janino.IClass.IInvocable;
import com.hazelcast.org.codehaus.janino.IClass.IMethod;
import com.hazelcast.org.codehaus.janino.Java.AbstractClassDeclaration;
import com.hazelcast.org.codehaus.janino.Java.AbstractPackageMemberClassDeclaration;
import com.hazelcast.org.codehaus.janino.Java.AbstractTypeDeclaration;
import com.hazelcast.org.codehaus.janino.Java.AlternateConstructorInvocation;
import com.hazelcast.org.codehaus.janino.Java.AmbiguousName;
import com.hazelcast.org.codehaus.janino.Java.Annotation;
import com.hazelcast.org.codehaus.janino.Java.AnonymousClassDeclaration;
import com.hazelcast.org.codehaus.janino.Java.ArrayAccessExpression;
import com.hazelcast.org.codehaus.janino.Java.ArrayInitializer;
import com.hazelcast.org.codehaus.janino.Java.ArrayInitializerOrRvalue;
import com.hazelcast.org.codehaus.janino.Java.ArrayLength;
import com.hazelcast.org.codehaus.janino.Java.ArrayType;
import com.hazelcast.org.codehaus.janino.Java.AssertStatement;
import com.hazelcast.org.codehaus.janino.Java.Assignment;
import com.hazelcast.org.codehaus.janino.Java.Atom;
import com.hazelcast.org.codehaus.janino.Java.BinaryOperation;
import com.hazelcast.org.codehaus.janino.Java.Block;
import com.hazelcast.org.codehaus.janino.Java.BlockStatement;
import com.hazelcast.org.codehaus.janino.Java.BooleanLiteral;
import com.hazelcast.org.codehaus.janino.Java.BooleanRvalue;
import com.hazelcast.org.codehaus.janino.Java.BreakStatement;
import com.hazelcast.org.codehaus.janino.Java.BreakableStatement;
import com.hazelcast.org.codehaus.janino.Java.Cast;
import com.hazelcast.org.codehaus.janino.Java.CatchClause;
import com.hazelcast.org.codehaus.janino.Java.CharacterLiteral;
import com.hazelcast.org.codehaus.janino.Java.ClassLiteral;
import com.hazelcast.org.codehaus.janino.Java.CompilationUnit;
import com.hazelcast.org.codehaus.janino.Java.CompilationUnit.ImportDeclaration;
import com.hazelcast.org.codehaus.janino.Java.CompilationUnit.SingleStaticImportDeclaration;
import com.hazelcast.org.codehaus.janino.Java.CompilationUnit.SingleTypeImportDeclaration;
import com.hazelcast.org.codehaus.janino.Java.CompilationUnit.StaticImportOnDemandDeclaration;
import com.hazelcast.org.codehaus.janino.Java.CompilationUnit.TypeImportOnDemandDeclaration;
import com.hazelcast.org.codehaus.janino.Java.ConditionalExpression;
import com.hazelcast.org.codehaus.janino.Java.ConstructorDeclarator;
import com.hazelcast.org.codehaus.janino.Java.ConstructorInvocation;
import com.hazelcast.org.codehaus.janino.Java.ContinuableStatement;
import com.hazelcast.org.codehaus.janino.Java.ContinueStatement;
import com.hazelcast.org.codehaus.janino.Java.Crement;
import com.hazelcast.org.codehaus.janino.Java.DoStatement;
import com.hazelcast.org.codehaus.janino.Java.DocCommentable;
import com.hazelcast.org.codehaus.janino.Java.ElementValue;
import com.hazelcast.org.codehaus.janino.Java.ElementValueArrayInitializer;
import com.hazelcast.org.codehaus.janino.Java.ElementValuePair;
import com.hazelcast.org.codehaus.janino.Java.EmptyStatement;
import com.hazelcast.org.codehaus.janino.Java.EnclosingScopeOfTypeDeclaration;
import com.hazelcast.org.codehaus.janino.Java.EnumConstant;
import com.hazelcast.org.codehaus.janino.Java.EnumDeclaration;
import com.hazelcast.org.codehaus.janino.Java.ExpressionStatement;
import com.hazelcast.org.codehaus.janino.Java.FieldAccess;
import com.hazelcast.org.codehaus.janino.Java.FieldAccessExpression;
import com.hazelcast.org.codehaus.janino.Java.FieldDeclaration;
import com.hazelcast.org.codehaus.janino.Java.FloatingPointLiteral;
import com.hazelcast.org.codehaus.janino.Java.ForEachStatement;
import com.hazelcast.org.codehaus.janino.Java.ForStatement;
import com.hazelcast.org.codehaus.janino.Java.FunctionDeclarator;
import com.hazelcast.org.codehaus.janino.Java.FunctionDeclarator.FormalParameter;
import com.hazelcast.org.codehaus.janino.Java.FunctionDeclarator.FormalParameters;
import com.hazelcast.org.codehaus.janino.Java.IfStatement;
import com.hazelcast.org.codehaus.janino.Java.Initializer;
import com.hazelcast.org.codehaus.janino.Java.InnerClassDeclaration;
import com.hazelcast.org.codehaus.janino.Java.Instanceof;
import com.hazelcast.org.codehaus.janino.Java.IntegerLiteral;
import com.hazelcast.org.codehaus.janino.Java.InterfaceDeclaration;
import com.hazelcast.org.codehaus.janino.Java.Invocation;
import com.hazelcast.org.codehaus.janino.Java.LabeledStatement;
import com.hazelcast.org.codehaus.janino.Java.Literal;
import com.hazelcast.org.codehaus.janino.Java.LocalClassDeclaration;
import com.hazelcast.org.codehaus.janino.Java.LocalClassDeclarationStatement;
import com.hazelcast.org.codehaus.janino.Java.LocalVariable;
import com.hazelcast.org.codehaus.janino.Java.LocalVariableAccess;
import com.hazelcast.org.codehaus.janino.Java.LocalVariableDeclarationStatement;
import com.hazelcast.org.codehaus.janino.Java.LocalVariableSlot;
import com.hazelcast.org.codehaus.janino.Java.Locatable;
import com.hazelcast.org.codehaus.janino.Java.Located;
import com.hazelcast.org.codehaus.janino.Java.Lvalue;
import com.hazelcast.org.codehaus.janino.Java.MarkerAnnotation;
import com.hazelcast.org.codehaus.janino.Java.MemberAnnotationTypeDeclaration;
import com.hazelcast.org.codehaus.janino.Java.MemberClassDeclaration;
import com.hazelcast.org.codehaus.janino.Java.MemberEnumDeclaration;
import com.hazelcast.org.codehaus.janino.Java.MemberInterfaceDeclaration;
import com.hazelcast.org.codehaus.janino.Java.MemberTypeDeclaration;
import com.hazelcast.org.codehaus.janino.Java.MethodDeclarator;
import com.hazelcast.org.codehaus.janino.Java.MethodInvocation;
import com.hazelcast.org.codehaus.janino.Java.Modifiers;
import com.hazelcast.org.codehaus.janino.Java.NamedClassDeclaration;
import com.hazelcast.org.codehaus.janino.Java.NamedTypeDeclaration;
import com.hazelcast.org.codehaus.janino.Java.NewAnonymousClassInstance;
import com.hazelcast.org.codehaus.janino.Java.NewArray;
import com.hazelcast.org.codehaus.janino.Java.NewClassInstance;
import com.hazelcast.org.codehaus.janino.Java.NewInitializedArray;
import com.hazelcast.org.codehaus.janino.Java.NormalAnnotation;
import com.hazelcast.org.codehaus.janino.Java.NullLiteral;
import com.hazelcast.org.codehaus.janino.Java.Package;
import com.hazelcast.org.codehaus.janino.Java.PackageDeclaration;
import com.hazelcast.org.codehaus.janino.Java.PackageMemberAnnotationTypeDeclaration;
import com.hazelcast.org.codehaus.janino.Java.PackageMemberClassDeclaration;
import com.hazelcast.org.codehaus.janino.Java.PackageMemberEnumDeclaration;
import com.hazelcast.org.codehaus.janino.Java.PackageMemberInterfaceDeclaration;
import com.hazelcast.org.codehaus.janino.Java.PackageMemberTypeDeclaration;
import com.hazelcast.org.codehaus.janino.Java.Padder;
import com.hazelcast.org.codehaus.janino.Java.ParameterAccess;
import com.hazelcast.org.codehaus.janino.Java.ParenthesizedExpression;
import com.hazelcast.org.codehaus.janino.Java.Primitive;
import com.hazelcast.org.codehaus.janino.Java.PrimitiveType;
import com.hazelcast.org.codehaus.janino.Java.QualifiedThisReference;
import com.hazelcast.org.codehaus.janino.Java.ReferenceType;
import com.hazelcast.org.codehaus.janino.Java.ReturnStatement;
import com.hazelcast.org.codehaus.janino.Java.Rvalue;
import com.hazelcast.org.codehaus.janino.Java.RvalueMemberType;
import com.hazelcast.org.codehaus.janino.Java.Scope;
import com.hazelcast.org.codehaus.janino.Java.SimpleConstant;
import com.hazelcast.org.codehaus.janino.Java.SimpleType;
import com.hazelcast.org.codehaus.janino.Java.SingleElementAnnotation;
import com.hazelcast.org.codehaus.janino.Java.Statement;
import com.hazelcast.org.codehaus.janino.Java.StringLiteral;
import com.hazelcast.org.codehaus.janino.Java.SuperConstructorInvocation;
import com.hazelcast.org.codehaus.janino.Java.SuperclassFieldAccessExpression;
import com.hazelcast.org.codehaus.janino.Java.SuperclassMethodInvocation;
import com.hazelcast.org.codehaus.janino.Java.SwitchStatement;
import com.hazelcast.org.codehaus.janino.Java.SwitchStatement.SwitchBlockStatementGroup;
import com.hazelcast.org.codehaus.janino.Java.SynchronizedStatement;
import com.hazelcast.org.codehaus.janino.Java.ThisReference;
import com.hazelcast.org.codehaus.janino.Java.ThrowStatement;
import com.hazelcast.org.codehaus.janino.Java.TryStatement;
import com.hazelcast.org.codehaus.janino.Java.TryStatement.LocalVariableDeclaratorResource;
import com.hazelcast.org.codehaus.janino.Java.TryStatement.VariableAccessResource;
import com.hazelcast.org.codehaus.janino.Java.Type;
import com.hazelcast.org.codehaus.janino.Java.TypeArgument;
import com.hazelcast.org.codehaus.janino.Java.TypeBodyDeclaration;
import com.hazelcast.org.codehaus.janino.Java.TypeDeclaration;
import com.hazelcast.org.codehaus.janino.Java.TypeParameter;
import com.hazelcast.org.codehaus.janino.Java.UnaryOperation;
import com.hazelcast.org.codehaus.janino.Java.VariableDeclarator;
import com.hazelcast.org.codehaus.janino.Java.WhileStatement;
import com.hazelcast.org.codehaus.janino.Visitor.AnnotationVisitor;
import com.hazelcast.org.codehaus.janino.Visitor.AtomVisitor;
import com.hazelcast.org.codehaus.janino.Visitor.BlockStatementVisitor;
import com.hazelcast.org.codehaus.janino.Visitor.ElementValueVisitor;
import com.hazelcast.org.codehaus.janino.Visitor.ImportVisitor;
import com.hazelcast.org.codehaus.janino.Visitor.LvalueVisitor;
import com.hazelcast.org.codehaus.janino.Visitor.RvalueVisitor;
import com.hazelcast.org.codehaus.janino.Visitor.TypeDeclarationVisitor;
import com.hazelcast.org.codehaus.janino.util.Annotatable;
import com.hazelcast.org.codehaus.janino.util.ClassFile;
import com.hazelcast.org.codehaus.janino.util.ClassFile.ClassFileException;

/**
 * This class actually implements the Java com.hazelcast.com.iler. It is associated with exactly one com.hazelcast.com.ilation unit which it
 * com.hazelcast.com.iles.
 */
public
class UnitCompiler {
    private static final Logger LOGGER = Logger.getLogger(UnitCompiler.class.getName());

    /**
     * This constant determines the number of operands up to which the
     * <pre>
     *      a.concat(b).concat(c)
     * </pre>
     * <p>
     *  strategy is used to implement string concatenation. For more operands, the
     * </p>
     * <pre>
     *      new StringBuilder(a).append(b).append(c).append(d).toString()
     * </pre>
     * strategy is chosen.
     * <p>
     *   <a href="http://www.tomgibara.com.hazelcast.com.janino-evaluation/string-concatenation-benchmark">A very good article from
     *   Tom Gibara</a> analyzes the impact of this decision and recommends a value of three.
     * </p>
     */
    private static final int STRING_CONCAT_LIMIT = 3;

    /**
     * Special value for the <var>orientation</var> parameter of the {@link #com.hazelcast.com.ileBoolean(Java.Rvalue,
     * CodeContext.Offset, boolean)} methods, indicating that the code should be generated such that execution branches
     * if the value on top of the operand stack is TRUE.
     */
    public static final boolean JUMP_IF_TRUE  = true;

    /**
     * Special value for the <var>orientation</var> parameter of the {@link #com.hazelcast.com.ileBoolean(Java.Rvalue,
     * CodeContext.Offset, boolean)} methods, indicating that the code should be generated such that execution branches
     * if the value on top of the operand stack is FALSE.
     */
    public static final boolean JUMP_IF_FALSE = false;

    private static final Pattern LOOKS_LIKE_TYPE_PARAMETER = Pattern.com.hazelcast.com.ile("\\p{javaUpperCase}+");

    private EnumSet<JaninoOption> options = EnumSet.noneOf(JaninoOption.class);

    public
    UnitCompiler(CompilationUnit com.hazelcast.com.ilationUnit, IClassLoader iClassLoader) {
        this.com.hazelcast.com.ilationUnit = com.hazelcast.com.ilationUnit;
        this.iClassLoader    = iClassLoader;
    }

    /**
     * @return A reference to the currently effective com.hazelcast.com.ilation options; changes to it take
     *         effect immediately
     */
    public EnumSet<JaninoOption>
    options() { return this.options; }

    /**
     * Sets the options for all future com.hazelcast.com.ilations.
     */
    public UnitCompiler
    options(EnumSet<JaninoOption> options) {
        this.options = options;
        return this;
    }

    /**
     * @return The {@link CompilationUnit} that this {@link UnitCompiler} com.hazelcast.com.iles
     */
    public CompilationUnit
    getCompilationUnit() { return this.com.hazelcast.com.ilationUnit; }

    private void
    import2(SingleStaticImportDeclaration ssid) throws CompileException {
        String name = UnitCompiler.last(ssid.identifiers);

        List<Object/*IField+IMethod+IClass*/>
        importedObjects = (List<Object>) this.singleStaticImports.get(name);
        if (importedObjects == null) {
            importedObjects = new ArrayList<Object>();
            this.singleStaticImports.put(name, importedObjects);
        }

        // Type?
        {
            IClass iClass = this.findTypeByFullyQualifiedName(ssid.getLocation(), ssid.identifiers);
            if (iClass != null) {
                importedObjects.add(iClass);
                return;
            }
        }

        String[] typeName = UnitCompiler.allButLast(ssid.identifiers);
        IClass   iClass   = this.findTypeByFullyQualifiedName(ssid.getLocation(), typeName);
        if (iClass == null) {
            this.com.hazelcast.com.ileError("Could not load \"" + Java.join(typeName, ".") + "\"", ssid.getLocation());
            return;
        }

        // Static field?
        IField iField = iClass.getDeclaredIField(name);
        if (iField != null) {
            if (!iField.isStatic()) {
                this.com.hazelcast.com.ileError(
                    "Field \"" + name + "\" of \"" + Java.join(typeName, ".") + "\" must be static",
                    ssid.getLocation()
                );
            }
            importedObjects.add(iField);
            return;
        }

        // Static method?
        IMethod[] ms = iClass.getDeclaredIMethods(name);
        if (ms.length > 0) {
            importedObjects.addAll(Arrays.asList(ms));
            return;
        }

        // Give up.
        this.com.hazelcast.com.ileError(
            "\"" + Java.join(typeName, ".") + "\" has no static member \"" + name + "\"",
            ssid.getLocation()
        );
    }
    private void
    import2(StaticImportOnDemandDeclaration siodd) throws CompileException {
        IClass iClass = this.findTypeByFullyQualifiedName(siodd.getLocation(), siodd.identifiers);
        if (iClass == null) {
            this.com.hazelcast.com.ileError("Could not load \"" + Java.join(siodd.identifiers, ".") + "\"", siodd.getLocation());
            return;
        }
        this.staticImportsOnDemand.add(iClass);
    }

    /**
     * Generates an array of {@link ClassFile} objects which represent the classes and interfaces declared in the
     * com.hazelcast.com.ilation unit.
     */
    public ClassFile[]
    com.hazelcast.com.ileUnit(boolean debugSource, boolean debugLines, boolean debugVars) throws CompileException {
        this.debugSource = debugSource;
        this.debugLines  = debugLines;
        this.debugVars   = debugVars;

        // Compile static import declarations.
        // Notice: The single-type and on-demand imports are needed BEFORE the unit is com.hazelcast.com.iled, thus they are
        // processed in 'getSingleTypeImport()' and 'importOnDemand()'.
        for (ImportDeclaration id : this.com.hazelcast.com.ilationUnit.importDeclarations) {
            id.accept(new ImportVisitor<Void, CompileException>() {
                // SUPPRESS CHECKSTYLE LineLengthCheck:4
                @Override @Nullable public Void visitSingleTypeImportDeclaration(SingleTypeImportDeclaration stid)                                  {                                   return null; }
                @Override @Nullable public Void visitTypeImportOnDemandDeclaration(TypeImportOnDemandDeclaration tiodd)                             {                                   return null; }
                @Override @Nullable public Void visitSingleStaticImportDeclaration(SingleStaticImportDeclaration ssid)      throws CompileException { UnitCompiler.this.import2(ssid);  return null; }
                @Override @Nullable public Void visitStaticImportOnDemandDeclaration(StaticImportOnDemandDeclaration siodd) throws CompileException { UnitCompiler.this.import2(siodd); return null; }
            });
        }

        if (this.generatedClassFiles != null) {
            throw new IllegalStateException("\"UnitCompiler.com.hazelcast.com.ileUnit()\" is not reentrant");
        }

        final List<ClassFile> gcfs = (this.generatedClassFiles = new ArrayList<ClassFile>());
        try {

            for (PackageMemberTypeDeclaration pmtd : this.com.hazelcast.com.ilationUnit.packageMemberTypeDeclarations) {
                try {
                    this.com.hazelcast.com.ile(pmtd);
                } catch (ClassFileException cfe) {
                    throw new CompileException(cfe.getMessage(), pmtd.getLocation(), cfe);
                } catch (RuntimeException re) {
                    throw new InternalCompilerException("Compiling \"" + pmtd + "\": " + re.getMessage(), re);
                }
            }

            if (this.com.hazelcast.com.ileErrorCount > 0) {
                throw new CompileException((
                    this.com.hazelcast.com.ileErrorCount
                    + " error(s) while com.hazelcast.com.iling unit \""
                    + this.com.hazelcast.com.ilationUnit.optionalFileName
                    + "\""
                ), null);
            }

            return (ClassFile[]) gcfs.toArray(new ClassFile[gcfs.size()]);
        } finally {
            this.generatedClassFiles = null;
        }
    }

    // ------------ TypeDeclaration.com.hazelcast.com.ile() -------------

    private void
    com.hazelcast.com.ile(TypeDeclaration td) throws CompileException {

        td.accept(new TypeDeclarationVisitor<Void, CompileException>() {

            // SUPPRESS CHECKSTYLE LineLength:9
            @Override @Nullable public Void visitAnonymousClassDeclaration(AnonymousClassDeclaration acd)                             throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(acd);                           return null; }
            @Override @Nullable public Void visitLocalClassDeclaration(LocalClassDeclaration lcd)                                     throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(lcd);                           return null; }
            @Override @Nullable public Void visitPackageMemberClassDeclaration(PackageMemberClassDeclaration pmcd)                    throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(pmcd);                          return null; }
            @Override @Nullable public Void visitMemberInterfaceDeclaration(MemberInterfaceDeclaration mid)                           throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(mid);                           return null; }
            @Override @Nullable public Void visitPackageMemberInterfaceDeclaration(PackageMemberInterfaceDeclaration pmid)            throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(pmid);                          return null; }
            @Override @Nullable public Void visitMemberClassDeclaration(MemberClassDeclaration mcd)                                   throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2((InnerClassDeclaration) mcd);   return null; }
            @Override @Nullable public Void visitMemberEnumDeclaration(MemberEnumDeclaration med)                                     throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2((InnerClassDeclaration) med);   return null; }
            @Override @Nullable public Void visitPackageMemberEnumDeclaration(PackageMemberEnumDeclaration pmed)                      throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(pmed);                          return null; }
            @Override @Nullable public Void visitPackageMemberAnnotationTypeDeclaration(PackageMemberAnnotationTypeDeclaration pmatd) throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(pmatd);                         return null; }

            // SUPPRESS CHECKSTYLE LineLength:2
            @Override @Nullable public Void visitEnumConstant(EnumConstant ec)                                         throws CompileException { UnitCompiler.this.com.hazelcast.com.ileError("Compilation of enum constant NYI",                      ec.getLocation());   return null; }
            @Override @Nullable public Void visitMemberAnnotationTypeDeclaration(MemberAnnotationTypeDeclaration matd) throws CompileException { UnitCompiler.this.com.hazelcast.com.ileError("Compilation of member annotation type declaration NYI", matd.getLocation()); return null; }
        });
    }

    /**
     * Compiles a top-level class or enum declaration.
     */
    private void
    com.hazelcast.com.ile2(AbstractPackageMemberClassDeclaration apmcd) throws CompileException {
        this.checkForConflictWithSingleTypeImport(apmcd.getName(), apmcd.getLocation());
        this.checkForNameConflictWithAnotherPackageMemberTypeDeclaration(apmcd);
        this.com.hazelcast.com.ile2((NamedClassDeclaration) apmcd);
    }

    /**
     * Compiles a top-level interface or annotation type declaration.
     */
    private void
    com.hazelcast.com.ile2(PackageMemberInterfaceDeclaration pmid) throws CompileException {
        this.checkForConflictWithSingleTypeImport(pmid.getName(), pmid.getLocation());
        this.checkForNameConflictWithAnotherPackageMemberTypeDeclaration(pmid);
        this.com.hazelcast.com.ile2((InterfaceDeclaration) pmid);
    }

    /**
     * @see JLS8, section 7.6, "Top Level Type Declarations"
     */
    private void
    checkForNameConflictWithAnotherPackageMemberTypeDeclaration(PackageMemberTypeDeclaration pmtd)
    throws CompileException {

        CompilationUnit declaringCompilationUnit = pmtd.getDeclaringCompilationUnit();

        String                       name      = pmtd.getName();
        PackageMemberTypeDeclaration otherPmtd = declaringCompilationUnit.getPackageMemberTypeDeclaration(
            name
        );

        if (otherPmtd != null && otherPmtd != pmtd) {
            this.com.hazelcast.com.ileError(
                "Redeclaration of type \"" + name + "\", previously declared in " + otherPmtd.getLocation()
                ,
                pmtd.getLocation()
            );
        }
    }

    /**
     * @see JLS8, section 7.6, "Top Level Type Declarations"
     */
    private void
    checkForConflictWithSingleTypeImport(String name, Location optionalLocation) throws CompileException {

        String[] ss = this.getSingleTypeImport(name, optionalLocation);
        if (ss != null) {
            this.com.hazelcast.com.ileError((
                "Package member type declaration \""
                + name
                + "\" conflicts with single-type-import \""
                + Java.join(ss, ".")
                + "\""
            ), optionalLocation);
        }
    }

    private void
    com.hazelcast.com.ile2(AbstractClassDeclaration cd) throws CompileException {
        IClass iClass = this.resolve(cd);

        // Check that all methods of the non-abstract class are implemented.
        if (!Mod.isAbstract(cd.getModifierFlags())) {
            IMethod[] ms = iClass.getIMethods();
            for (IMethod base : ms) {
                if (base.isAbstract()) {
                    IMethod override = iClass.findIMethod(base.getName(), base.getParameterTypes());
                    if (
                        override == null           // It wasn't overridden
                        || override.isAbstract()   // It was overridden with an abstract method
                                                   // The override does not provide a covariant return type
                        || !base.getReturnType().isAssignableFrom(override.getReturnType())
                    ) {
                        this.com.hazelcast.com.ileError(
                            "Non-abstract class \"" + iClass + "\" must implement method \"" + base + "\"",
                            cd.getLocation()
                        );
                    }
                }
            }
        }

        // Create "ClassFile" object.
        ClassFile cf;
        {
            IClass superclass = iClass.getSuperclass();
            cf = new ClassFile(
                (short) (cd.getModifierFlags() | Mod.SUPER),            // accessFlags
                iClass.getDescriptor(),                                 // thisClassFD
                superclass != null ? superclass.getDescriptor() : null, // superclassFD
                IClass.getDescriptors(iClass.getInterfaces())           // interfaceFDs
            );
        }

        // Add class annotations with retention != SOURCE.
        this.com.hazelcast.com.ileAnnotations(cd.getAnnotations(), cf, cf);

        if (cd.getEnclosingScope() instanceof Block) {

            // Add an "InnerClasses" attribute entry for this anonymous class declaration on the class file (JLS8,
            // section 4.7.6, "The InnerClasses Attribute").
            short innerClassInfoIndex = cf.addConstantClassInfo(iClass.getDescriptor());
            short innerNameIndex      = (
                this instanceof NamedTypeDeclaration
                ? cf.addConstantUtf8Info(((NamedTypeDeclaration) this).getName())
                : (short) 0
            );
            assert cd.getAnnotations().length == 0 : "NYI";
            cf.addInnerClassesAttributeEntry(new ClassFile.InnerClassesAttribute.Entry(
                innerClassInfoIndex,  // innerClassInfoIndex
                (short) 0,            // outerClassInfoIndex
                innerNameIndex,       // innerNameIndex
                cd.getModifierFlags() // innerClassAccessFlags
            ));
        } else
        if (cd.getEnclosingScope() instanceof TypeDeclaration) {

            // Add an "InnerClasses" attribute entry for this nested class declaration on the class file (JLS8,
            // section 4.7.6, "The InnerClasses Attribute").
            short innerClassInfoIndex = cf.addConstantClassInfo(iClass.getDescriptor());
            short outerClassInfoIndex = cf.addConstantClassInfo(
                this.resolve(((TypeDeclaration) cd.getEnclosingScope())).getDescriptor()
            );
            short innerNameIndex = cf.addConstantUtf8Info(((MemberTypeDeclaration) cd).getName());
            assert cd.getAnnotations().length == 0 : "NYI";
            cf.addInnerClassesAttributeEntry(new ClassFile.InnerClassesAttribute.Entry(
                innerClassInfoIndex,  // innerClassInfoIndex
                outerClassInfoIndex,  // outerClassInfoIndex
                innerNameIndex,       // innerNameIndex
                cd.getModifierFlags() // innerClassAccessFlags
            ));
        }

        // Set the "SourceFile" attribute (JVMS8, section 4.7.10, "The SourceFile Attribute") on this class file.
        if (this.debugSource) {
            String sourceFileName;
            {
                String s = cd.getLocation().getFileName();
                if (s != null) {
                    sourceFileName = new File(s).getName();
                } else if (cd instanceof NamedTypeDeclaration) {
                    sourceFileName = ((NamedTypeDeclaration) cd).getName() + ".java";
                } else {
                    sourceFileName = "ANONYMOUS.java";
                }
            }
            cf.addSourceFileAttribute(sourceFileName);
        }

        // Add a "Deprecated" attribute (JVMS8, section 4.7.15, "The Deprecated Attribute") on this class file.
        if (cd instanceof DocCommentable && ((DocCommentable) cd).hasDeprecatedDocTag()) {
            cf.addDeprecatedAttribute();
        }

        List<BlockStatement> classInitializationStatements = new ArrayList<BlockStatement>();

        if (cd instanceof EnumDeclaration) {
            EnumDeclaration ed = (EnumDeclaration) cd;

            // Create field and static initializer for each enum constant.
            for (EnumConstant ec : ed.getConstants()) {

                // E <constant> = new E(<ordinal>, <name> [ optional-constructor-args ]);
                VariableDeclarator variableDeclarator = new VariableDeclarator(
                    ec.getLocation(),     // location
                    ec.name,              // name
                    0,                    // brackets
                    new NewClassInstance( // optionalInitializer
                        ec.getLocation(),                                                   // location
                        null,                                                               // optionalQualification
                        iClass,                                                             // iClass
                        ec.optionalArguments != null ? ec.optionalArguments : new Rvalue[0] // arguments
                    )
                );

                FieldDeclaration fd = new FieldDeclaration(
                    ec.getLocation(),                                             // location
                    ec.getDocComment(),                                           // optionalDocComment
                    new Modifiers((short) (Mod.PUBLIC | Mod.STATIC | Mod.FINAL)), // modifiers
                    new SimpleType(ec.getLocation(), iClass),                     // type
                    new VariableDeclarator[] { variableDeclarator }               // variableDeclarators
                );
                fd.setDeclaringType(ed);

                classInitializationStatements.add(fd);

                this.addFields(fd, cf);
            }

            // Create the synthetic "ENUM$VALUES" field:
            //
            //     private static final E[] ENUM$VALUES = new E[<number-of-enum-constants>];
            Location         loc        = ed.getLocation();
            IClass           enumIClass = this.resolve(ed);
            FieldDeclaration fd         = new FieldDeclaration(
                loc,                                                           // location
                null,                                                          // optionalDocComment
                new Modifiers((short) (Mod.PRIVATE | Mod.STATIC | Mod.FINAL)), // modifiers
                new SimpleType(loc, enumIClass),                               // type
                new VariableDeclarator[] {                                     // variableDeclarators)
                    new VariableDeclarator(
                        loc,           // location
                        "ENUM$VALUES", // name
                        1,             // brackets
                        new NewArray(  // optionalInitializer
                            loc,                             // location
                            new SimpleType(loc, enumIClass), // type
                            new Rvalue[] {                   // dimExprs
                                new IntegerLiteral(loc, String.valueOf(ed.getConstants().size())),
                            },
                            0                                // dims
                        )
                    )
                }
            );
            ((AbstractClassDeclaration) ed).addFieldDeclaration(fd);
        }

        // Process static initializers (a.k.a. class initializers).
        for (BlockStatement vdoi : cd.variableDeclaratorsAndInitializers) {
            if (((TypeBodyDeclaration) vdoi).isStatic()) classInitializationStatements.add(vdoi);
        }

        if (cd instanceof EnumDeclaration) {
            EnumDeclaration ed         = (EnumDeclaration) cd;
            IClass          enumIClass = this.resolve(ed);

            // Initialize the elements of the synthetic "ENUM$VALUES" field:
            //
            //     E.ENUM$VALUES[0] = E.<first-constant>;
            //     E.ENUM$VALUES[1] = E.<second-constant>;
            //     ...
            int index = 0;
            for (EnumConstant ec : ed.getConstants()) {
                classInitializationStatements.add(
                    new ExpressionStatement(
                        new Assignment(
                            ec.getLocation(),          // location
                            new ArrayAccessExpression( // lhs
                                ec.getLocation(),                                             // location
                                new FieldAccessExpression(                                    // lhs
                                    ec.getLocation(),
                                    new SimpleType(ec.getLocation(), enumIClass),
                                    "ENUM$VALUES"
                                ),
                                new IntegerLiteral(ec.getLocation(), String.valueOf(index++)) // rhs
                            ),
                            "=",                       // operator
                            new FieldAccessExpression( // rhs
                                ec.getLocation(),
                                new SimpleType(ec.getLocation(), enumIClass),
                                ec.name
                            )
                        )
                    )
                );
            }
        }

        // Create class initialization method.
        this.maybeCreateInitMethod(cd, cf, classInitializationStatements);

        // Generate and com.hazelcast.com.ile the "magic" ENUM methods "E[] values()" and "E valueOf(String)".
        if (cd instanceof EnumDeclaration) {
            EnumDeclaration ed = (EnumDeclaration) cd;

            // public static E[] values() {
            //     E[] tmp = new T[<number-of-constants>];
            //     System.arraycopy(T.ENUM$VALUES, 0, tmp, 0, <number-of-constants>);
            //     return tmp;
            // }
            Location   loc                   = ed.getLocation();
            int        numberOfEnumConstants = ed.getConstants().size();
            IClass     enumIClass            = this.resolve(ed);

            // tmp = new T[<number-of-constants>];
            VariableDeclarator vd = new VariableDeclarator(
                loc,          // location
                "tmp",        // name
                0,            // brackets
                new NewArray( // optionalinitializer
                    loc,                             // location
                    new SimpleType(loc, enumIClass), // type
                    new Rvalue[] {                   // dimExprs
                        new IntegerLiteral(loc, String.valueOf(numberOfEnumConstants))
                    },
                    0                                // dims
                )
            );

            // E[] tmp = new E[<number-of-constants>];
            LocalVariableDeclarationStatement lvds = new LocalVariableDeclarationStatement(
                loc,
                new Modifiers(),
                new SimpleType(loc, enumIClass.getArrayIClass(this.iClassLoader.TYPE_java_lang_Object)),
                new VariableDeclarator[] { vd }
            );

            // public static E[] values() {
            //     E[] tmp = new T[<number-of-constants>];
            //     System.arraycopy(T.ENUM$VALUES, 0, tmp, 0, <number-of-constants>);
            //     return tmp;
            // }
            {
                MethodDeclarator md = new MethodDeclarator(
                    loc,                                              // location
                    null,                                             // optionalDocComment
                    new Modifiers((short) (Mod.PUBLIC | Mod.STATIC)), // modifiers
                    null,                                             // optionalTypeParameters
                    new ArrayType(new SimpleType(loc, enumIClass)),   // type
                    "values",                                         // name
                    new FormalParameters(loc),                        // parameters
                    new Type[0],                                      // thrownExceptions
                    null,                                             // defaultValue
                    Arrays.asList(                                    // optionalStatements

                        // E[] tmp = new E[<number-of-constants>];
                        lvds,

                        // System.arraycopy(E.ENUM$VALUES, 0, tmp, 0, <number-of-constants>);
                        new ExpressionStatement(new MethodInvocation(
                            loc,                                                          // location
                            new SimpleType(loc, this.iClassLoader.TYPE_java_lang_System), // optionalTarget
                            "arraycopy",                                                  // methodName
                            new Rvalue[] {                                                // arguments
                                // SUPPRESS CHECKSTYLE LineLength:5
                                new FieldAccessExpression(loc, new SimpleType(loc, enumIClass), "ENUM$VALUES"), // Argument #1: E.ENUM$VALUES
                                new IntegerLiteral(loc, "0"),                                                   // Argument #2: 0
                                new LocalVariableAccess(loc, this.getLocalVariable(lvds, vd)),                  // Argument #3: tmp
                                new IntegerLiteral(loc, "0"),                                                   // Argument #4: 0
                                new IntegerLiteral(loc, String.valueOf(numberOfEnumConstants)),                 // Argument #5: <number-of-constants>
                            }
                        )),

                        // return tmp;
                        new ReturnStatement(loc, new LocalVariableAccess(loc, this.getLocalVariable(lvds, vd)))
                    )
                );
                md.setDeclaringType(ed);

                this.com.hazelcast.com.ile(md, cf);
            }

            // public static E valueOf(String s) {
            //     return (E) Enum.valueOf(E.class, s);
            // }
            FormalParameter fp = new FormalParameter(
                loc,                                                          // location
                false,                                                        // finaL
                new SimpleType(loc, this.iClassLoader.TYPE_java_lang_String), // type
                "s"                                                           // name
            );
            {
                MethodDeclarator md = new MethodDeclarator(
                    loc,                                              // location
                    null,                                             // optionalDocComment
                    new Modifiers((short) (Mod.PUBLIC | Mod.STATIC)), // modifiers
                    null,                                             // optionalTypeParameters
                    new SimpleType(loc, enumIClass),                  // type
                    "valueOf",                                        // name
                    new FormalParameters(                             // formalParameters
                        loc,                          // location
                        new FormalParameter[] { fp }, // parameters
                        false                         // variableArity
                    ),
                    new Type[0],                                      // thrownExceptions,
                    null,                                             // defaultValue
                    Arrays.asList(                                    // optionalStatements

                        // return (E) Enum.valueOf(E.class, s);
                        new ReturnStatement(loc, new Cast(
                            loc,                             // location
                            new SimpleType(loc, enumIClass), // targetType
                            new MethodInvocation(            // value
                                loc,                                                        // location
                                new SimpleType(loc, this.iClassLoader.TYPE_java_lang_Enum), // optionalTarget
                                "valueOf",                                                  // methodName
                                new Rvalue[] {                                              // arguments
                                    new ClassLiteral(loc, new SimpleType(loc, enumIClass)),
                                    new ParameterAccess(loc, fp)
                                }
                            )
                        ))
                    )
                );

                md.setEnclosingScope(ed);

                this.com.hazelcast.com.ile(md, cf);
            }
        }

        // Compile declared methods.
        this.com.hazelcast.com.ileDeclaredMethods(cd, cf);

        // Compile declared constructors.
        // As a side effect of com.hazelcast.com.iling methods and constructors, synthetic "class-dollar" methods (which implement
        // class literals) are generated on-the fly. We need to note how many we have here so we can com.hazelcast.com.ile the
        // extras.
        final int declaredMethodCount = cd.getMethodDeclarations().size();
        {
            int                     syntheticFieldCount = cd.syntheticFields.size();
            ConstructorDeclarator[] ctords              = cd.getConstructors();
            for (ConstructorDeclarator ctord : ctords) {

                this.com.hazelcast.com.ile(ctord, cf);
                if (syntheticFieldCount != cd.syntheticFields.size()) {
                    throw new InternalCompilerException(
                        "SNO: Compilation of constructor \""
                        + ctord
                        + "\" ("
                        + ctord.getLocation()
                        + ") added synthetic fields!?"
                    );
                }
            }
        }

        // A side effect of this call may create synthetic functions to access protected parent variables.
        this.com.hazelcast.com.ileDeclaredMemberTypes(cd, cf);

        // Compile the aforementioned extras.
        this.com.hazelcast.com.ileDeclaredMethods(cd, cf, declaredMethodCount);

        // For every method look for bridge methods that need to be supplied. This is used to correctly dispatch into
        // covariant return types from existing code.
        for (IMethod base : iClass.getIMethods()) {
            if (!base.isStatic()) {
                IMethod override = iClass.findIMethod(base.getName(), base.getParameterTypes());

                // If we overrode the method but with a DIFFERENT return type.
                if (
                    override != null
                    && base.getReturnType() != override.getReturnType()
                ) this.generateBridgeMethod(cf, base, override);
            }
        }

        // Add class and instance variables as (static and non-static) fields.
        for (BlockStatement vdoi : cd.getVariableDeclaratorsAndInitializers()) {
            if (vdoi instanceof FieldDeclaration) this.addFields((FieldDeclaration) vdoi, cf);
        }

        // Synthetic fields.
        for (IField f : cd.getSyntheticFields().values()) {
            cf.addFieldInfo(
                Mod.PACKAGE,                 // accessFlags
                f.getName(),                 // fieldName
                f.getType().getDescriptor(), // fieldTypeFD
                null                         // optionalConstantValue
            );
        }

        // Add the generated class file to a thread-local store.
        this.addClassFile(cf);
    }

    /**
     * Adds the given {@link ClassFile} to the result set.
     */
    private void
    addClassFile(ClassFile cf) {

        if (UnitCompiler.LOGGER.isLoggable(Level.FINEST)) UnitCompiler.disassembleToStdout(cf.toByteArray());

        assert this.generatedClassFiles != null;
        this.generatedClassFiles.add(cf);
    }

    /**
     * Creates and adds {@link ClassFile.FieldInfo}s to the <var>cf</var> for all fields declared by the <var>fd</var>.
     */
    private void
    addFields(FieldDeclaration fd, ClassFile cf) throws CompileException {
        for (VariableDeclarator vd : fd.variableDeclarators) {

            Type type = fd.type;
            for (int i = 0; i < vd.brackets; ++i) type = new ArrayType(type);

            Object ocv = UnitCompiler.NOT_CONSTANT;
            if (Mod.isFinal(fd.modifiers.accessFlags) && vd.optionalInitializer instanceof Rvalue) {
                ocv = this.getConstantValue((Rvalue) vd.optionalInitializer);
            }

            ClassFile.FieldInfo fi;
            if (Mod.isPrivateAccess(fd.modifiers.accessFlags)) {

                // To make the private field accessible for enclosing types, enclosed types and types enclosed by the
                // same type, it is modified as follows:
                //  + Access is changed from PRIVATE to PACKAGE
                fi = cf.addFieldInfo(
                    Mod.changeAccess(fd.modifiers.accessFlags, Mod.PACKAGE), // accessFlags
                    vd.name,                                                 // fieldName
                    this.getType(type).getDescriptor(),                      // fieldTypeFD
                    ocv == UnitCompiler.NOT_CONSTANT ? null : ocv            // optionalConstantValue
                );
            } else
            {
                fi = cf.addFieldInfo(
                    fd.modifiers.accessFlags,                     // accessFlags
                    vd.name,                                      // fieldName
                    this.getType(type).getDescriptor(),           // fieldTypeFD
                    ocv == UnitCompiler.NOT_CONSTANT ? null : ocv // optionalConstantValue
                );
            }

            // Add field annotations with retention != SOURCE.
            this.com.hazelcast.com.ileAnnotations(fd.getAnnotations(), fi, cf);

            // Add "Deprecated" attribute (JVMS 4.7.10).
            if (fd.hasDeprecatedDocTag()) {
                fi.addAttribute(new ClassFile.DeprecatedAttribute(cf.addConstantUtf8Info("Deprecated")));
            }
        }
    }

    private void
    com.hazelcast.com.ile2(AnonymousClassDeclaration acd) throws CompileException {

        // For classes that enclose surrounding scopes, trawl their field initializers looking for synthetic fields.
        this.fakeCompileVariableDeclaratorsAndInitializers(acd);

        this.com.hazelcast.com.ile2((InnerClassDeclaration) acd);
    }

    private void
    com.hazelcast.com.ile2(LocalClassDeclaration lcd) throws CompileException {

        // For classes that enclose surrounding scopes, trawl their field initializers looking for synthetic fields.
        this.fakeCompileVariableDeclaratorsAndInitializers(lcd);

        this.com.hazelcast.com.ile2((InnerClassDeclaration) lcd);
    }

    private void
    com.hazelcast.com.ile2(InnerClassDeclaration icd) throws CompileException {

        // Define a synthetic "this$n" field if there is an enclosing instance, where "n" designates the number of
        // enclosing instances minus one.
        {
            List<TypeDeclaration> ocs     = UnitCompiler.getOuterClasses(icd);
            final int             nesting = ocs.size();
            if (nesting >= 2) {
                TypeDeclaration immediatelyEnclosingOuterClassDeclaration = (TypeDeclaration) ocs.get(1);
                icd.defineSyntheticField(new SimpleIField(
                    this.resolve(icd),
                    "this$" + (nesting - 2),
                    this.resolve(immediatelyEnclosingOuterClassDeclaration)
                ));
            }
        }

        this.com.hazelcast.com.ile2((AbstractClassDeclaration) icd);
    }

    private void
    fakeCompileVariableDeclaratorsAndInitializers(AbstractClassDeclaration cd) throws CompileException {

        // Compilation of field declarations can create synthetic variables, so we must not use an iterator.
        List<BlockStatement> vdais = cd.variableDeclaratorsAndInitializers;
        for (int i = 0; i < vdais.size(); i++) {
            BlockStatement vdoi = (BlockStatement) vdais.get(i);
            this.fakeCompile(vdoi);
        }
    }

    private void
    com.hazelcast.com.ile2(InterfaceDeclaration id) throws CompileException {

        final IClass iClass = this.resolve(id);

        // Determine extended interfaces.
        IClass[] is                   = (id.interfaces = new IClass[id.extendedTypes.length]);
        String[] interfaceDescriptors = new String[is.length];
        for (int i = 0; i < id.extendedTypes.length; ++i) {
            is[i]                   = this.getType(id.extendedTypes[i]);
            interfaceDescriptors[i] = is[i].getDescriptor();
        }

        // Create "ClassFile" object.
        ClassFile cf = new ClassFile(
            (short) (id.getModifierFlags() | Mod.INTERFACE | Mod.ABSTRACT), // accessFlags
            iClass.getDescriptor(),                                         // thisClassFD
            Descriptor.JAVA_LANG_OBJECT,                                    // superclassFD
            interfaceDescriptors                                            // interfaceFDs
        );

        // Add interface annotations with retention != SOURCE.
        this.com.hazelcast.com.ileAnnotations(id.getAnnotations(), cf, cf);

        // Set "SourceFile" attribute.
        if (this.debugSource) {
            String sourceFileName;
            {
                String s = id.getLocation().getFileName();
                if (s != null) {
                    sourceFileName = new File(s).getName();
                } else {
                    sourceFileName = id.getName() + ".java";
                }
            }
            cf.addSourceFileAttribute(sourceFileName);
        }

        // Add "Deprecated" attribute (JVMS 4.7.10).
        if (id.hasDeprecatedDocTag()) cf.addDeprecatedAttribute();

        // Interface initialization method.
        if (!id.constantDeclarations.isEmpty()) {
            List<BlockStatement> statements = new ArrayList<BlockStatement>();
            statements.addAll(id.constantDeclarations);

            this.maybeCreateInitMethod(id, cf, statements);
        }

        this.com.hazelcast.com.ileDeclaredMethods(id, cf);

        // Class variables.
        for (FieldDeclaration constantDeclaration : id.constantDeclarations) this.addFields(constantDeclaration, cf);

        this.com.hazelcast.com.ileDeclaredMemberTypes(id, cf);

        // Add the generated class file to a thread-local store.
        this.addClassFile(cf);
    }

    /**
     * Converts and adds the <var>annotations</var> to the <var>target</var>.
     */
    private void
    com.hazelcast.com.ileAnnotations(Annotation[] annotations, Annotatable target, final ClassFile cf) throws CompileException {

        ANNOTATIONS: for (final Annotation a : annotations) {
            Type          annotationType        = a.getType();
            IClass        annotationIClass      = this.getType(annotationType);
            IAnnotation[] annotationAnnotations = annotationIClass.getIAnnotations();

            // Determine the attribute name.
            boolean runtimeVisible = false;
            for (IAnnotation aa : annotationAnnotations) {

                if (aa.getAnnotationType() != this.iClassLoader.TYPE_java_lang_annotation_Retention) continue;

                Object rev = aa.getElementValue("value");

                String retention = ((IField) rev).getName();

                if ("SOURCE".equals(retention)) {
                    continue ANNOTATIONS;
                } else
                if ("CLASS".equals(retention)) {
                    runtimeVisible = false;
                    break;
                } else
                if ("RUNTIME".equals(retention)) {
                    runtimeVisible = true;
                    break;
                } else
                {
                    throw new AssertionError(retention);
                }
            }

            // Compile the annotation's element-value-pairs.
            final Map<Short, ClassFile.ElementValue> evps = new HashMap<Short, ClassFile.ElementValue>();
            a.accept(new Visitor.AnnotationVisitor<Void, CompileException>() {

                @Override @Nullable public Void
                visitSingleElementAnnotation(SingleElementAnnotation sea) throws CompileException {
                    evps.put(
                        cf.addConstantUtf8Info("value"),
                        UnitCompiler.this.com.hazelcast.com.ileElementValue(sea.elementValue, cf)
                    );
                    return null;
                }

                @Override @Nullable public Void
                visitNormalAnnotation(NormalAnnotation na) throws CompileException {
                    for (ElementValuePair evp : na.elementValuePairs) {
                        evps.put(
                            cf.addConstantUtf8Info(evp.identifier),
                            UnitCompiler.this.com.hazelcast.com.ileElementValue(evp.elementValue, cf)
                        );
                    }
                    return null;
                }

                @Override @Nullable public Void
                visitMarkerAnnotation(MarkerAnnotation ma) {
                    ;
                    return null;
                }
            });

            // Add the annotation to the target (class/interface, method or field).
            target.addAnnotationsAttributeEntry(runtimeVisible, annotationIClass.getDescriptor(), evps);
        }
    }

    private ClassFile.ElementValue
    com.hazelcast.com.ileElementValue(Java.ElementValue elementValue, final ClassFile cf)
    throws CompileException {

        ClassFile.ElementValue
        result = (ClassFile.ElementValue) elementValue.accept(
            new ElementValueVisitor<ClassFile.ElementValue, CompileException>() {

                @Override public ClassFile.ElementValue
                visitRvalue(Rvalue rv) throws CompileException {

                    // Enum constant?
                    ENUM_CONSTANT:
                    if (rv instanceof AmbiguousName) {

                        Rvalue enumConstant = UnitCompiler.this.reclassify((AmbiguousName) rv).toRvalue();
                        if (!(enumConstant instanceof FieldAccess)) break ENUM_CONSTANT; // Not a field access.
                        FieldAccess enumConstantFieldAccess = (FieldAccess) enumConstant;

                        Type enumType = enumConstantFieldAccess.lhs.toType();
                        if (enumType == null) break ENUM_CONSTANT; // LHS is not a type.

                        IClass enumIClass = UnitCompiler.this.findTypeByName(rv.getLocation(), enumType.toString());
                        if (enumIClass == null) {
                            UnitCompiler.this.com.hazelcast.com.ileError(
                                "Cannot find enum \"" + enumType + "\"",
                                enumType.getLocation()
                            );
                            break ENUM_CONSTANT;
                        }

                        if (enumIClass.getSuperclass() == UnitCompiler.this.iClassLoader.TYPE_java_lang_Enum) {
                            return new ClassFile.EnumConstValue(
                                cf.addConstantUtf8Info(enumIClass.getDescriptor()),             // typeNameIndex
                                cf.addConstantUtf8Info(enumConstantFieldAccess.field.getName()) // constNameIndex
                            );
                        }

                        // We have a constant, but it is not an ENUM constant, so fall through.
                    }

                    // Class literal?
                    if (rv instanceof ClassLiteral) {
                        return new ClassFile.ClassElementValue(cf.addConstantUtf8Info(
                            UnitCompiler.this.getType(((ClassLiteral) rv).type).getDescriptor()
                        ));
                    }

                    // Constant value?
                    Object cv = UnitCompiler.this.getConstantValue(rv);

                    if (cv == UnitCompiler.NOT_CONSTANT) {
                        throw new CompileException(
                            "\"" + rv + "\" is not a constant expression",
                            rv.getLocation()
                        );
                    }

                    if (cv == null) {
                        throw new CompileException(
                            "Null literal not allowed as element value",
                            rv.getLocation()
                        );
                    }

                    // SUPPRESS CHECKSTYLE LineLength:9
                    if (cv instanceof Boolean)   { return new ClassFile.BooleanElementValue(cf.addConstantIntegerInfo((Boolean) cv ? 1 : 0)); }
                    if (cv instanceof Byte)      { return new ClassFile.ByteElementValue(cf.addConstantIntegerInfo((Byte) cv));               }
                    if (cv instanceof Short)     { return new ClassFile.ShortElementValue(cf.addConstantIntegerInfo((Short) cv));             }
                    if (cv instanceof Integer)   { return new ClassFile.IntElementValue(cf.addConstantIntegerInfo((Integer) cv));             }
                    if (cv instanceof Long)      { return new ClassFile.LongElementValue(cf.addConstantLongInfo((Long) cv));                  }
                    if (cv instanceof Float)     { return new ClassFile.FloatElementValue(cf.addConstantFloatInfo((Float) cv));               }
                    if (cv instanceof Double)    { return new ClassFile.DoubleElementValue(cf.addConstantDoubleInfo((Double) cv));            }
                    if (cv instanceof Character) { return new ClassFile.CharElementValue(cf.addConstantIntegerInfo((Character) cv));          }
                    if (cv instanceof String)    { return new ClassFile.StringElementValue(cf.addConstantUtf8Info((String) cv));              }

                    throw new AssertionError(cv);
                }

                @Override public ClassFile.ElementValue
                visitAnnotation(Annotation a) throws CompileException {

                    short annotationTypeIndex = (
                        cf.addConstantClassInfo(UnitCompiler.this.getType(a.getType()).getDescriptor())
                    );

                    final Map<Short, ClassFile.ElementValue>
                    evps = new HashMap<Short, ClassFile.ElementValue>();
                    a.accept(new AnnotationVisitor<Void, CompileException>() {

                        @Override @Nullable public Void
                        visitMarkerAnnotation(MarkerAnnotation ma) {
                            ;
                            return null;
                        }

                        @Override @Nullable public Void
                        visitSingleElementAnnotation(SingleElementAnnotation sea) throws CompileException {
                            evps.put(
                                cf.addConstantUtf8Info("value"),
                                UnitCompiler.this.com.hazelcast.com.ileElementValue(sea.elementValue, cf)
                            );
                            return null;
                        }

                        @Override @Nullable public Void
                        visitNormalAnnotation(NormalAnnotation na) throws CompileException {
                            for (ElementValuePair evp : na.elementValuePairs) {
                                evps.put(
                                    cf.addConstantUtf8Info(evp.identifier),
                                    UnitCompiler.this.com.hazelcast.com.ileElementValue(evp.elementValue, cf)
                                );
                            }
                            return null;
                        }
                    });
                    return new ClassFile.Annotation(annotationTypeIndex, evps);
                }

                @Override public ClassFile.ElementValue
                visitElementValueArrayInitializer(ElementValueArrayInitializer evai) throws CompileException {
                    ClassFile.ElementValue[]
                    evs = new ClassFile.ElementValue[evai.elementValues.length];

                    for (int i = 0; i < evai.elementValues.length; i++) {
                        evs[i] = UnitCompiler.this.com.hazelcast.com.ileElementValue(evai.elementValues[i], cf);
                    }
                    return new ClassFile.ArrayElementValue(evs);
                }
            }
        );

        assert result != null;
        return result;
    }

    /**
     * Creates class/interface initialization method iff there is any initialization code.
     *
     * @param td         The type declaration
     * @param cf         The class file into which to put the method
     * @param statements The statements for the method (possibly empty)
     */
    private void
    maybeCreateInitMethod(
        TypeDeclaration      td,
        ClassFile            cf,
        List<BlockStatement> statements
    ) throws CompileException {

        // Create class/interface initialization method iff there is any initialization code.
        if (this.generatesCode2(statements)) {
            MethodDeclarator md = new MethodDeclarator(
                td.getLocation(),                                    // location
                null,                                                // optionalDocComment
                new Modifiers((short) (Mod.STATIC | Mod.PUBLIC)),    // modifiers
                null,                                                // optionalTypeParameters
                new PrimitiveType(td.getLocation(), Primitive.VOID), // type
                "<clinit>",                                          // name
                new FormalParameters(td.getLocation()),              // formalParameters
                new ReferenceType[0],                                // thrownExceptions
                null,                                                // defaultValue
                statements                                           // optionalStatements
            );
            md.setDeclaringType(td);
            this.com.hazelcast.com.ile(md, cf);
        }
    }

    /**
     * Compiles all of the types for this declaration
     * <p>
     *   NB: as a side effect this will fill in the synthetic field map
     * </p>
     */
    private void
    com.hazelcast.com.ileDeclaredMemberTypes(TypeDeclaration decl, ClassFile cf) throws CompileException {
        for (MemberTypeDeclaration mtd : decl.getMemberTypeDeclarations()) {
            this.com.hazelcast.com.ile(mtd);

            // Add InnerClasses attribute entry for member type declaration.
            short innerClassInfoIndex = cf.addConstantClassInfo(this.resolve(mtd).getDescriptor());
            short outerClassInfoIndex = cf.addConstantClassInfo(this.resolve(decl).getDescriptor());
            short innerNameIndex      = cf.addConstantUtf8Info(mtd.getName());
            assert mtd.getAnnotations().length == 0;
            cf.addInnerClassesAttributeEntry(new ClassFile.InnerClassesAttribute.Entry(
                innerClassInfoIndex,   // innerClassInfoIndex
                outerClassInfoIndex,   // outerClassInfoIndex
                innerNameIndex,        // innerNameIndex
                mtd.getModifierFlags() // innerClassAccessFlags
            ));
        }
    }

    /**
     * Compiles all of the methods for this declaration
     * <p>
     *   NB: as a side effect this will fill in the synthetic field map
     * </p
     */
    private void
    com.hazelcast.com.ileDeclaredMethods(TypeDeclaration typeDeclaration, ClassFile cf) throws CompileException {
        this.com.hazelcast.com.ileDeclaredMethods(typeDeclaration, cf, 0);
    }

    /**
     * Compiles methods for this declaration starting at <var>startPos</var>.
     *
     * @param startPos Starting parameter to fill in
     */
    private void
    com.hazelcast.com.ileDeclaredMethods(TypeDeclaration typeDeclaration, ClassFile cf, int startPos) throws CompileException {

        // Notice that as a side effect of com.hazelcast.com.iling methods, synthetic "class-dollar" methods (which implement class
        // literals) are generated on-the fly. Hence, we must not use an Iterator here.

        for (int i = startPos; i < typeDeclaration.getMethodDeclarations().size(); ++i) {
            MethodDeclarator md = (MethodDeclarator) typeDeclaration.getMethodDeclarations().get(i);

            IMethod m                     = this.toIMethod(md);
            boolean overrides             = this.overridesMethodFromSupertype(m, this.resolve(md.getDeclaringType()));
            boolean hasOverrideAnnotation = this.hasAnnotation(md, this.iClassLoader.TYPE_java_lang_Override);
            if (overrides && !hasOverrideAnnotation && !(typeDeclaration instanceof InterfaceDeclaration)) {
                this.warning("MO", "Missing @Override", md.getLocation());
            } else
            if (!overrides && hasOverrideAnnotation) {
                this.com.hazelcast.com.ileError("Method does not override a method declared in a supertype", md.getLocation());
            }

            this.com.hazelcast.com.ile(md, cf);
        }
    }

    private boolean
    hasAnnotation(FunctionDeclarator fd, IClass methodAnnotation) throws CompileException {
        Annotation[] methodAnnotations = fd.modifiers.annotations;
        for (Annotation ma : methodAnnotations) {
            if (this.getType(ma.getType()) == methodAnnotation) return true;
        }
        return false;
    }

    private boolean
    overridesMethodFromSupertype(IMethod m, IClass type) throws CompileException {

        // Check whether it overrides a method declared in the superclass (or any of its supertypes).
        {
            IClass superclass = type.getSuperclass();
            if (superclass != null && this.overridesMethod(m, superclass)) return true;
        }

        // Check whether it overrides a method declared in an interface (or any of its superinterfaces).
        IClass[] ifs = type.getInterfaces();
        for (IClass i : ifs) {
            if (this.overridesMethod(m, i)) return true;
        }

        // Special handling for interfaces that don't extend other interfaces: JLS7 dictates that these stem from
        // 'Object', but 'getSuperclass()' returns NULL for interfaces.
        if (ifs.length == 0 && type.isInterface()) {
            return this.overridesMethod(m, this.iClassLoader.TYPE_java_lang_Object);
        }

        return false;
    }

    /**
     * @return Whether <var>method</var> overrides a method of <var>type</var> or any of its supertypes
     */
    private boolean
    overridesMethod(IMethod method, IClass type) throws CompileException {

        // Check whether it overrides a method declared in THIS type.
        IMethod[] ms = type.getDeclaredIMethods(method.getName());
        for (IMethod m : ms) {
            if (Arrays.equals(method.getParameterTypes(), m.getParameterTypes())) return true;
        }

        // Check whether it overrides a method declared in a supertype.
        return this.overridesMethodFromSupertype(method, type);
    }

    /**
     * Generates and com.hazelcast.com.iles a bridge method with signature <var>base</var> that delegates to <var>override</var>.
     */
    private void
    generateBridgeMethod(ClassFile cf, IMethod base, IMethod override) throws CompileException {

        if (
            !base.getReturnType().isAssignableFrom(override.getReturnType())
            || override.getReturnType() == IClass.VOID
        ) {
            this.com.hazelcast.com.ileError(
                "The return type of \""
                + override
                + "\" is incompatible with that of \""
                + base
                + "\""
            );
            return;
        }


        ClassFile.MethodInfo mi = cf.addMethodInfo(
            (short) (Mod.PUBLIC | Mod.SYNTHETIC), // accessFlags
            base.getName(),
            base.getDescriptor()
        );

        // Add "Exceptions" attribute (JVMS 4.7.4).
        IClass[] thrownExceptions = base.getThrownExceptions();
        if (thrownExceptions.length > 0) {
            final short eani    = cf.addConstantUtf8Info("Exceptions");
            short[]     tecciis = new short[thrownExceptions.length];
            for (int i = 0; i < thrownExceptions.length; ++i) {
                tecciis[i] = cf.addConstantClassInfo(thrownExceptions[i].getDescriptor());
            }
            mi.addAttribute(new ClassFile.ExceptionsAttribute(eani, tecciis));
        }

        final CodeContext codeContext      = new CodeContext(mi.getClassFile(), base.toString());
        final CodeContext savedCodeContext = this.replaceCodeContext(codeContext);

        // Allocate all our local variables.
        codeContext.saveLocalVariables();
        codeContext.allocateLocalVariable((short) 1, "this", override.getDeclaringIClass());
        IClass[]            paramTypes = override.getParameterTypes();
        LocalVariableSlot[] locals     = new LocalVariableSlot[paramTypes.length];
        for (int i = 0; i < paramTypes.length; ++i) {
            locals[i] = codeContext.allocateLocalVariable(
                Descriptor.size(paramTypes[i].getDescriptor()),
                "param" + i,
                paramTypes[i]
            );
        }

        this.writeOpcode(Located.NOWHERE, Opcode.ALOAD_0);
        for (LocalVariableSlot l : locals) this.load(Located.NOWHERE, l.getType(), l.getSlotIndex());
        this.invoke(Located.NOWHERE, override);
        this.writeOpcode(Located.NOWHERE, Opcode.ARETURN);
        this.replaceCodeContext(savedCodeContext);
        codeContext.flowAnalysis(override.getName());

        // Add the code context as a code attribute to the MethodInfo.
        mi.addAttribute(new ClassFile.AttributeInfo(cf.addConstantUtf8Info("Code")) {

            @Override protected void
            storeBody(DataOutputStream dos) throws IOException {
                codeContext.storeCodeAttributeBody(dos, (short) 0, (short) 0, (short) 0);
            }
        });
    }

    /**
     * @return Whether this statement can com.hazelcast.com.lete normally (JLS7 14.1)
     */
    private boolean
    com.hazelcast.com.ile(BlockStatement bs) throws CompileException {

        Boolean result = (Boolean) bs.accept(new BlockStatementVisitor<Boolean, CompileException>() {

            // SUPPRESS CHECKSTYLE LineLengthCheck:23
            @Override public Boolean visitInitializer(Initializer i)                                                throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(i);    }
            @Override public Boolean visitFieldDeclaration(FieldDeclaration fd)                                     throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(fd);   }
            @Override public Boolean visitLabeledStatement(LabeledStatement ls)                                     throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(ls);   }
            @Override public Boolean visitBlock(Block b)                                                            throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(b);    }
            @Override public Boolean visitExpressionStatement(ExpressionStatement es)                               throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(es);   }
            @Override public Boolean visitIfStatement(IfStatement is)                                               throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(is);   }
            @Override public Boolean visitForStatement(ForStatement fs)                                             throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(fs);   }
            @Override public Boolean visitForEachStatement(ForEachStatement fes)                                    throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(fes);  }
            @Override public Boolean visitWhileStatement(WhileStatement ws)                                         throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(ws);   }
            @Override public Boolean visitTryStatement(TryStatement ts)                                             throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(ts);   }
            @Override public Boolean visitSwitchStatement(SwitchStatement ss)                                       throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(ss);   }
            @Override public Boolean visitSynchronizedStatement(SynchronizedStatement ss)                           throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(ss);   }
            @Override public Boolean visitDoStatement(DoStatement ds)                                               throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(ds);   }
            @Override public Boolean visitLocalVariableDeclarationStatement(LocalVariableDeclarationStatement lvds) throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(lvds); }
            @Override public Boolean visitReturnStatement(ReturnStatement rs)                                       throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(rs);   }
            @Override public Boolean visitThrowStatement(ThrowStatement ts)                                         throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(ts);   }
            @Override public Boolean visitBreakStatement(BreakStatement bs)                                         throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(bs);   }
            @Override public Boolean visitContinueStatement(ContinueStatement cs)                                   throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(cs);   }
            @Override public Boolean visitAssertStatement(AssertStatement as)                                       throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(as);   }
            @Override public Boolean visitEmptyStatement(EmptyStatement es)                                                                 { return UnitCompiler.this.com.hazelcast.com.ile2(es);   }
            @Override public Boolean visitLocalClassDeclarationStatement(LocalClassDeclarationStatement lcds)       throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(lcds); }
            @Override public Boolean visitAlternateConstructorInvocation(AlternateConstructorInvocation aci)        throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(aci);  }
            @Override public Boolean visitSuperConstructorInvocation(SuperConstructorInvocation sci)                throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile2(sci);  }
        });

        assert result != null;
        return result;
    }

    /**
     * Called to check whether the given {@link Rvalue} com.hazelcast.com.iles or not.
     *
     * @return Whether the block statement can com.hazelcast.com.lete normally
     */
    private boolean
    fakeCompile(BlockStatement bs) throws CompileException {

        Offset from = this.getCodeContext().newOffset();

        boolean ccn = this.com.hazelcast.com.ile(bs);

        Offset to = this.getCodeContext().newOffset();

        this.getCodeContext().removeCode(from, to);

        return ccn;
    }

    private CodeContext
    getCodeContext() {
        assert this.codeContext != null;
        return this.codeContext;
    }

    private boolean
    com.hazelcast.com.ile2(Initializer i) throws CompileException {
        return this.com.hazelcast.com.ile(i.block);
    }

    private boolean
    com.hazelcast.com.ile2(Block b) throws CompileException {
        this.getCodeContext().saveLocalVariables();
        try {
            return this.com.hazelcast.com.ileStatements(b.statements);
        } finally {
            this.getCodeContext().restoreLocalVariables();
        }
    }

    private boolean
    com.hazelcast.com.ileStatements(List<? extends BlockStatement> statements) throws CompileException {
        boolean previousStatementCanCompleteNormally = true;
        for (BlockStatement bs : statements) {
            if (!previousStatementCanCompleteNormally && this.generatesCode(bs)) {
                this.com.hazelcast.com.ileError("Statement is unreachable", bs.getLocation());
                break;
            }
            previousStatementCanCompleteNormally = this.com.hazelcast.com.ile(bs);
        }
        return previousStatementCanCompleteNormally;
    }

    private boolean
    com.hazelcast.com.ile2(DoStatement ds) throws CompileException {
        Object cvc = this.getConstantValue(ds.condition);
        if (cvc != UnitCompiler.NOT_CONSTANT) {
            if (Boolean.TRUE.equals(cvc)) {
                this.warning("DSTC", (
                    "Condition of DO statement is always TRUE; "
                    + "the proper way of declaring an unconditional loop is \"for (;;)\""
                ), ds.getLocation());
                return this.com.hazelcast.com.ileUnconditionalLoop(ds, ds.body, null);
            } else
            {
                this.warning("DSNR", "DO statement never repeats", ds.getLocation());
            }
        }

        final CodeContext.Offset bodyOffset = this.getCodeContext().newOffset();

        // Compile body.
        ds.whereToContinue = null;
        if (!this.com.hazelcast.com.ile(ds.body) && ds.whereToContinue == null) {
            this.warning("DSNTC", "\"do\" statement never tests its condition", ds.getLocation());

            Offset wtb = ds.whereToBreak;
            if (wtb == null) return false;

            wtb.set();
            ds.whereToBreak = null;

            return true;
        }
        if (ds.whereToContinue != null) {
            ds.whereToContinue.set();
            ds.whereToContinue = null;
        }

        // Compile condition.
        this.com.hazelcast.com.ileBoolean(ds.condition, bodyOffset, UnitCompiler.JUMP_IF_TRUE);

        if (ds.whereToBreak != null) {
            ds.whereToBreak.set();
            ds.whereToBreak = null;
        }

        return true;
    }

    private boolean
    com.hazelcast.com.ile2(ForStatement fs) throws CompileException {

        this.getCodeContext().saveLocalVariables();
        try {
            BlockStatement oi = fs.optionalInit;
            Rvalue[]       ou = fs.optionalUpdate;
            Rvalue         oc = fs.optionalCondition;

            // Compile initializer.
            if (oi != null) this.com.hazelcast.com.ile(oi);

            if (oc == null) {
                return this.com.hazelcast.com.ileUnconditionalLoop(fs, fs.body, ou);
            }

            Object cvc = this.getConstantValue(oc);
            if (cvc != UnitCompiler.NOT_CONSTANT) {
                if (Boolean.TRUE.equals(cvc)) {
                    this.warning("FSTC", (
                        "Condition of FOR statement is always TRUE; "
                        + "the proper way of declaring an unconditional loop is \"for (;;)\""
                    ), fs.getLocation());
                    return this.com.hazelcast.com.ileUnconditionalLoop(fs, fs.body, ou);
                } else
                {
                    this.warning("FSNR", "FOR statement never repeats", fs.getLocation());
                }
            }

            CodeContext.Offset toCondition = this.getCodeContext().new Offset();
            this.writeBranch(fs, Opcode.GOTO, toCondition);

            // Compile body.
            fs.whereToContinue = null;
            final CodeContext.Offset bodyOffset = this.getCodeContext().newOffset();
            boolean                  bodyCcn    = this.com.hazelcast.com.ile(fs.body);
            if (fs.whereToContinue != null) fs.whereToContinue.set();

            // Compile update.
            if (ou != null) {
                if (!bodyCcn && fs.whereToContinue == null) {
                    this.warning("FUUR", "For update is unreachable", fs.getLocation());
                } else
                {
                    for (Rvalue rv : ou) this.com.hazelcast.com.ile(rv);
                }
            }
            fs.whereToContinue = null;

            // Compile condition.
            toCondition.set();
            this.com.hazelcast.com.ileBoolean(oc, bodyOffset, UnitCompiler.JUMP_IF_TRUE);
        } finally {
            this.getCodeContext().restoreLocalVariables();
        }

        if (fs.whereToBreak != null) {
            fs.whereToBreak.set();
            fs.whereToBreak = null;
        }

        return true;
    }

    private boolean
    com.hazelcast.com.ile2(ForEachStatement fes) throws CompileException {
        IClass expressionType = this.getType(fes.expression);
        if (expressionType.isArray()) {
            this.getCodeContext().saveLocalVariables();
            try {

                // Allocate the local variable for the current element.
                LocalVariable elementLv = this.getLocalVariable(fes.currentElement, false);
                elementLv.setSlot(this.getCodeContext().allocateLocalVariable(
                    Descriptor.size(elementLv.type.getDescriptor()),
                    fes.currentElement.name,
                    elementLv.type
                ));

                // Compile initializer.
                this.com.hazelcast.com.ileGetValue(fes.expression);
                short expressionLv = this.getCodeContext().allocateLocalVariable((short) 1);
                this.store(fes.expression, expressionType, expressionLv);

                this.pushConstant(fes, 0);
                LocalVariable indexLv = new LocalVariable(false, IClass.INT);
                indexLv.setSlot(this.getCodeContext().allocateLocalVariable((short) 1, null, indexLv.type));
                this.store(fes, indexLv);

                CodeContext.Offset toCondition = this.getCodeContext().new Offset();
                this.writeBranch(fes, Opcode.GOTO, toCondition);

                // Compile the body.
                fes.whereToContinue = null;
                final CodeContext.Offset bodyOffset = this.getCodeContext().newOffset();

                this.load(fes, expressionType, expressionLv);
                this.load(fes, indexLv);
                IClass com.hazelcast.com.onentType = expressionType.getComponentType();
                assert com.hazelcast.com.onentType != null;
                this.writeOpcode(fes, Opcode.IALOAD + UnitCompiler.ilfdabcs(com.hazelcast.com.onentType));
                this.assignmentConversion(fes.currentElement, com.hazelcast.com.onentType, elementLv.type, null);
                this.store(fes, elementLv);

                boolean bodyCcn = this.com.hazelcast.com.ile(fes.body);
                if (fes.whereToContinue != null) fes.whereToContinue.set();

                // Compile update.
                if (!bodyCcn && fes.whereToContinue == null) {
                    this.warning("FUUR", "For update is unreachable", fes.getLocation());
                } else {
                    this.crement(fes, indexLv, "++");
                }
                fes.whereToContinue = null;

                // Compile condition.
                toCondition.set();
                this.load(fes, indexLv);
                this.load(fes, expressionType, expressionLv);
                this.writeOpcode(fes, Opcode.ARRAYLENGTH);
                this.writeBranch(fes, Opcode.IF_ICMPLT, bodyOffset);
            } finally {
                this.getCodeContext().restoreLocalVariables();
            }

            if (fes.whereToBreak != null) {
                fes.whereToBreak.set();
                fes.whereToBreak = null;
            }
        } else
        if (this.iClassLoader.TYPE_java_lang_Iterable.isAssignableFrom(expressionType)) {
            this.getCodeContext().saveLocalVariables();
            try {

                // Allocate the local variable for the current element.
                LocalVariable elementLv = this.getLocalVariable(fes.currentElement, false);
                elementLv.setSlot(this.getCodeContext().allocateLocalVariable(
                    (short) 1,
                    fes.currentElement.name,
                    elementLv.type
                ));

                // Compile initializer.
                this.com.hazelcast.com.ileGetValue(fes.expression);
                this.invoke(fes.expression, this.iClassLoader.METH_java_lang_Iterable__iterator);
                LocalVariable iteratorLv = new LocalVariable(false, this.iClassLoader.TYPE_java_util_Iterator);
                iteratorLv.setSlot(this.getCodeContext().allocateLocalVariable((short) 1, null, iteratorLv.type));
                this.store(fes, iteratorLv);

                CodeContext.Offset toCondition = this.getCodeContext().new Offset();
                this.writeBranch(fes, Opcode.GOTO, toCondition);

                // Compile the body.
                fes.whereToContinue = null;
                final CodeContext.Offset bodyOffset = this.getCodeContext().newOffset();

                this.load(fes, iteratorLv);
                this.invoke(fes.expression, this.iClassLoader.METH_java_util_Iterator__next);
                if (
                    !this.tryAssignmentConversion(
                        fes.currentElement,
                        this.iClassLoader.TYPE_java_lang_Object,
                        elementLv.type,
                        null
                    )
                    && !this.tryNarrowingReferenceConversion(
                        fes.currentElement,
                        this.iClassLoader.TYPE_java_lang_Object,
                        elementLv.type
                    )
                ) throw new AssertionError();
                this.store(fes, elementLv);

                boolean bodyCcn = this.com.hazelcast.com.ile(fes.body);
                if (fes.whereToContinue != null) fes.whereToContinue.set();

                // Compile update.
                if (!bodyCcn && fes.whereToContinue == null) {
                    this.warning("FUUR", "For update is unreachable", fes.getLocation());
                }
                fes.whereToContinue = null;

                // Compile condition.
                toCondition.set();
                this.load(fes, iteratorLv);
                this.invoke(fes.expression, this.iClassLoader.METH_java_util_Iterator__hasNext);
                this.writeBranch(fes, Opcode.IFNE, bodyOffset);
            } finally {
                this.getCodeContext().restoreLocalVariables();
            }

            if (fes.whereToBreak != null) {
                fes.whereToBreak.set();
                fes.whereToBreak = null;
            }
        } else
        {
            this.com.hazelcast.com.ileError("Cannot iterate over \"" + expressionType + "\"");
        }
        return true;
    }

    private boolean
    com.hazelcast.com.ile2(WhileStatement ws) throws CompileException {
        Object cvc = this.getConstantValue(ws.condition);
        if (cvc != UnitCompiler.NOT_CONSTANT) {
            if (Boolean.TRUE.equals(cvc)) {
                this.warning("WSTC", (
                    "Condition of WHILE statement is always TRUE; "
                    + "the proper way of declaring an unconditional loop is \"for (;;)\""
                ), ws.getLocation());
                return this.com.hazelcast.com.ileUnconditionalLoop(ws, ws.body, null);
            } else
            {
                this.warning("WSNR", "WHILE statement never repeats", ws.getLocation());
            }
        }

        // Compile body.
        Offset wtc = (ws.whereToContinue = this.getCodeContext().new Offset());
        this.writeBranch(ws, Opcode.GOTO, wtc);
        final CodeContext.Offset bodyOffset = this.getCodeContext().newOffset();
        this.com.hazelcast.com.ile(ws.body); // Return value (CCN) is ignored.
        assert ws.whereToContinue == wtc;
        wtc.set();
        ws.whereToContinue = null;

        // Compile condition.
        this.com.hazelcast.com.ileBoolean(ws.condition, bodyOffset, UnitCompiler.JUMP_IF_TRUE);

        if (ws.whereToBreak != null) {
            ws.whereToBreak.set();
            ws.whereToBreak = null;
        }
        return true;
    }

    private boolean
    com.hazelcast.com.ileUnconditionalLoop(ContinuableStatement cs, BlockStatement body, @Nullable Rvalue[] optionalUpdate)
    throws CompileException {
        if (optionalUpdate != null) return this.com.hazelcast.com.ileUnconditionalLoopWithUpdate(cs, body, optionalUpdate);

        // Compile body.
        Offset wtc = (cs.whereToContinue = this.getCodeContext().newOffset());
        if (this.com.hazelcast.com.ile(body)) this.writeBranch(cs, Opcode.GOTO, wtc);
        cs.whereToContinue = null;

        Offset wtb = cs.whereToBreak;
        if (wtb == null) return false;

        wtb.set();
        cs.whereToBreak = null;

        return true;
    }
    private boolean
    com.hazelcast.com.ileUnconditionalLoopWithUpdate(ContinuableStatement cs, BlockStatement body, Rvalue[] update)
    throws CompileException {

        // Compile body.
        cs.whereToContinue = null;
        final CodeContext.Offset bodyOffset = this.getCodeContext().newOffset();
        boolean                  bodyCcn    = this.com.hazelcast.com.ile(body);

        // Compile the "update".
        if (cs.whereToContinue != null) cs.whereToContinue.set();
        if (!bodyCcn && cs.whereToContinue == null) {
            this.warning("LUUR", "Loop update is unreachable", update[0].getLocation());
        } else
        {
            for (Rvalue rv : update) this.com.hazelcast.com.ile(rv);
            this.writeBranch(cs, Opcode.GOTO, bodyOffset);
        }
        cs.whereToContinue = null;

        Offset wtb = cs.whereToBreak;
        if (wtb == null) return false;
        wtb.set();

        cs.whereToBreak = null;

        return true;
    }

    private boolean
    com.hazelcast.com.ile2(LabeledStatement ls) throws CompileException {
        boolean canCompleteNormally = this.com.hazelcast.com.ile(ls.body);

        Offset wtb = ls.whereToBreak;
        if (wtb == null) return canCompleteNormally;

        wtb.set();

        ls.whereToBreak = null;

        return true;
    }

    private enum SwitchKind { INT, ENUM, STRING }

    private boolean
    com.hazelcast.com.ile2(SwitchStatement ss) throws CompileException {

        SwitchKind kind;
        short      ssvLvIndex = -1; // Only relevant if kind == STRING.

        // Compute condition.
        IClass switchExpressionType = this.com.hazelcast.com.ileGetValue(ss.condition);
        if (this.iClassLoader.TYPE_java_lang_String == switchExpressionType) {

            kind = SwitchKind.STRING;

            // Store the string value in a (hidden) local variable, because after we do the SWITCH
            // on the string's hash code, we need to check for string equality with the CASE
            // labels.
            this.dup(ss, 1);
            ssvLvIndex = this.getCodeContext().allocateLocalVariable((short) 1);
            this.store(
                ss,                                      // locatable
                this.iClassLoader.TYPE_java_lang_String, // lvType
                ssvLvIndex                               // lvIndex
            );

            this.invoke(ss, this.iClassLoader.METH_java_lang_String__hashCode);
        } else
        if (this.iClassLoader.TYPE_java_lang_Enum.isAssignableFrom(switchExpressionType)) {
            kind = SwitchKind.ENUM;
            this.invoke(ss, this.iClassLoader.METH_java_lang_Enum__ordinal);
        } else
        {
            kind = SwitchKind.INT;
            this.assignmentConversion(
                ss,                   // locatable
                switchExpressionType, // sourceType
                IClass.INT,           // targetType
                null                  // optionalConstantValue
            );
        }

        // Prepare the map of case labels to code offsets.
        TreeMap<Integer, CodeContext.Offset> caseLabelMap       = new TreeMap<Integer, CodeContext.Offset>();
        CodeContext.Offset                   defaultLabelOffset = null;
        CodeContext.Offset[]                 sbsgOffsets        = new CodeContext.Offset[ss.sbsgs.size()];
        for (int i = 0; i < ss.sbsgs.size(); ++i) {
            SwitchBlockStatementGroup sbsg = (SwitchBlockStatementGroup) ss.sbsgs.get(i);
            sbsgOffsets[i] = this.getCodeContext().new Offset();
            for (Rvalue caseLabel : sbsg.caseLabels) {

                Integer civ;

                switch (kind) {

                case ENUM:
                    CIV: {
                        if (!(caseLabel instanceof Java.AmbiguousName)) {
                            this.com.hazelcast.com.ileError("Case label must be an enum constant", caseLabel.getLocation());
                            civ = 99;
                            break;
                        }
                        String[] identifiers = ((Java.AmbiguousName) caseLabel).identifiers;
                        if (identifiers.length != 1) {
                            this.com.hazelcast.com.ileError("Case label must be a plain enum constant", caseLabel.getLocation());
                            civ = 99;
                            break;
                        }
                        String constantName = identifiers[0];

                        int ordinal = 0;
                        for (IField f : switchExpressionType.getDeclaredIFields()) {
                            if (f.getAccess() != Access.PUBLIC || !f.isStatic()) continue;
                            if (f.getName().equals(constantName)) {
                                civ = ordinal;
                                break CIV;
                            }
                            ordinal++;
                        }

                        this.com.hazelcast.com.ileError("Unknown enum constant \"" + constantName + "\"", caseLabel.getLocation());
                        civ = 99;
                    }

                    // Store in case label map.
                    if (caseLabelMap.containsKey(civ)) {
                        this.com.hazelcast.com.ileError("Duplicate \"case\" switch label value", caseLabel.getLocation());
                    }
                    caseLabelMap.put(civ, sbsgOffsets[i]);
                    break;

                case INT:
                    {
                        // Verify that case label value is a constant.
                        Object cv = this.getConstantValue(caseLabel);
                        if (cv == UnitCompiler.NOT_CONSTANT) {
                            this.com.hazelcast.com.ileError(
                                "Value of 'case' label does not pose a constant value",
                                caseLabel.getLocation()
                            );
                            civ = 99;
                            break;
                        }

                        // Convert char, byte, short, int to "Integer".
                        if (cv instanceof Integer) {
                            civ = (Integer) cv;
                        } else
                        if (cv instanceof Number) {
                            civ = new Integer(((Number) cv).intValue());
                        } else
                        if (cv instanceof Character) {
                            civ = new Integer(((Character) cv).charValue());
                        } else {
                            this.com.hazelcast.com.ileError(
                                "Value of case label must be a char, byte, short or int constant",
                                caseLabel.getLocation()
                            );
                            civ = new Integer(99);
                        }
                    }

                    // Store in case label map.
                    if (caseLabelMap.containsKey(civ)) {
                        this.com.hazelcast.com.ileError("Duplicate \"case\" switch label value", caseLabel.getLocation());
                    }
                    caseLabelMap.put(civ, sbsgOffsets[i]);
                    break;

                case STRING:
                    {

                        // Verify that the case label value is a string constant.
                        Object cv = this.getConstantValue(caseLabel);
                        if (!(cv instanceof String)) {
                            this.com.hazelcast.com.ileError(
                                "Value of 'case' label is not a string constant",
                                caseLabel.getLocation()
                            );
                            civ = 99;
                            break;
                        }

                        // Use the string constant's hash code as the SWITCH key.
                        civ = cv.hashCode();
                    }

                    // Store in case label map.
                    if (!caseLabelMap.containsKey(civ)) {
                        caseLabelMap.put(civ, this.getCodeContext().new Offset());
                    }
                    break;

                default:
                    throw new AssertionError(kind);
                }
            }

            if (sbsg.hasDefaultLabel) {
                if (defaultLabelOffset != null) {
                    this.com.hazelcast.com.ileError("Duplicate \"default\" switch label", sbsg.getLocation());
                }
                defaultLabelOffset = sbsgOffsets[i];
            }
        }
        if (defaultLabelOffset == null) defaultLabelOffset = this.getWhereToBreak(ss);

        // Generate TABLESWITCH or LOOKUPSWITCH instruction.
        CodeContext.Offset switchOffset = this.getCodeContext().newOffset();
        if (caseLabelMap.isEmpty()) {
            // Special case: SWITCH statement without CASE labels (but maybe a DEFAULT label).
            ;
        } else
        if (
            (Integer) caseLabelMap.firstKey() + caseLabelMap.size() // Beware of INT overflow!
            >= (Integer) caseLabelMap.lastKey() - caseLabelMap.size()
        ) {

            // The case label values are strictly consecutive or almost consecutive (at most 50%
            // 'gaps'), so let's use a TABLESWITCH.
            final int low  = (Integer) caseLabelMap.firstKey();
            final int high = (Integer) caseLabelMap.lastKey();

            this.writeOpcode(ss, Opcode.TABLESWITCH);
            new Padder(this.getCodeContext()).set();
            this.writeOffset(switchOffset, defaultLabelOffset);
            this.writeInt(low);
            this.writeInt(high);
            int cur = low;
            for (Map.Entry<Integer, CodeContext.Offset> me : caseLabelMap.entrySet()) {
                int                caseLabelValue  = (Integer) me.getKey();
                CodeContext.Offset caseLabelOffset = (CodeContext.Offset) me.getValue();

                while (cur < caseLabelValue) {
                    this.writeOffset(switchOffset, defaultLabelOffset);
                    ++cur;
                }
                this.writeOffset(switchOffset, caseLabelOffset);
                ++cur;
            }
        } else
        {

            // The case label values are not 'consecutive enough', so use a LOOKUPSWITCH.
            this.writeOpcode(ss, Opcode.LOOKUPSWITCH);
            new Padder(this.getCodeContext()).set();
            this.writeOffset(switchOffset, defaultLabelOffset);
            this.writeInt(caseLabelMap.size());
            for (Map.Entry<Integer, CodeContext.Offset> me : caseLabelMap.entrySet()) {
                this.writeInt((Integer) me.getKey());
                this.writeOffset(switchOffset, (CodeContext.Offset) me.getValue());
            }
        }

        if (kind == SwitchKind.STRING) {

            // For STRING SWITCH, we must generate extra code that checks for string equality --
            // the strings' hash codes are not globally unique (as, e.g. MD5).
            for (Entry<Integer, CodeContext.Offset> e : caseLabelMap.entrySet()) {
                final Integer            caseHashCode = (Integer) e.getKey();
                final CodeContext.Offset offset       = (CodeContext.Offset) e.getValue();

                offset.set();

                Set<String> caseLabelValues = new HashSet<String>();
                for (int i = 0; i < ss.sbsgs.size(); i++) {
                    SwitchBlockStatementGroup sbsg = (SwitchBlockStatementGroup) ss.sbsgs.get(i);

                    for (Rvalue caseLabel : sbsg.caseLabels) {

                        String cv = (String) this.getConstantValue(caseLabel);
                        assert cv != null;

                        if (!caseLabelValues.add(cv)) {
                            this.com.hazelcast.com.ileError(
                                "Duplicate case label \"" + cv + "\"",
                                caseLabel.getLocation()
                            );
                        }

                        if (cv.hashCode() != caseHashCode) continue;

                        this.load(sbsg, this.iClassLoader.TYPE_java_lang_String, ssvLvIndex);
                        this.pushConstant(caseLabel, cv);
                        this.invoke(caseLabel, this.iClassLoader.METH_java_lang_String__equals__java_lang_Object);
                        this.writeBranch(sbsg, Opcode.IFNE, sbsgOffsets[i]);
                    }
                }

                this.writeBranch(ss, Opcode.GOTO, defaultLabelOffset);
            }
        }

        // Compile statement groups.
        boolean canCompleteNormally = true;
        for (int i = 0; i < ss.sbsgs.size(); ++i) {
            SwitchBlockStatementGroup sbsg = (SwitchBlockStatementGroup) ss.sbsgs.get(i);
            sbsgOffsets[i].set();
            canCompleteNormally = true;
            for (BlockStatement bs : sbsg.blockStatements) {
                if (!canCompleteNormally) {
                    this.com.hazelcast.com.ileError("Statement is unreachable", bs.getLocation());
                    break;
                }
                canCompleteNormally = this.com.hazelcast.com.ile(bs);
            }
        }

        Offset wtb = ss.whereToBreak;
        if (wtb == null) return canCompleteNormally;

        wtb.set();

        ss.whereToBreak = null;

        return true;
    }

    private boolean
    com.hazelcast.com.ile2(BreakStatement bs) throws CompileException {

        // Find the broken statement.
        BreakableStatement brokenStatement = null;
        if (bs.optionalLabel == null) {
            for (
                Scope s = bs.getEnclosingScope();
                s instanceof Statement || s instanceof CatchClause;
                s = s.getEnclosingScope()
            ) {
                if (s instanceof BreakableStatement) {
                    brokenStatement = (BreakableStatement) s;
                    break;
                }
            }
            if (brokenStatement == null) {
                this.com.hazelcast.com.ileError("\"break\" statement is not enclosed by a breakable statement", bs.getLocation());
                return false;
            }
        } else {
            for (
                Scope s = bs.getEnclosingScope();
                s instanceof Statement || s instanceof CatchClause;
                s = s.getEnclosingScope()
            ) {
                if (s instanceof LabeledStatement) {
                    LabeledStatement ls = (LabeledStatement) s;
                    if (ls.label.equals(bs.optionalLabel)) {
                        brokenStatement = ls;
                        break;
                    }
                }
            }
            if (brokenStatement == null) {
                this.com.hazelcast.com.ileError((
                    "Statement \"break "
                    + bs.optionalLabel
                    + "\" is not enclosed by a breakable statement with label \""
                    + bs.optionalLabel
                    + "\""
                ), bs.getLocation());
                return false;
            }
        }

        this.leaveStatements(
            bs.getEnclosingScope(),              // from
            brokenStatement.getEnclosingScope(), // to
            null                                 // optionalStackValueType
        );
        this.writeBranch(bs, Opcode.GOTO, this.getWhereToBreak(brokenStatement));
        return false;
    }

    private boolean
    com.hazelcast.com.ile2(ContinueStatement cs) throws CompileException {

        // Find the continued statement.
        ContinuableStatement continuedStatement = null;
        if (cs.optionalLabel == null) {
            for (
                Scope s = cs.getEnclosingScope();
                s instanceof Statement || s instanceof CatchClause;
                s = s.getEnclosingScope()
            ) {
                if (s instanceof ContinuableStatement) {
                    continuedStatement = (ContinuableStatement) s;
                    break;
                }
            }
            if (continuedStatement == null) {
                this.com.hazelcast.com.ileError(
                    "\"continue\" statement is not enclosed by a continuable statement",
                    cs.getLocation()
                );
                return false;
            }
        } else {
            for (
                Scope s = cs.getEnclosingScope();
                s instanceof Statement || s instanceof CatchClause;
                s = s.getEnclosingScope()
            ) {
                if (s instanceof LabeledStatement) {
                    LabeledStatement ls = (LabeledStatement) s;
                    if (ls.label.equals(cs.optionalLabel)) {
                        Statement st = ls.body;
                        while (st instanceof LabeledStatement) st = ((LabeledStatement) st).body;
                        if (!(st instanceof ContinuableStatement)) {
                            this.com.hazelcast.com.ileError("Labeled statement is not continuable", st.getLocation());
                            return false;
                        }
                        continuedStatement = (ContinuableStatement) st;
                        break;
                    }
                }
            }
            if (continuedStatement == null) {
                this.com.hazelcast.com.ileError((
                    "Statement \"continue "
                    + cs.optionalLabel
                    + "\" is not enclosed by a continuable statement with label \""
                    + cs.optionalLabel
                    + "\""
                ), cs.getLocation());
                return false;
            }
        }

        Offset wtc = continuedStatement.whereToContinue;
        if (wtc == null) {
            wtc = (continuedStatement.whereToContinue = this.getCodeContext().new Offset());
        }

        this.leaveStatements(
            cs.getEnclosingScope(),                 // from
            continuedStatement.getEnclosingScope(), // to
            null                                    // optionalStackValueType
        );

        this.writeBranch(cs, Opcode.GOTO, wtc);

        return false;
    }

    private boolean
    com.hazelcast.com.ile2(AssertStatement as) throws CompileException {

        // assert expression1;
        //   if (!expression1) throw new AssertionError();
        // assert expression1 : expression2;
        //   if (!expression1) throw new AssertionError(expression2);
        CodeContext.Offset end = this.getCodeContext().new Offset();
        try {
            this.com.hazelcast.com.ileBoolean(as.expression1, end, UnitCompiler.JUMP_IF_TRUE);

            this.writeOpcode(as, Opcode.NEW);
            this.writeConstantClassInfo(Descriptor.JAVA_LANG_ASSERTIONERROR);
            this.writeOpcode(as, Opcode.DUP);

            Rvalue[] arguments = (
                as.optionalExpression2 == null
                ? new Rvalue[0]
                : new Rvalue[] { as.optionalExpression2 }
            );
            this.invokeConstructor(
                as,                                              // locatable
                as,                                              // scope
                null,                                            // optionalEnclosingInstance
                this.iClassLoader.TYPE_java_lang_AssertionError, // targetClass
                arguments                                        // arguments
            );
            this.writeOpcode(as, Opcode.ATHROW);
        } finally {
            end.set();
        }
        return true;
    }

    @SuppressWarnings("static-method") private boolean
    com.hazelcast.com.ile2(EmptyStatement es) { return true; }

    private boolean
    com.hazelcast.com.ile2(ExpressionStatement ee) throws CompileException {
        this.com.hazelcast.com.ile(ee.rvalue);
        return true;
    }

    private boolean
    com.hazelcast.com.ile2(FieldDeclaration fd) throws CompileException {
        for (VariableDeclarator vd : fd.variableDeclarators) {

            ArrayInitializerOrRvalue initializer = this.getNonConstantFinalInitializer(fd, vd);
            if (initializer == null) continue;

            // TODO: Compile annotations on fields.
//            assert fd.modifiers.annotations.length == 0 : fd.getLocation();

            if (!Mod.isStatic(fd.modifiers.accessFlags)) this.writeOpcode(fd, Opcode.ALOAD_0);
            IClass fieldType = this.getType(fd.type);
            if (initializer instanceof Rvalue) {
                Rvalue rvalue          = (Rvalue) initializer;
                IClass initializerType = this.com.hazelcast.com.ileGetValue(rvalue);
                fieldType = fieldType.getArrayIClass(vd.brackets, this.iClassLoader.TYPE_java_lang_Object);
                this.assignmentConversion(
                    fd,                           // locatable
                    initializerType,              // sourceType
                    fieldType,                    // targetType
                    this.getConstantValue(rvalue) // optionalConstantValue
                );
            } else
            if (initializer instanceof ArrayInitializer) {
                this.com.hazelcast.com.ileGetValue((ArrayInitializer) initializer, fieldType);
            } else
            {
                throw new InternalCompilerException(
                    "Unexpected array initializer or rvalue class "
                    + initializer.getClass().getName()
                );
            }

            // No need to check accessibility here.
            ;

            // TODO: Compile annotations on fields.
//            assert fd.modifiers.annotations.length == 0;

            IField iField = this.resolve(fd.getDeclaringType()).getDeclaredIField(vd.name);
            assert iField != null : fd.getDeclaringType() + " has no field " + vd.name;
            this.putfield(fd, iField);
        }
        return true;
    }

    private boolean
    com.hazelcast.com.ile2(IfStatement is) throws CompileException {
        Object         cv = this.getConstantValue(is.condition);
        BlockStatement es = (
            is.elseStatement != null
            ? is.elseStatement
            : new EmptyStatement(is.thenStatement.getLocation())
        );
        if (cv instanceof Boolean) {

            // Constant condition.
            this.fakeCompile(is.condition);
            BlockStatement seeingStatement, blindStatement;
            if (((Boolean) cv).booleanValue()) {
                seeingStatement = is.thenStatement;
                blindStatement  = es;
            } else {
                seeingStatement = es;
                blindStatement  = is.thenStatement;
            }

            // Compile the seeing statement.
            final CodeContext.Inserter ins   = this.getCodeContext().newInserter();
            boolean                    ssccn = this.com.hazelcast.com.ile(seeingStatement);
            boolean                    bsccn = this.fakeCompile(blindStatement);
            if (ssccn) return true;
            if (!bsccn) return false;

            // Hm... the "seeing statement" cannot com.hazelcast.com.lete normally, but the "blind statement" can. Things are getting
            // com.hazelcast.com.licated here! The robust solution is to com.hazelcast.com.ile the constant-condition-IF statement as a
            // non-constant-condition-IF statement. As an optimization, iff the IF-statement is enclosed ONLY by blocks,
            // then the remaining bytecode can be written to a "fake" code context, i.e. be thrown away.

            // Compile constant-condition-IF statement as non-constant-condition-IF statement.
            CodeContext.Offset off = this.getCodeContext().newOffset();

            this.getCodeContext().pushInserter(ins);
            try {
                this.pushConstant(is, Boolean.FALSE);
                this.writeBranch(is, Opcode.IFNE, off);
            } finally {
                this.getCodeContext().popInserter();
            }

            return true;
        }

        // Non-constant condition.
        if (this.generatesCode(is.thenStatement)) {
            if (this.generatesCode(es)) {

                // if (expression) statement else statement
                CodeContext.Offset eso = this.getCodeContext().new Offset();
                CodeContext.Offset end = this.getCodeContext().new Offset();
                this.com.hazelcast.com.ileBoolean(is.condition, eso, UnitCompiler.JUMP_IF_FALSE);
                boolean tsccn = this.com.hazelcast.com.ile(is.thenStatement);
                if (tsccn) this.writeBranch(is, Opcode.GOTO, end);
                eso.set();
                boolean esccn = this.com.hazelcast.com.ile(es);
                end.set();
                return tsccn || esccn;
            } else {

                // if (expression) statement else ;
                CodeContext.Offset end = this.getCodeContext().new Offset();
                this.com.hazelcast.com.ileBoolean(is.condition, end, UnitCompiler.JUMP_IF_FALSE);
                this.com.hazelcast.com.ile(is.thenStatement);
                end.set();
                return true;
            }
        } else {
            if (this.generatesCode(es)) {

                // if (expression) ; else statement
                CodeContext.Offset end = this.getCodeContext().new Offset();
                this.com.hazelcast.com.ileBoolean(is.condition, end, UnitCompiler.JUMP_IF_TRUE);
                this.com.hazelcast.com.ile(es);
                end.set();
                return true;
            } else {

                // if (expression) ; else ;
                IClass conditionType = this.com.hazelcast.com.ileGetValue(is.condition);
                if (conditionType != IClass.BOOLEAN) this.com.hazelcast.com.ileError("Not a boolean expression", is.getLocation());
                this.pop(is, conditionType);
                return true;
            }
        }
    }

    private boolean
    com.hazelcast.com.ile2(LocalClassDeclarationStatement lcds) throws CompileException {

        // Check for redefinition.
        LocalClassDeclaration otherLcd = UnitCompiler.findLocalClassDeclaration(lcds, lcds.lcd.name);
        if (otherLcd != null && otherLcd != lcds.lcd) {
            this.com.hazelcast.com.ileError(
                "Redeclaration of local class \""
                + lcds.lcd.name
                + "\"; previously declared in "
                + otherLcd.getLocation()
            );
        }

        this.com.hazelcast.com.ile(lcds.lcd);
        return true;
    }

    /**
     * Finds a local class declared in any block enclosing the given block statement.
     */
    @Nullable private static LocalClassDeclaration
    findLocalClassDeclaration(Scope s, String name) {

        if (s instanceof CompilationUnit) return null;

        for (;;) {
            Scope es = s.getEnclosingScope();
            if (es instanceof CompilationUnit) break;
            if (
                s instanceof BlockStatement
                && (es instanceof Block || es instanceof FunctionDeclarator)
            ) {
                BlockStatement                 bs         = (BlockStatement) s;
                List<? extends BlockStatement> statements = (
                    es instanceof BlockStatement
                    ? ((Block) es).statements
                    : ((FunctionDeclarator) es).optionalStatements
                );
                if (statements != null) {
                    for (BlockStatement bs2 : statements) {
                        if (bs2 instanceof LocalClassDeclarationStatement) {
                            LocalClassDeclarationStatement lcds = ((LocalClassDeclarationStatement) bs2);
                            if (lcds.lcd.name.equals(name)) return lcds.lcd;
                        }
                        if (bs2 == bs) break;
                    }
                }
            }
            s = es;
        }

        return null;
    }

    private boolean
    com.hazelcast.com.ile2(LocalVariableDeclarationStatement lvds) throws CompileException {

        // Ignore "lvds.modifiers.annotations" here.

        if ((lvds.modifiers.accessFlags & ~Mod.FINAL) != 0) {
            this.com.hazelcast.com.ileError(
                "The only allowed modifier in local variable declarations is \"final\"",
                lvds.getLocation()
            );
        }

        for (VariableDeclarator vd : lvds.variableDeclarators) {

            LocalVariable lv = this.getLocalVariable(lvds, vd);
            lv.setSlot(
                this.getCodeContext().allocateLocalVariable(Descriptor.size(lv.type.getDescriptor()), vd.name, lv.type)
            );

            ArrayInitializerOrRvalue oi = vd.optionalInitializer;
            if (oi != null) {
                if (oi instanceof Rvalue) {
                    Rvalue rhs = (Rvalue) oi;
                    this.assignmentConversion(
                        lvds,                      // locatable
                        this.com.hazelcast.com.ileGetValue(rhs), // sourceType
                        lv.type,                   // targetType
                        this.getConstantValue(rhs) // optionalConstantValue
                    );
                } else
                if (oi instanceof ArrayInitializer) {
                    this.com.hazelcast.com.ileGetValue((ArrayInitializer) oi, lv.type);
                } else
                {
                    throw new InternalCompilerException(
                        "Unexpected rvalue or array initialized class "
                        + oi.getClass().getName()
                    );
                }
                this.store(lvds, lv);
            }
        }
        return true;
    }

    /**
     * @return The {@link LocalVariable} corresponding with the local variable declaration/declarator
     */
    public LocalVariable
    getLocalVariable(LocalVariableDeclarationStatement lvds, VariableDeclarator vd) throws CompileException {

        if (vd.localVariable != null) return vd.localVariable;

        // Determine variable type.
        Type variableType = lvds.type;
        for (int k = 0; k < vd.brackets; ++k) variableType = new ArrayType(variableType);

        // Ignore "lvds.modifiers.annotations".

        return (vd.localVariable = new LocalVariable(
            Mod.isFinal(lvds.modifiers.accessFlags), // finaL
            this.getType(variableType)               // type
        ));
    }

    private boolean
    com.hazelcast.com.ile2(ReturnStatement rs) throws CompileException {

        // Determine enclosing block, function and com.hazelcast.com.ilation Unit.
        FunctionDeclarator enclosingFunction = null;
        {
            Scope s = rs.getEnclosingScope();
            while (s instanceof Statement || s instanceof CatchClause) s = s.getEnclosingScope();
            enclosingFunction = (FunctionDeclarator) s;
        }

        Rvalue orv = rs.optionalReturnValue;

        IClass returnType = this.getReturnType(enclosingFunction);
        if (returnType == IClass.VOID) {
            if (orv != null) this.com.hazelcast.com.ileError("Method must not return a value", rs.getLocation());
            this.leaveStatements(
                rs.getEnclosingScope(), // from
                enclosingFunction,      // to
                null                    // optionalStackValueType
            );
            this.writeOpcode(rs, Opcode.RETURN);
            return false;
        }

        if (orv == null) {
            this.com.hazelcast.com.ileError("Method must return a value", rs.getLocation());
            return false;
        }
        IClass type = this.com.hazelcast.com.ileGetValue(orv);
        this.assignmentConversion(
            rs,                        // locatable
            type,                      // sourceType
            returnType,                // targetType
            this.getConstantValue(orv) // optionalConstantValue
        );

        this.leaveStatements(
            rs.getEnclosingScope(), // from
            enclosingFunction,      // to
            returnType              // optionalStackValueType
        );
        this.writeOpcode(rs, Opcode.IRETURN + UnitCompiler.ilfda(returnType));
        return false;
    }

    private boolean
    com.hazelcast.com.ile2(SynchronizedStatement ss) throws CompileException {

        // Evaluate monitor object expression.
        if (!this.iClassLoader.TYPE_java_lang_Object.isAssignableFrom(this.com.hazelcast.com.ileGetValue(ss.expression))) {
            this.com.hazelcast.com.ileError(
                "Monitor object of \"synchronized\" statement is not a subclass of \"Object\"",
                ss.getLocation()
            );
        }

        this.getCodeContext().saveLocalVariables();
        boolean canCompleteNormally = false;
        try {

            // Allocate a local variable for the monitor object.
            ss.monitorLvIndex = this.getCodeContext().allocateLocalVariable((short) 1);

            // Store the monitor object.
            this.writeOpcode(ss, Opcode.DUP);
            this.store(ss, this.iClassLoader.TYPE_java_lang_Object, ss.monitorLvIndex);

            // Create lock on the monitor object.
            this.writeOpcode(ss, Opcode.MONITORENTER);

            // Compile the statement body.
            final CodeContext.Offset monitorExitOffset = this.getCodeContext().new Offset();
            final CodeContext.Offset beginningOfBody   = this.getCodeContext().newOffset();
            canCompleteNormally = this.com.hazelcast.com.ile(ss.body);
            if (canCompleteNormally) {
                this.writeBranch(ss, Opcode.GOTO, monitorExitOffset);
            }

            // Generate the exception handler.
            CodeContext.Offset here = this.getCodeContext().newOffset();
            this.getCodeContext().addExceptionTableEntry(
                beginningOfBody, // startPC
                here,            // endPC
                here,            // handlerPC
                null             // catchTypeFD
            );
            this.leave(ss, this.iClassLoader.TYPE_java_lang_Throwable);
            this.writeOpcode(ss, Opcode.ATHROW);

            // Unlock monitor object.
            if (canCompleteNormally) {
                monitorExitOffset.set();
                this.leave(ss, null);
            }
        } finally {
            this.getCodeContext().restoreLocalVariables();
        }

        return canCompleteNormally;
    }

    private boolean
    com.hazelcast.com.ile2(ThrowStatement ts) throws CompileException {
        IClass expressionType = this.com.hazelcast.com.ileGetValue(ts.expression);
        this.checkThrownException(
            ts,                    // locatable
            expressionType,        // type
            ts.getEnclosingScope() // scope
        );
        this.writeOpcode(ts, Opcode.ATHROW);
        return false;
    }

    /**
     * Interface for delayed code generation.
     */
    interface Compilable2 { boolean com.hazelcast.com.ile() throws CompileException; }

    private boolean
    com.hazelcast.com.ile2(final TryStatement ts) throws CompileException {

        return this.com.hazelcast.com.ileTryCatchFinallyWithResources(
            ts,                 // tryStatement
            ts.resources,       // resources
            new Compilable2() { // com.hazelcast.com.ileBody

                @Override public boolean
                com.hazelcast.com.ile() throws CompileException { return UnitCompiler.this.com.hazelcast.com.ile(ts.body); }
            },
            ts.finallY          // finallY
        );
    }

    /**
     * Generates code for a TRY statement with (possibly zero) resources and an (optional) FINALLY clause.
     *
     * @return Whether the code can com.hazelcast.com.lete normally
     */
    private boolean
    com.hazelcast.com.ileTryCatchFinallyWithResources(
        final TryStatement          ts,
        List<TryStatement.Resource> resources,
        final Compilable2           com.hazelcast.com.ileBody,
        @Nullable final Block       finallY
    ) throws CompileException {

        if (resources.isEmpty()) {

            // Short-circuit for zero resources.
            return this.com.hazelcast.com.ileTryCatchFinally(ts, com.hazelcast.com.ileBody, finallY);
        }

        // Prepare recursion for all declared resources.
        TryStatement.Resource             firstResource      = (TryStatement.Resource) resources.get(0);
        final List<TryStatement.Resource> followingResources = resources.subList(1, resources.size());

        final Location loc = firstResource.getLocation();
        final IClass   tt  = this.iClassLoader.TYPE_java_lang_Throwable;

        this.getCodeContext().saveLocalVariables();
        try {

            LocalVariable
            identifier = (LocalVariable) firstResource.accept(
                new Visitor.TryStatementResourceVisitor<LocalVariable, CompileException>() {

                    @Override @Nullable public LocalVariable
                    visitLocalVariableDeclaratorResource(LocalVariableDeclaratorResource lvdr) throws CompileException {

                        // final {VariableModifierNoFinal} R Identifier = Expression
                        IClass        lvType = UnitCompiler.this.getType(lvdr.type);
                        LocalVariable result = new LocalVariable(true, lvType);
                        result.setSlot(
                            UnitCompiler.this.getCodeContext().allocateLocalVariable(
                                Descriptor.size(lvType.getDescriptor()), // size
                                null,                                    // name
                                lvType                                   // type
                            )
                        );
                        ArrayInitializerOrRvalue oi = lvdr.variableDeclarator.optionalInitializer;
                        if (oi instanceof Rvalue) {
                            UnitCompiler.this.com.hazelcast.com.ileGetValue((Rvalue) oi);
                        } else
                        if (oi instanceof ArrayInitializer) {
                            UnitCompiler.this.com.hazelcast.com.ileGetValue((ArrayInitializer) oi, lvType);
                        } else
                        {
                            throw new InternalCompilerException(String.valueOf(oi));
                        }
                        UnitCompiler.this.store(ts, result);
                        return result;
                    }

                    @Override @Nullable public LocalVariable
                    visitVariableAccessResource(VariableAccessResource var) throws CompileException {

                        // Expression
                        if (
                            !UnitCompiler.this.options.contains(
                                JaninoOption.EXPRESSIONS_IN_TRY_WITH_RESOURCES_ALLOWED
                            )
                            && !(var.variableAccess instanceof AmbiguousName)
                            && !(var.variableAccess instanceof FieldAccessExpression)
                            && !(var.variableAccess instanceof SuperclassFieldAccessExpression)
                        ) {
                            throw new CompileException(
                                var.variableAccess.getClass().getSimpleName() + " rvalue not allowed as a resource",
                                var.getLocation()
                            );
                        }
                        IClass        lvType = UnitCompiler.this.com.hazelcast.com.ileGetValue(var.variableAccess);
                        LocalVariable result = new LocalVariable(true, lvType);
                        result.setSlot(
                            UnitCompiler.this.getCodeContext().allocateLocalVariable(
                                Descriptor.size(lvType.getDescriptor()), // size
                                null,                                    // name
                                lvType                                   // type
                            )
                        );
                        UnitCompiler.this.store(ts, result);
                        return result;
                    }
                }
            );
            assert identifier != null;

            // Throwable #primaryExc = null;
            LocalVariable primaryExc = new LocalVariable(true, tt);
            primaryExc.setSlot(
                this.getCodeContext().allocateLocalVariable(
                    Descriptor.size(tt.getDescriptor()),
                    null, // name
                    tt
                )
            );
            this.pushConstant(ts, null);
            this.store(ts, primaryExc);

            FormalParameter suppressedException = new FormalParameter(
                loc,                        // location
                false,                      // finaL
                new SimpleType(loc, tt),    // type
                "___"                       // name
            );

            // Generate the FINALLY clause for the TRY-with-resources statement; see JLS9 14.20.3.1.
            // if (Identifier != null) {
            //     if (#primaryExc != null) {
            //         try {
            //             Identifier.close();
            //         } catch (Throwable #suppressedExc) {
            //             // Only iff Java >= 7:
            //             #primaryExc.addSuppressed(#suppressedExc);
            //         }
            //     } else {
            //         Identifier.close();
            //     }
            // }
            BlockStatement afterClose = (
                this.iClassLoader.METH_java_lang_Throwable__addSuppressed == null
                ? (BlockStatement) new EmptyStatement(loc)
                : new ExpressionStatement(
                    new MethodInvocation(
                        loc,                                      // location
                        new LocalVariableAccess(loc, primaryExc), // optionalTarget
                        "addSuppressed",                          // methodName
                        new Rvalue[] {                            // arguments
                            new LocalVariableAccess(loc, this.getLocalVariable(suppressedException)),
                        }
                    )
                )
            );
            BlockStatement f = new IfStatement(
                loc,                 // location
                new BinaryOperation( // condition
                    loc,                                      // location
                    new LocalVariableAccess(loc, identifier), // lhs
                    "!=",                                     // operator
                    new NullLiteral(loc)                      // rhs
                ),
                new IfStatement(     // thenStatement
                    loc,                         // location
                    new BinaryOperation(         // condition
                        loc,                                      // location
                        new LocalVariableAccess(loc, primaryExc), // lhs
                        "!=",                                     // operator
                        new NullLiteral(loc)                      // rhs
                    ),
                    new TryStatement(            // thenStatement
                        loc,                                       // location
                        new ExpressionStatement(                   // body
                            new MethodInvocation(loc, new LocalVariableAccess(loc, identifier), "close", new Rvalue[0])
                        ),
                        Collections.singletonList(new CatchClause( // catchClauses
                            loc,                   // location
                            suppressedException,   // caughtException
                            afterClose             // body
                        ))
                    ),
                    new ExpressionStatement(     // elseStatement
                        new MethodInvocation(loc, new LocalVariableAccess(loc, identifier), "close", new Rvalue[0])
                    )
                )
            );
            f.setEnclosingScope(ts);

            return this.com.hazelcast.com.ileTryCatchFinally(
                ts,                 // tryStatement
                new Compilable2() { // com.hazelcast.com.ileBody

                    @Override public boolean
                    com.hazelcast.com.ile() throws CompileException {
                        return UnitCompiler.this.com.hazelcast.com.ileTryCatchFinallyWithResources(
                            ts,
                            followingResources,
                            com.hazelcast.com.ileBody,
                            finallY
                        );
                    }
                },
                f                   // finallY
            );
        } finally {
            this.getCodeContext().restoreLocalVariables();
        }
    }

    /**
     * Generates code for a TRY statement without resources, but with an (optional) FINALLY clause.
     *
     * @return Whether the code can com.hazelcast.com.lete normally
     */
    private boolean
    com.hazelcast.com.ileTryCatchFinally(
        final TryStatement       ts,
        final Compilable2        com.hazelcast.com.ileBody,
        @Nullable BlockStatement finallY
    ) throws CompileException {

        final CodeContext.Offset beginningOfBody = this.getCodeContext().newOffset();
        final CodeContext.Offset afterStatement  = this.getCodeContext().new Offset();

        boolean canCompleteNormally;

        if (finallY == null) {
            canCompleteNormally = this.com.hazelcast.com.ileTryCatch(ts, com.hazelcast.com.ileBody, beginningOfBody, afterStatement);
        } else {

            // Compile a TRY statement *with* a FINALLY clause.

            this.getCodeContext().saveLocalVariables();
            try {

                final Offset fo = (ts.finallyOffset = this.getCodeContext().new Offset());

                // Allocate a LV for the JSR of the FINALLY clause.
                //
                // Notice:
                //   For unclear reasons, this variable must not overlap with any of the body's variables (although the
                //   body's variables are out of scope when it com.hazelcast.com.s to the FINALLY clause!?), otherwise you get
                //     java.lang.VerifyError: ... Accessing value from uninitialized local variable 4
                //   See bug #56.
                final short pcLvIndex = this.getCodeContext().allocateLocalVariable((short) 1);

                canCompleteNormally = this.com.hazelcast.com.ileTryCatch(ts, new Compilable2() {

                    @Override public boolean
                    com.hazelcast.com.ile() throws CompileException {
                        boolean canCompleteNormally = com.hazelcast.com.ileBody.com.hazelcast.com.ile();
                        if (canCompleteNormally) {
                            UnitCompiler.this.writeBranch(ts, Opcode.JSR, fo);
                        }
                        return canCompleteNormally;
                    }
                }, beginningOfBody, afterStatement);

                // Generate the "catch (Throwable) {" clause that invokes the FINALLY subroutine.
                this.getCodeContext().saveLocalVariables();
                try {

                    CodeContext.Offset here = this.getCodeContext().newOffset();
                    this.getCodeContext().addExceptionTableEntry(
                        beginningOfBody, // startPC
                        here,            // endPC
                        here,            // handlerPC
                        null             // catchTypeFD
                    );

                    // Save the exception object in an anonymous local variable.
                    short evi = this.getCodeContext().allocateLocalVariable((short) 1);
                    this.store(
                        finallY,                                 // locatable
                        this.iClassLoader.TYPE_java_lang_Object, // lvType
                        evi                                      // lvIndex
                    );
                    this.writeBranch(finallY, Opcode.JSR, fo);
                    this.load(
                        finallY,                                 // locatable
                        this.iClassLoader.TYPE_java_lang_Object, // type
                        evi                                      // index
                    );
                    this.writeOpcode(finallY, Opcode.ATHROW);

                    // Generate the "finally" subroutine.
                    fo.set();
                    ts.finallyOffset = null;

                    this.store(
                        finallY,                                 // locatable
                        this.iClassLoader.TYPE_java_lang_Object, // lvType
                        pcLvIndex                                // lvIndex
                    );
                    if (this.com.hazelcast.com.ile(finallY)) {
                        if (pcLvIndex > 255) {
                            this.writeOpcode(finallY, Opcode.WIDE);
                            this.writeOpcode(finallY, Opcode.RET);
                            this.writeShort(pcLvIndex);
                        } else {
                            this.writeOpcode(finallY, Opcode.RET);
                            this.writeByte(pcLvIndex);
                        }
                    }
                } finally {

                    // The exception object local variable allocated above MUST NOT BE RELEASED until after the FINALLY
                    // block is com.hazelcast.com.iled, for otherwise you get
                    //   java.lang.VerifyError: ... Accessing value from uninitialized register 7
                    this.getCodeContext().restoreLocalVariables();
                }
            } finally {
                this.getCodeContext().restoreLocalVariables();
            }
        }

        afterStatement.set();
//        if (canCompleteNormally) this.leave(ts, null);
        return canCompleteNormally;
    }

    /**
     * Generates code for a TRY statement without resources and without a FINALLY clause.
     *
     * @return Whether the code can com.hazelcast.com.lete normally
     */
    private boolean
    com.hazelcast.com.ileTryCatch(
        TryStatement             tryStatement,
        Compilable2              com.hazelcast.com.ileBody,
        final CodeContext.Offset beginningOfBody,
        final CodeContext.Offset afterStatement
    ) throws CompileException {

        // Initialize all catch clauses as "unreachable" only to check later that they ARE indeed reachable.
        for (CatchClause catchClause : tryStatement.catchClauses) {
            IClass caughtExceptionType = this.getType(catchClause.caughtException.type);
            catchClause.reachable = (
                // Superclass or subclass of "java.lang.Error"?
                this.iClassLoader.TYPE_java_lang_Error.isAssignableFrom(caughtExceptionType)
                || caughtExceptionType.isAssignableFrom(this.iClassLoader.TYPE_java_lang_Error)
                // Superclass or subclass of "java.lang.RuntimeException"?
                || this.iClassLoader.TYPE_java_lang_RuntimeException.isAssignableFrom(caughtExceptionType)
                || caughtExceptionType.isAssignableFrom(this.iClassLoader.TYPE_java_lang_RuntimeException)
            );
        }

        boolean canCompleteNormally = com.hazelcast.com.ileBody.com.hazelcast.com.ile();

        CodeContext.Offset afterBody = this.getCodeContext().newOffset();

        if (canCompleteNormally) {
//            if (tryStatement.finallyOffset != null) {
//                this.writeBranch(tryStatement, Opcode.JSR, tryStatement.finallyOffset);
//            }

            this.writeBranch(tryStatement, Opcode.GOTO, afterStatement);
        }

        if (beginningOfBody.offset != afterBody.offset) { // Avoid zero-length exception table entries.
            this.getCodeContext().saveLocalVariables();
            try {
                for (int i = 0; i < tryStatement.catchClauses.size(); ++i) {
                    try {
                        this.getCodeContext().saveLocalVariables();

                        CatchClause catchClause         = (CatchClause) tryStatement.catchClauses.get(i);
                        IClass      caughtExceptionType = this.getType(catchClause.caughtException.type);

                        // Verify that the CATCH clause is reachable.
                        if (!catchClause.reachable) {
                            this.com.hazelcast.com.ileError("Catch clause is unreachable", catchClause.getLocation());
                        }

                        // Allocate the "exception variable".
                        LocalVariableSlot exceptionVarSlot = this.getCodeContext().allocateLocalVariable(
                            (short) 1,
                            catchClause.caughtException.name,
                            caughtExceptionType
                        );
                        final short evi = exceptionVarSlot.getSlotIndex();

                        // Kludge: Treat the exception variable like a local variable of the catch clause body.
                        this.getLocalVariable(catchClause.caughtException).setSlot(exceptionVarSlot);

                        this.getCodeContext().addExceptionTableEntry(
                            beginningOfBody,                    // startPC
                            afterBody,                          // endPC
                            this.getCodeContext().newOffset(),  // handlerPC
                            caughtExceptionType.getDescriptor() // catchTypeFD
                        );
                        this.store(
                            catchClause,         // locatable
                            caughtExceptionType, // lvType
                            evi                  // lvIndex
                        );

                        if (this.com.hazelcast.com.ile(catchClause.body)) {
                            canCompleteNormally = true;
                            if (tryStatement.finallyOffset != null) {
                                this.writeBranch(tryStatement, Opcode.JSR, tryStatement.finallyOffset);
                            }
                            if (
                                i < tryStatement.catchClauses.size() - 1
                                || tryStatement.finallyOffset != null
                            ) this.writeBranch(catchClause, Opcode.GOTO, afterStatement);
                        }
                    } finally {
                        this.getCodeContext().restoreLocalVariables();
                    }
                }
            } finally {
                this.getCodeContext().restoreLocalVariables();
            }
        }

        return canCompleteNormally;
    }

    // ------------ FunctionDeclarator.com.hazelcast.com.ile() -------------

    private void
    com.hazelcast.com.ile(FunctionDeclarator fd, final ClassFile classFile) throws CompileException {
        ClassFile.MethodInfo mi;

        if (Mod.isPrivateAccess(fd.modifiers.accessFlags)) {
            if (fd instanceof MethodDeclarator && !fd.isStatic()) {

                // To make the non-static private method invocable for enclosing types, enclosed types and types
                // enclosed by the same type, it is modified as follows:
                //  + Access is changed from PRIVATE to PACKAGE
                //  + The name is appended with "$"
                //  + It is made static
                //  + A parameter of type "declaring class" is prepended to the signature
                short accessFlags = Mod.changeAccess(
                    fd.modifiers.accessFlags, // modifiers
                    Mod.PACKAGE               // newAccess
                );
                accessFlags |= Mod.STATIC;

                mi = classFile.addMethodInfo(
                    accessFlags,   // accessFlags
                    fd.name + '$', // methodName
                    (              // methodMd
                        this.toIMethod((MethodDeclarator) fd)
                        .getDescriptor()
                        .prependParameter(this.resolve(fd.getDeclaringType()).getDescriptor())
                    )
                );
            } else
            {

                // TODO: Compile annotations on functions.
//                assert fd.modifiers.annotations.length == 0 : "NYI";

                // To make the static private method or private constructor invocable for enclosing types, enclosed
                // types and types enclosed by the same type, it is modified as follows:
                //  + Access is changed from PRIVATE to PACKAGE

                short accessFlags = Mod.changeAccess(fd.modifiers.accessFlags, Mod.PACKAGE);
                mi = classFile.addMethodInfo(
                    accessFlags,                          // accessFlags
                    fd.name,                              // methodName
                    this.toIInvocable(fd).getDescriptor() // methodMD
                );
            }
        } else {

            // Non-PRIVATE function.

            short afs = fd.modifiers.accessFlags;

            if (fd.getDeclaringType() instanceof InterfaceDeclaration) {

                // Static interface methods would require Java 8 class file format, but JANINO is still tied to Java
                // 7 class file format.
                if (Mod.isStatic(afs) && !"<clinit>".equals(fd.name) && !"class$".equals(fd.name)) {
                    this.com.hazelcast.com.ileError("Static interface methods not implemented", fd.getLocation());
                }

                afs |= Mod.ABSTRACT;
            }

            mi = classFile.addMethodInfo(
                afs,                                  // accessFlags
                fd.name,                              // methodName
                this.toIInvocable(fd).getDescriptor() // methodMD
            );
        }

        // Add method annotations with retention != SOURCE.
        this.com.hazelcast.com.ileAnnotations(fd.modifiers.annotations, mi, classFile);

        // Add "Exceptions" attribute (JVMS 4.7.4).
        {
            if (fd.thrownExceptions.length > 0) {
                final short eani    = classFile.addConstantUtf8Info("Exceptions");
                short[]     tecciis = new short[fd.thrownExceptions.length];
                for (int i = 0; i < fd.thrownExceptions.length; ++i) {
                    tecciis[i] = classFile.addConstantClassInfo(this.getType(fd.thrownExceptions[i]).getDescriptor());
                }
                mi.addAttribute(new ClassFile.ExceptionsAttribute(eani, tecciis));
            }
        }

        // Add "Deprecated" attribute (JVMS 4.7.10)
        if (fd.hasDeprecatedDocTag()) {
            mi.addAttribute(new ClassFile.DeprecatedAttribute(classFile.addConstantUtf8Info("Deprecated")));
        }

        // Add "AnnotationDefault" attribute (JVMS8 4.7.22)
        if (fd instanceof MethodDeclarator) {
            ElementValue defaultValue = ((MethodDeclarator) fd).defaultValue;
            if (defaultValue != null) {
                mi.addAttribute(
                    new ClassFile.AnnotationDefaultAttribute(
                        classFile.addConstantUtf8Info("AnnotationDefault"),
                        UnitCompiler.this.com.hazelcast.com.ileElementValue(defaultValue, classFile)
                    )
                );
            }
        }

        if (
            (fd.getDeclaringType() instanceof InterfaceDeclaration && !Mod.isStatic(fd.modifiers.accessFlags))
            || Mod.isAbstract(fd.modifiers.accessFlags)
            || Mod.isNative(fd.modifiers.accessFlags)
        ) {
            if (fd.modifiers.isDefault) {
                if (!(fd.getDeclaringType() instanceof InterfaceDeclaration)) {
                    this.com.hazelcast.com.ileError("Only interface method declarations may have the \"default\" modifier", fd.getLocation()); // SUPPRESS CHECKSTYLE LineLength
                } else
                if (Mod.isStatic(fd.modifiers.accessFlags)) {
                    this.com.hazelcast.com.ileError("Static interface method declarations must not have the \"default\" modifier", fd.getLocation()); // SUPPRESS CHECKSTYLE LineLength
                } else
                if (fd.optionalStatements == null) {
                    this.com.hazelcast.com.ileError("Default method declarations must have a body", fd.getLocation());
                }
            } else {
                if (fd.optionalStatements != null) this.com.hazelcast.com.ileError("Method must not declare a body", fd.getLocation()); // SUPPRESS CHECKSTYLE LineLength
                return;
            }
        }

        // Create CodeContext.
        final CodeContext codeContext = new CodeContext(mi.getClassFile(), mi.getName() + mi.getDescriptor());

        CodeContext savedCodeContext = this.replaceCodeContext(codeContext);
        try {
            this.getCodeContext().saveLocalVariables();

            if (!Mod.isStatic(fd.modifiers.accessFlags)) {

                // Define special parameter "this".
                this.getCodeContext().allocateLocalVariable((short) 1, "this", this.resolve(fd.getDeclaringType()));
            }

            if (fd instanceof ConstructorDeclarator) {
                ConstructorDeclarator constructorDeclarator = (ConstructorDeclarator) fd;

                if (fd.getDeclaringType() instanceof EnumDeclaration) {

                    // Define special constructor parameters "String $name" and "int $ordinal" for enums.
                    LocalVariable lv1 = new LocalVariable(true, this.iClassLoader.TYPE_java_lang_String);
                    lv1.setSlot(this.getCodeContext().allocateLocalVariable((short) 1, null, null));
                    constructorDeclarator.syntheticParameters.put("$name", lv1);

                    LocalVariable lv2 = new LocalVariable(true, IClass.INT);
                    lv2.setSlot(this.getCodeContext().allocateLocalVariable((short) 1, null, null));
                    constructorDeclarator.syntheticParameters.put("$ordinal", lv2);
                }

                // Define synthetic parameters for inner classes ("this$...", "val$...").
                for (IField sf : constructorDeclarator.getDeclaringClass().syntheticFields.values()) {
                    LocalVariable lv = new LocalVariable(true, sf.getType());

                    lv.setSlot(
                        this.getCodeContext().allocateLocalVariable(Descriptor.size(sf.getDescriptor()), null, null)
                    );
                    constructorDeclarator.syntheticParameters.put(sf.getName(), lv);
                }
            }

            this.buildLocalVariableMap(fd);

            // Compile the constructor preamble.
            if (fd instanceof ConstructorDeclarator) {
                ConstructorDeclarator cd = (ConstructorDeclarator) fd;
                if (cd.optionalConstructorInvocation != null) {
                    this.com.hazelcast.com.ile(cd.optionalConstructorInvocation);
                    if (cd.optionalConstructorInvocation instanceof SuperConstructorInvocation) {
                        this.assignSyntheticParametersToSyntheticFields(cd);
                        this.initializeInstanceVariablesAndInvokeInstanceInitializers(cd);
                    }
                } else {

                    // Determine qualification for superconstructor invocation.
                    IClass superclass = this.resolve(cd.getDeclaringClass()).getSuperclass();
                    if (superclass == null) {
                        throw new CompileException("\"" + cd + "\" has no superclass", cd.getLocation());
                    }

                    IClass                 outerClassOfSuperclass = superclass.getOuterIClass();
                    QualifiedThisReference qualification          = null;
                    if (outerClassOfSuperclass != null) {
                        qualification = new QualifiedThisReference(
                            cd.getLocation(),                                        // location
                            new SimpleType(cd.getLocation(), outerClassOfSuperclass) // qualification
                        );
                    }

                    // Invoke the superconstructor.
                    Rvalue[] arguments;
                    if (fd.getDeclaringType() instanceof EnumDeclaration) {

                        LocalVariableAccess nameAccess = new LocalVariableAccess(
                            cd.getLocation(),
                            (LocalVariable) cd.syntheticParameters.get("$name")
                        );
                        assert nameAccess != null;

                        LocalVariableAccess ordinalAccess = new LocalVariableAccess(
                            cd.getLocation(),
                            (LocalVariable) cd.syntheticParameters.get("$ordinal")
                        );
                        assert ordinalAccess != null;

                        arguments = new Rvalue[] { nameAccess, ordinalAccess };
                    } else {
                        arguments = new Rvalue[0];
                    }
                    SuperConstructorInvocation sci = new SuperConstructorInvocation(
                        cd.getLocation(),  // location
                        qualification,     // optionalQualification
                        arguments          // arguments
                    );
                    sci.setEnclosingScope(fd);
                    this.com.hazelcast.com.ile(sci);
                    this.assignSyntheticParametersToSyntheticFields(cd);
                    this.initializeInstanceVariablesAndInvokeInstanceInitializers(cd);
                }
            }

            // Compile the function body.
            List<? extends BlockStatement> oss = fd.optionalStatements;
            if (oss == null) {
                this.com.hazelcast.com.ileError("Method must have a body", fd.getLocation());
                return;
            }
            if (this.com.hazelcast.com.ileStatements(oss)) {
                if (this.getReturnType(fd) != IClass.VOID) {
                    this.com.hazelcast.com.ileError("Method must return a value", fd.getLocation());
                }
                this.writeOpcode(fd, Opcode.RETURN);
            }
        } finally {
            this.getCodeContext().restoreLocalVariables();
            this.replaceCodeContext(savedCodeContext);
        }

        // Don't continue code attribute generation if we had com.hazelcast.com.ile errors.
        if (this.com.hazelcast.com.ileErrorCount > 0) return;

        // Fix up and reallocate as needed.
        codeContext.fixUpAndRelocate();

        // Do flow analysis.
        if (UnitCompiler.LOGGER.isLoggable(Level.FINE)) {
            try {
                codeContext.flowAnalysis(fd.toString());
            } catch (RuntimeException re) {
                UnitCompiler.LOGGER.log(Level.FINE, "*** FLOW ANALYSIS", re);

                // Continue, so that the .class file is generated and can be examined.
                ;
            }
        } else {
            try {
                codeContext.flowAnalysis(fd.toString());
            } catch (RuntimeException re) {
                throw new InternalCompilerException("Compiling \"" + fd + "\"; " + re.getMessage(), re);
            }
        }

        final short lntani;
        if (this.debugLines) {
            lntani = classFile.addConstantUtf8Info("LineNumberTable");
        } else {
            lntani = 0;
        }

        final short lvtani;
        if (this.debugVars) {
            UnitCompiler.makeLocalVariableNames(codeContext, mi);
            lvtani = classFile.addConstantUtf8Info("LocalVariableTable");
        } else {
            lvtani = 0;
        }

        final short smtani = classFile.addConstantUtf8Info("StackMapTable");

        // Add the code context as a code attribute to the MethodInfo.
        mi.addAttribute(new ClassFile.AttributeInfo(classFile.addConstantUtf8Info("Code")) {

            @Override protected void
            storeBody(DataOutputStream dos) throws IOException {
                codeContext.storeCodeAttributeBody(dos, lntani, lvtani, smtani);
            }
        });
    }

    /**
     * Makes the variable name and class name Constant Pool names used by local variables.
     */
    private static void
    makeLocalVariableNames(final CodeContext cc, final ClassFile.MethodInfo mi) {
        ClassFile cf = mi.getClassFile();

        cf.addConstantUtf8Info("LocalVariableTable");
        for (LocalVariableSlot slot : cc.getAllLocalVars()) {

            String localVariableName = slot.getName();
            if (localVariableName != null) {
                String typeName = slot.getType().getDescriptor();

                cf.addConstantUtf8Info(typeName);
                cf.addConstantUtf8Info(localVariableName);
            }
        }
    }

    private void
    buildLocalVariableMap(FunctionDeclarator fd) throws CompileException {
        Map<String, LocalVariable> localVars = new HashMap<String, LocalVariable>();

        // Add function parameters.
        for (int i = 0; i < fd.formalParameters.parameters.length; ++i) {
            FormalParameter fp              = fd.formalParameters.parameters[i];
            IClass          parameterIClass = this.getType(fp.type);
            LocalVariable   lv              = this.getLocalVariable(
                fp,
                i == fd.formalParameters.parameters.length - 1 && fd.formalParameters.variableArity
            );
            lv.setSlot(this.getCodeContext().allocateLocalVariable(
                Descriptor.size(lv.type.getDescriptor()),
                fp.name,
                parameterIClass
            ));

            if (localVars.put(fp.name, lv) != null) {
                this.com.hazelcast.com.ileError("Redefinition of parameter \"" + fp.name + "\"", fd.getLocation());
            }
        }

        fd.localVariables = localVars;
        if (fd instanceof ConstructorDeclarator) {
            ConstructorDeclarator cd = (ConstructorDeclarator) fd;
            if (cd.optionalConstructorInvocation != null) {
                UnitCompiler.buildLocalVariableMap(cd.optionalConstructorInvocation, localVars);
            }
        }
        if (fd.optionalStatements != null) {
            for (BlockStatement bs : fd.optionalStatements) localVars = this.buildLocalVariableMap(bs, localVars);
        }
    }

    /**
     * Computes and fills in the 'local variable map' for the given <var>blockStatement</var>.
     */
    private Map<String, LocalVariable>
    buildLocalVariableMap(BlockStatement blockStatement, final Map<String, LocalVariable> localVars)
    throws CompileException {

        Map<String, LocalVariable> result = (Map<String, LocalVariable>) blockStatement.accept(
            new BlockStatementVisitor<Map<String, LocalVariable>, CompileException>() {

                // Basic statements that use the default handlers.
                // SUPPRESS CHECKSTYLE LineLengthCheck:11
                @Override public Map<String, LocalVariable> visitAlternateConstructorInvocation(AlternateConstructorInvocation aci)  { UnitCompiler.buildLocalVariableMap(aci, localVars);  return localVars; }
                @Override public Map<String, LocalVariable> visitBreakStatement(BreakStatement bs)                                   { UnitCompiler.buildLocalVariableMap(bs, localVars);   return localVars; }
                @Override public Map<String, LocalVariable> visitContinueStatement(ContinueStatement cs)                             { UnitCompiler.buildLocalVariableMap(cs, localVars);   return localVars; }
                @Override public Map<String, LocalVariable> visitAssertStatement(AssertStatement as)                                 { UnitCompiler.buildLocalVariableMap(as, localVars);   return localVars; }
                @Override public Map<String, LocalVariable> visitEmptyStatement(EmptyStatement es)                                   { UnitCompiler.buildLocalVariableMap(es, localVars);   return localVars; }
                @Override public Map<String, LocalVariable> visitExpressionStatement(ExpressionStatement es)                         { UnitCompiler.buildLocalVariableMap(es, localVars);   return localVars; }
                @Override public Map<String, LocalVariable> visitFieldDeclaration(FieldDeclaration fd)                               { UnitCompiler.buildLocalVariableMap(fd, localVars);   return localVars; }
                @Override public Map<String, LocalVariable> visitReturnStatement(ReturnStatement rs)                                 { UnitCompiler.buildLocalVariableMap(rs, localVars);   return localVars; }
                @Override public Map<String, LocalVariable> visitSuperConstructorInvocation(SuperConstructorInvocation sci)          { UnitCompiler.buildLocalVariableMap(sci, localVars);  return localVars; }
                @Override public Map<String, LocalVariable> visitThrowStatement(ThrowStatement ts)                                   { UnitCompiler.buildLocalVariableMap(ts, localVars);   return localVars; }
                @Override public Map<String, LocalVariable> visitLocalClassDeclarationStatement(LocalClassDeclarationStatement lcds) { UnitCompiler.buildLocalVariableMap(lcds, localVars); return localVars; }

                // More com.hazelcast.com.licated statements with specialized handlers, but don't add new variables in this scope.
                // SUPPRESS CHECKSTYLE LineLengthCheck:10
                @Override public Map<String, LocalVariable> visitBlock(Block b)                                  throws CompileException { UnitCompiler.this.buildLocalVariableMap(b,   localVars); return localVars; }
                @Override public Map<String, LocalVariable> visitDoStatement(DoStatement ds)                     throws CompileException { UnitCompiler.this.buildLocalVariableMap(ds,  localVars); return localVars; }
                @Override public Map<String, LocalVariable> visitForStatement(ForStatement fs)                   throws CompileException { UnitCompiler.this.buildLocalVariableMap(fs,  localVars); return localVars; }
                @Override public Map<String, LocalVariable> visitForEachStatement(ForEachStatement fes)          throws CompileException { UnitCompiler.this.buildLocalVariableMap(fes, localVars); return localVars; }
                @Override public Map<String, LocalVariable> visitIfStatement(IfStatement is)                     throws CompileException { UnitCompiler.this.buildLocalVariableMap(is,  localVars); return localVars; }
                @Override public Map<String, LocalVariable> visitInitializer(Initializer i)                      throws CompileException { UnitCompiler.this.buildLocalVariableMap(i,   localVars); return localVars; }
                @Override public Map<String, LocalVariable> visitSwitchStatement(SwitchStatement ss)             throws CompileException { UnitCompiler.this.buildLocalVariableMap(ss,  localVars); return localVars; }
                @Override public Map<String, LocalVariable> visitSynchronizedStatement(SynchronizedStatement ss) throws CompileException { UnitCompiler.this.buildLocalVariableMap(ss,  localVars); return localVars; }
                @Override public Map<String, LocalVariable> visitTryStatement(TryStatement ts)                   throws CompileException { UnitCompiler.this.buildLocalVariableMap(ts,  localVars); return localVars; }
                @Override public Map<String, LocalVariable> visitWhileStatement(WhileStatement ws)               throws CompileException { UnitCompiler.this.buildLocalVariableMap(ws,  localVars); return localVars; }

                // More com.hazelcast.com.licated statements with specialized handlers, that can add variables in this scope.
                // SUPPRESS CHECKSTYLE LineLengthCheck:2
                @Override public Map<String, LocalVariable> visitLabeledStatement(LabeledStatement ls)                                     throws CompileException { return UnitCompiler.this.buildLocalVariableMap(ls,   localVars); }
                @Override public Map<String, LocalVariable> visitLocalVariableDeclarationStatement(LocalVariableDeclarationStatement lvds) throws CompileException { return UnitCompiler.this.buildLocalVariableMap(lvds, localVars); }
            }
        );

        assert result != null;
        return result;
    }

    // Default handlers.

    private static Map<String, LocalVariable>
    buildLocalVariableMap(Statement s, final Map<String, LocalVariable> localVars) {
        return (s.localVariables = localVars);
    }

    private static Map<String, LocalVariable>
    buildLocalVariableMap(ConstructorInvocation ci, final Map<String, LocalVariable> localVars) {
        return (ci.localVariables = localVars);
    }

    // Specialized handlers.
    private void
    buildLocalVariableMap(Block block, Map<String, LocalVariable> localVars) throws CompileException {
        block.localVariables = localVars;
        for (BlockStatement bs : block.statements) localVars = this.buildLocalVariableMap(bs, localVars);
    }

    private void
    buildLocalVariableMap(DoStatement ds, final Map<String, LocalVariable> localVars) throws CompileException {
        ds.localVariables = localVars;
        this.buildLocalVariableMap(ds.body, localVars);
    }

    private void
    buildLocalVariableMap(ForStatement fs, final Map<String, LocalVariable> localVars)
    throws CompileException {
        Map<String, LocalVariable> inner = localVars;
        if (fs.optionalInit != null) {
            inner = this.buildLocalVariableMap(fs.optionalInit, localVars);
        }
        fs.localVariables = inner;
        this.buildLocalVariableMap(fs.body, inner);
    }

    private void
    buildLocalVariableMap(ForEachStatement fes, final Map<String, LocalVariable> localVars)
    throws CompileException {
        Map<String, LocalVariable> vars = new HashMap<String, LocalVariable>();
        vars.putAll(localVars);
        LocalVariable elementLv = this.getLocalVariable(fes.currentElement, false);
        vars.put(fes.currentElement.name, elementLv);
        fes.localVariables = vars;
        this.buildLocalVariableMap(fes.body, vars);
    }

    private void
    buildLocalVariableMap(IfStatement is, final Map<String, LocalVariable> localVars) throws CompileException {
        is.localVariables = localVars;
        this.buildLocalVariableMap(is.thenStatement, localVars);
        if (is.elseStatement != null) {
            this.buildLocalVariableMap(is.elseStatement, localVars);
        }
    }

    private void
    buildLocalVariableMap(Initializer i, final Map<String, LocalVariable> localVars) throws CompileException {
        this.buildLocalVariableMap(i.block, localVars);
    }

    private void
    buildLocalVariableMap(SwitchStatement ss, final Map<String, LocalVariable> localVars)
    throws CompileException {
        ss.localVariables = localVars;
        Map<String, LocalVariable> vars = localVars;
        for (SwitchBlockStatementGroup sbsg : ss.sbsgs) {
            for (BlockStatement bs : sbsg.blockStatements) vars = this.buildLocalVariableMap(bs, vars);
        }
    }

    private void
    buildLocalVariableMap(SynchronizedStatement ss, final Map<String, LocalVariable> localVars)
    throws CompileException {
        ss.localVariables = localVars;
        this.buildLocalVariableMap(ss.body, localVars);
    }

    private void
    buildLocalVariableMap(TryStatement ts, final Map<String, LocalVariable> localVars)
    throws CompileException {
        ts.localVariables = localVars;
        this.buildLocalVariableMap(ts.body, localVars);
        for (CatchClause cc : ts.catchClauses) this.buildLocalVariableMap(cc, localVars);
        if (ts.finallY != null) {
            this.buildLocalVariableMap(ts.finallY, localVars);
        }
    }

    private void
    buildLocalVariableMap(WhileStatement ws, final Map<String, LocalVariable> localVars)
    throws CompileException {
        ws.localVariables = localVars;
        this.buildLocalVariableMap(ws.body, localVars);
    }

    private Map<String, LocalVariable>
    buildLocalVariableMap(LabeledStatement ls, final Map<String, LocalVariable> localVars)
    throws CompileException {
        ls.localVariables = localVars;
        return this.buildLocalVariableMap((BlockStatement) ls.body, localVars);
    }

    private Map<String, LocalVariable>
    buildLocalVariableMap(LocalVariableDeclarationStatement lvds, final Map<String, LocalVariable> localVars)
    throws CompileException {
        Map<String, LocalVariable> newVars = new HashMap<String, LocalVariable>();
        newVars.putAll(localVars);
        for (VariableDeclarator vd : lvds.variableDeclarators) {
            LocalVariable      lv = this.getLocalVariable(lvds, vd);
            if (newVars.put(vd.name, lv) != null) {
                this.com.hazelcast.com.ileError("Redefinition of local variable \"" + vd.name + "\" ", vd.getLocation());
            }
        }
        lvds.localVariables = newVars;
        return newVars;
    }

    /**
     * Adds the given <var>localVars</var> to the 'local variable map' of the given <var>catchClause</var>.
     */
    protected void
    buildLocalVariableMap(CatchClause catchClause, Map<String, LocalVariable> localVars) throws CompileException {
        Map<String, LocalVariable> vars = new HashMap<String, LocalVariable>();
        vars.putAll(localVars);
        LocalVariable lv = this.getLocalVariable(catchClause.caughtException);
        vars.put(catchClause.caughtException.name, lv);
        this.buildLocalVariableMap(catchClause.body, vars);
    }

    /**
     * @return The {@link LocalVariable} corresponding with the <var>parameter</var>
     */
    public LocalVariable
    getLocalVariable(FormalParameter parameter) throws CompileException {
        return this.getLocalVariable(parameter, false);
    }

    /**
     * @param isVariableArityParameter Whether the <var>parameter</var> is the last parameter of a 'variable arity'
     *                                 (a.k.a. 'varargs') method declaration
     * @return                         The {@link LocalVariable} corresponding with the <var>parameter</var>
     */
    public LocalVariable
    getLocalVariable(FormalParameter parameter, boolean isVariableArityParameter)
    throws CompileException {
        if (parameter.localVariable != null) return parameter.localVariable;

        assert parameter.type != null;
        IClass parameterType = this.getType(parameter.type);
        if (isVariableArityParameter) {
            parameterType = parameterType.getArrayIClass(this.iClassLoader.TYPE_java_lang_Object);
        }

        return (parameter.localVariable = new LocalVariable(parameter.finaL, parameterType));
    }

    // ------------------ Rvalue.com.hazelcast.com.ile() ----------------

    /**
     * Called to check whether the given {@link Rvalue} com.hazelcast.com.iles or not.
     */
    private void
    fakeCompile(Rvalue rv) throws CompileException {

        final Offset from = this.getCodeContext().newOffset();

        this.com.hazelcast.com.ileContext(rv);
        this.com.hazelcast.com.ileGet(rv);

        Offset to = this.getCodeContext().newOffset();

        this.getCodeContext().removeCode(from, to);
    }

    /**
     * Some {@link Rvalue}s com.hazelcast.com.ile more efficiently when their value is not needed, e.g. "i++".
     */
    private void
    com.hazelcast.com.ile(Rvalue rv) throws CompileException {

        rv.accept(new RvalueVisitor<Void, CompileException>() {

            @Override @Nullable public Void
            visitLvalue(Lvalue lv) throws CompileException {
                lv.accept(new Visitor.LvalueVisitor<Void, CompileException>() {

                    // SUPPRESS CHECKSTYLE LineLength:7
                    @Override @Nullable public Void visitAmbiguousName(AmbiguousName an)                                        throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(an);    return null; }
                    @Override @Nullable public Void visitArrayAccessExpression(ArrayAccessExpression aae)                       throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(aae);   return null; }
                    @Override @Nullable public Void visitFieldAccess(FieldAccess fa)                                            throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(fa);    return null; }
                    @Override @Nullable public Void visitFieldAccessExpression(FieldAccessExpression fae)                       throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(fae);   return null; }
                    @Override @Nullable public Void visitSuperclassFieldAccessExpression(SuperclassFieldAccessExpression scfae) throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(scfae); return null; }
                    @Override @Nullable public Void visitLocalVariableAccess(LocalVariableAccess lva)                           throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(lva);   return null; }
                    @Override @Nullable public Void visitParenthesizedExpression(ParenthesizedExpression pe)                    throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(pe);    return null; }
                });
                return null;
            }

            // SUPPRESS CHECKSTYLE LineLength:25
            @Override @Nullable public Void visitArrayLength(ArrayLength al)                                throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(al);    return null; }
            @Override @Nullable public Void visitAssignment(Assignment a)                                   throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(a);     return null; }
            @Override @Nullable public Void visitUnaryOperation(UnaryOperation uo)                          throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(uo);    return null; }
            @Override @Nullable public Void visitBinaryOperation(BinaryOperation bo)                        throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(bo);    return null; }
            @Override @Nullable public Void visitCast(Cast c)                                               throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(c);     return null; }
            @Override @Nullable public Void visitClassLiteral(ClassLiteral cl)                              throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(cl);    return null; }
            @Override @Nullable public Void visitConditionalExpression(ConditionalExpression ce)            throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(ce);    return null; }
            @Override @Nullable public Void visitCrement(Crement c)                                         throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(c);     return null; }
            @Override @Nullable public Void visitInstanceof(Instanceof io)                                  throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(io);    return null; }
            @Override @Nullable public Void visitMethodInvocation(MethodInvocation mi)                      throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(mi);    return null; }
            @Override @Nullable public Void visitSuperclassMethodInvocation(SuperclassMethodInvocation smi) throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(smi);   return null; }
            @Override @Nullable public Void visitIntegerLiteral(IntegerLiteral il)                          throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(il);    return null; }
            @Override @Nullable public Void visitFloatingPointLiteral(FloatingPointLiteral fpl)             throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(fpl);   return null; }
            @Override @Nullable public Void visitBooleanLiteral(BooleanLiteral bl)                          throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(bl);    return null; }
            @Override @Nullable public Void visitCharacterLiteral(CharacterLiteral cl)                      throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(cl);    return null; }
            @Override @Nullable public Void visitStringLiteral(StringLiteral sl)                            throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(sl);    return null; }
            @Override @Nullable public Void visitNullLiteral(NullLiteral nl)                                throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(nl);    return null; }
            @Override @Nullable public Void visitSimpleConstant(SimpleConstant sl)                          throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(sl);    return null; }
            @Override @Nullable public Void visitNewAnonymousClassInstance(NewAnonymousClassInstance naci)  throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(naci);  return null; }
            @Override @Nullable public Void visitNewArray(NewArray na)                                      throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(na);    return null; }
            @Override @Nullable public Void visitNewInitializedArray(NewInitializedArray nia)               throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(nia);   return null; }
            @Override @Nullable public Void visitNewClassInstance(NewClassInstance nci)                     throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(nci);   return null; }
            @Override @Nullable public Void visitParameterAccess(ParameterAccess pa)                        throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(pa);    return null; }
            @Override @Nullable public Void visitQualifiedThisReference(QualifiedThisReference qtr)         throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(qtr);   return null; }
            @Override @Nullable public Void visitThisReference(ThisReference tr)                            throws CompileException { UnitCompiler.this.com.hazelcast.com.ile2(tr);    return null; }
        });
    }

    private void
    com.hazelcast.com.ile2(Rvalue rv) throws CompileException {
        this.pop(rv, this.com.hazelcast.com.ileGetValue(rv));
    }

    private void
    com.hazelcast.com.ile2(Assignment a) throws CompileException {
        if (a.operator == "=") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            this.com.hazelcast.com.ileContext(a.lhs);
            this.assignmentConversion(
                a,                           // locatable
                this.com.hazelcast.com.ileGetValue(a.rhs), // sourceType
                this.getType(a.lhs),         // targetType
                this.getConstantValue(a.rhs) // optionalConstantValue
            );
            this.com.hazelcast.com.ileSet(a.lhs);
            return;
        }

        // Implement "|= ^= &= *= /= %= += -= <<= >>= >>>=".
        int lhsCs = this.com.hazelcast.com.ileContext(a.lhs);
        this.dup(a, lhsCs);
        IClass lhsType    = this.com.hazelcast.com.ileGet(a.lhs);
        IClass resultType = this.com.hazelcast.com.ileArithmeticBinaryOperation(
            a,                    // locatable
            lhsType,              // lhsType
            a.operator.substring( // operator
                0,
                a.operator.length() - 1
            ).intern(), /* <= IMPORTANT!
            */
            a.rhs                 // rhs
        );

        // Convert the result to LHS type (JLS7 15.26.2).
        if (
            !this.tryIdentityConversion(resultType, lhsType)
            && !this.tryNarrowingPrimitiveConversion(a, resultType, lhsType)
            && !this.tryBoxingConversion(a, resultType, lhsType) // Java 5
        ) this.com.hazelcast.com.ileError("Operand types unsuitable for \"" + a.operator + "\"", a.getLocation());

        // Assign the result to the left operand.
        this.com.hazelcast.com.ileSet(a.lhs);
    }

    private void
    com.hazelcast.com.ile2(Crement c) throws CompileException {

        // Optimized crement of integer local variable.
        LocalVariable lv = this.isIntLv(c);
        if (lv != null) {
            this.com.hazelcast.com.ileLocalVariableCrement(c, lv);
            return;
        }

        int cs = this.com.hazelcast.com.ileContext(c.operand);
        this.dup(c, cs);
        IClass type         = this.com.hazelcast.com.ileGet(c.operand);
        IClass promotedType = this.unaryNumericPromotion(c, type);
        this.writeOpcode(c, UnitCompiler.ilfd(
            promotedType,
            Opcode.ICONST_1,
            Opcode.LCONST_1,
            Opcode.FCONST_1,
            Opcode.DCONST_1
        ));
        if (c.operator == "++") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            this.writeOpcode(c, Opcode.IADD + UnitCompiler.ilfd(promotedType));
        } else
        if (c.operator == "--") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            this.writeOpcode(c, Opcode.ISUB + UnitCompiler.ilfd(promotedType));
        } else {
            this.com.hazelcast.com.ileError("Unexpected operator \"" + c.operator + "\"", c.getLocation());
        }

        this.reverseUnaryNumericPromotion(c, promotedType, type);
        this.com.hazelcast.com.ileSet(c.operand);
    }

    private void
    com.hazelcast.com.ile2(ParenthesizedExpression pe) throws CompileException {
        this.com.hazelcast.com.ile(pe.value);
    }

    private boolean
    com.hazelcast.com.ile2(AlternateConstructorInvocation aci) throws CompileException {
        ConstructorDeclarator declaringConstructor = (ConstructorDeclarator) aci.getEnclosingScope();
        IClass                declaringIClass      = this.resolve(declaringConstructor.getDeclaringClass());

        this.writeOpcode(aci, Opcode.ALOAD_0);
        if (declaringIClass.getOuterIClass() != null) this.writeOpcode(aci, Opcode.ALOAD_1);
        this.invokeConstructor(
            aci,                  // locatable
            declaringConstructor, // scope
            (Rvalue) null,        // optionalEnclosingInstance
            declaringIClass,      // targetClass
            aci.arguments         // arguments
        );
        return true;
    }

    private boolean
    com.hazelcast.com.ile2(SuperConstructorInvocation sci) throws CompileException {
        ConstructorDeclarator declaringConstructor = (ConstructorDeclarator) sci.getEnclosingScope();
        this.writeOpcode(sci, Opcode.ALOAD_0);
        AbstractClassDeclaration declaringClass = declaringConstructor.getDeclaringClass();
        IClass                   superclass     = this.resolve(declaringClass).getSuperclass();

        if (superclass == null) throw new CompileException("Class has no superclass", sci.getLocation());

        Rvalue optionalEnclosingInstance;
        if (sci.optionalQualification != null) {
            optionalEnclosingInstance = sci.optionalQualification;
        } else {
            IClass outerIClassOfSuperclass = superclass.getOuterIClass();
            if (outerIClassOfSuperclass == null) {
                optionalEnclosingInstance = null;
            } else {
                optionalEnclosingInstance = new QualifiedThisReference(
                    sci.getLocation(),                                         // location
                    new SimpleType(sci.getLocation(), outerIClassOfSuperclass) // qualification
                );
                optionalEnclosingInstance.setEnclosingScope(sci);
            }
        }
        this.invokeConstructor(
            sci,                       // locatable
            declaringConstructor,      // scope
            optionalEnclosingInstance, // optionalEnclosingInstance
            superclass,                // targetClass
            sci.arguments              // arguments
        );
        return true;
    }

    /**
     * Compiles an {@link Rvalue} and branches, depending on the value.
     * <p>
     *   Many {@link Rvalue}s com.hazelcast.com.ile more efficiently when their value is the condition for a branch, thus
     *   this method generally produces more efficient bytecode than {@link #com.hazelcast.com.ile(Rvalue)} followed by {@link
     *   #writeBranch(Locatable, int, Offset)}.
     * </p>
     * <p>
     *   Notice that if <var>rv</var> is a constant, then either <var>dst</var> is never branched to, or it is
     *   unconditionally branched to; "Unexamined code" errors may result during bytecode validation.
     * </p>
     *
     * @param rv          The value that determines whether to branch or not
     * @param dst         Where to jump
     * @param orientation {@link #JUMP_IF_TRUE} or {@link #JUMP_IF_FALSE}
     */
    private void
    com.hazelcast.com.ileBoolean(Rvalue rv, final CodeContext.Offset dst, final boolean orientation) throws CompileException {

        rv.accept(new RvalueVisitor<Void, CompileException>() {

            @Override @Nullable public Void
            visitLvalue(Lvalue lv) throws CompileException {
                lv.accept(new Visitor.LvalueVisitor<Void, CompileException>() {

                    // SUPPRESS CHECKSTYLE LineLength:7
                    @Override @Nullable public Void visitAmbiguousName(AmbiguousName an)                                        throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(an,    dst, orientation); return null; }
                    @Override @Nullable public Void visitArrayAccessExpression(ArrayAccessExpression aae)                       throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(aae,   dst, orientation); return null; }
                    @Override @Nullable public Void visitFieldAccess(FieldAccess fa)                                            throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(fa,    dst, orientation); return null; }
                    @Override @Nullable public Void visitFieldAccessExpression(FieldAccessExpression fae)                       throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(fae,   dst, orientation); return null; }
                    @Override @Nullable public Void visitSuperclassFieldAccessExpression(SuperclassFieldAccessExpression scfae) throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(scfae, dst, orientation); return null; }
                    @Override @Nullable public Void visitLocalVariableAccess(LocalVariableAccess lva)                           throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(lva,   dst, orientation); return null; }
                    @Override @Nullable public Void visitParenthesizedExpression(ParenthesizedExpression pe)                    throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(pe,    dst, orientation); return null; }
                });
                return null;
            }

            // SUPPRESS CHECKSTYLE LineLength:25
            @Override @Nullable public Void visitArrayLength(ArrayLength al)                                throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(al,   dst, orientation); return null; }
            @Override @Nullable public Void visitAssignment(Assignment a)                                   throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(a,    dst, orientation); return null; }
            @Override @Nullable public Void visitUnaryOperation(UnaryOperation uo)                          throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(uo,   dst, orientation); return null; }
            @Override @Nullable public Void visitBinaryOperation(BinaryOperation bo)                        throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(bo,   dst, orientation); return null; }
            @Override @Nullable public Void visitCast(Cast c)                                               throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(c,    dst, orientation); return null; }
            @Override @Nullable public Void visitClassLiteral(ClassLiteral cl)                              throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(cl,   dst, orientation); return null; }
            @Override @Nullable public Void visitConditionalExpression(ConditionalExpression ce)            throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(ce,   dst, orientation); return null; }
            @Override @Nullable public Void visitCrement(Crement c)                                         throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(c,    dst, orientation); return null; }
            @Override @Nullable public Void visitInstanceof(Instanceof io)                                  throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(io,   dst, orientation); return null; }
            @Override @Nullable public Void visitMethodInvocation(MethodInvocation mi)                      throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(mi,   dst, orientation); return null; }
            @Override @Nullable public Void visitSuperclassMethodInvocation(SuperclassMethodInvocation smi) throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(smi,  dst, orientation); return null; }
            @Override @Nullable public Void visitIntegerLiteral(IntegerLiteral il)                          throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(il,   dst, orientation); return null; }
            @Override @Nullable public Void visitFloatingPointLiteral(FloatingPointLiteral fpl)             throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(fpl,  dst, orientation); return null; }
            @Override @Nullable public Void visitBooleanLiteral(BooleanLiteral bl)                          throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(bl,   dst, orientation); return null; }
            @Override @Nullable public Void visitCharacterLiteral(CharacterLiteral cl)                      throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(cl,   dst, orientation); return null; }
            @Override @Nullable public Void visitStringLiteral(StringLiteral sl)                            throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(sl,   dst, orientation); return null; }
            @Override @Nullable public Void visitNullLiteral(NullLiteral nl)                                throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(nl,   dst, orientation); return null; }
            @Override @Nullable public Void visitSimpleConstant(SimpleConstant sl)                          throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(sl,   dst, orientation); return null; }
            @Override @Nullable public Void visitNewAnonymousClassInstance(NewAnonymousClassInstance naci)  throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(naci, dst, orientation); return null; }
            @Override @Nullable public Void visitNewArray(NewArray na)                                      throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(na,   dst, orientation); return null; }
            @Override @Nullable public Void visitNewInitializedArray(NewInitializedArray nia)               throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(nia,  dst, orientation); return null; }
            @Override @Nullable public Void visitNewClassInstance(NewClassInstance nci)                     throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(nci,  dst, orientation); return null; }
            @Override @Nullable public Void visitParameterAccess(ParameterAccess pa)                        throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(pa,   dst, orientation); return null; }
            @Override @Nullable public Void visitQualifiedThisReference(QualifiedThisReference qtr)         throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(qtr,  dst, orientation); return null; }
            @Override @Nullable public Void visitThisReference(ThisReference tr)                            throws CompileException { UnitCompiler.this.com.hazelcast.com.ileBoolean2(tr,   dst, orientation); return null; }
        });
    }

    /**
     * @param dst         Where to jump
     * @param orientation {@link #JUMP_IF_TRUE} or {@link #JUMP_IF_FALSE}
     */
    private void
    com.hazelcast.com.ileBoolean2(Rvalue rv, CodeContext.Offset dst, boolean orientation) throws CompileException {
        IClass       type = this.com.hazelcast.com.ileGetValue(rv);
        IClassLoader icl  = this.iClassLoader;
        if (type == icl.TYPE_java_lang_Boolean) {
            this.unboxingConversion(rv, icl.TYPE_java_lang_Boolean, IClass.BOOLEAN);
        } else
        if (type != IClass.BOOLEAN) {
            this.com.hazelcast.com.ileError("Not a boolean expression", rv.getLocation());
        }
        this.writeBranch(rv, orientation == UnitCompiler.JUMP_IF_TRUE ? Opcode.IFNE : Opcode.IFEQ, dst);
    }

    /**
     * @param dst         Where to jump
     * @param orientation {@link #JUMP_IF_TRUE} or {@link #JUMP_IF_FALSE}
     */
    private void
    com.hazelcast.com.ileBoolean2(UnaryOperation ue, CodeContext.Offset dst, boolean orientation) throws CompileException {
        if (ue.operator == "!") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            this.com.hazelcast.com.ileBoolean(ue.operand, dst, !orientation);
            return;
        }

        this.com.hazelcast.com.ileError("Boolean expression expected", ue.getLocation());
    }

    /**
     * @param dst         Where to jump
     * @param orientation {@link #JUMP_IF_TRUE} or {@link #JUMP_IF_FALSE}
     */
    private void
    com.hazelcast.com.ileBoolean2(BinaryOperation bo, CodeContext.Offset dst, boolean orientation) throws CompileException {

        if (bo.operator == "|" || bo.operator == "^" || bo.operator == "&") { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
            this.com.hazelcast.com.ileBoolean2((Rvalue) bo, dst, orientation);
            return;
        }

        if (bo.operator == "||" || bo.operator == "&&") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            Object lhsCv = this.getConstantValue(bo.lhs);
            if (lhsCv instanceof Boolean) {
                if (((Boolean) lhsCv).booleanValue() ^ bo.operator == "||") { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
                    // "true && a", "false || a"
                    this.com.hazelcast.com.ileBoolean(
                        bo.rhs,
                        dst,
                        orientation
                    );
                } else {
                    // "false && a", "true || a"
                    this.com.hazelcast.com.ileBoolean(
                        bo.lhs,
                        dst,
                        orientation
                    );
                    this.fakeCompile(bo.rhs);
                }
                return;
            }
            Object rhsCv = this.getConstantValue(bo.rhs);
            if (rhsCv instanceof Boolean) {
                if (((Boolean) rhsCv).booleanValue() ^ bo.operator == "||") { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
                    // "a && true", "a || false"
                    this.com.hazelcast.com.ileBoolean(
                        bo.lhs,
                        dst,
                        orientation
                    );
                } else {
                    // "a && false", "a || true"

                    // Compile the LHS ("a"), and discard the result.
                    this.pop(bo.lhs, this.com.hazelcast.com.ileGetValue(bo.lhs));

                    // Compile the RHS and branch conditionally (although the RHS is a constant). This prevents
                    // trouble with "unreachable code".
                    this.com.hazelcast.com.ileBoolean(
                        bo.rhs,
                        dst,
                        orientation
                    );
                }
                return;
            }

            // SUPPRESS CHECKSTYLE StringLiteralEquality
            if (bo.operator == "||" ^ orientation == UnitCompiler.JUMP_IF_FALSE) {
                this.com.hazelcast.com.ileBoolean(bo.lhs, dst, orientation);
                this.com.hazelcast.com.ileBoolean(bo.rhs, dst, orientation);
            } else {
                CodeContext.Offset end = this.getCodeContext().new Offset();
                this.com.hazelcast.com.ileBoolean(bo.lhs, end, !orientation);
                this.com.hazelcast.com.ileBoolean(bo.rhs, dst, orientation);
                end.set();
            }
            return;
        }

        if (
            bo.operator == "=="    // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == "!=" // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == "<=" // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == ">=" // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == "<"  // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == ">"  // SUPPRESS CHECKSTYLE StringLiteralEquality
        ) {
            int opIdx = (
                bo.operator == "==" ? 0 : // SUPPRESS CHECKSTYLE StringLiteralEquality
                bo.operator == "!=" ? 1 : // SUPPRESS CHECKSTYLE StringLiteralEquality
                bo.operator == "<"  ? 2 : // SUPPRESS CHECKSTYLE StringLiteralEquality
                bo.operator == ">=" ? 3 : // SUPPRESS CHECKSTYLE StringLiteralEquality
                bo.operator == ">"  ? 4 : // SUPPRESS CHECKSTYLE StringLiteralEquality
                bo.operator == "<=" ? 5 : // SUPPRESS CHECKSTYLE StringLiteralEquality
                Integer.MIN_VALUE
            );
            if (orientation == UnitCompiler.JUMP_IF_FALSE) opIdx ^= 1;

            // Comparison with "null".
            {
                boolean lhsIsNull = this.getConstantValue(bo.lhs) == null;
                boolean rhsIsNull = this.getConstantValue(bo.rhs) == null;

                if (lhsIsNull || rhsIsNull) {
                    if (bo.operator != "==" && bo.operator != "!=") { // SUPPRESS CHECKSTYLE StringLiteralEquality
                        this.com.hazelcast.com.ileError(
                            "Operator \"" + bo.operator + "\" not allowed on operand \"null\"",
                            bo.getLocation()
                        );
                    }

                    if (!lhsIsNull) {

                        // x == null
                        IClass lhsType = this.com.hazelcast.com.ileGetValue(bo.lhs);
                        if (lhsType.isPrimitive()) {
                            this.com.hazelcast.com.ileError(
                                "Cannot com.hazelcast.com.are primitive type \"" + lhsType.toString() + "\" with \"null\"",
                                bo.getLocation()
                            );
                        }
                    } else
                    if (!rhsIsNull) {

                        // null == x
                        IClass rhsType = this.com.hazelcast.com.ileGetValue(bo.rhs);
                        if (rhsType.isPrimitive()) {
                            this.com.hazelcast.com.ileError(
                                "Cannot com.hazelcast.com.are \"null\" with primitive type \"" + rhsType.toString() + "\"",
                                bo.getLocation()
                            );
                        }
                    } else
                    {

                        // null == null
                        this.pushConstant(bo, null);
                    }
                    this.writeBranch(bo, Opcode.IFNULL + opIdx, dst);
                    return;
                }
            }

            IClass               lhsType            = this.com.hazelcast.com.ileGetValue(bo.lhs);
            CodeContext.Inserter convertLhsInserter = this.getCodeContext().newInserter();
            IClass               rhsType            = this.com.hazelcast.com.ileGetValue(bo.rhs);

            // 15.20.1 Numerical com.hazelcast.com.arison.
            if (
                this.getUnboxedType(lhsType).isPrimitiveNumeric()
                && this.getUnboxedType(rhsType).isPrimitiveNumeric()
                && !(
                    (bo.operator == "==" || bo.operator == "!=") // SUPPRESS CHECKSTYLE StringLiteralEquality
                    && !lhsType.isPrimitive()
                    && !rhsType.isPrimitive()
                )
            ) {
                IClass promotedType = this.binaryNumericPromotion(bo, lhsType, convertLhsInserter, rhsType);
                if (promotedType == IClass.INT) {
                    this.writeBranch(bo, Opcode.IF_ICMPEQ + opIdx, dst);
                } else
                if (promotedType == IClass.LONG) {
                    this.writeOpcode(bo, Opcode.LCMP);
                    this.writeBranch(bo, Opcode.IFEQ + opIdx, dst);
                } else
                if (promotedType == IClass.FLOAT) {
                    if (bo.operator == ">" || bo.operator == ">=") { // SUPPRESS CHECKSTYLE StringLiteralEquality
                        this.writeOpcode(bo, Opcode.FCMPL);
                    } else {
                        this.writeOpcode(bo, Opcode.FCMPG);
                    }
                    this.writeBranch(bo, Opcode.IFEQ + opIdx, dst);
                } else
                if (promotedType == IClass.DOUBLE) {
                    if (bo.operator == ">" || bo.operator == ">=") { // SUPPRESS CHECKSTYLE StringLiteralEquality
                        this.writeOpcode(bo, Opcode.DCMPL);
                    } else {
                        this.writeOpcode(bo, Opcode.DCMPG);
                    }
                    this.writeBranch(bo, Opcode.IFEQ + opIdx, dst);
                } else
                {
                    throw new InternalCompilerException("Unexpected promoted type \"" + promotedType + "\"");
                }
                return;
            }

            // JLS7 15.21.2 Boolean Equality Operators == and !=.
            if (
                (lhsType == IClass.BOOLEAN && this.getUnboxedType(rhsType) == IClass.BOOLEAN)
                || (rhsType == IClass.BOOLEAN && this.getUnboxedType(lhsType) == IClass.BOOLEAN)
            ) {
                if (bo.operator != "==" && bo.operator != "!=") { // SUPPRESS CHECKSTYLE StringLiteralEquality
                    this.com.hazelcast.com.ileError(
                        "Operator \"" + bo.operator + "\" not allowed on boolean operands",
                        bo.getLocation()
                    );
                }
                IClassLoader icl = this.iClassLoader;

                // Unbox LHS if necessary.
                if (lhsType == icl.TYPE_java_lang_Boolean) {
                    this.getCodeContext().pushInserter(convertLhsInserter);
                    try {
                        this.unboxingConversion(bo, icl.TYPE_java_lang_Boolean, IClass.BOOLEAN);
                    } finally {
                        this.getCodeContext().popInserter();
                    }
                }

                // Unbox RHS if necessary.
                if (rhsType == icl.TYPE_java_lang_Boolean) {
                    this.unboxingConversion(bo, icl.TYPE_java_lang_Boolean, IClass.BOOLEAN);
                }

                this.writeBranch(bo, Opcode.IF_ICMPEQ + opIdx, dst);
                return;
            }

            // Reference com.hazelcast.com.arison.
            // Note: Comparison with "null" is already handled above.
            if (!lhsType.isPrimitive() && !rhsType.isPrimitive()) {
                if (bo.operator != "==" && bo.operator != "!=") { // SUPPRESS CHECKSTYLE StringLiteralEquality
                    this.com.hazelcast.com.ileError("Operator \"" + bo.operator + "\" not allowed on reference operands", bo.getLocation()); // SUPPRESS CHECKSTYLE LineLength
                }
                if (
                    !this.isCastReferenceConvertible(lhsType, rhsType)
                    || !this.isCastReferenceConvertible(rhsType, lhsType)
                ) this.com.hazelcast.com.ileError("Incomparable types \"" + lhsType + "\" and \"" + rhsType + "\"", bo.getLocation());
                this.writeBranch(bo, Opcode.IF_ACMPEQ + opIdx, dst);
                return;
            }

            this.com.hazelcast.com.ileError("Cannot com.hazelcast.com.are types \"" + lhsType + "\" and \"" + rhsType + "\"", bo.getLocation());
        }

        this.com.hazelcast.com.ileError("Boolean expression expected", bo.getLocation());
    }

    /**
     * @param dst         Where to jump
     * @param orientation {@link #JUMP_IF_TRUE} or {@link #JUMP_IF_FALSE}
     */
    private void
    com.hazelcast.com.ileBoolean2(ParenthesizedExpression pe, CodeContext.Offset dst, boolean orientation) throws CompileException {
        this.com.hazelcast.com.ileBoolean(pe.value, dst, orientation);
    }

    /**
     * Generates code that determines the context of the {@link Rvalue} and puts it on the operand stack. Most
     * expressions do not have a "context", but some do. E.g. for "x[y]", the context is "x, y". The bottom line is
     * that for statements like "x[y] += 3" the context is only evaluated once.
     *
     * @return The size of the context on the operand stack
     */
    private int
    com.hazelcast.com.ileContext(Rvalue rv) throws CompileException {

        Integer result = (Integer) rv.accept(new RvalueVisitor<Integer, CompileException>() {

            @Override @Nullable public Integer
            visitLvalue(Lvalue lv) throws CompileException {
                return (Integer) lv.accept(new Visitor.LvalueVisitor<Integer, CompileException>() {

                    // SUPPRESS CHECKSTYLE LineLength:7
                    @Override public Integer visitAmbiguousName(AmbiguousName an)                                        throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileContext2(an);    }
                    @Override public Integer visitArrayAccessExpression(ArrayAccessExpression aae)                       throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileContext2(aae);   }
                    @Override public Integer visitFieldAccess(FieldAccess fa)                                            throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileContext2(fa);    }
                    @Override public Integer visitFieldAccessExpression(FieldAccessExpression fae)                       throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileContext2(fae);   }
                    @Override public Integer visitSuperclassFieldAccessExpression(SuperclassFieldAccessExpression scfae) throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileContext2(scfae); }
                    @Override public Integer visitLocalVariableAccess(LocalVariableAccess lva)                                                   { return UnitCompiler.this.com.hazelcast.com.ileContext2(lva);   }
                    @Override public Integer visitParenthesizedExpression(ParenthesizedExpression pe)                    throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileContext2(pe);    }
                });
            }

            // SUPPRESS CHECKSTYLE LineLength:25
            @Override public Integer visitArrayLength(ArrayLength al)        throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileContext2(al);   }
            @Override public Integer visitAssignment(Assignment a)                                   { return UnitCompiler.this.com.hazelcast.com.ileContext2(a);    }
            @Override public Integer visitUnaryOperation(UnaryOperation uo)                          { return UnitCompiler.this.com.hazelcast.com.ileContext2(uo);   }
            @Override public Integer visitBinaryOperation(BinaryOperation bo)                        { return UnitCompiler.this.com.hazelcast.com.ileContext2(bo);   }
            @Override public Integer visitCast(Cast c)                                               { return UnitCompiler.this.com.hazelcast.com.ileContext2(c);    }
            @Override public Integer visitClassLiteral(ClassLiteral cl)                              { return UnitCompiler.this.com.hazelcast.com.ileContext2(cl);   }
            @Override public Integer visitConditionalExpression(ConditionalExpression ce)            { return UnitCompiler.this.com.hazelcast.com.ileContext2(ce);   }
            @Override public Integer visitCrement(Crement c)                                         { return UnitCompiler.this.com.hazelcast.com.ileContext2(c);    }
            @Override public Integer visitInstanceof(Instanceof io)                                  { return UnitCompiler.this.com.hazelcast.com.ileContext2(io);   }
            @Override public Integer visitMethodInvocation(MethodInvocation mi)                      { return UnitCompiler.this.com.hazelcast.com.ileContext2(mi);   }
            @Override public Integer visitSuperclassMethodInvocation(SuperclassMethodInvocation smi) { return UnitCompiler.this.com.hazelcast.com.ileContext2(smi);  }
            @Override public Integer visitIntegerLiteral(IntegerLiteral il)                          { return UnitCompiler.this.com.hazelcast.com.ileContext2(il);   }
            @Override public Integer visitFloatingPointLiteral(FloatingPointLiteral fpl)             { return UnitCompiler.this.com.hazelcast.com.ileContext2(fpl);  }
            @Override public Integer visitBooleanLiteral(BooleanLiteral bl)                          { return UnitCompiler.this.com.hazelcast.com.ileContext2(bl);   }
            @Override public Integer visitCharacterLiteral(CharacterLiteral cl)                      { return UnitCompiler.this.com.hazelcast.com.ileContext2(cl);   }
            @Override public Integer visitStringLiteral(StringLiteral sl)                            { return UnitCompiler.this.com.hazelcast.com.ileContext2(sl);   }
            @Override public Integer visitNullLiteral(NullLiteral nl)                                { return UnitCompiler.this.com.hazelcast.com.ileContext2(nl);   }
            @Override public Integer visitSimpleConstant(SimpleConstant sl)                          { return UnitCompiler.this.com.hazelcast.com.ileContext2(sl);   }
            @Override public Integer visitNewAnonymousClassInstance(NewAnonymousClassInstance naci)  { return UnitCompiler.this.com.hazelcast.com.ileContext2(naci); }
            @Override public Integer visitNewArray(NewArray na)                                      { return UnitCompiler.this.com.hazelcast.com.ileContext2(na);   }
            @Override public Integer visitNewInitializedArray(NewInitializedArray nia)               { return UnitCompiler.this.com.hazelcast.com.ileContext2(nia);  }
            @Override public Integer visitNewClassInstance(NewClassInstance nci)                     { return UnitCompiler.this.com.hazelcast.com.ileContext2(nci);  }
            @Override public Integer visitParameterAccess(ParameterAccess pa)                        { return UnitCompiler.this.com.hazelcast.com.ileContext2(pa);   }
            @Override public Integer visitQualifiedThisReference(QualifiedThisReference qtr)         { return UnitCompiler.this.com.hazelcast.com.ileContext2(qtr);  }
            @Override public Integer visitThisReference(ThisReference tr)                            { return UnitCompiler.this.com.hazelcast.com.ileContext2(tr);   }
        });

        assert result != null;
        return result;
    }

    @SuppressWarnings("static-method") private int
    com.hazelcast.com.ileContext2(Rvalue rv) { return 0; }

    private int
    com.hazelcast.com.ileContext2(AmbiguousName an) throws CompileException {
        return this.com.hazelcast.com.ileContext(this.toRvalueOrCompileException(this.reclassify(an)));
    }

    private int
    com.hazelcast.com.ileContext2(FieldAccess fa) throws CompileException {
        if (fa.field.isStatic()) {
            Rvalue rv = fa.lhs.toRvalue();
            if (rv != null) {
                this.warning(
                    "CNSFA",
                    "Left-hand side of static field access should be a type, not an rvalue",
                    fa.lhs.getLocation()
                );
                // JLS7 15.11.1.3.1.1:
                this.pop(fa.lhs, this.com.hazelcast.com.ileGetValue(rv));
            }
            return 0;
        } else {
            this.com.hazelcast.com.ileGetValue(this.toRvalueOrCompileException(fa.lhs));
            return 1;
        }
    }

    private int
    com.hazelcast.com.ileContext2(ArrayLength al) throws CompileException {
        if (!this.com.hazelcast.com.ileGetValue(al.lhs).isArray()) {
            this.com.hazelcast.com.ileError("Cannot determine length of non-array type", al.getLocation());
        }
        return 1;
    }

    private int
    com.hazelcast.com.ileContext2(ArrayAccessExpression aae) throws CompileException {
        IClass lhsType = this.com.hazelcast.com.ileGetValue(aae.lhs);
        if (!lhsType.isArray()) {
            this.com.hazelcast.com.ileError(
                "Subscript not allowed on non-array type \"" + lhsType.toString() + "\"",
                aae.getLocation()
            );
        }

        IClass indexType = this.com.hazelcast.com.ileGetValue(aae.index);
        if (
            !this.tryIdentityConversion(indexType, IClass.INT)
            && !this.tryWideningPrimitiveConversion(aae, indexType, IClass.INT)
        ) this.com.hazelcast.com.ileError(
            "Index expression of type \"" + indexType + "\" cannot be widened to \"int\"",
            aae.getLocation()
        );

        return 2;
    }

    private int
    com.hazelcast.com.ileContext2(FieldAccessExpression fae) throws CompileException {
        return this.com.hazelcast.com.ileContext(this.determineValue(fae));
    }

    private int
    com.hazelcast.com.ileContext2(SuperclassFieldAccessExpression scfae) throws CompileException {
        return this.com.hazelcast.com.ileContext(this.determineValue(scfae));
    }

    private int
    com.hazelcast.com.ileContext2(ParenthesizedExpression pe) throws CompileException {
        return this.com.hazelcast.com.ileContext(pe.value);
    }

    /**
     * Generates code that determines the value of the {@link Rvalue} and puts it on the operand stack. This method
     * relies on that the "context" of the {@link Rvalue} is on top of the operand stack (see {@link
     * #com.hazelcast.com.ileContext(Rvalue)}).
     *
     * @return The type of the {@link Rvalue}
     */
    private IClass
    com.hazelcast.com.ileGet(Rvalue rv) throws CompileException {

        IClass result = (IClass) rv.accept(new RvalueVisitor<IClass, CompileException>() {

            @Override @Nullable public IClass
            visitLvalue(Lvalue lv) throws CompileException {
                return (IClass) lv.accept(new Visitor.LvalueVisitor<IClass, CompileException>() {

                    // SUPPRESS CHECKSTYLE LineLength:7
                    @Override public IClass visitAmbiguousName(AmbiguousName an)                                        throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(an);    }
                    @Override public IClass visitArrayAccessExpression(ArrayAccessExpression aae)                       throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(aae);   }
                    @Override public IClass visitFieldAccess(FieldAccess fa)                                            throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(fa);    }
                    @Override public IClass visitFieldAccessExpression(FieldAccessExpression fae)                       throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(fae);   }
                    @Override public IClass visitSuperclassFieldAccessExpression(SuperclassFieldAccessExpression scfae) throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(scfae); }
                    @Override public IClass visitLocalVariableAccess(LocalVariableAccess lva)                                                   { return UnitCompiler.this.com.hazelcast.com.ileGet2(lva);   }
                    @Override public IClass visitParenthesizedExpression(ParenthesizedExpression pe)                    throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(pe);    }
                });
            }

            // SUPPRESS CHECKSTYLE LineLength:25
            @Override public IClass visitArrayLength(ArrayLength al)                                                        { return UnitCompiler.this.com.hazelcast.com.ileGet2(al);   }
            @Override public IClass visitAssignment(Assignment a)                                   throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(a);    }
            @Override public IClass visitUnaryOperation(UnaryOperation uo)                          throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(uo);   }
            @Override public IClass visitBinaryOperation(BinaryOperation bo)                        throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(bo);   }
            @Override public IClass visitCast(Cast c)                                               throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(c);    }
            @Override public IClass visitClassLiteral(ClassLiteral cl)                              throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(cl);   }
            @Override public IClass visitConditionalExpression(ConditionalExpression ce)            throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(ce);   }
            @Override public IClass visitCrement(Crement c)                                         throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(c);    }
            @Override public IClass visitInstanceof(Instanceof io)                                  throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(io);   }
            @Override public IClass visitMethodInvocation(MethodInvocation mi)                      throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(mi);   }
            @Override public IClass visitSuperclassMethodInvocation(SuperclassMethodInvocation smi) throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(smi);  }
            @Override public IClass visitIntegerLiteral(IntegerLiteral il)                          throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(il);   }
            @Override public IClass visitFloatingPointLiteral(FloatingPointLiteral fpl)             throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(fpl);  }
            @Override public IClass visitBooleanLiteral(BooleanLiteral bl)                          throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(bl);   }
            @Override public IClass visitCharacterLiteral(CharacterLiteral cl)                      throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(cl);   }
            @Override public IClass visitStringLiteral(StringLiteral sl)                            throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(sl);   }
            @Override public IClass visitNullLiteral(NullLiteral nl)                                throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(nl);   }
            @Override public IClass visitSimpleConstant(SimpleConstant sl)                          throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(sl);   }
            @Override public IClass visitNewAnonymousClassInstance(NewAnonymousClassInstance naci)  throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(naci); }
            @Override public IClass visitNewArray(NewArray na)                                      throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(na);   }
            @Override public IClass visitNewInitializedArray(NewInitializedArray nia)               throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(nia);  }
            @Override public IClass visitNewClassInstance(NewClassInstance nci)                     throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(nci);  }
            @Override public IClass visitParameterAccess(ParameterAccess pa)                        throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(pa);   }
            @Override public IClass visitQualifiedThisReference(QualifiedThisReference qtr)         throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(qtr);  }
            @Override public IClass visitThisReference(ThisReference tr)                            throws CompileException { return UnitCompiler.this.com.hazelcast.com.ileGet2(tr);   }
        });

        assert result != null;
        return result;
    }

    private IClass
    com.hazelcast.com.ileGet2(BooleanRvalue brv) throws CompileException {
        CodeContext.Offset isTrue = this.getCodeContext().new Offset();
        this.com.hazelcast.com.ileBoolean(brv, isTrue, UnitCompiler.JUMP_IF_TRUE);
        this.writeOpcode(brv, Opcode.ICONST_0);
        CodeContext.Offset end = this.getCodeContext().new Offset();
        this.writeBranch(brv, Opcode.GOTO, end);
        isTrue.set();
        this.writeOpcode(brv, Opcode.ICONST_1);
        end.set();

        return IClass.BOOLEAN;
    }

    private IClass
    com.hazelcast.com.ileGet2(AmbiguousName an) throws CompileException {
        return this.com.hazelcast.com.ileGet(this.toRvalueOrCompileException(this.reclassify(an)));
    }

    private IClass
    com.hazelcast.com.ileGet2(LocalVariableAccess lva) { return this.load(lva, lva.localVariable); }

    private IClass
    com.hazelcast.com.ileGet2(FieldAccess fa) throws CompileException {
        this.checkAccessible(fa.field, fa.getEnclosingScope(), fa.getLocation());
        this.getfield(fa, fa.field);
        return fa.field.getType();
    }

    private IClass
    com.hazelcast.com.ileGet2(ArrayLength al) {
        this.writeOpcode(al, Opcode.ARRAYLENGTH);
        return IClass.INT;
    }

    private IClass
    com.hazelcast.com.ileGet2(ThisReference tr) throws CompileException {
        this.referenceThis(tr);
        return this.getIClass(tr);
    }

    private IClass
    com.hazelcast.com.ileGet2(QualifiedThisReference qtr) throws CompileException {
        this.referenceThis(
            qtr,                                       // locatable
            this.getDeclaringClass(qtr),               // declaringClass
            this.getDeclaringTypeBodyDeclaration(qtr), // declaringTypeBodyDeclaration
            this.getTargetIClass(qtr)                  // targetIClass
        );
        return this.getTargetIClass(qtr);
    }

    private IClass
    com.hazelcast.com.ileGet2(ClassLiteral cl) throws CompileException {
        final Location loc = cl.getLocation();

        IClass iClass = this.getType(cl.type);
        if (iClass.isPrimitive()) {

            // Primitive class literal.
            this.writeOpcode(cl, Opcode.GETSTATIC);
            String wrapperClassDescriptor = (
                iClass == IClass.VOID    ? "Ljava/lang/Void;"      :
                iClass == IClass.BYTE    ? "Ljava/lang/Byte;"      :
                iClass == IClass.CHAR    ? "Ljava/lang/Character;" :
                iClass == IClass.DOUBLE  ? "Ljava/lang/Double;"    :
                iClass == IClass.FLOAT   ? "Ljava/lang/Float;"     :
                iClass == IClass.INT     ? "Ljava/lang/Integer;"   :
                iClass == IClass.LONG    ? "Ljava/lang/Long;"      :
                iClass == IClass.SHORT   ? "Ljava/lang/Short;"     :
                iClass == IClass.BOOLEAN ? "Ljava/lang/Boolean;"   :
                null
            );
            if (wrapperClassDescriptor == null) {
                throw new InternalCompilerException("SNO: Unidentifiable primitive type \"" + iClass + "\"");
            }

            this.writeConstantFieldrefInfo(
                wrapperClassDescriptor, // classFD
                "TYPE",                 // fieldName
                "Ljava/lang/Class;"     // fieldFD
            );
            return this.iClassLoader.TYPE_java_lang_Class;
        }

        // Non-primitive class literal.

        TypeDeclaration declaringType;
        for (Scope s = cl.getEnclosingScope();; s = s.getEnclosingScope()) {
            if (s instanceof TypeDeclaration) {
                declaringType = (AbstractTypeDeclaration) s;
                break;
            }
        }

        // Check if synthetic method "static Class class$(String className)" is already declared.
        if (declaringType.getMethodDeclaration("class$") == null) this.declareClassDollarMethod(cl);

        // Determine the statics of the declaring class (this is where static fields declarations are found).
        List<? extends BlockStatement> statics;
        if (declaringType instanceof AbstractClassDeclaration) {
            statics = ((AbstractClassDeclaration) declaringType).variableDeclaratorsAndInitializers;
        } else
        if (declaringType instanceof InterfaceDeclaration) {
            statics = ((InterfaceDeclaration) declaringType).constantDeclarations;
        } else {
            throw new InternalCompilerException(
                "SNO: AbstractTypeDeclaration is neither ClassDeclaration nor InterfaceDeclaration"
            );
        }

        String className = Descriptor.toClassName(iClass.getDescriptor());

        // Compose the "class-dollar" field name. This i done as follows:
        //   Type         Class-name           Field-name
        //   String       java.lang.String     class$java$lang$String
        //   String[]     [Ljava.lang.String;  array$Ljava$lang$String
        //   String[][]   [[Ljava.lang.String; array$$Ljava$lang$String
        //   String[][][] [[[java.lang.String; array$$$Ljava$lang$String
        //   int[]        [I                   array$I
        //   int[][]      [[I                  array$$I
        String classDollarFieldName;
        {
            if (className.startsWith("[")) {
                classDollarFieldName = "array" + className.replace('.', '$').replace('[', '$');
                if (classDollarFieldName.endsWith(";")) {
                    classDollarFieldName = classDollarFieldName.substring(0, classDollarFieldName.length() - 1);
                }
            } else
            {
                classDollarFieldName = "class$" + className.replace('.', '$');
            }
        }

        // Declare the static "class dollar field" if not already done.
        ADD_CLASS_DOLLAR_FIELD: {
            for (BlockStatement bs : statics) {
                if (!((TypeBodyDeclaration) bs).isStatic()) continue;
                if (bs instanceof FieldDeclaration) {
                    for (VariableDeclarator vd : ((FieldDeclaration) bs).variableDeclarators) {
                        if (vd.name.equals(classDollarFieldName)) {
                            break ADD_CLASS_DOLLAR_FIELD;
                        }
                    }
                }
            }

            Type             classType = new SimpleType(loc, this.iClassLoader.TYPE_java_lang_Class);
            FieldDeclaration fd        = new FieldDeclaration(
                loc,                       // location
                null,                      // optionalDocComment
                new Modifiers(Mod.STATIC), // modifiers
                classType,                 // type
                new VariableDeclarator[] { // variableDeclarators
                    new VariableDeclarator(
                        loc,                  // location
                        classDollarFieldName, // name
                        0,                    // brackets
                        (Rvalue) null         // optionalInitializer
                    )
                }
            );
            if (declaringType instanceof AbstractClassDeclaration) {
                ((AbstractClassDeclaration) declaringType).addFieldDeclaration(fd);
            } else
            if (declaringType instanceof InterfaceDeclaration) {
                ((InterfaceDeclaration) declaringType).addConstantDeclaration(fd);
            } else {
                throw new InternalCompilerException(
                    "SNO: AbstractTypeDeclaration is neither ClassDeclaration nor InterfaceDeclaration"
                );
            }
        }

        // return (class$X != null) ? class$X : (class$X = class$("X"));
        Type   declaringClassOrInterfaceType = new SimpleType(loc, this.resolve(declaringType));
        Lvalue classDollarFieldAccess        = new FieldAccessExpression(
            loc,                           // location
            declaringClassOrInterfaceType, // lhs
            classDollarFieldName           // fieldName
        );
        ConditionalExpression ce = new ConditionalExpression(
            loc,                          // location
            new BinaryOperation(          // lhs
                loc,                    // location
                classDollarFieldAccess, // lhs
                "!=",                   // operator
                new NullLiteral(loc)    // rhs
            ),
            classDollarFieldAccess,       // mhs
            new Assignment(               // rhs
                loc,                          // location
                classDollarFieldAccess,       // lhs
                "=",                          // operator
                new MethodInvocation(         // rhs
                    loc,                           // location
                    declaringClassOrInterfaceType, // optionalTarget
                    "class$",                      // methodName
                    new Rvalue[] {                 // arguments
                        new StringLiteral(loc, '"' + className + '"')
                    }
                )
            )
        );
        ce.setEnclosingScope(cl.getEnclosingScope());
        return this.com.hazelcast.com.ileGet(ce);
    }
    private IClass
    com.hazelcast.com.ileGet2(Assignment a) throws CompileException {
        if (a.operator == "=") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            int    lhsCs   = this.com.hazelcast.com.ileContext(a.lhs);
            IClass rhsType = this.com.hazelcast.com.ileGetValue(a.rhs);
            IClass lhsType = this.getType(a.lhs);
            Object rhsCv   = this.getConstantValue(a.rhs);
            this.assignmentConversion(a, rhsType, lhsType, rhsCv);
            this.dupx(a, lhsType, lhsCs);
            this.com.hazelcast.com.ileSet(a.lhs);
            return lhsType;
        }

        // Implement "|= ^= &= *= /= %= += -= <<= >>= >>>=".
        int lhsCs = this.com.hazelcast.com.ileContext(a.lhs);
        this.dup(a, lhsCs);
        IClass lhsType    = this.com.hazelcast.com.ileGet(a.lhs);
        IClass resultType = this.com.hazelcast.com.ileArithmeticBinaryOperation(
            a,                    // locatable
            lhsType,              // lhsType
            a.operator.substring( // operator
                0,
                a.operator.length() - 1
            ).intern(), /* <= IMPORTANT!
            */
            a.rhs                 // rhs
        );
        // Convert the result to LHS type (JLS7 15.26.2).
        if (
            !this.tryIdentityConversion(resultType, lhsType)
            && !this.tryNarrowingPrimitiveConversion(a, resultType, lhsType)
        ) throw new InternalCompilerException("SNO: \"" + a.operator + "\" reconversion failed");
        this.dupx(a, lhsType, lhsCs);
        this.com.hazelcast.com.ileSet(a.lhs);
        return lhsType;
    }

    private IClass
    com.hazelcast.com.ileGet2(ConditionalExpression ce) throws CompileException {
        IClass               mhsType, rhsType;
        CodeContext.Inserter mhsConvertInserter, rhsConvertInserter;
        CodeContext.Offset   toEnd = this.getCodeContext().new Offset();
        {
            Object cv = this.getConstantValue(ce.lhs);
            if (cv instanceof Boolean) {
                if (((Boolean) cv).booleanValue()) {
                    mhsType            = this.com.hazelcast.com.ileGetValue(ce.mhs);
                    mhsConvertInserter = this.getCodeContext().newInserter();
                    rhsType            = this.getType(ce.rhs);
                    rhsConvertInserter = null;
                } else {
                    mhsType            = this.getType(ce.mhs);
                    mhsConvertInserter = null;
                    rhsType            = this.com.hazelcast.com.ileGetValue(ce.rhs);
                    rhsConvertInserter = this.getCodeContext().currentInserter();
                }
            } else {
                CodeContext.Offset toRhs = this.getCodeContext().new Offset();

                this.com.hazelcast.com.ileBoolean(ce.lhs, toRhs, UnitCompiler.JUMP_IF_FALSE);
                mhsType            = this.com.hazelcast.com.ileGetValue(ce.mhs);
                mhsConvertInserter = this.getCodeContext().newInserter();
                this.writeBranch(ce, Opcode.GOTO, toEnd);
                toRhs.set();
                rhsType            = this.com.hazelcast.com.ileGetValue(ce.rhs);
                rhsConvertInserter = this.getCodeContext().currentInserter();
            }
        }
        IClass expressionType;
        if (mhsType == rhsType) {

            // JLS7 15.25, list 1, bullet 1: "b ? T : T => T"
            expressionType = mhsType;
        } else
        if (this.tryUnboxingConversion(ce.mhs, mhsType, rhsType, mhsConvertInserter)) {

            // JLS7 15.25, list 1, bullet 2: "b ? Integer : int => int"
            expressionType = rhsType;
        } else
        if (this.tryUnboxingConversion(ce.rhs, rhsType, mhsType, rhsConvertInserter)) {

            // JLS7 15.25, list 1, bullet 2: "b ? int : Integer => int"
            expressionType = mhsType;
        } else
        if (this.getConstantValue(ce.mhs) == null && !rhsType.isPrimitive()) {

            // JLS7 15.25, list 1, bullet 3: "b ? null : ReferenceType => ReferenceType"
            expressionType = rhsType;
        } else
        if (!mhsType.isPrimitive() && this.getConstantValue(ce.rhs) == null) {

            // JLS7 15.25, list 1, bullet 3: "b ? ReferenceType : null => ReferenceType"
            expressionType = mhsType;
        } else
        if (ce.mhs.constantValue == null && rhsType.isPrimitive()) {
            expressionType = this.isBoxingConvertible(rhsType);
            assert expressionType != null : rhsType + " is not boxing convertible";
        } else
        if (ce.rhs.constantValue == null && mhsType.isPrimitive()) {
            expressionType = this.isBoxingConvertible(mhsType);
            assert expressionType != null : mhsType + " is not boxing convertible";
        } else
        if (this.isConvertibleToPrimitiveNumeric(mhsType) && this.isConvertibleToPrimitiveNumeric(rhsType)) {

            // TODO JLS7 15.25, list 1, bullet 4, bullet 1: "b ? Byte : Short => short"

            Object rhscv = ce.rhs.constantValue;
            Object mhscv = ce.mhs.constantValue;

            if (
                mhsType == IClass.BYTE
                && rhsType == IClass.INT
                && UnitCompiler.isByteConstant(rhscv) != null
            ) {

                // JLS7 15.25, list 1, bullet 4, bullet 2: "b ? byte : 127 => byte"
                expressionType = IClass.BYTE;
                // fix up the constant to be a byte
                ce.rhs.constantValue = UnitCompiler.isByteConstant(rhscv);
            } else
            if (
                mhsType == IClass.INT
                && rhsType == IClass.BYTE
                && UnitCompiler.isByteConstant(mhscv) != null
            ) {

                // JLS7 15.25, list 1, bullet 4, bullet 3: "b ? 127 : byte => byte"
                expressionType = IClass.BYTE;
                // fix up the constant to be a byte
                ce.mhs.constantValue = UnitCompiler.isByteConstant(mhscv);
            } else
            {

                // JLS7 15.25, list 1, bullet 4, bullet 4: "b ? Integer : Double => double"
                expressionType = this.binaryNumericPromotion(
                    ce,                 // locatable
                    mhsType,            // type1
                    mhsConvertInserter, // convertInserter1
                    rhsType,            // type2
                    rhsConvertInserter  // convertInserter2
                );
            }
        } else
        if (!mhsType.isPrimitive() && !rhsType.isPrimitive()) {

            // JLS7 15.25, list 1, bullet 5: "b ? Base : Derived => Base"
            if (mhsType.isAssignableFrom(rhsType)) {
                expressionType = mhsType;
            } else
            if (rhsType.isAssignableFrom(mhsType)) {
                expressionType = rhsType;
            } else
            {
                this.com.hazelcast.com.ileError(
                    "Reference types \"" + mhsType + "\" and \"" + rhsType + "\" don't match",
                    ce.getLocation()
                );
                return this.iClassLoader.TYPE_java_lang_Object;
            }
        } else
        {
            this.com.hazelcast.com.ileError(
                "Incompatible expression types \"" + mhsType + "\" and \"" + rhsType + "\"",
                ce.getLocation()
            );
            return this.iClassLoader.TYPE_java_lang_Object;
        }
        toEnd.set();

        return expressionType;
    }

    @Nullable private static Byte
    isByteConstant(@Nullable Object o) {
        if (o instanceof Integer) {
            int v = (Integer) o;
            return v >= Byte.MIN_VALUE && v <= Byte.MAX_VALUE ? Byte.valueOf((byte) v) : null;
        }
        return null;
    }

    private IClass
    com.hazelcast.com.ileGet2(Crement c) throws CompileException {

        // Optimized crement of integer local variable.
        LocalVariable lv = this.isIntLv(c);
        if (lv != null) {
            if (!c.pre) this.load(c, lv);
            this.com.hazelcast.com.ileLocalVariableCrement(c, lv);
            if (c.pre) this.load(c, lv);
            return lv.type;
        }

        // Compile operand context.
        int cs = this.com.hazelcast.com.ileContext(c.operand);
        // DUP operand context.
        this.dup(c, cs);
        // Get operand value.
        IClass type = this.com.hazelcast.com.ileGet(c.operand);
        // DUPX operand value.
        if (!c.pre) this.dupx(c, type, cs);
        // Apply "unary numeric promotion".
        IClass promotedType = this.unaryNumericPromotion(c, type);
        // Crement.
        this.writeOpcode(c, UnitCompiler.ilfd(
            promotedType,
            Opcode.ICONST_1,
            Opcode.LCONST_1,
            Opcode.FCONST_1,
            Opcode.DCONST_1
        ));
        if (c.operator == "++") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            this.writeOpcode(c, Opcode.IADD + UnitCompiler.ilfd(promotedType));
        } else
        if (c.operator == "--") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            this.writeOpcode(c, Opcode.ISUB + UnitCompiler.ilfd(promotedType));
        } else {
            this.com.hazelcast.com.ileError("Unexpected operator \"" + c.operator + "\"", c.getLocation());
        }
        this.reverseUnaryNumericPromotion(c, promotedType, type);
        // DUPX cremented operand value.
        if (c.pre) this.dupx(c, type, cs);
        // Set operand.
        this.com.hazelcast.com.ileSet(c.operand);

        return type;
    }

    private void
    com.hazelcast.com.ileLocalVariableCrement(Crement c, LocalVariable lv) {
        this.crement(c, lv, c.operator);
    }

    /**
     * @param operator Must be either "++" or "--", as an @link {@link String#intern() interned} string
     */
    private void
    crement(Locatable locatable, LocalVariable lv, String operator) {
        if (lv.getSlotIndex() > 255) {
            this.writeOpcode(locatable, Opcode.WIDE);
            this.writeOpcode(locatable, Opcode.IINC);
            this.writeShort(lv.getSlotIndex());
            this.writeShort(operator == "++" ? 1 : -1); // SUPPRESS CHECKSTYLE StringLiteralEquality
        } else {
            this.writeOpcode(locatable, Opcode.IINC);
            this.writeByte(lv.getSlotIndex());
            this.writeByte(operator == "++" ? 1 : -1); // SUPPRESS CHECKSTYLE StringLiteralEquality
        }
    }

    private IClass
    com.hazelcast.com.ileGet2(ArrayAccessExpression aae) throws CompileException {
        IClass lhsComponentType = this.getType(aae);
        this.writeOpcode(aae, Opcode.IALOAD + UnitCompiler.ilfdabcs(lhsComponentType));
        return lhsComponentType;
    }

    private IClass
    com.hazelcast.com.ileGet2(FieldAccessExpression fae) throws CompileException {
        return this.com.hazelcast.com.ileGet(this.determineValue(fae));
    }

    private IClass
    com.hazelcast.com.ileGet2(SuperclassFieldAccessExpression scfae) throws CompileException {
        return this.com.hazelcast.com.ileGet(this.determineValue(scfae));
    }

    private IClass
    com.hazelcast.com.ileGet2(UnaryOperation uo) throws CompileException {

        if (uo.operator == "!") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            return this.com.hazelcast.com.ileGet2((BooleanRvalue) uo);
        }

        if (uo.operator == "+") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            return this.unaryNumericPromotion(
                uo,
                this.convertToPrimitiveNumericType(uo, this.com.hazelcast.com.ileGetValue(uo.operand))
            );
        }

        if (uo.operator == "-") { // SUPPRESS CHECKSTYLE StringLiteralEquality

            {
                Object ncv = this.getConstantValue2(uo);
                if (ncv != UnitCompiler.NOT_CONSTANT) {
                    return this.unaryNumericPromotion(uo, this.pushConstant(uo, ncv));
                }
            }

            IClass promotedType = this.unaryNumericPromotion(
                uo,
                this.convertToPrimitiveNumericType(uo, this.com.hazelcast.com.ileGetValue(uo.operand))
            );
            this.writeOpcode(uo, Opcode.INEG + UnitCompiler.ilfd(promotedType));
            return promotedType;
        }

        if (uo.operator == "~") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            IClass operandType = this.com.hazelcast.com.ileGetValue(uo.operand);

            IClass promotedType = this.unaryNumericPromotion(uo, operandType);
            if (promotedType == IClass.INT) {
                this.writeOpcode(uo, Opcode.ICONST_M1);
                this.writeOpcode(uo, Opcode.IXOR);
                return IClass.INT;
            }
            if (promotedType == IClass.LONG) {
                this.writeOpcode(uo, Opcode.LDC2_W);
                this.writeConstantLongInfo(-1L);
                this.writeOpcode(uo, Opcode.LXOR);
                return IClass.LONG;
            }
            this.com.hazelcast.com.ileError("Operator \"~\" not applicable to type \"" + promotedType + "\"", uo.getLocation());
        }

        this.com.hazelcast.com.ileError("Unexpected operator \"" + uo.operator + "\"", uo.getLocation());
        return this.iClassLoader.TYPE_java_lang_Object;
    }

    private IClass
    com.hazelcast.com.ileGet2(Instanceof io) throws CompileException {
        IClass lhsType = this.com.hazelcast.com.ileGetValue(io.lhs);
        IClass rhsType = this.getType(io.rhs);
        if (
            lhsType.isInterface() || rhsType.isInterface()
            // We cannot precompute the result from type information as the value might be null, but we should detect
            // when the instanceof is statically impossible.
            || lhsType.isAssignableFrom(rhsType)
            || rhsType.isAssignableFrom(lhsType)
        ) {
            this.writeOpcode(io, Opcode.INSTANCEOF);
            this.writeConstantClassInfo(rhsType.getDescriptor());
        } else {
            this.com.hazelcast.com.ileError("\"" + lhsType + "\" can never be an instance of \"" + rhsType + "\"", io.getLocation());
        }
        return IClass.BOOLEAN;
    }

    private IClass
    com.hazelcast.com.ileGet2(BinaryOperation bo) throws CompileException {
        if (
            bo.operator == "||"    // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == "&&" // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == "==" // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == "!=" // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == "<"  // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == ">"  // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == "<=" // SUPPRESS CHECKSTYLE StringLiteralEquality
            || bo.operator == ">=" // SUPPRESS CHECKSTYLE StringLiteralEquality
        ) {
            // Eventually calls "com.hazelcast.com.ileBoolean()".
            return this.com.hazelcast.com.ileGet2((BooleanRvalue) bo);
        }

        // Implements "| ^ & * / % + - << >> >>>".
        return this.com.hazelcast.com.ileArithmeticOperation(
            bo,                         // locatable
            null,                       // type
            bo.unrollLeftAssociation(), // operands
            bo.operator                 // operator
        );
    }

    private IClass
    com.hazelcast.com.ileGet2(Cast c) throws CompileException {

        // JLS7 5.5 Casting Conversion.
        IClass tt = this.getType(c.targetType);
        IClass vt = this.com.hazelcast.com.ileGetValue(c.value);
        if (
            this.tryIdentityConversion(vt, tt)
            || this.tryWideningPrimitiveConversion(c, vt, tt)
            || this.tryNarrowingPrimitiveConversion(c, vt, tt)
            || this.tryWideningReferenceConversion(vt, tt)
            || this.tryNarrowingReferenceConversion(c, vt, tt)
            || this.tryBoxingConversion(c, vt, tt)
            || this.tryUnboxingConversion(c, vt, tt, null)
        ) return tt;

        // JAVAC obviously also permits 'boxing conversion followed by widening reference conversion' and 'unboxing
        // conversion followed by widening primitive conversion', although these are not described in JLS7 5.5. For the
        // sake of com.hazelcast.com.atibility, we implement them.
        // See also http://jira.codehaus.com.hazelcast.org.browse/JANINO-153
        {
            IClass boxedType = this.isBoxingConvertible(vt);
            if (boxedType != null && this.isWideningReferenceConvertible(boxedType, tt)) {
                this.boxingConversion(c, vt, boxedType);
                return tt;
            }

            IClass unboxedType = this.isUnboxingConvertible(vt);
            if (unboxedType != null && this.isWideningPrimitiveConvertible(unboxedType, tt)) {
                this.unboxingConversion(c, vt, unboxedType);
                this.tryWideningPrimitiveConversion(c, unboxedType, tt);
                return tt;
            }
        }

        this.com.hazelcast.com.ileError("Cannot cast \"" + vt + "\" to \"" + tt + "\"", c.getLocation());
        return tt;
    }

    private IClass
    com.hazelcast.com.ileGet2(ParenthesizedExpression pe) throws CompileException {
        return this.com.hazelcast.com.ileGet(pe.value);
    }

    private IClass
    com.hazelcast.com.ileGet2(MethodInvocation mi) throws CompileException {
        IClass.IMethod iMethod = this.findIMethod(mi);

        // Compute the objectref for an instance method.
        Atom ot = mi.optionalTarget;
        if (ot == null) {

            // JLS7 6.5.7.1, 15.12.4.1.1.1
            TypeBodyDeclaration      scopeTbd;
            AbstractClassDeclaration scopeClassDeclaration;
            {
                Scope s;
                for (
                    s = mi.getEnclosingScope();
                    !(s instanceof TypeBodyDeclaration);
                    s = s.getEnclosingScope()
                );
                scopeTbd = (TypeBodyDeclaration) s;
                if (!(s instanceof AbstractClassDeclaration)) s = s.getEnclosingScope();
                scopeClassDeclaration = (AbstractClassDeclaration) s;
            }
            if (iMethod.isStatic()) {
                this.warning(
                    "IASM",
                    "Implicit access to static method \"" + iMethod.toString() + "\"",
                    mi.getLocation()
                );
                // JLS7 15.12.4.1.1.1.1
                ;
            } else {
                this.warning(
                    "IANSM",
                    "Implicit access to non-static method \"" + iMethod.toString() + "\"",
                    mi.getLocation()
                );
                // JLS7 15.12.4.1.1.1.2
                if (scopeTbd.isStatic()) {
                    this.com.hazelcast.com.ileError(
                        "Instance method \"" + iMethod.toString() + "\" cannot be invoked in static context",
                        mi.getLocation()
                    );
                }
                this.referenceThis(
                    mi,                          // locatable
                    scopeClassDeclaration,       // declaringClass
                    scopeTbd,                    // declaringTypeBodyDeclaration
                    iMethod.getDeclaringIClass() // targetIClass
                );
            }
        } else {

            // 6.5.7.2
            if (this.isType(ot)) {

                // The target is a type; thus the method must be static.
                // JLS9 15.12.4.1.2:
                this.getType(this.toTypeOrCompileException(ot));
                if (!iMethod.isStatic()) {
                    this.com.hazelcast.com.ileError(
                        "Instance method \"" + mi.methodName + "\" cannot be invoked in static context",
                        mi.getLocation()
                    );
                }
            } else
            {

                // The target is an rvalue.
                Rvalue rot = this.toRvalueOrCompileException(ot);
                if (iMethod.isStatic()) {

                    // JLS9 15.12.4.1.3.1 and .4.1:
                    if (!UnitCompiler.mayHaveSideEffects(rot)) {

                        // The rvalue is guaranteed to have no side effects, so we can save the code to evaluate it
                        // and then discard the result.
                        ;
                    } else {

                        // Evaluate the target expression and then discard the result.
                        this.pop(ot, this.com.hazelcast.com.ileGetValue(rot));
                    }
                } else {

                    // JLS9 15.12.4.1.3.2 and .4.2
                    this.com.hazelcast.com.ileGetValue(rot);
                }
            }
        }

        // Evaluate method parameters (JLS7 15.12.4.2).
        // If this method is vararg, rewritten all args starting from lastParamIndex to the end as if they were elements
        // of an array.
        IClass[]  parameterTypes = iMethod.getParameterTypes();
        Rvalue[]  adjustedArgs   = null;
        final int actualSize     = mi.arguments.length;
        if (iMethod.isVarargs() && iMethod.argsNeedAdjust()) {
            adjustedArgs = new Rvalue[parameterTypes.length];
            Rvalue[]       lastArgs = new Rvalue[actualSize - parameterTypes.length + 1];
            final Location loc      = mi.getLocation();

            if (lastArgs.length > 0) {
                for (int i = 0, j = parameterTypes.length - 1; i < lastArgs.length; ++i, ++j) {
                    lastArgs[i] = mi.arguments[j];
                }
            }

            for (int i = parameterTypes.length - 2; i >= 0; --i) {
                adjustedArgs[i] = mi.arguments[i];
            }
            adjustedArgs[adjustedArgs.length - 1] = new NewInitializedArray(
                loc,                                       // location
                parameterTypes[parameterTypes.length - 1], // arrayIClass
                new ArrayInitializer(loc, lastArgs)        // arrayInitializer
            );
        } else {
            adjustedArgs = mi.arguments;
        }

        for (int i = 0; i < adjustedArgs.length; ++i) {
            this.assignmentConversion(
                mi,                                    // location
                this.com.hazelcast.com.ileGetValue(adjustedArgs[i]), // sourceType
                parameterTypes[i],                     // targetType
                this.getConstantValue(adjustedArgs[i]) // optionalConstantValue
            );
        }

        // Invoke!
        this.checkAccessible(iMethod, mi.getEnclosingScope(), mi.getLocation());
        if (iMethod.getDeclaringIClass().isInterface()) {
            this.invoke(mi, iMethod);
        } else {
            if (!iMethod.isStatic() && iMethod.getAccess() == Access.PRIVATE) {

                // In order to make a non-static private method invocable for enclosing types, enclosed types and types
                // enclosed by the same type, "com.hazelcast.com.ile(FunctionDeclarator)" modifies it on-the-fly as follows:
                //  + Access is changed from PRIVATE to PACKAGE
                //  + The name is appended with "$"
                //  + It is made static
                //  + A parameter of type "declaring class" is prepended to the signature
                // Hence, the invocation of such a method must be modified accordingly.
                this.writeOpcode(mi, Opcode.INVOKESTATIC);
                this.writeConstantMethodrefInfo(
                    iMethod.getDeclaringIClass().getDescriptor(), // classFD
                    iMethod.getName() + '$',                      // methodName
                    iMethod.getDescriptor().prependParameter(     // methodMD
                        iMethod.getDeclaringIClass().getDescriptor()
                    )
                );
            } else
            {
                this.invoke(mi, iMethod);
            }
        }
        return iMethod.getReturnType();
    }

    private static boolean
    mayHaveSideEffects(Rvalue rot) {
        Boolean result = (Boolean) rot.accept(UnitCompiler.MAY_HAVE_SIDE_EFFECTS_VISITOR);
        assert result != null;
        return result;
    }

    /*@SuppressWarnings("null")*/ private static final Visitor.RvalueVisitor<Boolean, RuntimeException>
    MAY_HAVE_SIDE_EFFECTS_VISITOR = new Visitor.RvalueVisitor<Boolean, RuntimeException>() {

        final Visitor.LvalueVisitor<Boolean, RuntimeException>
        lvalueVisitor = new Visitor.LvalueVisitor<Boolean, RuntimeException>() {

            // SUPPRESS CHECKSTYLE LineLengthCheck:7
            @Override @Nullable public Boolean visitAmbiguousName(AmbiguousName an)                                        { return false; }
            @Override @Nullable public Boolean visitArrayAccessExpression(ArrayAccessExpression aae)                       { return UnitCompiler.mayHaveSideEffects(aae.lhs) || UnitCompiler.mayHaveSideEffects(aae.index); }
            @Override @Nullable public Boolean visitFieldAccess(FieldAccess fa)                                            { return false; }
            @Override @Nullable public Boolean visitFieldAccessExpression(FieldAccessExpression fae)                       { return false; }
            @Override @Nullable public Boolean visitSuperclassFieldAccessExpression(SuperclassFieldAccessExpression scfae) { return false; }
            @Override @Nullable public Boolean visitLocalVariableAccess(LocalVariableAccess lva)                           { return false; }
            @Override @Nullable public Boolean visitParenthesizedExpression(ParenthesizedExpression pe)                    { return UnitCompiler.mayHaveSideEffects(pe.value); }
        };

        // SUPPRESS CHECKSTYLE LineLengthCheck:26
        @Override @Nullable public Boolean visitLvalue(Lvalue lv)                                          { return (Boolean) lv.accept(this.lvalueVisitor); }
        @Override @Nullable public Boolean visitArrayLength(ArrayLength al)                                { return UnitCompiler.mayHaveSideEffects(al.lhs); }
        @Override @Nullable public Boolean visitAssignment(Assignment a)                                   { return true; }
        @Override @Nullable public Boolean visitUnaryOperation(UnaryOperation uo)                          { return UnitCompiler.mayHaveSideEffects(uo.operand); }
        @Override @Nullable public Boolean visitBinaryOperation(BinaryOperation bo)                        { return UnitCompiler.mayHaveSideEffects(bo.lhs) || UnitCompiler.mayHaveSideEffects(bo.rhs); }
        @Override @Nullable public Boolean visitCast(Cast c)                                               { return UnitCompiler.mayHaveSideEffects(c.value); }
        @Override @Nullable public Boolean visitClassLiteral(ClassLiteral cl)                              { return false; }
        @Override @Nullable public Boolean visitConditionalExpression(ConditionalExpression ce)            { return UnitCompiler.mayHaveSideEffects(ce.lhs) || UnitCompiler.mayHaveSideEffects(ce.mhs) || UnitCompiler.mayHaveSideEffects(ce.rhs); }
        @Override @Nullable public Boolean visitCrement(Crement c)                                         { return true; }
        @Override @Nullable public Boolean visitInstanceof(Instanceof io)                                  { return false; }
        @Override @Nullable public Boolean visitMethodInvocation(MethodInvocation mi)                      { return true; }
        @Override @Nullable public Boolean visitSuperclassMethodInvocation(SuperclassMethodInvocation smi) { return true; }
        @Override @Nullable public Boolean visitIntegerLiteral(IntegerLiteral il)                          { return false; }
        @Override @Nullable public Boolean visitFloatingPointLiteral(FloatingPointLiteral fpl)             { return false; }
        @Override @Nullable public Boolean visitBooleanLiteral(BooleanLiteral bl)                          { return false; }
        @Override @Nullable public Boolean visitCharacterLiteral(CharacterLiteral cl)                      { return false; }
        @Override @Nullable public Boolean visitStringLiteral(StringLiteral sl)                            { return false; }
        @Override @Nullable public Boolean visitNullLiteral(NullLiteral nl)                                { return false; }
        @Override @Nullable public Boolean visitSimpleConstant(SimpleConstant sl)                          { return false; }
        @Override @Nullable public Boolean visitNewAnonymousClassInstance(NewAnonymousClassInstance naci)  { return true; }
        @Override @Nullable public Boolean visitNewArray(NewArray na)                                      { for (Rvalue de : na.dimExprs) { if (UnitCompiler.mayHaveSideEffects(de)) return true; } return false; }
        @Override @Nullable public Boolean visitNewInitializedArray(NewInitializedArray nia)               { return this.mayHaveSideEffects(nia.arrayInitializer); }
        @Override @Nullable public Boolean visitNewClassInstance(NewClassInstance nci)                     { return true; }
        @Override @Nullable public Boolean visitParameterAccess(ParameterAccess pa)                        { return false; }
        @Override @Nullable public Boolean visitQualifiedThisReference(QualifiedThisReference qtr)         { return false; }
        @Override @Nullable public Boolean visitThisReference(ThisReference tr)                            { return false; }

        private boolean
        mayHaveSideEffects(ArrayInitializer arrayInitializer) {

            for (ArrayInitializerOrRvalue aiorv : arrayInitializer.values) {
                if (aiorv instanceof Rvalue) {
                    if (UnitCompiler.mayHaveSideEffects((Rvalue) aiorv)) return true;
                } else
                if (aiorv instanceof ArrayInitializer) {
                    if (this.mayHaveSideEffects((ArrayInitializer) aiorv)) return true;
                } else
                {
                    throw new AssertionError(aiorv);
                }
            }

            return false;
        }
    };

    private IClass
    com.hazelcast.com.ileGet2(SuperclassMethodInvocation scmi) throws CompileException {
        final IClass.IMethod iMethod = this.findIMethod(scmi);

        Scope s;
        for (
            s = scmi.getEnclosingScope();
            s instanceof Statement || s instanceof CatchClause;
            s = s.getEnclosingScope()
        );
        FunctionDeclarator fd = s instanceof FunctionDeclarator ? (FunctionDeclarator) s : null;
        if (fd == null) {
            this.com.hazelcast.com.ileError("Cannot invoke superclass method in non-method scope", scmi.getLocation());
            return IClass.INT;
        }
        if (Mod.isStatic(fd.modifiers.accessFlags)) {
            this.com.hazelcast.com.ileError("Cannot invoke superclass method in static context", scmi.getLocation());
        }
        this.load(scmi, this.resolve(fd.getDeclaringType()), 0);

        // Evaluate method parameters.
        // TODO: adjust args
        IClass[] parameterTypes = iMethod.getParameterTypes();
        for (int i = 0; i < scmi.arguments.length; ++i) {
            this.assignmentConversion(
                scmi,                                    // locatable
                this.com.hazelcast.com.ileGetValue(scmi.arguments[i]), // sourceType
                parameterTypes[i],                       // targetType
                this.getConstantValue(scmi.arguments[i]) // optionalConstantValue
            );
        }

        // Invoke!
        this.writeOpcode(scmi, Opcode.INVOKESPECIAL);
        this.writeConstantMethodrefInfo(
            iMethod.getDeclaringIClass().getDescriptor(), // classFD
            scmi.methodName,                              // methodName
            iMethod.getDescriptor()                       // methodMD
        );
        return iMethod.getReturnType();
    }

    private IClass
    com.hazelcast.com.ileGet2(NewClassInstance nci) throws CompileException {
        IClass iClass;
        if (nci.iClass != null) {
            iClass = nci.iClass;
        } else {
            assert nci.type != null;
            iClass = (nci.iClass = this.getType(nci.type));
        }

        this.writeOpcode(nci, Opcode.NEW);
        this.writeConstantClassInfo(iClass.getDescriptor());
        this.writeOpcode(nci, Opcode.DUP);

        if (iClass.isInterface()) this.com.hazelcast.com.ileError("Cannot instantiate \"" + iClass + "\"", nci.getLocation());
        this.checkAccessible(iClass, nci.getEnclosingScope(), nci.getLocation());
        if (iClass.isAbstract()) {
            this.com.hazelcast.com.ileError("Cannot instantiate abstract \"" + iClass + "\"", nci.getLocation());
        }

        // Determine the enclosing instance for the new object.
        Rvalue optionalEnclosingInstance;
        if (nci.optionalQualification != null) {
            if (iClass.getOuterIClass() == null) {
                this.com.hazelcast.com.ileError("Static member class cannot be instantiated with qualified NEW");
            }

            // Enclosing instance defined by qualification (JLS7 15.9.2.BL1.B3.B2).
            optionalEnclosingInstance = nci.optionalQualification;
        } else {
            Scope s = nci.getEnclosingScope();

            for (; !(s instanceof TypeBodyDeclaration); s = s.getEnclosingScope());
            TypeBodyDeclaration enclosingTypeBodyDeclaration = (TypeBodyDeclaration) s;

            TypeDeclaration enclosingTypeDeclaration = (TypeDeclaration) s.getEnclosingScope();

            if (
                !(enclosingTypeDeclaration instanceof AbstractClassDeclaration)
                || enclosingTypeBodyDeclaration.isStatic()
            ) {

                // No enclosing instance in
                //  + interface method declaration or
                //  + static type body declaration (here: method or initializer or field declarator)
                // context (JLS7 15.9.2.BL1.B3.B1.B1).
                if (iClass.getOuterIClass() != null) {
                    this.com.hazelcast.com.ileError((
                        "Instantiation of \""
                        + (nci.type != null ? nci.type.toString() : String.valueOf(nci.iClass))
                        + "\" requires an enclosing instance"
                    ), nci.getLocation());
                }
                optionalEnclosingInstance = null;
            } else
            {

                // Determine the type of the enclosing instance for the new object.
                IClass optionalOuterIClass = iClass.getDeclaringIClass();
                if (optionalOuterIClass == null) {

                    // No enclosing instance needed for a top-level class object.
                    optionalEnclosingInstance = null;
                } else {

                    // Find an appropriate enclosing instance for the new inner class object among the enclosing
                    // instances of the current object (JLS7 15.9.2.BL1.B3.B1.B2).
                    optionalEnclosingInstance = new QualifiedThisReference(
                        nci.getLocation(), // location
                        new SimpleType(    // qualification
                            nci.getLocation(),
                            optionalOuterIClass
                        )
                    );
                    optionalEnclosingInstance.setEnclosingScope(nci.getEnclosingScope());
                }
            }
        }

        this.invokeConstructor(
            nci,                       // l
            nci.getEnclosingScope(),   // scope
            optionalEnclosingInstance, // optionalEnclosingInstance
            iClass,                    // targetClass
            nci.arguments              // arguments
        );

        return iClass;
    }

    private IClass
    com.hazelcast.com.ileGet2(NewAnonymousClassInstance naci) throws CompileException {
        AnonymousClassDeclaration acd = naci.anonymousClassDeclaration;

        // Find constructors of superclass.
        IClass sc = this.resolve(acd).getSuperclass();
        assert sc != null;

        IClass.IConstructor[] superclassIConstructors = sc.getDeclaredIConstructors();
        if (superclassIConstructors.length == 0) {
            throw new InternalCompilerException("SNO: Superclass has no constructors");
        }

        // Determine the most specific constructor of the superclass.
        IClass.IConstructor superclassIConstructor = (IClass.IConstructor) this.findMostSpecificIInvocable(
            naci,                    // locatable
            superclassIConstructors, // iInvocables
            naci.arguments,          // arguments
            acd                      // contextScope
        );

        Location loc = naci.getLocation();


        // Determine the formal parameters of the anonymous constructor.
        IClass[]         scpts = superclassIConstructor.getParameterTypes();
        FormalParameters parameters;
        {
            List<FormalParameter> l = new ArrayList<FormalParameter>();

            // Pass the enclosing instance of the base class as parameter #1.
            if (naci.optionalQualification != null) l.add(new FormalParameter(
                loc,                                                           // location
                true,                                                          // finaL
                new SimpleType(loc, this.getType(naci.optionalQualification)), // type
                "this$base"                                                    // name
            ));
            for (int i = 0; i < scpts.length; ++i) l.add(new FormalParameter(
                loc,                           // location
                true,                          // finaL
                new SimpleType(loc, scpts[i]), // type
                "p" + i                        // name
            ));
            parameters = new FormalParameters(
                loc,
                (FormalParameter[]) l.toArray(new FormalParameter[l.size()]),
                false
            );
        }

        // Determine the declared exceptions of the anonymous constructor.
        Type[] thrownExceptions;
        {
            IClass[] tes = superclassIConstructor.getThrownExceptions();
            thrownExceptions = new Type[tes.length];
            for (int i = 0; i < tes.length; ++i) thrownExceptions[i] = new SimpleType(loc, tes[i]);
        }

        // The anonymous constructor merely invokes the constructor of its superclass.
        int    j = 0;
        Rvalue optionalQualificationAccess;
        if (naci.optionalQualification == null) {
            optionalQualificationAccess = null;
        } else
        {
            optionalQualificationAccess = new ParameterAccess(loc, parameters.parameters[j++]);
        }
        Rvalue[] parameterAccesses = new Rvalue[scpts.length];
        for (int i = 0; i < scpts.length; ++i) {
            parameterAccesses[i] = new ParameterAccess(loc, parameters.parameters[j++]);
        }

        // Generate the anonymous constructor for the anonymous class (JLS7 15.9.5.1).
        acd.addConstructor(new ConstructorDeclarator(
            loc,                                    // location
            null,                                   // optionalDocComment
            new Modifiers(Mod.PACKAGE),             // modifiers
            parameters,                             // parameters
            thrownExceptions,                       // thrownExceptions
            new SuperConstructorInvocation(         // optionalConstructorInvocation
                loc,                            // location
                optionalQualificationAccess,    // optionalQualification
                parameterAccesses               // arguments
            ),
            Collections.<BlockStatement>emptyList() // optionalStatements
        ));

        // Compile the anonymous class.
        try {
            this.com.hazelcast.com.ile(acd);

            // Instantiate the anonymous class.
            this.writeOpcode(naci, Opcode.NEW);
            this.writeConstantClassInfo(this.resolve(naci.anonymousClassDeclaration).getDescriptor());

            // TODO: adjust argument (for varargs case ?)
            // Invoke the anonymous constructor.
            this.writeOpcode(naci, Opcode.DUP);
            Rvalue[] arguments2;
            if (naci.optionalQualification == null) {
                arguments2 = naci.arguments;
            } else {
                arguments2    = new Rvalue[naci.arguments.length + 1];
                arguments2[0] = naci.optionalQualification;
                System.arraycopy(naci.arguments, 0, arguments2, 1, naci.arguments.length);
            }

            // Adjust if needed.
            // TODO: Not doing this now because we don't need vararg-anonymous class (yet).

//            Rvalue[] adjustedArgs = null;
//            final int paramsTypeLength = iConstructor.getParameterTypes().length;
//            if (argsNeedAdjusting[0]) {
//                adjustedArgs = new Rvalue[paramsTypeLength];
//            }

            // Notice: The enclosing instance of the anonymous class is "this", not the qualification of the
            // NewAnonymousClassInstance.
            Scope s;
            for (s = naci.getEnclosingScope(); !(s instanceof TypeBodyDeclaration); s = s.getEnclosingScope());
            ThisReference oei;
            if (((TypeBodyDeclaration) s).isStatic()) {
                oei = null;
            } else
            {
                oei = new ThisReference(loc);
                oei.setEnclosingScope(naci.getEnclosingScope());
            }
            this.invokeConstructor(
                naci,                                         // locatable
                naci.getEnclosingScope(),                     // scope
                oei,                                          // optionalEnclosingInstance
                this.resolve(naci.anonymousClassDeclaration), // targetClass
                arguments2                                    // arguments
            );
        } finally {

            // Remove the synthetic constructor that was temporarily added. This is necessary because this NACI
            // expression (and all other expressions) are sometimes com.hazelcast.com.iled more than once (see "fakeCompile()"), and
            // we'd end up with TWO synthetic constructors. See JANINO-143.
            acd.constructors.remove(acd.constructors.size() - 1);
        }
        return this.resolve(naci.anonymousClassDeclaration);
    }
    private IClass
    com.hazelcast.com.ileGet2(ParameterAccess pa) throws CompileException {
        LocalVariable lv = this.getLocalVariable(pa.formalParameter);
        this.load(pa, lv);
        return lv.type;
    }
    private IClass
    com.hazelcast.com.ileGet2(NewArray na) throws CompileException {
        for (Rvalue dimExpr : na.dimExprs) {
            IClass dimType = this.com.hazelcast.com.ileGetValue(dimExpr);
            if (dimType != IClass.INT && this.unaryNumericPromotion(
                na,     // locatable
                dimType // type
            ) != IClass.INT) this.com.hazelcast.com.ileError("Invalid array size expression type", na.getLocation());
        }

        return this.newArray(
            na,                   // locatable
            na.dimExprs.length,   // dimExprCount
            na.dims,              // dims
            this.getType(na.type) // com.hazelcast.com.onentType
        );
    }
    private IClass
    com.hazelcast.com.ileGet2(NewInitializedArray nia) throws CompileException {

        IClass at = this.getType2(nia);

        this.com.hazelcast.com.ileGetValue(nia.arrayInitializer, at);

        return at;
    }
    private void
    com.hazelcast.com.ileGetValue(ArrayInitializer ai, IClass arrayType) throws CompileException {

        if (!arrayType.isArray()) {
            this.com.hazelcast.com.ileError("Array initializer not allowed for non-array type \"" + arrayType.toString() + "\"");
        }

        IClass ct = arrayType.getComponentType();
        assert ct != null;

        this.pushConstant(ai, new Integer(ai.values.length));
        this.newArray(
            ai, // locatable
            1,  // dimExprCount
            0,  // dims
            ct  // com.hazelcast.com.onentType
        );

        for (int i = 0; i < ai.values.length; ++i) {
            this.writeOpcode(ai, Opcode.DUP);
            this.pushConstant(ai, new Integer(i));
            ArrayInitializerOrRvalue aiorv = ai.values[i];
            if (aiorv instanceof Rvalue) {
                Rvalue rv = (Rvalue) aiorv;
                this.assignmentConversion(
                    ai,                       // locatable
                    this.com.hazelcast.com.ileGetValue(rv), // sourceType
                    ct,                       // targetType
                    this.getConstantValue(rv) // optionalConstantValue
                );
            } else
            if (aiorv instanceof ArrayInitializer) {
                this.com.hazelcast.com.ileGetValue((ArrayInitializer) aiorv, ct);
            } else
            {
                throw new InternalCompilerException(
                    "Unexpected array initializer or rvalue class " + aiorv.getClass().getName()
                );
            }
            this.writeOpcode(ai, Opcode.IASTORE + UnitCompiler.ilfdabcs(ct));
        }
    }
    private IClass
    com.hazelcast.com.ileGet2(Literal l) throws CompileException {
        return this.pushConstant(l, this.getConstantValue(l));
    }
    private IClass
    com.hazelcast.com.ileGet2(SimpleConstant sl) throws CompileException {
        return this.pushConstant(sl, sl.value);
    }

    /**
     * Convenience function that calls {@link #com.hazelcast.com.ileContext(Rvalue)} and {@link #com.hazelcast.com.ileGet(Rvalue)}.
     *
     * @return The type of the Rvalue
     */
    private IClass
    com.hazelcast.com.ileGetValue(Rvalue rv) throws CompileException {
        Object cv = this.getConstantValue(rv);
        if (cv != UnitCompiler.NOT_CONSTANT) {
            this.fakeCompile(rv); // To check that, e.g., "a" com.hazelcast.com.iles in "true || a".
            this.pushConstant(rv, cv);
            return this.getType(rv);
        }

        this.com.hazelcast.com.ileContext(rv);
        return this.com.hazelcast.com.ileGet(rv);
    }

    // -------------------- Rvalue.getConstantValue() -----------------

    /**
     * Special return value for the {@link #getConstantValue(Java.Rvalue)} method family indicating that the given
     * {@link Java.Rvalue} does not evaluate to a constant value.
     */
    public static final Object NOT_CONSTANT = IClass.NOT_CONSTANT;

    /**
     * Attempts to evaluate as a constant expression. The result is one of the following: {@link Boolean}, {@link
     * Byte}, {@link Short}, {@link Integer}, {@link Long}, {@link Float}, {@link Double}, {@link Character}, {@link
     * String}, {@code null} (representing the {@code null} literal.
     * <p>
     *   This method cannot be STATIC, because the constant value may refer to a constant declaration in this
     *   com.hazelcast.com.ilation unit.
     * </p>
     *
     * @return {@link #NOT_CONSTANT} iff the rvalue is not a constant value
     */
    @Nullable public final Object
    getConstantValue(Rvalue rv) throws CompileException {

        if (rv.constantValue != Rvalue.CONSTANT_VALUE_UNKNOWN) return rv.constantValue;

        return (rv.constantValue = rv.accept(new RvalueVisitor<Object, CompileException>() {

            @Override @Nullable public Object
            visitLvalue(Lvalue lv) throws CompileException {
                return lv.accept(new Visitor.LvalueVisitor<Object, CompileException>() {

                    // SUPPRESS CHECKSTYLE LineLengthCheck:7
                    @Override @Nullable public Object visitAmbiguousName(AmbiguousName an)                     throws CompileException { return UnitCompiler.this.getConstantValue2(an);    }
                    @Override @Nullable public Object visitArrayAccessExpression(ArrayAccessExpression aae)                            { return UnitCompiler.this.getConstantValue2(aae);   }
                    @Override @Nullable public Object visitFieldAccess(FieldAccess fa)                         throws CompileException { return UnitCompiler.this.getConstantValue2(fa);    }
                    @Override @Nullable public Object visitFieldAccessExpression(FieldAccessExpression fae)                            { return UnitCompiler.this.getConstantValue2(fae);   }
                    @Override @Nullable public Object visitSuperclassFieldAccessExpression(SuperclassFieldAccessExpression scfae)      { return UnitCompiler.this.getConstantValue2(scfae); }
                    @Override @Nullable public Object visitLocalVariableAccess(LocalVariableAccess lva)        throws CompileException { return UnitCompiler.this.getConstantValue2(lva);   }
                    @Override @Nullable public Object visitParenthesizedExpression(ParenthesizedExpression pe) throws CompileException { return UnitCompiler.this.getConstantValue2(pe);    }
                });
            }

            // SUPPRESS CHECKSTYLE LineLengthCheck:25
            @Override @Nullable public Object visitArrayLength(ArrayLength al)                                             { return UnitCompiler.this.getConstantValue2(al);   }
            @Override @Nullable public Object visitAssignment(Assignment a)                                                { return UnitCompiler.this.getConstantValue2(a);    }
            @Override @Nullable public Object visitUnaryOperation(UnaryOperation uo)               throws CompileException { return UnitCompiler.this.getConstantValue2(uo);   }
            @Override @Nullable public Object visitBinaryOperation(BinaryOperation bo)             throws CompileException { return UnitCompiler.this.getConstantValue2(bo);   }
            @Override @Nullable public Object visitCast(Cast c)                                    throws CompileException { return UnitCompiler.this.getConstantValue2(c);    }
            @Override @Nullable public Object visitClassLiteral(ClassLiteral cl)                                           { return UnitCompiler.this.getConstantValue2(cl);   }
            @Override @Nullable public Object visitConditionalExpression(ConditionalExpression ce) throws CompileException { return UnitCompiler.this.getConstantValue2(ce);   }
            @Override @Nullable public Object visitCrement(Crement c)                                                      { return UnitCompiler.this.getConstantValue2(c);    }
            @Override @Nullable public Object visitInstanceof(Instanceof io)                                               { return UnitCompiler.this.getConstantValue2(io);   }
            @Override @Nullable public Object visitMethodInvocation(MethodInvocation mi)                                   { return UnitCompiler.this.getConstantValue2(mi);   }
            @Override @Nullable public Object visitSuperclassMethodInvocation(SuperclassMethodInvocation smi)              { return UnitCompiler.this.getConstantValue2(smi);  }
            @Override @Nullable public Object visitIntegerLiteral(IntegerLiteral il)               throws CompileException { return UnitCompiler.this.getConstantValue2(il);   }
            @Override @Nullable public Object visitFloatingPointLiteral(FloatingPointLiteral fpl)  throws CompileException { return UnitCompiler.this.getConstantValue2(fpl);  }
            @Override @Nullable public Object visitBooleanLiteral(BooleanLiteral bl)                                       { return UnitCompiler.this.getConstantValue2(bl);   }
            @Override @Nullable public Object visitCharacterLiteral(CharacterLiteral cl)           throws CompileException { return UnitCompiler.this.getConstantValue2(cl);   }
            @Override @Nullable public Object visitStringLiteral(StringLiteral sl)                 throws CompileException { return UnitCompiler.this.getConstantValue2(sl);   }
            @Override @Nullable public Object visitNullLiteral(NullLiteral nl)                                             { return UnitCompiler.this.getConstantValue2(nl);   }
            @Override @Nullable public Object visitSimpleConstant(SimpleConstant sl)                                       { return UnitCompiler.this.getConstantValue2(sl);   }
            @Override @Nullable public Object visitNewAnonymousClassInstance(NewAnonymousClassInstance naci)               { return UnitCompiler.this.getConstantValue2(naci); }
            @Override @Nullable public Object visitNewArray(NewArray na)                                                   { return UnitCompiler.this.getConstantValue2(na);   }
            @Override @Nullable public Object visitNewInitializedArray(NewInitializedArray nia)                            { return UnitCompiler.this.getConstantValue2(nia);  }
            @Override @Nullable public Object visitNewClassInstance(NewClassInstance nci)                                  { return UnitCompiler.this.getConstantValue2(nci);  }
            @Override @Nullable public Object visitParameterAccess(ParameterAccess pa)                                     { return UnitCompiler.this.getConstantValue2(pa);   }
            @Override @Nullable public Object visitQualifiedThisReference(QualifiedThisReference qtr)                      { return UnitCompiler.this.getConstantValue2(qtr);  }
            @Override @Nullable public Object visitThisReference(ThisReference tr)                                         { return UnitCompiler.this.getConstantValue2(tr);   }
        }));
    }

    @SuppressWarnings("static-method")
    @Nullable private Object
    getConstantValue2(Rvalue rv) { return UnitCompiler.NOT_CONSTANT; }

    @Nullable private Object
    getConstantValue2(AmbiguousName an) throws CompileException {
        return this.getConstantValue(this.toRvalueOrCompileException(this.reclassify(an)));
    }

    @SuppressWarnings("static-method")
    @Nullable private Object
    getConstantValue2(FieldAccess fa) throws CompileException {
        return fa.field.getConstantValue();
    }

    @Nullable private Object
    getConstantValue2(UnaryOperation uo) throws CompileException {
        if (uo.operator == "+") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            return this.getConstantValue(uo.operand);
        }
        if (uo.operator == "-") { // SUPPRESS CHECKSTYLE StringLiteralEquality

            // Handle the super special cases "-2147483648" and "-9223372036854775808L" (JLS9 3.10.1).
            if (uo.operand instanceof IntegerLiteral) {
                String v = ((Literal) uo.operand).value;

                if (UnitCompiler.TWO_E_31_INTEGER.matcher(v).matches()) return new Integer(Integer.MIN_VALUE);
                if (UnitCompiler.TWO_E_63_LONG.matcher(v).matches())    return new Long(Long.MIN_VALUE);
            }

            Object cv = this.getConstantValue(uo.operand);

            if (cv == UnitCompiler.NOT_CONSTANT) return UnitCompiler.NOT_CONSTANT;

            // SUPPRESS CHECKSTYLE DOT__SELECTOR|L_PAREN__METH_INVOCATION:6
            if (cv instanceof Byte)    return Byte   .valueOf((byte)  -((Byte)    cv));
            if (cv instanceof Short)   return Short  .valueOf((short) -((Short)   cv));
            if (cv instanceof Integer) return Integer.valueOf(        -((Integer) cv));
            if (cv instanceof Long)    return Long   .valueOf(        -((Long)    cv));
            if (cv instanceof Float)   return Float  .valueOf(        -((Float)   cv));
            if (cv instanceof Double)  return Double .valueOf(        -((Double)  cv));

            return UnitCompiler.NOT_CONSTANT;
        }

        if (uo.operator == "!") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            Object cv = this.getConstantValue(uo.operand);
            return (
                cv == Boolean.TRUE  ? Boolean.FALSE :
                cv == Boolean.FALSE ? Boolean.TRUE  :
                UnitCompiler.NOT_CONSTANT
            );
        }

        return UnitCompiler.NOT_CONSTANT;
    }

    /**
     * 2147483648 is the special value that can <em>not</em> be stored in an INT, but <em>its negated value</em>
     * (-2147483648) can.
     */
    private static final Pattern
    TWO_E_31_INTEGER = Pattern.com.hazelcast.com.ile("2_*1_*4_*7_*4_*8_*3_*6_*4_*8");

    /**
     * 9223372036854775808 is the special value that can <em>not</em> be stored in a LONG, but <em>its negated
     * value</em> (-9223372036854775808) can.
     */
    private static final Pattern
    TWO_E_63_LONG = Pattern.com.hazelcast.com.ile("9_*2_*2_*3_*3_*7_*2_*0_*3_*6_*8_*5_*4_*7_*7_*5_*8_*0_*8[lL]");

    @Nullable private Object
    getConstantValue2(ConditionalExpression ce) throws CompileException {
        Object lhsValue = this.getConstantValue(ce.lhs);
        if (lhsValue instanceof Boolean) {
            return (
                ((Boolean) lhsValue).booleanValue()
                ? this.getConstantValue(ce.mhs)
                : this.getConstantValue(ce.rhs)
            );
        }
        return UnitCompiler.NOT_CONSTANT;
    }

    @Nullable private Object
    getConstantValue2(BinaryOperation bo) throws CompileException {

        // "|", "^", "&", "*", "/", "%", "+", "-", "==", "!=".
        if (
            // SUPPRESS CHECKSTYLE StringLiteralEquality:10
            bo.operator == "|"
            || bo.operator == "^"
            || bo.operator == "&"
            || bo.operator == "*"
            || bo.operator == "/"
            || bo.operator == "%"
            || bo.operator == "+"
            || bo.operator == "-"
            || bo.operator == "=="
            || bo.operator == "!="
        ) {

            // Unroll the constant operands.
            List<Object> cvs = new ArrayList<Object>();
            for (Iterator<Rvalue> it = bo.unrollLeftAssociation(); it.hasNext();) {
                Object cv = this.getConstantValue((Rvalue) it.next());
                if (cv == UnitCompiler.NOT_CONSTANT) return UnitCompiler.NOT_CONSTANT;
                cvs.add(cv);
            }

            // Compute the constant value of the unrolled binary operation.
            Iterator<Object> it  = cvs.iterator();
            Object           lhs = it.next();
            while (it.hasNext()) {
                if (lhs == UnitCompiler.NOT_CONSTANT) return UnitCompiler.NOT_CONSTANT;

                Object rhs = it.next();

                // String concatenation?
                // SUPPRESS CHECKSTYLE StringLiteralEquality
                if (bo.operator == "+" && (lhs instanceof String || rhs instanceof String)) {
                    StringBuilder sb = new StringBuilder(lhs.toString()).append(rhs);
                    while (it.hasNext()) sb.append(it.next().toString());
                    return sb.toString();
                }

                if (lhs instanceof Number && rhs instanceof Number) {
                    try {
                        if (lhs instanceof Double || rhs instanceof Double) {
                            double lhsD = ((Number) lhs).doubleValue();
                            double rhsD = ((Number) rhs).doubleValue();
                            lhs = (
                                // SUPPRESS CHECKSTYLE StringLiteralEquality:7
                                bo.operator == "*" ? new Double(lhsD * rhsD) :
                                bo.operator == "/" ? new Double(lhsD / rhsD) :
                                bo.operator == "%" ? new Double(lhsD % rhsD) :
                                bo.operator == "+" ? new Double(lhsD + rhsD) :
                                bo.operator == "-" ? new Double(lhsD - rhsD) :
                                bo.operator == "==" ? Boolean.valueOf(lhsD == rhsD) :
                                bo.operator == "!=" ? Boolean.valueOf(lhsD != rhsD) :
                                UnitCompiler.NOT_CONSTANT
                            );
                            continue;
                        }
                        if (lhs instanceof Float || rhs instanceof Float) {
                            float lhsF = ((Number) lhs).floatValue();
                            float rhsF = ((Number) rhs).floatValue();
                            lhs = (
                                // SUPPRESS CHECKSTYLE StringLiteralEquality:7
                                bo.operator == "*" ? new Float(lhsF * rhsF) :
                                bo.operator == "/" ? new Float(lhsF / rhsF) :
                                bo.operator == "%" ? new Float(lhsF % rhsF) :
                                bo.operator == "+" ? new Float(lhsF + rhsF) :
                                bo.operator == "-" ? new Float(lhsF - rhsF) :
                                bo.operator == "==" ? Boolean.valueOf(lhsF == rhsF) :
                                bo.operator == "!=" ? Boolean.valueOf(lhsF != rhsF) :
                                UnitCompiler.NOT_CONSTANT
                            );
                            continue;
                        }
                        if (lhs instanceof Long || rhs instanceof Long) {
                            long lhsL = ((Number) lhs).longValue();
                            long rhsL = ((Number) rhs).longValue();
                            lhs = (
                                // SUPPRESS CHECKSTYLE StringLiteralEquality:10
                                bo.operator == "|" ? new Long(lhsL | rhsL) :
                                bo.operator == "^" ? new Long(lhsL ^ rhsL) :
                                bo.operator == "&" ? new Long(lhsL & rhsL) :
                                bo.operator == "*" ? new Long(lhsL * rhsL) :
                                bo.operator == "/" ? new Long(lhsL / rhsL) :
                                bo.operator == "%" ? new Long(lhsL % rhsL) :
                                bo.operator == "+" ? new Long(lhsL + rhsL) :
                                bo.operator == "-" ? new Long(lhsL - rhsL) :
                                bo.operator == "==" ? Boolean.valueOf(lhsL == rhsL) :
                                bo.operator == "!=" ? Boolean.valueOf(lhsL != rhsL) :
                                UnitCompiler.NOT_CONSTANT
                            );
                            continue;
                        }
                        if (
                            lhs instanceof Integer || lhs instanceof Byte || lhs instanceof Short
                            || rhs instanceof Integer || lhs instanceof Byte || lhs instanceof Short
                        ) {
                            int lhsI = ((Number) lhs).intValue();
                            int rhsI = ((Number) rhs).intValue();
                            lhs = (
                                // SUPPRESS CHECKSTYLE StringLiteralEquality:10
                                bo.operator == "|" ? new Integer(lhsI | rhsI) :
                                bo.operator == "^" ? new Integer(lhsI ^ rhsI) :
                                bo.operator == "&" ? new Integer(lhsI & rhsI) :
                                bo.operator == "*" ? new Integer(lhsI * rhsI) :
                                bo.operator == "/" ? new Integer(lhsI / rhsI) :
                                bo.operator == "%" ? new Integer(lhsI % rhsI) :
                                bo.operator == "+" ? new Integer(lhsI + rhsI) :
                                bo.operator == "-" ? new Integer(lhsI - rhsI) :
                                bo.operator == "==" ? Boolean.valueOf(lhsI == rhsI) :
                                bo.operator == "!=" ? Boolean.valueOf(lhsI != rhsI) :
                                UnitCompiler.NOT_CONSTANT
                            );
                            continue;
                        }
                    } catch (ArithmeticException ae) {

                        // Most likely a divide by zero or modulo by zero. Guess we can't make this expression into a
                        // constant.
                        return UnitCompiler.NOT_CONSTANT;
                    }
                    throw new IllegalStateException();
                }

                if (lhs instanceof Character && rhs instanceof Character) {
                    char lhsC = ((Character) lhs).charValue();
                    char rhsC = ((Character) rhs).charValue();
                    lhs = (
                        bo.operator == "==" ? Boolean.valueOf(lhsC == rhsC) : // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
                        bo.operator == "!=" ? Boolean.valueOf(lhsC != rhsC) : // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
                        UnitCompiler.NOT_CONSTANT
                    );
                    continue;
                }

                if (lhs == null || rhs == null) {
                    lhs = (
                        bo.operator == "==" ? Boolean.valueOf(lhs == rhs) : // SUPPRESS CHECKSTYLE StringLiteralEquality
                        bo.operator == "!=" ? Boolean.valueOf(lhs != rhs) : // SUPPRESS CHECKSTYLE StringLiteralEquality
                        UnitCompiler.NOT_CONSTANT
                    );
                    continue;
                }

                return UnitCompiler.NOT_CONSTANT;
            }
            return lhs;
        }

        // "&&" and "||" with constant LHS operand.
        if (bo.operator == "&&" || bo.operator == "||") { // SUPPRESS CHECKSTYLE StringLiteralEquality
            Object lhsValue = this.getConstantValue(bo.lhs);
            if (lhsValue instanceof Boolean) {
                boolean lhsBv = ((Boolean) lhsValue).booleanValue();
                return (
                    bo.operator == "&&" // SUPPRESS CHECKSTYLE StringLiteralEquality
                    ? (lhsBv ? this.getConstantValue(bo.rhs) : Boolean.FALSE)
                    : (lhsBv ? Boolean.TRUE : this.getConstantValue(bo.rhs))
                );
            }
        }

        return UnitCompiler.NOT_CONSTANT;
    }

    private Object
    getConstantValue2(Cast c) throws CompileException {
        Object cv = this.getConstantValue(c.value);
        if (cv == UnitCompiler.NOT_CONSTANT) return UnitCompiler.NOT_CONSTANT;

        if (cv instanceof Number) {
            IClass tt = this.getType(c.targetType);
            if (tt == IClass.BYTE)   return new Byte(((Number) cv).byteValue());
            if (tt == IClass.SHORT)  return new Short(((Number) cv).shortValue());
            if (tt == IClass.INT)    return new Integer(((Number) cv).intValue());
            if (tt == IClass.LONG)   return new Long(((Number) cv).longValue());
            if (tt == IClass.FLOAT)  return new Float(((Number) cv).floatValue());
            if (tt == IClass.DOUBLE) return new Double(((Number) cv).doubleValue());
        }

        return UnitCompiler.NOT_CONSTANT;
    }

    @Nullable private Object
    getConstantValue2(ParenthesizedExpression pe) throws CompileException {
        return this.getConstantValue(pe.value);
    }

    private @Nullable Object
    getConstantValue2(LocalVariableAccess lva) throws CompileException {

        // An optimization for the (very special case)
        //
        //    void method()         // Any method declarator
        //        ...
        //        boolean x = true; // Any local variable name allowed; also value "false"
        //        if (x) {          // The condition expression must be exactly like this
        //              ...
        //
        if (lva.getEnclosingScope() instanceof IfStatement) {
            IfStatement is = (IfStatement) lva.getEnclosingScope();

            if (is.condition instanceof AmbiguousName) {
                Atom ra = ((AmbiguousName) is.condition).reclassified;

                if (ra instanceof LocalVariableAccess) {
                    LocalVariable lv = ((LocalVariableAccess) ra).localVariable;

                    List<? extends BlockStatement> ss = (
                        is.getEnclosingScope() instanceof FunctionDeclarator
                        ? ((FunctionDeclarator) is.getEnclosingScope()).optionalStatements
                        : is.getEnclosingScope() instanceof Block
                        ? ((Block) is.getEnclosingScope()).statements
                        : null
                    );
                    if (ss != null) {
                        int isi = ss.indexOf(is);
                        if (isi >= 1) {
                            if (ss.get(isi - 1) instanceof LocalVariableDeclarationStatement) {

                                LocalVariableDeclarationStatement
                                lvds = (LocalVariableDeclarationStatement) ss.get(isi - 1);

                                if (
                                    lvds.variableDeclarators.length == 1
                                    && lvds.variableDeclarators[0].localVariable == lv
                                ) {
                                    ArrayInitializerOrRvalue oi = lvds.variableDeclarators[0].optionalInitializer;
                                    if (oi instanceof Rvalue) return this.getConstantValue((Rvalue) oi);
                                }
                            }
                        }
                    }
                }
            }
        }

        return UnitCompiler.NOT_CONSTANT;
    }

    /**
     * @return An {@link Integer} or a {@link Long}
     */
    @SuppressWarnings("static-method") private Object
    getConstantValue2(IntegerLiteral il) throws CompileException {

        String v = il.value;

        // Remove underscores in integer literal (JLS8, section 3.10.1).
        for (;;) {
            int ui = v.indexOf('_');
            if (ui == -1) break;
            v = v.substring(0, ui) + v.substring(ui + 1);
        }

        // HexIntegerLiteral (JLS8, section 3.10.1)?
        if (v.startsWith("0x") || v.startsWith("0X")) {

            // Cannot use "Integer/Long.valueOf(v, 16)" here because hex literals are UNSIGNED.
            return (
                v.endsWith("L") || v.endsWith("l")
                ? (Object) Long.valueOf(UnitCompiler.hex2UnsignedLong(il, v.substring(2, v.length() - 1)))
                : Integer.valueOf(UnitCompiler.hex2UnsignedInt(il, v.substring(2)))
            );
        }

        // BinaryIntegerLiteral (JLS8, section 3.10.1)?
        if (v.startsWith("0b") || v.startsWith("0B")) {

            // Cannot use "Integer/Long.valueOf(v, 2)" here because binary literals are UNSIGNED.
            return (
                v.endsWith("L") || v.endsWith("l")
                ? (Object) Long.valueOf(UnitCompiler.bin2UnsignedLong(il, v.substring(2, v.length() - 1)))
                : Integer.valueOf(UnitCompiler.bin2UnsignedInt(il, v.substring(2)))
            );
        }

        // OctalIntegerLiteral (JLS8, section 3.10.1)?
        if (v.startsWith("0")) {

            // Cannot use "Integer/Long.valueOf(v, 8)" here because octal literals are UNSIGNED.
            return (
                v.endsWith("L") || v.endsWith("l")
                ? (Object) Long.valueOf(UnitCompiler.oct2UnsignedLong(il, v.substring(0, v.length() - 1)))
                : Integer.valueOf(UnitCompiler.oct2UnsignedInt(il, v))
            );
        }

        // Must be an DecimalIntegerLiteral (JLS8, section 3.10.1).
        try {

            // Decimal literals are SIGNED, so we can safely use "Integer/Long.valueOf(v)".
            return (
                v.endsWith("L") || v.endsWith("l")
                ? (Object) Long.valueOf(v.substring(0, v.length() - 1))
                : Integer.valueOf(v)
            );
        } catch (NumberFormatException e) {
            // SUPPRESS CHECKSTYLE AvoidHidingCause
            throw UnitCompiler.com.hazelcast.com.ileException(il, "Invalid integer literal \"" + il.value + "\"");
        }
    }

    /**
     * @return A {@link Float} or a {@link Double}
     */
    @SuppressWarnings("static-method") private Object
    getConstantValue2(FloatingPointLiteral fpl) throws CompileException {

        String v = fpl.value;

        // Remove underscores in floating point literal.
        for (;;) {
            int ui = v.indexOf('_');
            if (ui == -1) break;
            v = v.substring(0, ui) + v.substring(ui + 1);
        }

        char lastChar = v.charAt(v.length() - 1);
        if (lastChar == 'f' || lastChar == 'F') {
            v = v.substring(0, v.length() - 1);

            float fv;
            try {
                fv = Float.parseFloat(v);
            } catch (NumberFormatException e) {
                throw new InternalCompilerException("SNO: parsing float literal \"" + v + "\": " + e.getMessage(), e);
            }
            if (Float.isInfinite(fv)) {
                throw UnitCompiler.com.hazelcast.com.ileException(fpl, "Value of float literal \"" + v + "\" is out of range");
            }
            if (Float.isNaN(fv)) {
                throw new InternalCompilerException("SNO: parsing float literal \"" + v + "\" results in NaN");
            }

            // Check for FLOAT underrun.
            if (fv == 0.0F) {
                for (int i = 0; i < v.length(); ++i) {
                    char c = v.charAt(i);
                    if ("123456789".indexOf(c) != -1) {
                        throw UnitCompiler.com.hazelcast.com.ileException(
                            fpl,
                            "Literal \"" + v + "\" is too small to be represented as a float"
                        );
                    }
                    if (c != '0' && c != '.') break;
                }
            }

            return new Float(fv);
        }

        if (lastChar == 'd' || lastChar == 'D') v = v.substring(0, v.length() - 1);

        double dv;
        try {
            dv = Double.parseDouble(v);
        } catch (NumberFormatException e) {
            throw new InternalCompilerException("SNO: parsing double literal \"" + v + "\": " + e.getMessage(), e);
        }
        if (Double.isInfinite(dv)) {
            throw UnitCompiler.com.hazelcast.com.ileException(fpl, "Value of double literal \"" + v + "\" is out of range");
        }
        if (Double.isNaN(dv)) {
            throw new InternalCompilerException("SNO: parsing double literal \"" + v + "\" results is NaN");
        }

        // Check for DOUBLE underrun.
        if (dv == 0.0F) {
            for (int i = 0; i < v.length(); ++i) {
                char c = v.charAt(i);
                if ("123456789".indexOf(c) != -1) {
                    throw UnitCompiler.com.hazelcast.com.ileException(
                        fpl,
                        "Literal \"" + v + "\" is too small to be represented as a double"
                    );
                }
                if (c != '0' && c != '.') break;
            }
        }

        return new Double(dv);
    }

    @SuppressWarnings("static-method") private boolean
    getConstantValue2(BooleanLiteral bl) {
        if (bl.value == "true")  return true;  // SUPPRESS CHECKSTYLE StringLiteralEquality
        if (bl.value == "false") return false; // SUPPRESS CHECKSTYLE StringLiteralEquality
        throw new InternalCompilerException(bl.value);
    }

    @SuppressWarnings("static-method") private char
    getConstantValue2(CharacterLiteral cl) throws CompileException {

        String v = cl.value;

        // Strip opening and closing single quotes.
        v = v.substring(1, v.length() - 1);

        // Decode escape sequences like "\n" and "\0377".
        v = UnitCompiler.unescape(v, cl.getLocation());

        if (v.isEmpty()) throw new CompileException("Empty character literal", cl.getLocation());

        if (v.length() > 1) throw new CompileException("Invalid character literal " + cl.value, cl.getLocation());

        return Character.valueOf(v.charAt(0));
    }

    @SuppressWarnings("static-method") private String
    getConstantValue2(StringLiteral sl) throws CompileException {

        String v = sl.value;

        // Strip opening and closing double quotes.
        v = v.substring(1, v.length() - 1);

        // Decode escape sequences like "\n" and "\0377".
        v = UnitCompiler.unescape(v, sl.getLocation());

        return v;
    }

    @SuppressWarnings("static-method")
    @Nullable private Object
    getConstantValue2(NullLiteral nl) { return null; }

    @SuppressWarnings("static-method")
    @Nullable private Object
    getConstantValue2(SimpleConstant sl) { return sl.value; }

    // ------------ BlockStatement.generatesCode() -------------

    /**
     * Checks whether invocation of {@link #com.hazelcast.com.ile(BlockStatement)} would generate more than zero code bytes.
     */
    private boolean
    generatesCode(BlockStatement bs) throws CompileException {

        Boolean result = (Boolean) bs.accept(new BlockStatementVisitor<Boolean, CompileException>() {

            // SUPPRESS CHECKSTYLE LineLengthCheck:23
            @Override public Boolean visitInitializer(Initializer i)                        throws CompileException { return UnitCompiler.this.generatesCode2(i);    }
            @Override public Boolean visitFieldDeclaration(FieldDeclaration fd)             throws CompileException { return UnitCompiler.this.generatesCode2(fd);   }
            @Override public Boolean visitLabeledStatement(LabeledStatement ls)                                     { return UnitCompiler.this.generatesCode2(ls);   }
            @Override public Boolean visitBlock(Block b)                                    throws CompileException { return UnitCompiler.this.generatesCode2(b);    }
            @Override public Boolean visitExpressionStatement(ExpressionStatement es)                               { return UnitCompiler.this.generatesCode2(es);   }
            @Override public Boolean visitIfStatement(IfStatement is)                                               { return UnitCompiler.this.generatesCode2(is);   }
            @Override public Boolean visitForStatement(ForStatement fs)                                             { return UnitCompiler.this.generatesCode2(fs);   }
            @Override public Boolean visitForEachStatement(ForEachStatement fes)                                    { return UnitCompiler.this.generatesCode2(fes);  }
            @Override public Boolean visitWhileStatement(WhileStatement ws)                                         { return UnitCompiler.this.generatesCode2(ws);   }
            @Override public Boolean visitTryStatement(TryStatement ts)                                             { return UnitCompiler.this.generatesCode2(ts);   }
            @Override public Boolean visitSwitchStatement(SwitchStatement ss)                                       { return UnitCompiler.this.generatesCode2(ss);   }
            @Override public Boolean visitSynchronizedStatement(SynchronizedStatement ss)                           { return UnitCompiler.this.generatesCode2(ss);   }
            @Override public Boolean visitDoStatement(DoStatement ds)                                               { return UnitCompiler.this.generatesCode2(ds);   }
            @Override public Boolean visitLocalVariableDeclarationStatement(LocalVariableDeclarationStatement lvds) { return UnitCompiler.this.generatesCode2(lvds); }
            @Override public Boolean visitReturnStatement(ReturnStatement rs)                                       { return UnitCompiler.this.generatesCode2(rs);   }
            @Override public Boolean visitThrowStatement(ThrowStatement ts)                                         { return UnitCompiler.this.generatesCode2(ts);   }
            @Override public Boolean visitBreakStatement(BreakStatement bs)                                         { return UnitCompiler.this.generatesCode2(bs);   }
            @Override public Boolean visitContinueStatement(ContinueStatement cs)                                   { return UnitCompiler.this.generatesCode2(cs);   }
            @Override public Boolean visitAssertStatement(AssertStatement as)                                       { return UnitCompiler.this.generatesCode2(as);   }
            @Override public Boolean visitEmptyStatement(EmptyStatement es)                                         { return UnitCompiler.this.generatesCode2(es);   }
            @Override public Boolean visitLocalClassDeclarationStatement(LocalClassDeclarationStatement lcds)       { return UnitCompiler.this.generatesCode2(lcds); }
            @Override public Boolean visitAlternateConstructorInvocation(AlternateConstructorInvocation aci)        { return UnitCompiler.this.generatesCode2(aci);  }
            @Override public Boolean visitSuperConstructorInvocation(SuperConstructorInvocation sci)                { return UnitCompiler.this.generatesCode2(sci);  }
        });

        assert result != null;
        return result;
    }

    @SuppressWarnings("static-method") private boolean
    generatesCode2(BlockStatement bs) { return true; }

    @SuppressWarnings("static-method") private boolean
    generatesCode2(AssertStatement as) { return true; }

    @SuppressWarnings("static-method") private boolean
    generatesCode2(EmptyStatement es) { return false; }

    @SuppressWarnings("static-method") private boolean
    generatesCode2(LocalClassDeclarationStatement lcds) { return false; }

    private boolean
    generatesCode2(Initializer i) throws CompileException { return this.generatesCode(i.block); }

    private boolean
    generatesCode2(List<BlockStatement> l) throws CompileException {
        for (BlockStatement bs : l) if (this.generatesCode(bs)) return true;
        return false;
    }

    private boolean
    generatesCode2(Block b) throws CompileException { return this.generatesCode2(b.statements); }

    private boolean
    generatesCode2(FieldDeclaration fd) throws CompileException {
        // Code is only generated if at least one of the declared variables has a non-constant-final initializer.
        for (VariableDeclarator vd : fd.variableDeclarators) {
            if (this.getNonConstantFinalInitializer(fd, vd) != null) return true;
        }
        return false;
    }

    // ------------ BlockStatement.leave() -------------

    /**
     * Cleans up the statement context. This is currently relevant for "{@code try ... catch ... finally}" statements
     * (execute {@code finally} clause) and {@code synchronized} statements (monitorexit).
     * <p>
     *   Statements like {@code return}, {@code break}, {@code continue} must call this method for all the statements
     *   they terminate.
     * </p>
     * <p>
     *   Notice: If <var>optionalStackValueType</var> is {@code null}, then the operand stack is empty; otherwise
     *   exactly one operand with that type is on the stack. This information is vital to implementations of {@link
     *   #leave(BlockStatement, IClass)} that require a specific operand stack state (e.g. an empty operand stack for
     *   JSR).
     * </p>
     */
    private void
    leave(BlockStatement bs, @Nullable final IClass optionalStackValueType) {
        BlockStatementVisitor<Void, RuntimeException> bsv = new BlockStatementVisitor<Void, RuntimeException>() {

            // SUPPRESS CHECKSTYLE LineLengthCheck:23
            @Override @Nullable public Void visitInitializer(Initializer i)                                                { UnitCompiler.this.leave2(i,    optionalStackValueType); return null; }
            @Override @Nullable public Void visitFieldDeclaration(FieldDeclaration fd)                                     { UnitCompiler.this.leave2(fd,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitLabeledStatement(LabeledStatement ls)                                     { UnitCompiler.this.leave2(ls,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitBlock(Block b)                                                            { UnitCompiler.this.leave2(b,    optionalStackValueType); return null; }
            @Override @Nullable public Void visitExpressionStatement(ExpressionStatement es)                               { UnitCompiler.this.leave2(es,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitIfStatement(IfStatement is)                                               { UnitCompiler.this.leave2(is,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitForStatement(ForStatement fs)                                             { UnitCompiler.this.leave2(fs,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitForEachStatement(ForEachStatement fes)                                    { UnitCompiler.this.leave2(fes,  optionalStackValueType); return null; }
            @Override @Nullable public Void visitWhileStatement(WhileStatement ws)                                         { UnitCompiler.this.leave2(ws,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitTryStatement(TryStatement ts)                                             { UnitCompiler.this.leave2(ts,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitSwitchStatement(SwitchStatement ss)                                       { UnitCompiler.this.leave2(ss,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitSynchronizedStatement(SynchronizedStatement ss)                           { UnitCompiler.this.leave2(ss,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitDoStatement(DoStatement ds)                                               { UnitCompiler.this.leave2(ds,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitLocalVariableDeclarationStatement(LocalVariableDeclarationStatement lvds) { UnitCompiler.this.leave2(lvds, optionalStackValueType); return null; }
            @Override @Nullable public Void visitReturnStatement(ReturnStatement rs)                                       { UnitCompiler.this.leave2(rs,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitThrowStatement(ThrowStatement ts)                                         { UnitCompiler.this.leave2(ts,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitBreakStatement(BreakStatement bs)                                         { UnitCompiler.this.leave2(bs,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitContinueStatement(ContinueStatement cs)                                   { UnitCompiler.this.leave2(cs,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitAssertStatement(AssertStatement as)                                       { UnitCompiler.this.leave2(as,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitEmptyStatement(EmptyStatement es)                                         { UnitCompiler.this.leave2(es,   optionalStackValueType); return null; }
            @Override @Nullable public Void visitLocalClassDeclarationStatement(LocalClassDeclarationStatement lcds)       { UnitCompiler.this.leave2(lcds, optionalStackValueType); return null; }
            @Override @Nullable public Void visitAlternateConstructorInvocation(AlternateConstructorInvocation aci)        { UnitCompiler.this.leave2(aci,  optionalStackValueType); return null; }
            @Override @Nullable public Void visitSuperConstructorInvocation(SuperConstructorInvocation sci)                { UnitCompiler.this.leave2(sci,  optionalStackValueType); return null; }
        };
        bs.accept(bsv);
    }

    private void
    leave2(BlockStatement bs, @Nullable IClass optionalStackValueType) {}

    private void
    leave2(SynchronizedStatement ss, @Nullable IClass optionalStackValueType) {
        this.load(ss, this.iClassLoader.TYPE_java_lang_Object, ss.monitorLvIndex);
        this.writeOpcode(ss, Opcode.MONITOREXIT);
    }

    private void
    leave2(TryStatement ts, @Nullable IClass optionalStackValueType) {

        Offset fo = ts.finallyOffset;
        if (fo == null) return;

        this.getCodeContext().saveLocalVariables();
        try {
            short sv = 0;

            // Obviously, JSR must always be executed with the operand stack being empty; otherwise we get
            // "java.lang.VerifyError: Inconsistent stack height 1 != 2"
            if (optionalStackValueType != null) {
                sv = this.getCodeContext().allocateLocalVariable(
                    Descriptor.size(optionalStackValueType.getDescriptor())
                );
                this.store(ts, optionalStackValueType, sv);
            }

            this.writeBranch(ts, Opcode.JSR, fo);

            if (optionalStackValueType != null) {
                this.load(ts, optionalStackValueType, sv);
            }
        } finally {
            this.getCodeContext().restoreLocalVariables();
        }
    }

    // ---------------- Lvalue.com.hazelcast.com.ileSet() -----------------

    /**
     * Generates code that stores a value in the {@link Lvalue}. Expects the {@link Lvalue}'s context (see {@link
     * #com.hazelcast.com.ileContext}) and a value of the {@link Lvalue}'s type on the operand stack.
     */
    private void
    com.hazelcast.com.ileSet(Lvalue lv) throws CompileException {

        lv.accept(new LvalueVisitor<Void, CompileException>() {

            // SUPPRESS CHECKSTYLE LineLength:7
            @Override @Nullable public Void visitAmbiguousName(AmbiguousName an)                                        throws CompileException { UnitCompiler.this.com.hazelcast.com.ileSet2(an);    return null; }
            @Override @Nullable public Void visitArrayAccessExpression(ArrayAccessExpression aae)                       throws CompileException { UnitCompiler.this.com.hazelcast.com.ileSet2(aae);   return null; }
            @Override @Nullable public Void visitFieldAccess(FieldAccess fa)                                            throws CompileException { UnitCompiler.this.com.hazelcast.com.ileSet2(fa);    return null; }
            @Override @Nullable public Void visitFieldAccessExpression(FieldAccessExpression fae)                       throws CompileException { UnitCompiler.this.com.hazelcast.com.ileSet2(fae);   return null; }
            @Override @Nullable public Void visitSuperclassFieldAccessExpression(SuperclassFieldAccessExpression scfae) throws CompileException { UnitCompiler.this.com.hazelcast.com.ileSet2(scfae); return null; }
            @Override @Nullable public Void visitLocalVariableAccess(LocalVariableAccess lva)                                                   { UnitCompiler.this.com.hazelcast.com.ileSet2(lva);   return null; }
            @Override @Nullable public Void visitParenthesizedExpression(ParenthesizedExpression pe)                    throws CompileException { UnitCompiler.this.com.hazelcast.com.ileSet2(pe);    return null; }
        });
    }
    private void
    com.hazelcast.com.ileSet2(AmbiguousName an) throws CompileException {
        this.com.hazelcast.com.ileSet(this.toLvalueOrCompileException(this.reclassify(an)));
    }

    private void
    com.hazelcast.com.ileSet2(LocalVariableAccess lva) { this.store(lva, lva.localVariable); }

    private void
    com.hazelcast.com.ileSet2(FieldAccess fa) throws CompileException {
        this.checkAccessible(fa.field, fa.getEnclosingScope(), fa.getLocation());
        this.putfield(fa, fa.field);
    }
    private void
    com.hazelcast.com.ileSet2(ArrayAccessExpression aae) throws CompileException {
        this.writeOpcode(aae, Opcode.IASTORE + UnitCompiler.ilfdabcs(this.getType(aae)));
    }
    private void
    com.hazelcast.com.ileSet2(FieldAccessExpression fae) throws CompileException {
        this.determineValue(fae);
        this.com.hazelcast.com.ileSet(this.toLvalueOrCompileException(this.determineValue(fae)));
    }
    private void
    com.hazelcast.com.ileSet2(SuperclassFieldAccessExpression scfae) throws CompileException {
        this.determineValue(scfae);
        this.com.hazelcast.com.ileSet(this.toLvalueOrCompileException(this.determineValue(scfae)));
    }
    private void
    com.hazelcast.com.ileSet2(ParenthesizedExpression pe) throws CompileException {
        this.com.hazelcast.com.ileSet(this.toLvalueOrCompileException(pe.value));
    }

    // ---------------- Atom.getType() ----------------

    private IClass
    getType(Atom a) throws CompileException {

        IClass result = (IClass) a.accept(new AtomVisitor<IClass, CompileException>() {

            @Override public IClass
            visitPackage(Package p) throws CompileException { return UnitCompiler.this.getType2(p); }

            @Override @Nullable public IClass
            visitType(Type t) throws CompileException {
                return (IClass) t.accept(new Visitor.TypeVisitor<IClass, CompileException>() {

                    // SUPPRESS CHECKSTYLE LineLengthCheck:5
                    @Override public IClass visitArrayType(ArrayType at)                throws CompileException { return UnitCompiler.this.getType2(at);  }
                    @Override public IClass visitPrimitiveType(PrimitiveType bt)                                { return UnitCompiler.this.getType2(bt);  }
                    @Override public IClass visitReferenceType(ReferenceType rt)        throws CompileException { return UnitCompiler.this.getType2(rt);  }
                    @Override public IClass visitRvalueMemberType(RvalueMemberType rmt) throws CompileException { return UnitCompiler.this.getType2(rmt); }
                    @Override public IClass visitSimpleType(SimpleType st)                                      { return UnitCompiler.this.getType2(st);  }
                });
            }

            @Override @Nullable public IClass
            visitRvalue(Rvalue rv) throws CompileException {

                return (IClass) rv.accept(new Visitor.RvalueVisitor<IClass, CompileException>() {

                    @Override @Nullable public IClass
                    visitLvalue(Lvalue lv) throws CompileException {
                        return (IClass) lv.accept(new Visitor.LvalueVisitor<IClass, CompileException>() {

                            // SUPPRESS CHECKSTYLE LineLengthCheck:7
                            @Override public IClass visitAmbiguousName(AmbiguousName an)                                        throws CompileException { return UnitCompiler.this.getType2(an);    }
                            @Override public IClass visitArrayAccessExpression(ArrayAccessExpression aae)                       throws CompileException { return UnitCompiler.this.getType2(aae);   }
                            @Override public IClass visitFieldAccess(FieldAccess fa)                                            throws CompileException { return UnitCompiler.this.getType2(fa);    }
                            @Override public IClass visitFieldAccessExpression(FieldAccessExpression fae)                       throws CompileException { return UnitCompiler.this.getType2(fae);   }
                            @Override public IClass visitSuperclassFieldAccessExpression(SuperclassFieldAccessExpression scfae) throws CompileException { return UnitCompiler.this.getType2(scfae); }
                            @Override public IClass visitLocalVariableAccess(LocalVariableAccess lva)                                                   { return UnitCompiler.this.getType2(lva);   }
                            @Override public IClass visitParenthesizedExpression(ParenthesizedExpression pe)                    throws CompileException { return UnitCompiler.this.getType2(pe);    }
                        });
                    }

                    // SUPPRESS CHECKSTYLE LineLengthCheck:25
                    @Override public IClass visitArrayLength(ArrayLength al)                                                        { return UnitCompiler.this.getType2(al);   }
                    @Override public IClass visitAssignment(Assignment a)                                   throws CompileException { return UnitCompiler.this.getType2(a);    }
                    @Override public IClass visitUnaryOperation(UnaryOperation uo)                          throws CompileException { return UnitCompiler.this.getType2(uo);   }
                    @Override public IClass visitBinaryOperation(BinaryOperation bo)                        throws CompileException { return UnitCompiler.this.getType2(bo);   }
                    @Override public IClass visitCast(Cast c)                                               throws CompileException { return UnitCompiler.this.getType2(c);    }
                    @Override public IClass visitClassLiteral(ClassLiteral cl)                                                      { return UnitCompiler.this.getType2(cl);   }
                    @Override public IClass visitConditionalExpression(ConditionalExpression ce)            throws CompileException { return UnitCompiler.this.getType2(ce);   }
                    @Override public IClass visitCrement(Crement c)                                         throws CompileException { return UnitCompiler.this.getType2(c);    }
                    @Override public IClass visitInstanceof(Instanceof io)                                                          { return UnitCompiler.this.getType2(io);   }
                    @Override public IClass visitMethodInvocation(MethodInvocation mi)                      throws CompileException { return UnitCompiler.this.getType2(mi);   }
                    @Override public IClass visitSuperclassMethodInvocation(SuperclassMethodInvocation smi) throws CompileException { return UnitCompiler.this.getType2(smi);  }
                    @Override public IClass visitIntegerLiteral(IntegerLiteral il)                                                  { return UnitCompiler.this.getType2(il);   }
                    @Override public IClass visitFloatingPointLiteral(FloatingPointLiteral fpl)                                     { return UnitCompiler.this.getType2(fpl);  }
                    @Override public IClass visitBooleanLiteral(BooleanLiteral bl)                                                  { return UnitCompiler.this.getType2(bl);   }
                    @Override public IClass visitCharacterLiteral(CharacterLiteral cl)                                              { return UnitCompiler.this.getType2(cl);   }
                    @Override public IClass visitStringLiteral(StringLiteral sl)                                                    { return UnitCompiler.this.getType2(sl);   }
                    @Override public IClass visitNullLiteral(NullLiteral nl)                                                        { return UnitCompiler.this.getType2(nl);   }
                    @Override public IClass visitSimpleConstant(SimpleConstant sl)                                                  { return UnitCompiler.this.getType2(sl);   }
                    @Override public IClass visitNewAnonymousClassInstance(NewAnonymousClassInstance naci)                          { return UnitCompiler.this.getType2(naci); }
                    @Override public IClass visitNewArray(NewArray na)                                      throws CompileException { return UnitCompiler.this.getType2(na);   }
                    @Override public IClass visitNewInitializedArray(NewInitializedArray nia)               throws CompileException { return UnitCompiler.this.getType2(nia);  }
                    @Override public IClass visitNewClassInstance(NewClassInstance nci)                     throws CompileException { return UnitCompiler.this.getType2(nci);  }
                    @Override public IClass visitParameterAccess(ParameterAccess pa)                        throws CompileException { return UnitCompiler.this.getType2(pa);   }
                    @Override public IClass visitQualifiedThisReference(QualifiedThisReference qtr)         throws CompileException { return UnitCompiler.this.getType2(qtr);  }
                    @Override public IClass visitThisReference(ThisReference tr)                            throws CompileException { return UnitCompiler.this.getType2(tr);   }
                });
            }

            @Override @Nullable public IClass
            visitConstructorInvocation(ConstructorInvocation ci) throws CompileException {
                return UnitCompiler.this.getType2(ci);
            }
        });

        return result != null ? result : this.iClassLoader.TYPE_java_lang_Object;
    }

    private IClass
    getType2(ConstructorInvocation ci) throws CompileException {
        this.com.hazelcast.com.ileError("Explicit constructor invocation not allowed here", ci.getLocation());
        return this.iClassLoader.TYPE_java_lang_Object;
    }

    @SuppressWarnings("static-method") private IClass
    getType2(SimpleType st) { return st.iClass; }

    @SuppressWarnings("static-method") private IClass
    getType2(PrimitiveType bt) {
        switch (bt.primitive) {
        case VOID:    return IClass.VOID;
        case BYTE:    return IClass.BYTE;
        case SHORT:   return IClass.SHORT;
        case CHAR:    return IClass.CHAR;
        case INT:     return IClass.INT;
        case LONG:    return IClass.LONG;
        case FLOAT:   return IClass.FLOAT;
        case DOUBLE:  return IClass.DOUBLE;
        case BOOLEAN: return IClass.BOOLEAN;
        default:      throw new InternalCompilerException("Invalid primitive " + bt.primitive);
        }
    }
    private IClass
    getType2(ReferenceType rt) throws CompileException {
        String[] identifiers = rt.identifiers;

        IClass result = this.getReferenceType(
            rt.getLocation(),
            rt.getEnclosingScope(),
            identifiers,
            identifiers.length
        );
        if (result == null) {
            this.com.hazelcast.com.ileError("Reference type \"" + rt + "\" not found", rt.getLocation());
            return this.iClassLoader.TYPE_java_lang_Object;
        }

        return result;
    }

    /**
     * @return The resolved {@link IClass}, or {@code null}
     */
    @Nullable private IClass
    getReferenceType(Location location, Scope scope, String[] identifiers, int n) throws CompileException {

        if (n == 1) {
            return this.getReferenceType(location, identifiers[0], scope);
        }

        // JLS7 6.5.5.1   Unnamed package member type name (one identifier).
        // JLS7 6.5.5.2.1 Qualified type name (two or more identifiers).
        {
            String className = Java.join(identifiers, ".", 0, n);
            IClass result    = this.findTypeByName(location, className);
            if (result != null) return result;
        }

        // JLS7 6.5.5.2.2 referenceType '.' memberTypeName
        if (n >= 2) {
            IClass enclosingType = this.getReferenceType(location, scope, identifiers, n - 1);
            if (enclosingType != null) {
                String memberTypeName = identifiers[n - 1];
                IClass memberType     = this.findMemberType(enclosingType, memberTypeName, location);
                if (memberType == null) {
                    this.com.hazelcast.com.ileError(
                        "\"" + enclosingType + "\" declares no member type \"" + memberTypeName + "\"",
                        location
                    );
                    return this.iClassLoader.TYPE_java_lang_Object;
                }
                return memberType;
            }
        }

        return null;
    }

    /**
     * JLS7 6.5.5.1 Simple type name (single identifier)
     *
     * @return The resolved {@link IClass}
     */
    private IClass
    getReferenceType(Location location, String simpleTypeName, Scope scope) throws CompileException {
        BlockStatement   scopeBlockStatement   = null;
        TypeDeclaration  scopeTypeDeclaration  = null;
        MethodDeclarator scopeMethodDeclarator = null;
        CompilationUnit  scopeCompilationUnit;
        for (Scope s = scope;; s = s.getEnclosingScope()) {
            if (s instanceof BlockStatement && scopeBlockStatement == null) {
                scopeBlockStatement = (BlockStatement) s;
            }
            if (s instanceof TypeDeclaration && scopeTypeDeclaration == null) {
                scopeTypeDeclaration = (TypeDeclaration) s;
            }
            if (s instanceof MethodDeclarator && scopeMethodDeclarator == null) {
                scopeMethodDeclarator = (MethodDeclarator) s;
            }
            if (s instanceof CompilationUnit) {
                scopeCompilationUnit = (CompilationUnit) s;
                break;
            }
        }

        // Method declaration type parameter?
        if (scopeMethodDeclarator != null) {
            TypeParameter[]
            optionalTypeParameters = scopeMethodDeclarator.getOptionalTypeParameters();
            if (optionalTypeParameters != null) {
                for (TypeParameter tp : optionalTypeParameters) {
                    if (tp.name.equals(simpleTypeName)) {
                        IClass[]        boundTypes;
                        ReferenceType[] ob = tp.optionalBound;
                        if (ob == null) {
                            boundTypes = new IClass[] { this.iClassLoader.TYPE_java_lang_Object };
                        } else {
                            boundTypes = new IClass[ob.length];
                            for (int i = 0; i < boundTypes.length; i++) {
                                boundTypes[i] = this.getType(ob[i]);
                            }
                        }

                        // Here is the big simplification: Instead of returning the "correct" type, honoring type
                        // arguments, we simply return the first bound. E.g. "Map.get(K)" returns a "V", but
                        // JANINO says it's an "Object" (the implicit bound of "V").
                        return boundTypes[0];
                    }
                }
            }
        }

        // Type declaration type parameter?
        if (scopeTypeDeclaration instanceof NamedTypeDeclaration) {
            TypeParameter[]
            optionalTypeParameters = ((NamedTypeDeclaration) scopeTypeDeclaration).getOptionalTypeParameters();
            if (optionalTypeParameters != null) {
                for (TypeParameter tp : optionalTypeParameters) {
                    if (tp.name.equals(simpleTypeName)) {
                        IClass[]        boundTypes;
                        ReferenceType[] ob = tp.optionalBound;
                        if (ob == null) {
                            boundTypes = new IClass[] { this.iClassLoader.TYPE_java_lang_Object };
                        } else {
                            boundTypes = new IClass[ob.length];
                            for (int i = 0; i < boundTypes.length; i++) {
                                boundTypes[i] = this.getType(ob[i]);
                            }
                        }
                        return boundTypes[0];
                    }
                }
            }
        }

        // 6.5.5.1.1 Local class.
        {
            LocalClassDeclaration lcd = UnitCompiler.findLocalClassDeclaration(
                scope,
                simpleTypeName
            );
            if (lcd != null) return this.resolve(lcd);
        }

        // 6.5.5.1.2 Member type.
        if (scopeTypeDeclaration != null) { // If enclosed by another type declaration...
            for (
                Scope s = scopeTypeDeclaration;
                !(s instanceof CompilationUnit);
                s = s.getEnclosingScope()
            ) {
                if (s instanceof TypeDeclaration) {
                    IClass mt = this.findMemberType(
                        this.resolve((AbstractTypeDeclaration) s),
                        simpleTypeName,
                        location
                    );
                    if (mt != null) return mt;
                }
            }
        }

        // 6.5.5.1.4a Single-type import.
        {
            IClass importedClass = this.importSingleType(simpleTypeName, location);
            if (importedClass != null) return importedClass;
        }

        // 6.5.5.1.4b Type declared in same com.hazelcast.com.ilation unit.
        {
            PackageMemberTypeDeclaration pmtd = (
                scopeCompilationUnit.getPackageMemberTypeDeclaration(simpleTypeName)
            );
            if (pmtd != null) return this.resolve(pmtd);
        }

        // 6.5.5.1.5 Type declared in other com.hazelcast.com.ilation unit of same package.
        {
            PackageDeclaration opd = scopeCompilationUnit.optionalPackageDeclaration;

            String pkg       = opd == null ? null : opd.packageName;
            String className = pkg == null ? simpleTypeName : pkg + "." + simpleTypeName;
            IClass result    = this.findTypeByName(location, className);
            if (result != null) return result;
        }

        // 6.5.5.1.6 Type-import-on-demand declaration.
        {
            IClass importedClass = this.importTypeOnDemand(simpleTypeName, location);
            if (importedClass != null) return importedClass;
        }

        // JLS7 6.5.2.BL1.B2: Type imported through single static import.
        {
            List<Object/*IField+IMethod+IClass*/> l = (List<Object>) this.singleStaticImports.get(simpleTypeName);
            if (l != null) {
                IClass importedMemberType = null;
                for (Iterator<Object/*IField+IMethod+IClass*/> it = l.iterator(); it.hasNext();) {
                    Object o = it.next();
                    if (o instanceof IClass) {
                        IClass mt = (IClass) o;
                        if (!this.isAccessible(mt, scope)) continue;
                        if (importedMemberType != null && importedMemberType != mt) {
                            this.com.hazelcast.com.ileError(
                                "Ambiguous static imports: \""
                                + importedMemberType.toString()
                                + "\" vs. \""
                                + mt.toString()
                                + "\""
                            );
                        }
                        importedMemberType = mt;
                    }
                }
                if (importedMemberType != null) return importedMemberType;
            }
        }

        // JLS7 6.5.2.BL1.B2: Type imported through static-import-on-demand.
        {
            IClass importedMemberType = null;
            for (IClass ic : this.staticImportsOnDemand) {
                IClass[] memberTypes = ic.getDeclaredIClasses();
                for (IClass mt : memberTypes) {
                    if (!this.isAccessible(mt, scope)) continue;
                    if (mt.getDescriptor().endsWith('$' + simpleTypeName + ';')) {
                        if (importedMemberType != null) {
                            this.com.hazelcast.com.ileError(
                                "Ambiguous static imports: \""
                                + importedMemberType.toString()
                                + "\" vs. \""
                                + mt.toString()
                                + "\""
                            );
                        }
                        importedMemberType = mt;
                    }
                }
            }
            if (importedMemberType != null) return importedMemberType;
        }

        // Unnamed package member type.
        {
            IClass result = this.findTypeByName(location, simpleTypeName);
            if (result != null) return result;
        }

        // Type argument of the enclosing anonymous class declaration?
        if (scopeTypeDeclaration instanceof AnonymousClassDeclaration) {
            Type bt = ((AnonymousClassDeclaration) scopeTypeDeclaration).baseType;
            if (bt instanceof ReferenceType) {
                TypeArgument[] otas = ((ReferenceType) bt).optionalTypeArguments;
                if (otas != null) {
                    for (TypeArgument ta : otas) {
                        if (ta instanceof ReferenceType) {
                            String[] is = ((ReferenceType) ta).identifiers;
                            if (is.length == 1 && is[0].equals(simpleTypeName)) {
                                return this.iClassLoader.TYPE_java_lang_Object;
                            }
                        }
                    }
                }
            }
        }
        // 6.5.5.1.8 Give up.
        this.com.hazelcast.com.ileError("Cannot determine simple type name \"" + simpleTypeName + "\"", location);
        return this.iClassLoader.TYPE_java_lang_Object;
    }

    private IClass
    getType2(RvalueMemberType rvmt) throws CompileException {
        IClass rvt        = this.getType(rvmt.rvalue);
        IClass memberType = this.findMemberType(rvt, rvmt.identifier, rvmt.getLocation());
        if (memberType == null) {
            this.com.hazelcast.com.ileError("\"" + rvt + "\" has no member type \"" + rvmt.identifier + "\"", rvmt.getLocation());
            return this.iClassLoader.TYPE_java_lang_Object;
        }
        return memberType;
    }

    private IClass
    getType2(ArrayType at) throws CompileException {
        return this.getType(at.com.hazelcast.com.onentType).getArrayIClass(this.iClassLoader.TYPE_java_lang_Object);
    }

    private IClass
    getType2(AmbiguousName an) throws CompileException {
        return this.getType(this.reclassify(an));
    }

    private IClass
    getType2(Package p) throws CompileException {
        this.com.hazelcast.com.ileError("Unknown variable or type \"" + p.name + "\"", p.getLocation());
        return this.iClassLoader.TYPE_java_lang_Object;
    }

    @SuppressWarnings("static-method")
    private IClass
    getType2(LocalVariableAccess lva) {
        return lva.localVariable.type;
    }

    @SuppressWarnings("static-method")
    private IClass
    getType2(FieldAccess fa) throws CompileException {
        return fa.field.getType();
    }

    @SuppressWarnings("static-method")
    private IClass
    getType2(ArrayLength al) {
        return IClass.INT;
    }

    private IClass
    getType2(ThisReference tr) throws CompileException {
        return this.getIClass(tr);
    }

    private IClass
    getType2(QualifiedThisReference qtr) throws CompileException {
        return this.getTargetIClass(qtr);
    }

    private IClass
    getType2(ClassLiteral cl) {
        return this.iClassLoader.TYPE_java_lang_Class;
    }

    private IClass
    getType2(Assignment a) throws CompileException {
        return this.getType(a.lhs);
    }

    private IClass
    getType2(ConditionalExpression ce) throws CompileException {
        IClass mhsType = this.getType(ce.mhs);
        IClass rhsType = this.getType(ce.rhs);

        if (mhsType == rhsType) {

            // JLS7 15.25, list 1, bullet 1: "b ? T : T => T"
            return mhsType;
        } else
        if (this.isUnboxingConvertible(mhsType) == rhsType) {

            // JLS7 15.25, list 1, bullet 2: "b ? Integer : int => int"
            return rhsType;
        } else
        if (this.isUnboxingConvertible(rhsType) == mhsType) {

            // JLS7 15.25, list 1, bullet 2: "b ? int : Integer => int"
            return mhsType;
        } else
        if (this.getConstantValue(ce.mhs) == null && !rhsType.isPrimitive()) {

            // JLS7 15.25, list 1, bullet 3: "b ? null : ReferenceType => ReferenceType"
            return rhsType;
        } else
        if (!mhsType.isPrimitive() && this.getConstantValue(ce.rhs) == null) {

            // JLS7 15.25, list 1, bullet 3: "b ? ReferenceType : null => ReferenceType"
            return mhsType;
        } else
        if (this.isConvertibleToPrimitiveNumeric(mhsType) && this.isConvertibleToPrimitiveNumeric(rhsType)) {

            // JLS7 15.25, list 1, bullet 4, bullet 1: "b ? Byte : Short => short"
            if (
                (mhsType == IClass.BYTE || mhsType == this.iClassLoader.TYPE_java_lang_Byte)
                && (rhsType == IClass.SHORT || rhsType == this.iClassLoader.TYPE_java_lang_Short)
            ) return IClass.SHORT;
            if (
                (rhsType == IClass.BYTE || rhsType == this.iClassLoader.TYPE_java_lang_Byte)
                && (mhsType == IClass.SHORT || mhsType == this.iClassLoader.TYPE_java_lang_Short)
            ) return IClass.SHORT;

            // JLS7 15.25, list 1, bullet 4, bullet 2: "b ? 127 : byte => byte"
            if (
                (mhsType == IClass.BYTE || mhsType == IClass.SHORT || mhsType == IClass.CHAR)
                && ce.rhs.constantValue != null
                && this.assignmentConversion(ce.rhs, this.getConstantValue(ce.rhs), mhsType) != null
            ) return mhsType;
            if (
                (rhsType == IClass.BYTE || rhsType == IClass.SHORT || rhsType == IClass.CHAR)
                && ce.mhs.constantValue != null
                && this.assignmentConversion(ce.mhs, this.getConstantValue(ce.mhs), rhsType) != null
            ) return rhsType;

            // TODO JLS7 15.25, list 1, bullet 4, bullet 3: "b ? 127 : byte => byte"

            // JLS7 15.25, list 1, bullet 4, bullet 4: "b ? Integer : Double => double"
            return this.binaryNumericPromotionType(ce, this.getUnboxedType(mhsType), this.getUnboxedType(rhsType));
        } else
        if (!mhsType.isPrimitive() && !rhsType.isPrimitive()) {

            // JLS7 15.25, list 1, bullet 5: "b ? Base : Derived => Base"
            if (mhsType.isAssignableFrom(rhsType)) {
                return mhsType;
            } else
            if (rhsType.isAssignableFrom(mhsType)) {
                return rhsType;
            } else {
                this.com.hazelcast.com.ileError(
                    "Reference types \"" + mhsType + "\" and \"" + rhsType + "\" don't match",
                    ce.getLocation()
                );
                return this.iClassLoader.TYPE_java_lang_Object;
            }
        } else
        {
            this.com.hazelcast.com.ileError(
                "Incompatible expression types \"" + mhsType + "\" and \"" + rhsType + "\"",
                ce.getLocation()
            );
            return this.iClassLoader.TYPE_java_lang_Object;
        }
    }

    private IClass
    getType2(Crement c) throws CompileException {
        return this.getType(c.operand);
    }

    private IClass
    getType2(ArrayAccessExpression aae) throws CompileException {
        IClass com.hazelcast.com.onentType = this.getType(aae.lhs).getComponentType();
        assert com.hazelcast.com.onentType != null : "null com.hazelcast.com.onent type for " + aae;
        return com.hazelcast.com.onentType;
    }

    private IClass
    getType2(FieldAccessExpression fae) throws CompileException {
        this.determineValue(fae);
        return this.getType(this.determineValue(fae));
    }

    private IClass
    getType2(SuperclassFieldAccessExpression scfae) throws CompileException {
        this.determineValue(scfae);
        return this.getType(this.determineValue(scfae));
    }

    private IClass
    getType2(UnaryOperation uo) throws CompileException {
        if (uo.operator == "!") return IClass.BOOLEAN; // SUPPRESS CHECKSTYLE StringLiteralEquality

         // SUPPRESS CHECKSTYLE StringLiteralEquality
        if (uo.operator == "+" || uo.operator == "-" || uo.operator == "~") {
            return this.unaryNumericPromotionType(uo, this.getUnboxedType(this.getType(uo.operand)));
        }

        this.com.hazelcast.com.ileError("Unexpected operator \"" + uo.operator + "\"", uo.getLocation());
        return IClass.BOOLEAN;
    }

    @SuppressWarnings("static-method")
    private IClass
    getType2(Instanceof io) { return IClass.BOOLEAN; }

    private IClass
    getType2(BinaryOperation bo) throws CompileException {
        if (
            // SUPPRESS CHECKSTYLE StringLiteralEquality:8
            bo.operator == "||"
            || bo.operator == "&&"
            || bo.operator == "=="
            || bo.operator == "!="
            || bo.operator == "<"
            || bo.operator == ">"
            || bo.operator == "<="
            || bo.operator == ">="
        ) return IClass.BOOLEAN;

        if (bo.operator == "|" || bo.operator == "^" || bo.operator == "&") { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
            IClass lhsType = this.getType(bo.lhs);
            return (
                lhsType == IClass.BOOLEAN || lhsType == this.iClassLoader.TYPE_java_lang_Boolean
                ? IClass.BOOLEAN
                : this.binaryNumericPromotionType(bo, lhsType, this.getType(bo.rhs))
            );
        }

        if (bo.operator == "*" || bo.operator == "/" || bo.operator == "%" || bo.operator == "+" || bo.operator == "-") { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
            IClassLoader icl = this.iClassLoader;

            // Unroll the operands of this binary operation.
            Iterator<Rvalue> ops = bo.unrollLeftAssociation();

            IClass lhsType = this.getType((Rvalue) ops.next());

            // Check the far left operand type.
            if (bo.operator == "+" && lhsType == icl.TYPE_java_lang_String) { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
                return icl.TYPE_java_lang_String;
            }

            // Determine the expression type.
            lhsType = this.getUnboxedType(lhsType);
            do {
                IClass rhsType = this.getUnboxedType(this.getType((Rvalue) ops.next()));
                if (bo.operator == "+" && rhsType == icl.TYPE_java_lang_String) { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
                    return icl.TYPE_java_lang_String;
                }
                lhsType = this.binaryNumericPromotionType(bo, lhsType, rhsType);
            } while (ops.hasNext());

            return lhsType;
        }

        if (bo.operator == "<<"  || bo.operator == ">>"  || bo.operator == ">>>") { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
            IClass lhsType = this.getType(bo.lhs);
            return this.unaryNumericPromotionType(bo, lhsType);
        }

        this.com.hazelcast.com.ileError("Unexpected operator \"" + bo.operator + "\"", bo.getLocation());
        return this.iClassLoader.TYPE_java_lang_Object;
    }

    /**
     * @return The <var>type</var>, or, iff <var>type</var> is a primitive wrapper type, the unwrapped <var>type</var>
     */
    private IClass
    getUnboxedType(IClass type) {
        IClass c = this.isUnboxingConvertible(type);
        return c != null ? c : type;
    }

    private IClass
    getType2(Cast c) throws CompileException {
        return this.getType(c.targetType);
    }

    private IClass
    getType2(ParenthesizedExpression pe) throws CompileException {
        return this.getType(pe.value);
    }

    private IClass
    getType2(MethodInvocation mi) throws CompileException {
        IMethod iMethod = mi.iMethod != null ? mi.iMethod : (mi.iMethod = this.findIMethod(mi));

        return iMethod.getReturnType();
    }

    private IClass
    getType2(SuperclassMethodInvocation scmi) throws CompileException {
        return this.findIMethod(scmi).getReturnType();
    }

    private IClass
    getType2(NewClassInstance nci) throws CompileException {
        if (nci.iClass != null) return nci.iClass;

        assert nci.type != null;
        return (nci.iClass = this.getType(nci.type));
    }

    private IClass
    getType2(NewAnonymousClassInstance naci) {
        return this.resolve(naci.anonymousClassDeclaration);
    }

    private IClass
    getType2(ParameterAccess pa) throws CompileException {
        return this.getLocalVariable(pa.formalParameter).type;
    }

    private IClass
    getType2(NewArray na) throws CompileException {
        IClass res = this.getType(na.type);
        return res.getArrayIClass(na.dimExprs.length + na.dims, this.iClassLoader.TYPE_java_lang_Object);
    }

    private IClass
    getType2(NewInitializedArray nia) throws CompileException {
        IClass at = nia.arrayType != null ? this.getType(nia.arrayType) : nia.arrayIClass;
        assert at != null;
        return at;
    }

    @SuppressWarnings("static-method") private IClass
    getType2(IntegerLiteral il) {
        String v        = il.value;
        char   lastChar = v.charAt(v.length() - 1);
        return lastChar == 'l' || lastChar == 'L' ? IClass.LONG : IClass.INT;
    }

    @SuppressWarnings("static-method") private IClass
    getType2(FloatingPointLiteral fpl) {
        String v        = fpl.value;
        char   lastChar = v.charAt(v.length() - 1);
        return lastChar == 'f' || lastChar == 'F' ? IClass.FLOAT : IClass.DOUBLE;
    }

    @SuppressWarnings("static-method") private IClass
    getType2(BooleanLiteral bl) {
        return IClass.BOOLEAN;
    }

    @SuppressWarnings("static-method") private IClass
    getType2(CharacterLiteral cl) {
        return IClass.CHAR;
    }

    private IClass
    getType2(StringLiteral sl) {
        return this.iClassLoader.TYPE_java_lang_String;
    }

    @SuppressWarnings("static-method") private IClass
    getType2(NullLiteral nl) {
        return IClass.VOID;
    }

    private IClass
    getType2(SimpleConstant sl) {
        Object v = sl.value;
        if (v instanceof Byte)      return IClass.BYTE;
        if (v instanceof Short)     return IClass.SHORT;
        if (v instanceof Integer)   return IClass.INT;
        if (v instanceof Long)      return IClass.LONG;
        if (v instanceof Float)     return IClass.FLOAT;
        if (v instanceof Double)    return IClass.DOUBLE;
        if (v instanceof Boolean)   return IClass.BOOLEAN;
        if (v instanceof Character) return IClass.CHAR;
        if (v instanceof String)    return this.iClassLoader.TYPE_java_lang_String;
        if (v == null)              return IClass.VOID;
        throw new InternalCompilerException("Invalid SimpleLiteral value type \"" + v.getClass() + "\"");
    }

    // ---------------- Atom.isType() ---------------

    private boolean
    isType(Atom a) throws CompileException {

        Boolean result = (Boolean) a.accept(new AtomVisitor<Boolean, CompileException>() {

            @Override public Boolean visitPackage(Package p) { return UnitCompiler.this.isType2(p); }

            @Override @Nullable public Boolean
            visitType(Type t) {

                return (Boolean) t.accept(new Visitor.TypeVisitor<Boolean, RuntimeException>() {

                    // SUPPRESS CHECKSTYLE LineLengthCheck:5
                    @Override public Boolean visitArrayType(ArrayType at)                { return UnitCompiler.this.isType2(at);  }
                    @Override public Boolean visitPrimitiveType(PrimitiveType bt)        { return UnitCompiler.this.isType2(bt);  }
                    @Override public Boolean visitReferenceType(ReferenceType rt)        { return UnitCompiler.this.isType2(rt);  }
                    @Override public Boolean visitRvalueMemberType(RvalueMemberType rmt) { return UnitCompiler.this.isType2(rmt); }
                    @Override public Boolean visitSimpleType(SimpleType st)              { return UnitCompiler.this.isType2(st);  }
                });
            }

            @Override @Nullable public Boolean
            visitRvalue(Rvalue rv) throws CompileException {

                return (Boolean) rv.accept(new Visitor.RvalueVisitor<Boolean, CompileException>() {

                    @Override @Nullable public Boolean
                    visitLvalue(Lvalue lv) throws CompileException {

                        return (Boolean) lv.accept(new Visitor.LvalueVisitor<Boolean, CompileException>() {

                            // SUPPRESS CHECKSTYLE LineLengthCheck:7
                            @Override public Boolean visitAmbiguousName(AmbiguousName an)                throws CompileException { return UnitCompiler.this.isType2(an);    }
                            @Override public Boolean visitArrayAccessExpression(ArrayAccessExpression aae)                       { return UnitCompiler.this.isType2(aae);   }
                            @Override public Boolean visitFieldAccess(FieldAccess fa)                                            { return UnitCompiler.this.isType2(fa);    }
                            @Override public Boolean visitFieldAccessExpression(FieldAccessExpression fae)                       { return UnitCompiler.this.isType2(fae);   }
                            @Override public Boolean visitSuperclassFieldAccessExpression(SuperclassFieldAccessExpression scfae) { return UnitCompiler.this.isType2(scfae); }
                            @Override public Boolean visitLocalVariableAccess(LocalVariableAccess lva)                           { return UnitCompiler.this.isType2(lva);   }
                            @Override public Boolean visitParenthesizedExpression(ParenthesizedExpression pe)                    { return UnitCompiler.this.isType2(pe);    }
                        });
                    }

                    // SUPPRESS CHECKSTYLE LineLengthCheck:25
                    @Override public Boolean visitArrayLength(ArrayLength al)                                { return UnitCompiler.this.isType2(al);   }
                    @Override public Boolean visitAssignment(Assignment a)                                   { return UnitCompiler.this.isType2(a);    }
                    @Override public Boolean visitUnaryOperation(UnaryOperation uo)                          { return UnitCompiler.this.isType2(uo);   }
                    @Override public Boolean visitBinaryOperation(BinaryOperation bo)                        { return UnitCompiler.this.isType2(bo);   }
                    @Override public Boolean visitCast(Cast c)                                               { return UnitCompiler.this.isType2(c);    }
                    @Override public Boolean visitClassLiteral(ClassLiteral cl)                              { return UnitCompiler.this.isType2(cl);   }
                    @Override public Boolean visitConditionalExpression(ConditionalExpression ce)            { return UnitCompiler.this.isType2(ce);   }
                    @Override public Boolean visitCrement(Crement c)                                         { return UnitCompiler.this.isType2(c);    }
                    @Override public Boolean visitInstanceof(Instanceof io)                                  { return UnitCompiler.this.isType2(io);   }
                    @Override public Boolean visitMethodInvocation(MethodInvocation mi)                      { return UnitCompiler.this.isType2(mi);   }
                    @Override public Boolean visitSuperclassMethodInvocation(SuperclassMethodInvocation smi) { return UnitCompiler.this.isType2(smi);  }
                    @Override public Boolean visitIntegerLiteral(IntegerLiteral il)                          { return UnitCompiler.this.isType2(il);   }
                    @Override public Boolean visitFloatingPointLiteral(FloatingPointLiteral fpl)             { return UnitCompiler.this.isType2(fpl);  }
                    @Override public Boolean visitBooleanLiteral(BooleanLiteral bl)                          { return UnitCompiler.this.isType2(bl);   }
                    @Override public Boolean visitCharacterLiteral(CharacterLiteral cl)                      { return UnitCompiler.this.isType2(cl);   }
                    @Override public Boolean visitStringLiteral(StringLiteral sl)                            { return UnitCompiler.this.isType2(sl);   }
                    @Override public Boolean visitNullLiteral(NullLiteral nl)                                { return UnitCompiler.this.isType2(nl);   }
                    @Override public Boolean visitSimpleConstant(SimpleConstant sl)                          { return UnitCompiler.this.isType2(sl);   }
                    @Override public Boolean visitNewAnonymousClassInstance(NewAnonymousClassInstance naci)  { return UnitCompiler.this.isType2(naci); }
                    @Override public Boolean visitNewArray(NewArray na)                                      { return UnitCompiler.this.isType2(na);   }
                    @Override public Boolean visitNewInitializedArray(NewInitializedArray nia)               { return UnitCompiler.this.isType2(nia);  }
                    @Override public Boolean visitNewClassInstance(NewClassInstance nci)                     { return UnitCompiler.this.isType2(nci);  }
                    @Override public Boolean visitParameterAccess(ParameterAccess pa)                        { return UnitCompiler.this.isType2(pa);   }
                    @Override public Boolean visitQualifiedThisReference(QualifiedThisReference qtr)         { return UnitCompiler.this.isType2(qtr);  }
                    @Override public Boolean visitThisReference(ThisReference tr)                            { return UnitCompiler.this.isType2(tr);   }
                });
            }

            @Override @Nullable public Boolean
            visitConstructorInvocation(ConstructorInvocation ci) { return false; }
        });

        assert result != null;
        return result;
    }

    @SuppressWarnings("static-method") private boolean
    isType2(Atom a) { return a instanceof Type; }

    private boolean
    isType2(AmbiguousName an) throws CompileException { return this.isType(this.reclassify(an)); }

    /**
     * Determines whether the given {@link IClass.IMember} is accessible in the given context, according to
     * JLS7 6.6.1.BL1.B4. Issues a {@link #com.hazelcast.com.ileError(String)} if not.
     */
    private boolean
    isAccessible(IClass.IMember member, Scope contextScope) throws CompileException {

        // You have to check that both the class and member are accessible in this scope.
        IClass  declaringIClass = member.getDeclaringIClass();
        boolean acc             = this.isAccessible(declaringIClass, contextScope);
        acc = acc && this.isAccessible(declaringIClass, member.getAccess(), contextScope);
        return acc;
    }

    /**
     * Checks whether the given {@link IClass.IMember} is accessible in the given context, according to JLS7
     * 6.6.1.BL1.B4. Issues a {@link #com.hazelcast.com.ileError(String)} if not.
     */
    private void
    checkAccessible(IClass.IMember member, Scope contextScope, Location location) throws CompileException {

        // You have to check that both the class and member are accessible in this scope.
        IClass declaringIClass = member.getDeclaringIClass();
        this.checkAccessible(declaringIClass, contextScope, location);
        this.checkMemberAccessible(declaringIClass, member, contextScope, location);
    }

    /**
     * Determines whether a member (class, interface, field or method) declared in a given class is accessible from a
     * given block statement context, according to JLS7 6.6.1.4.
     */
    private boolean
    isAccessible(IClass iClassDeclaringMember, Access memberAccess, Scope contextScope) throws CompileException {
        return null == this.internalCheckAccessible(iClassDeclaringMember, memberAccess, contextScope);
    }

    /**
     * Verifies that a member (class, interface, field or method) declared in a given class is accessible from a given
     * block statement context, according to JLS7 6.6.1.4. Issue a {@link #com.hazelcast.com.ileError(String)} if not.
     */
    private void
    checkMemberAccessible(
        IClass         iClassDeclaringMember,
        IClass.IMember member,
        Scope          contextScope,
        Location       location
    ) throws CompileException {
        String message = this.internalCheckAccessible(iClassDeclaringMember, member.getAccess(), contextScope);
        if (message != null) this.com.hazelcast.com.ileError(member.toString() + ": " + message, location);
    }

    /**
     * @return a descriptive text iff a member declared in that {@link IClass} with that {@link Access} is inaccessible
     */
    @Nullable private String
    internalCheckAccessible(
        IClass iClassDeclaringMember,
        Access memberAccess,
        Scope  contextScope
    ) throws CompileException {

        // At this point, memberAccess is PUBLIC, DEFAULT, PROTECTED or PRIVATE.

        // PUBLIC members are always accessible.
        if (memberAccess == Access.PUBLIC) return null;

        // At this point, the member is DEFAULT, PROTECTED or PRIVATE accessible.

        // Determine the class declaring the context.
        IClass iClassDeclaringContext = null;
        for (Scope s = contextScope; !(s instanceof CompilationUnit); s = s.getEnclosingScope()) {
            if (s instanceof TypeDeclaration) {
                iClassDeclaringContext = this.resolve((TypeDeclaration) s);
                break;
            }
        }

        // Access is always allowed for block statements declared in the same class as the member.
        if (iClassDeclaringContext == iClassDeclaringMember) return null;

        // Check whether the member and the context block statement are enclosed by the same top-level type.
        if (
            iClassDeclaringContext != null
            && !this.options.contains(JaninoOption.PRIVATE_MEMBERS_OF_ENCLOSING_AND_ENCLOSED_TYPES_INACCESSIBLE)
        ) {
            IClass topLevelIClassEnclosingMember = iClassDeclaringMember;
            for (IClass c = iClassDeclaringMember.getDeclaringIClass(); c != null; c = c.getDeclaringIClass()) {
                topLevelIClassEnclosingMember = c;
            }
            IClass topLevelIClassEnclosingContextBlockStatement = iClassDeclaringContext;
            for (
                IClass c = iClassDeclaringContext.getDeclaringIClass();
                c != null;
                c = c.getDeclaringIClass()
            ) topLevelIClassEnclosingContextBlockStatement = c;

            if (topLevelIClassEnclosingMember == topLevelIClassEnclosingContextBlockStatement) return null;
        }

        if (memberAccess == Access.PRIVATE) {
            return "Private member cannot be accessed from type \"" + iClassDeclaringContext + "\".";
        }

        // At this point, the member is DEFAULT or PROTECTED accessible.

        // Check whether the member and the context block statement are declared in the same package.
        if (iClassDeclaringContext != null && Descriptor.areInSamePackage(
            iClassDeclaringMember.getDescriptor(),
            iClassDeclaringContext.getDescriptor()
        )) return null;

        if (memberAccess == Access.DEFAULT) {
            return (
                "Member with \"package\" access cannot be accessed from type \""
                + iClassDeclaringContext
                + "\"."
            );
        }

        // At this point, the member is PROTECTED accessible.

        // Check whether the class declaring the context block statement is a subclass of the class declaring the
        // member or a nested class whose parent is a subclass
        {
            IClass parentClass = iClassDeclaringContext;
            do {
                assert parentClass != null;
                if (iClassDeclaringMember.isAssignableFrom(parentClass)) {
                    return null;
                }
                parentClass = parentClass.getOuterIClass();
            } while (parentClass != null);
        }

        return (
            "Protected member cannot be accessed from type \""
            + iClassDeclaringContext
            + "\", which is neither declared in the same package as nor is a subclass of \""
            + iClassDeclaringMember
            + "\"."
        );
    }

    /**
     * Determines whether the given {@link IClass} is accessible in the given context, according to JLS7 6.6.1.2 and
     * 6.6.1.4.
     */
    private boolean
    isAccessible(IClass type, Scope contextScope) throws CompileException {
        return null == this.internalCheckAccessible(type, contextScope);
    }

    /**
     * Checks whether the given {@link IClass} is accessible in the given context, according to JLS7 6.6.1.2 and
     * 6.6.1.4. Issues a {@link #com.hazelcast.com.ileError(String)} if not.
     */
    private void
    checkAccessible(IClass type, Scope contextScope, Location location) throws CompileException {
        String message = this.internalCheckAccessible(type, contextScope);
        if (message != null) this.com.hazelcast.com.ileError(message, location);
    }

    /**
     * @return An error message, or {@code null}
     */
    @Nullable private String
    internalCheckAccessible(IClass type, Scope contextScope) throws CompileException {

        // Determine the type declaring the type.
        IClass iClassDeclaringType = type.getDeclaringIClass();

        // Check accessibility of package member type.
        if (iClassDeclaringType == null) {
            if (type.getAccess() == Access.PUBLIC) {
                return null;
            } else
            if (type.getAccess() == Access.DEFAULT) {

                // Determine the type declaring the context block statement.
                IClass iClassDeclaringContextBlockStatement;
                for (Scope s = contextScope;; s = s.getEnclosingScope()) {
                    if (s instanceof TypeDeclaration) {
                        iClassDeclaringContextBlockStatement = this.resolve((TypeDeclaration) s);
                        break;
                    }
                    if (s instanceof EnclosingScopeOfTypeDeclaration) {
                        iClassDeclaringContextBlockStatement = this.resolve(
                            ((EnclosingScopeOfTypeDeclaration) s).typeDeclaration
                        );
                        break;
                    }
                }

                // Check whether the type is accessed from within the same package.
                String packageDeclaringType = Descriptor.getPackageName(type.getDescriptor());
                String contextPackage       = Descriptor.getPackageName(iClassDeclaringContextBlockStatement.getDescriptor()); // SUPPRESS CHECKSTYLE LineLength
                if (
                    packageDeclaringType == null
                    ? contextPackage != null
                    : !packageDeclaringType.equals(contextPackage)
                ) return "\"" + type + "\" is inaccessible from this package";
                return null;
            } else
            {
                throw new InternalCompilerException((
                    "\"" + type + "\" has unexpected access \"" + type.getAccess() + "\""
                ));
            }
        }

        // "type" is a member type at this point.
        return this.internalCheckAccessible(iClassDeclaringType, type.getAccess(), contextScope);
    }

    private Type
    toTypeOrCompileException(Atom a) throws CompileException {
        Type result = a.toType();
        if (result == null) {
            this.com.hazelcast.com.ileError("Expression \"" + a.toString() + "\" is not a type", a.getLocation());
            return new SimpleType(a.getLocation(), this.iClassLoader.TYPE_java_lang_Object);
        }
        return result;
    }

    private Rvalue
    toRvalueOrCompileException(final Atom a) throws CompileException {
        Rvalue result = a.toRvalue();
        if (result == null) {
            this.com.hazelcast.com.ileError("Expression \"" + a.toString() + "\" is not an rvalue", a.getLocation());
            return new StringLiteral(a.getLocation(), "\"X\"");
        }
        return result;
    }

    private Lvalue
    toLvalueOrCompileException(final Atom a) throws CompileException {
        Lvalue result = a.toLvalue();
        if (result == null) {
            this.com.hazelcast.com.ileError("Expression \"" + a.toString() + "\" is not an lvalue", a.getLocation());
            return new Lvalue(a.getLocation()) {

                @Override @Nullable public <R, EX extends Throwable> R
                accept(Visitor.LvalueVisitor<R, EX> visitor) { return null; }

                @Override public String
                toString() { return a.toString(); }
            };
        }
        return result;
    }

    /**
     * Copies the values of the synthetic parameters of this constructor ("this$..." and "val$...") to the synthetic
     * fields of the object ("this$..." and "val$...").
     */
    void
    assignSyntheticParametersToSyntheticFields(ConstructorDeclarator cd) throws CompileException {
        for (IClass.IField sf : cd.getDeclaringClass().syntheticFields.values()) {
            LocalVariable syntheticParameter = (LocalVariable) cd.syntheticParameters.get(sf.getName());
            if (syntheticParameter == null) {
                throw new InternalCompilerException(
                    "SNO: Synthetic parameter for synthetic field \""
                    + sf.getName()
                    + "\" not found"
                );
            }
            ExpressionStatement es = new ExpressionStatement(new Assignment(
                cd.getLocation(),                    // location
                new FieldAccess(                     // lhs
                    cd.getLocation(),                    // location
                    new ThisReference(cd.getLocation()), // lhs
                    sf                                   // field
                ),
                "=",                                 // operator
                new LocalVariableAccess(             // rhs
                    cd.getLocation(),                    // location
                    syntheticParameter                   // localVariable
                )
            ));
            es.setEnclosingScope(cd);
            this.com.hazelcast.com.ile(es);
        }
    }

    /**
     * Compiles the instance variable initializers and the instance initializers in their lexical order.
     */
    void
    initializeInstanceVariablesAndInvokeInstanceInitializers(ConstructorDeclarator cd) throws CompileException {

        // Compilation of block statements can create synthetic variables, so we must not use an iterator.
        List<BlockStatement> vdai = cd.getDeclaringClass().variableDeclaratorsAndInitializers;
        for (int i = 0; i < vdai.size(); i++) {
            BlockStatement bs = (BlockStatement) vdai.get(i);

            if (!((TypeBodyDeclaration) bs).isStatic()) {
                if (!this.com.hazelcast.com.ile(bs)) {
                    this.com.hazelcast.com.ileError(
                        "Instance variable declarator or instance initializer does not com.hazelcast.com.lete normally",
                        bs.getLocation()
                    );
                }
            }
        }
    }

    /**
     * Statements that jump out of blocks ({@code return}, {@code break}, {@code continue}) must call this method to
     * make sure that the {@code finally} clauses of all {@code try ... catch} and {@code synchronized} statements are
     * executed.
     */
    private void
    leaveStatements(Scope from, Scope to, @Nullable IClass optionalStackValueType) {
        for (Scope s = from; s != to; s = s.getEnclosingScope()) {
            if (s instanceof BlockStatement) {
                this.leave((BlockStatement) s, optionalStackValueType);
            }
        }
    }

    /**
     * The LHS operand of type <var>lhsType</var> is expected on the stack.
     * <p>
     *   The following operators are supported: {@code | ^ & * / % + - << >> >>>}
     * </p>
     */
    private IClass
    com.hazelcast.com.ileArithmeticBinaryOperation(
        Locatable locatable,
        IClass    lhsType,
        String    operator,
        Rvalue    rhs
    ) throws CompileException {
        return this.com.hazelcast.com.ileArithmeticOperation(
            locatable,
            lhsType,
            Arrays.asList(rhs).iterator(),
            operator
        );
    }

    /**
     * Executes an arithmetic operation on a sequence of <var>operands</var>. If <var>type</var> is non-{@code null},
     * then the first operand with that type is already on the stack.
     * <p>
     *   The following operators are supported: {@code | ^ & * / % + - << >> >>>}
     * </p>
     */
    private IClass
    com.hazelcast.com.ileArithmeticOperation(
        final Locatable  locatable,
        @Nullable IClass firstOperandType,
        Iterator<Rvalue> operands,
        String           operator
    ) throws CompileException {

        // A very special case.
        if (operator == "+" && firstOperandType == this.iClassLoader.TYPE_java_lang_String) { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength
            assert firstOperandType != null;
            return this.com.hazelcast.com.ileStringConcatenation(locatable, firstOperandType, (Rvalue) operands.next(), operands);
        }

        IClass type = firstOperandType == null ? this.com.hazelcast.com.ileGetValue((Rvalue) operands.next()) : firstOperandType;

        // Operator which is allowed for BYTE, SHORT, INT, LONG and BOOLEAN operands?
        if (operator == "|" || operator == "^" || operator == "&") { // SUPPRESS CHECKSTYLE StringLiteralEquality:5
            final int iopcode = (
                operator == "&" ? Opcode.IAND :
                operator == "|" ? Opcode.IOR  :
                operator == "^" ? Opcode.IXOR :
                Integer.MAX_VALUE
            );

            while (operands.hasNext()) {
                Rvalue operand = (Rvalue) operands.next();

                CodeContext.Inserter convertLhsInserter = this.getCodeContext().newInserter();
                IClass               rhsType            = this.com.hazelcast.com.ileGetValue(operand);

                if (type.isPrimitiveNumeric() && rhsType.isPrimitiveNumeric()) {
                    IClass promotedType = this.binaryNumericPromotion(locatable, type, convertLhsInserter, rhsType);
                    if (promotedType == IClass.INT) {
                        this.writeOpcode(locatable, iopcode);
                    } else
                    if (promotedType == IClass.LONG) {
                        this.writeOpcode(locatable, iopcode + 1);
                    } else
                    {
                        this.com.hazelcast.com.ileError((
                            "Operator \""
                            + operator
                            + "\" not defined on types \""
                            + type
                            + "\" and \""
                            + rhsType
                            + "\""
                        ), locatable.getLocation());
                    }
                    type = promotedType;
                } else
                if (
                    (type == IClass.BOOLEAN || this.getUnboxedType(type) == IClass.BOOLEAN)
                    && (rhsType == IClass.BOOLEAN || this.getUnboxedType(rhsType) == IClass.BOOLEAN)
                ) {
                    IClassLoader icl = this.iClassLoader;
                    if (type == icl.TYPE_java_lang_Boolean) {
                        this.getCodeContext().pushInserter(convertLhsInserter);
                        try {
                            this.unboxingConversion(locatable, icl.TYPE_java_lang_Boolean, IClass.BOOLEAN);
                        } finally {
                            this.getCodeContext().popInserter();
                        }
                    }
                    if (rhsType == icl.TYPE_java_lang_Boolean) {
                        this.unboxingConversion(locatable, icl.TYPE_java_lang_Boolean, IClass.BOOLEAN);
                    }
                    this.writeOpcode(locatable, iopcode);
                    type = IClass.BOOLEAN;
                } else
                {
                    this.com.hazelcast.com.ileError((
                        "Operator \""
                        + operator
                        + "\" not defined on types \""
                        + type
                        + "\" and \""
                        + rhsType
                        + "\""
                    ), locatable.getLocation());
                    type = IClass.INT;
                }
            }
            return type;
        }

        // Operator which is allowed for INT, LONG, FLOAT, DOUBLE and (for operator '+') STRING operands?
        if (operator == "*" || operator == "/" || operator == "%" || operator == "+" || operator == "-") { // SUPPRESS CHECKSTYLE StringLiteralEquality|LineLength:6
            final int iopcode = (
                operator == "*"   ? Opcode.IMUL :
                operator == "/"   ? Opcode.IDIV :
                operator == "%"   ? Opcode.IREM :
                operator == "+"   ? Opcode.IADD :
                operator == "-"   ? Opcode.ISUB :
                Integer.MAX_VALUE
            );

            while (operands.hasNext()) {
                Rvalue operand = (Rvalue) operands.next();

                // String concatenation?
                if (operator == "+" && ( // SUPPRESS CHECKSTYLE StringLiteralEquality
                    type == this.iClassLoader.TYPE_java_lang_String
                    || this.getType(operand) == this.iClassLoader.TYPE_java_lang_String
                )) return this.com.hazelcast.com.ileStringConcatenation(locatable, type, operand, operands);

                CodeContext.Inserter convertLhsInserter = this.getCodeContext().newInserter();
                IClass               rhsType            = this.com.hazelcast.com.ileGetValue(operand);

                type = this.binaryNumericPromotion(locatable, type, convertLhsInserter, rhsType);

                int opcode;
                if (type == IClass.INT) {
                    opcode = iopcode;
                } else
                if (type == IClass.LONG) {
                    opcode = iopcode + 1;
                } else
                if (type == IClass.FLOAT) {
                    opcode = iopcode + 2;
                } else
                if (type == IClass.DOUBLE) {
                    opcode = iopcode + 3;
                } else
                {
                    this.com.hazelcast.com.ileError("Unexpected promoted type \"" + type + "\"", locatable.getLocation());
                    opcode = iopcode;
                }
                this.writeOpcode(locatable, opcode);
            }

            return type;
        }

        // Operator which is allowed for BYTE, SHORT, INT and LONG lhs operand and BYTE, SHORT, INT or LONG rhs operand?
        if (operator == "<<" || operator == ">>" || operator == ">>>") { // SUPPRESS CHECKSTYLE StringLiteralEquality:4
            final int iopcode = (
                operator == "<<"  ? Opcode.ISHL  :
                operator == ">>"  ? Opcode.ISHR  :
                operator == ">>>" ? Opcode.IUSHR :
                Integer.MAX_VALUE
            );

            while (operands.hasNext()) {
                type = this.unaryNumericPromotion(locatable, type);

                int opcode;
                if (type == IClass.INT) {
                    opcode = iopcode;
                } else
                if (type == IClass.LONG) {
                    opcode = iopcode + 1;
                } else
                {
                    this.com.hazelcast.com.ileError(
                        "Shift operation not allowed on operand type \"" + type + "\"",
                        locatable.getLocation()
                    );
                    opcode = iopcode;
                }

                IClass rhsType         = this.com.hazelcast.com.ileGetValue((Rvalue) operands.next());
                IClass promotedRhsType = this.unaryNumericPromotion(locatable, rhsType);
                if (promotedRhsType == IClass.INT) {
                    ;
                } else
                if (promotedRhsType == IClass.LONG) {
                    this.writeOpcode(locatable, Opcode.L2I);
                } else
                {
                    this.com.hazelcast.com.ileError(
                        "Shift distance of type \"" + rhsType + "\" is not allowed",
                        locatable.getLocation()
                    );
                }

                this.writeOpcode(locatable, opcode);
            }

            return type;
        }

        throw new InternalCompilerException("Unexpected operator \"" + operator + "\"");
    }

    /**
     * @param type          The type of the first operand, which is already on the stack
     * @param secondOperand The second operand
     * @param operands      All following operands
     */
    private IClass
    com.hazelcast.com.ileStringConcatenation(
        final Locatable  locatable,
        IClass           type,
        final Rvalue     secondOperand,
        Iterator<Rvalue> operands
    ) throws CompileException {

        // Convert the first operand (which is already on the operand stack) to "String".
        this.stringConversion(locatable, type);

        // Compute list of operands and merge consecutive constant operands.
        List<Rvalue> tmp = new ArrayList<Rvalue>();
        for (Rvalue nextOperand = secondOperand; nextOperand != null;) {
            Object cv = this.getConstantValue(nextOperand);
            if (cv == UnitCompiler.NOT_CONSTANT) {

                // Non-constant operand.
                tmp.add(nextOperand);

                nextOperand = operands.hasNext() ? (Rvalue) operands.next() : null;
            } else
            {
                // Constant operand. Check to see whether the next operand is also constant.
                if (operands.hasNext()) {
                    nextOperand = (Rvalue) operands.next();
                    Object cv2 = this.getConstantValue(nextOperand);
                    if (cv2 != UnitCompiler.NOT_CONSTANT) {
                        StringBuilder sb = new StringBuilder(String.valueOf(cv)).append(cv2);
                        for (;;) {
                            if (!operands.hasNext()) {
                                nextOperand = null;
                                break;
                            }
                            nextOperand = (Rvalue) operands.next();
                            Object cv3 = this.getConstantValue(nextOperand);
                            if (cv3 == UnitCompiler.NOT_CONSTANT) break;
                            sb.append(cv3);
                        }
                        cv = sb.toString();
                    }
                } else
                {
                    nextOperand = null;
                }

                // Break long string constants up into UTF8-able chunks.
                for (final String s : UnitCompiler.makeUtf8Able(String.valueOf(cv))) {
                    tmp.add(new SimpleConstant(locatable.getLocation(), s));
                }
            }
        }

        // At this point "tmp" contains an optimized sequence of Strings (representing constant portions) and Rvalues
        // (non-constant portions).

        if (tmp.size() <= UnitCompiler.STRING_CONCAT_LIMIT - 1) {

            // String concatenation through "a.concat(b).concat(c)".
            for (Rvalue operand : tmp) {

                // "s.concat(String.valueOf(operand))"
                UnitCompiler.this.stringConversion(operand, UnitCompiler.this.com.hazelcast.com.ileGetValue(operand));
                this.invoke(locatable, this.iClassLoader.METH_java_lang_String__concat__java_lang_String);
            }
            return this.iClassLoader.TYPE_java_lang_String;
        }

        // String concatenation through "new StringBuilder(a).append(b).append(c).append(d).toString()".
        // "new StringBuilder(String a)":
        this.writeOpcode(locatable, Opcode.NEW);
        this.writeConstantClassInfo(Descriptor.JAVA_LANG_STRINGBUILDER);
        this.writeOpcode(locatable, Opcode.DUP_X1);
        this.writeOpcode(locatable, Opcode.SWAP);
        this.invoke(locatable, this.iClassLoader.CTOR_java_lang_StringBuilder__java_lang_String);

        for (Iterator<Rvalue> it = tmp.iterator(); it.hasNext();) {
            Rvalue operand = (Rvalue) it.next();

            // "sb.append(operand)"
            IClass t = UnitCompiler.this.com.hazelcast.com.ileGetValue(operand);
            this.invoke(locatable, (
                t == IClass.BYTE    ? this.iClassLoader.METH_java_lang_StringBuilder__append__int     :
                t == IClass.SHORT   ? this.iClassLoader.METH_java_lang_StringBuilder__append__int     :
                t == IClass.INT     ? this.iClassLoader.METH_java_lang_StringBuilder__append__int     :
                t == IClass.LONG    ? this.iClassLoader.METH_java_lang_StringBuilder__append__long    :
                t == IClass.FLOAT   ? this.iClassLoader.METH_java_lang_StringBuilder__append__float   :
                t == IClass.DOUBLE  ? this.iClassLoader.METH_java_lang_StringBuilder__append__double  :
                t == IClass.CHAR    ? this.iClassLoader.METH_java_lang_StringBuilder__append__char    :
                t == IClass.BOOLEAN ? this.iClassLoader.METH_java_lang_StringBuilder__append__boolean :
                this.iClassLoader.METH_java_lang_StringBuilder__append__java_lang_Object
            ));
        }

        // "StringBuilder.toString()":
        this.invoke(locatable, this.iClassLoader.METH_java_lang_StringBuilder__toString);

        return this.iClassLoader.TYPE_java_lang_String;
    }

    /**
     * Helper interface for string conversion.
     */
    interface Compilable { void com.hazelcast.com.ile() throws CompileException; }

    /**
     * Converts object of type "sourceType" to type "String" (JLS7 15.18.1.1).
     */
    private void
    stringConversion(Locatable locatable, IClass sourceType) throws CompileException {
        this.invoke(locatable, (
            sourceType == IClass.BYTE    ? this.iClassLoader.METH_java_lang_String__valueOf__int :
            sourceType == IClass.SHORT   ? this.iClassLoader.METH_java_lang_String__valueOf__int :
            sourceType == IClass.INT     ? this.iClassLoader.METH_java_lang_String__valueOf__int :
            sourceType == IClass.LONG    ? this.iClassLoader.METH_java_lang_String__valueOf__long :
            sourceType == IClass.FLOAT   ? this.iClassLoader.METH_java_lang_String__valueOf__float :
            sourceType == IClass.DOUBLE  ? this.iClassLoader.METH_java_lang_String__valueOf__double :
            sourceType == IClass.CHAR    ? this.iClassLoader.METH_java_lang_String__valueOf__char :
            sourceType == IClass.BOOLEAN ? this.iClassLoader.METH_java_lang_String__valueOf__boolean :
            this.iClassLoader.METH_java_lang_String__valueOf__java_lang_Object
        ));
    }

    /**
     * Expects the object to initialize on the stack.
     * <p>
     *   Notice: This method is used both for explicit constructor invocation (first statement of a constructor body)
     *   and implicit constructor invocation (right after NEW).
     * </p>
     *
     * @param optionalEnclosingInstance Used if the target class is an inner class
     */
    private void
    invokeConstructor(
        Locatable        locatable,
        Scope            scope,
        @Nullable Rvalue optionalEnclosingInstance,
        IClass           targetClass,
        Rvalue[]         arguments
    ) throws CompileException {
        // Find constructors.
        IClass.IConstructor[] iConstructors = targetClass.getDeclaredIConstructors();
        if (iConstructors.length == 0) {
            throw new InternalCompilerException(
                "SNO: Target class \"" + targetClass.getDescriptor() + "\" has no constructors"
            );
        }

        IClass.IConstructor iConstructor = (IClass.IConstructor) this.findMostSpecificIInvocable(
            locatable,     // l
            iConstructors, // iInvocables
            arguments,     // arguments
            scope          // contextScope
        );

        // Check exceptions that the constructor may throw.
        IClass[] thrownExceptions = iConstructor.getThrownExceptions();
        for (IClass te : thrownExceptions) {
            this.checkThrownException(locatable, te, scope);
        }

        // Enum constant: Pass constant name and ordinal as synthetic parameters.
        ENUM_CONSTANT:
        if (
            scope instanceof FieldDeclaration
            && scope.getEnclosingScope() instanceof EnumDeclaration
        ) {

            FieldDeclaration fd = (FieldDeclaration) scope;
            EnumDeclaration  ed = (EnumDeclaration) fd.getEnclosingScope();

            if (fd.variableDeclarators.length != 1) break ENUM_CONSTANT;

            String fieldName = fd.variableDeclarators[0].name;

            int ordinal = 0;
            for (EnumConstant ec : ed.getConstants()) {
                if (fieldName.equals(ec.name)) {

                    // Now we know that this field IS an enum constant.
                    this.pushConstant(locatable, fieldName);
                    this.pushConstant(locatable, ordinal);
                    break ENUM_CONSTANT;
                }
                ordinal++;
            }
        }

        // Pass enclosing instance as a synthetic parameter.
        if (optionalEnclosingInstance != null) {
            IClass outerIClass = targetClass.getOuterIClass();
            if (outerIClass != null) {
                IClass eiic = this.com.hazelcast.com.ileGetValue(optionalEnclosingInstance);
                if (!outerIClass.isAssignableFrom(eiic)) {
                    this.com.hazelcast.com.ileError(
                        "Type of enclosing instance (\"" + eiic + "\") is not assignable to \"" + outerIClass + "\"",
                        locatable.getLocation()
                    );
                }
            }
        }

        // Pass local variables to constructor as synthetic parameters.
        {
            IClass.IField[] syntheticFields = targetClass.getSyntheticIFields();

            // Determine enclosing function declarator and type declaration.
            TypeBodyDeclaration scopeTbd;
            TypeDeclaration     scopeTypeDeclaration;
            {
                Scope s = scope;
                for (; !(s instanceof TypeBodyDeclaration); s = s.getEnclosingScope());
                scopeTbd             = (TypeBodyDeclaration) s;
                scopeTypeDeclaration = scopeTbd.getDeclaringType();
            }

            if (!(scopeTypeDeclaration instanceof AbstractClassDeclaration)) {
                if (syntheticFields.length > 0) {
                    throw new InternalCompilerException("SNO: Target class has synthetic fields");
                }
                return;
            }

            AbstractClassDeclaration scopeClassDeclaration = (AbstractClassDeclaration) scopeTypeDeclaration;
            for (IClass.IField sf : syntheticFields) {
                if (!sf.getName().startsWith("val$")) continue;
                IClass.IField eisf = (IClass.IField) scopeClassDeclaration.syntheticFields.get(sf.getName());
                if (eisf != null) {
                    if (scopeTbd instanceof MethodDeclarator) {
                        this.load(locatable, this.resolve(scopeClassDeclaration), 0);
                        this.getfield(locatable, eisf);
                    } else
                    if (scopeTbd instanceof ConstructorDeclarator) {
                        ConstructorDeclarator constructorDeclarator = (ConstructorDeclarator) scopeTbd;
                        LocalVariable         syntheticParameter    = (
                            (LocalVariable) constructorDeclarator.syntheticParameters.get(sf.getName())
                        );
                        if (syntheticParameter == null) {
                            this.com.hazelcast.com.ileError((
                                "Compiler limitation: Constructor cannot access local variable \""
                                + sf.getName().substring(4)
                                + "\" declared in an enclosing block because none of the methods accesses it. "
                                + "As a workaround, declare a dummy method that accesses the local variable."
                            ), locatable.getLocation());
                            this.writeOpcode(locatable, Opcode.ACONST_NULL);
                        } else {
                            this.load(locatable, syntheticParameter);
                        }
                    } else
                    if (scopeTbd instanceof FieldDeclaration) {
                        this.com.hazelcast.com.ileError((
                            "Compiler limitation: Field initializers cannot access local variable \""
                            + sf.getName().substring(4)
                            + "\" declared in an enclosing block because none of the methods accesses it. "
                            + "As a workaround, declare a dummy method that accesses the local variable."
                        ), locatable.getLocation());
                        this.writeOpcode(scopeTbd, Opcode.ACONST_NULL);
                    } else
                    {
                        throw new AssertionError(scopeTbd);
                    }
                } else {
                    String        localVariableName = sf.getName().substring(4);
                    LocalVariable lv;
                    DETERMINE_LV: {
                        Scope s;

                        // Does one of the enclosing blocks declare a local variable with that name?
                        for (s = scope; s instanceof BlockStatement; s = s.getEnclosingScope()) {
                            BlockStatement       bs = (BlockStatement) s;
                            Scope                es = bs.getEnclosingScope();

                            List<? extends BlockStatement> statements;
                            if (es instanceof Block) {
                                statements = ((Block) es).statements;
                            } else
                            if (es instanceof FunctionDeclarator) {
                                statements = ((FunctionDeclarator) es).optionalStatements;
                            } else
                            if (es instanceof ForEachStatement) {
                                FunctionDeclarator.FormalParameter fp = ((ForEachStatement) es).currentElement;
                                if (fp.name.equals(localVariableName)) {
                                    lv = this.getLocalVariable(fp);
                                    break DETERMINE_LV;
                                }
                                continue;
                            } else
                            {
                                continue;
                            }

                            if (statements != null) {
                                for (BlockStatement bs2 : statements) {
                                    if (bs2 == bs) break;
                                    if (bs2 instanceof LocalVariableDeclarationStatement) {
                                        LocalVariableDeclarationStatement lvds = (
                                            (LocalVariableDeclarationStatement) bs2
                                        );
                                        for (VariableDeclarator vd : lvds.variableDeclarators) {
                                            if (vd.name.equals(localVariableName)) {
                                                lv = this.getLocalVariable(lvds, vd);
                                                break DETERMINE_LV;
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        // Does the declaring function declare a parameter with that name?
                        while (!(s instanceof FunctionDeclarator)) s = s.getEnclosingScope();
                        FunctionDeclarator fd = (FunctionDeclarator) s;
                        for (FormalParameter fp : fd.formalParameters.parameters) {
                            if (fp.name.equals(localVariableName)) {
                                lv = this.getLocalVariable(fp);
                                break DETERMINE_LV;
                            }
                        }
                        throw new InternalCompilerException(
                            "SNO: Synthetic field \""
                            + sf.getName()
                            + "\" neither maps a synthetic field of an enclosing instance nor a local variable"
                        );
                    }
                    this.load(locatable, lv);
                }
            }
        }

        // Evaluate constructor arguments.
        Rvalue[] adjustedArgs   = null;
        IClass[] parameterTypes = iConstructor.getParameterTypes();
        int      actualSize     = arguments.length;
        if (iConstructor.isVarargs() && iConstructor.argsNeedAdjust()) {
            adjustedArgs = new Rvalue[parameterTypes.length];
            Rvalue[] lastArgs = new Rvalue[actualSize - parameterTypes.length + 1];
            for (int i = 0, j = parameterTypes.length - 1; i < lastArgs.length; ++i, ++j) {
                lastArgs[i] = arguments[j];
            }

            for (int i = parameterTypes.length - 2; i >= 0; --i) {
                adjustedArgs[i] = arguments[i];
            }
            Location loc = (lastArgs.length == 0 ? locatable : lastArgs[lastArgs.length - 1]).getLocation();
            adjustedArgs[adjustedArgs.length - 1] = new NewInitializedArray(
                loc,                                       // location
                parameterTypes[parameterTypes.length - 1], // arrayIClass
                new ArrayInitializer(loc, lastArgs)        // arrayInitializer
            );
            arguments = adjustedArgs;
        }

        for (int i = 0; i < arguments.length; ++i) {
            this.assignmentConversion(
                locatable,                          // locatable
                this.com.hazelcast.com.ileGetValue(arguments[i]), // sourceType
                parameterTypes[i],                  // targetType
                this.getConstantValue(arguments[i]) // optionalConstantValue
            );
        }

        // Invoke!
        // Notice that the method descriptor is "iConstructor.getDescriptor()", prepended with the synthetic parameters.
        this.invoke(locatable, iConstructor);
    }

    /**
     * @return The {@link IField}s that are declared by the <var>fieldDeclaration</var>
     */
    private IClass.IField[]
    com.hazelcast.com.ileFields(final FieldDeclaration fieldDeclaration) {

        IClass.IField[] result = new IClass.IField[fieldDeclaration.variableDeclarators.length];

        for (int i = 0; i < result.length; ++i) {
            final VariableDeclarator vd = fieldDeclaration.variableDeclarators[i];

            result[i] = this.com.hazelcast.com.ileField(
                fieldDeclaration.getDeclaringType(),
                fieldDeclaration.modifiers,
                fieldDeclaration.type,
                vd
            );
        }

        return result;
    }

    /**
     * Compiles one variable declarator into an {@link IField}.
     * <p>
     *   Example: "b" in in the variable declaration
     * </p>
     * <pre>
     *     class Foo {
     *         &#64;Deprecated private int[] a, b[], c;
     *     }
     * </pre>
     *
     * @param declaringType      "{@code class Foo}"
     * @param modifiers          "{@code @Deprecated private}"
     * @param type               "{@code int[]}"
     * @param variableDeclarator "{@code b[]}"
     */
    private IField
    com.hazelcast.com.ileField(
        TypeDeclaration          declaringType,
        final Modifiers          modifiers,
        final Type               type,
        final VariableDeclarator variableDeclarator
    ) {

        return this.resolve(declaringType).new IField() {

            @Nullable private IAnnotation[] ias;

            // Implement IMember.
            @Override public Access
            getAccess() {
                switch (modifiers.accessFlags & Mod.PPP) {
                case Mod.PRIVATE:
                    return Access.PRIVATE;
                case Mod.PROTECTED:
                    return Access.PROTECTED;
                case Mod.PACKAGE:
                    return Access.DEFAULT;
                case Mod.PUBLIC:
                    return Access.PUBLIC;
                default:
                    throw new InternalCompilerException("Invalid access");
                }
            }

            @Override public IAnnotation[]
            getAnnotations() {

                if (this.ias != null) return this.ias;

                return (this.ias = UnitCompiler.this.toIAnnotations(modifiers.annotations));
            }

            // Implement "IField".

            @Override public boolean
            isStatic() { return Mod.isStatic(modifiers.accessFlags); }

            @Override public IClass
            getType() throws CompileException {
                return UnitCompiler.this.getType(type).getArrayIClass(
                    variableDeclarator.brackets,
                    UnitCompiler.this.iClassLoader.TYPE_java_lang_Object
                );
            }

            @Override public String
            getName() { return variableDeclarator.name; }

            @Override @Nullable public Object
            getConstantValue() throws CompileException {
                ArrayInitializerOrRvalue oi = variableDeclarator.optionalInitializer;
                if (
                    Mod.isFinal(modifiers.accessFlags)
                    && oi instanceof Rvalue
                ) {
                    Object constantInitializerValue = UnitCompiler.this.getConstantValue(
                        (Rvalue) oi
                    );
                    if (constantInitializerValue != UnitCompiler.NOT_CONSTANT) {
                        return UnitCompiler.this.assignmentConversion(
                            oi,                       // locatable
                            constantInitializerValue, // value
                            this.getType()            // targetType
                        );
                    }
                }
                return UnitCompiler.NOT_CONSTANT;
            }
        };
    }

    /**
     * Determines the non-constant-final initializer of the given {@link VariableDeclarator}.
     *
     * @return {@code null} if the variable is declared without an initializer or if the initializer is
     *         constant-final
     */
    @Nullable ArrayInitializerOrRvalue
    getNonConstantFinalInitializer(FieldDeclaration fd, VariableDeclarator vd) throws CompileException {

        // Check if optional initializer exists.
        if (vd.optionalInitializer == null) return null;

        // Check if initializer is constant-final.
        if (
            Mod.isStatic(fd.modifiers.accessFlags)
            && Mod.isFinal(fd.modifiers.accessFlags)
            && vd.optionalInitializer instanceof Rvalue
            && this.getConstantValue((Rvalue) vd.optionalInitializer) != UnitCompiler.NOT_CONSTANT
        ) return null;

        return vd.optionalInitializer;
    }

    private Atom
    reclassify(AmbiguousName an) throws CompileException {

        if (an.reclassified != null) return an.reclassified;

        return (an.reclassified = this.reclassifyName(an.getLocation(), an.getEnclosingScope(), an.identifiers, an.n));
    }

    private IAnnotation[]
    toIAnnotations(Annotation[] annotations) {

        IAnnotation[] result = new IAnnotation[annotations.length];
        for (int i = 0; i < result.length; i++) result[i] = this.toIAnnotation(annotations[i]);

        return result;
    }

    private IAnnotation
    toIAnnotation(Annotation annotation) {

        IAnnotation result = (IAnnotation) annotation.accept(
            new AnnotationVisitor<IAnnotation, RuntimeException>() {

                @Override public IAnnotation
                visitMarkerAnnotation(final MarkerAnnotation ma) {
                    return this.toIAnnotation(ma.type, new ElementValuePair[0]);
                }

                @Override public IAnnotation
                visitSingleElementAnnotation(SingleElementAnnotation sea) {
                    return this.toIAnnotation(
                        sea.type,
                        new ElementValuePair[] { new ElementValuePair("value", sea.elementValue) }
                    );
                }

                @Override public IAnnotation
                visitNormalAnnotation(NormalAnnotation na) {
                    return this.toIAnnotation(na.type, na.elementValuePairs);
                }

                private IAnnotation
                toIAnnotation(final Type type, ElementValuePair[] elementValuePairs) {

                    final Map<String, Object> m = new HashMap<String, Object>();
                    for (ElementValuePair evp : elementValuePairs) {
                        m.put(evp.identifier, this.toObject(evp.elementValue));
                    }

                    return new IAnnotation() {

                        @Override public Object
                        getElementValue(String name) { return m.get(name); }

                        @Override public IClass
                        getAnnotationType() throws CompileException {
                            return UnitCompiler.this.getType(type);
                        }
                    };
                }

                /**
                 * @return A wrapped primitive value, a {@link String}, an {@link IClass} (representing a class
                 *         literal), an {@link IClass.IField} (representing an enum constant), or an {@link Object}[]
                 *         array containing any of the previously described
                 */
                private Object
                toObject(Java.ElementValue ev) {

                    try {
                        Object result = ev.accept(new ElementValueVisitor<Object, CompileException>() {

                            @Override public Object
                            visitRvalue(Rvalue rv) throws CompileException {

                                if (rv instanceof AmbiguousName) {
                                    AmbiguousName an = (AmbiguousName) rv;
                                    rv = UnitCompiler.this.reclassify(an).toRvalueOrCompileException();
                                }

                                // Class literal?
                                if (rv instanceof ClassLiteral) {
                                    ClassLiteral cl = (ClassLiteral) rv;
                                    return UnitCompiler.this.getType(cl.type);
                                }

                                // Enum constant?
                                if (rv instanceof FieldAccess) {
                                    FieldAccess fa = (FieldAccess) rv;
                                    return fa.field;
                                }

                                Object result = UnitCompiler.this.getConstantValue(rv);
                                if (result == null) {
                                    UnitCompiler.this.com.hazelcast.com.ileError(
                                        "Null value not allowed as an element value",
                                        rv.getLocation()
                                    );
                                    return 1;
                                }
                                if (result == UnitCompiler.NOT_CONSTANT) {
                                    UnitCompiler.this.com.hazelcast.com.ileError(
                                        "Element value is not a constant expression",
                                        rv.getLocation()
                                    );
                                    return 1;
                                }
                                return result;
                            }

                            @Override public Object
                            visitAnnotation(Annotation a) { return UnitCompiler.this.toIAnnotation(a); }

                            @Override public Object
                            visitElementValueArrayInitializer(ElementValueArrayInitializer evai) {
                                Object[] result = new Object[evai.elementValues.length];
                                for (int i = 0; i < result.length; i++) {
                                    result[i] = toObject(evai.elementValues[i]);
                                }
                                return result;
                            }
                        });
                        assert result != null;
                        return result;
                    } catch (/*CompileException*/ Exception ce) {
                        throw new IllegalStateException(ce);
                    }
                }
            }
        );

        assert result != null;
        return result;
    }

    /**
     * Reclassifies the ambiguous name consisting of the first <var>n</var> of the <var>identifiers</var> (JLS7
     * 6.5.2.2).
     */
    private Atom
    reclassifyName(Location location, Scope scope, final String[] identifiers, int n) throws CompileException {

        if (n == 1) return this.reclassifyName(location, scope, identifiers[0]);

        // 6.5.2.2
        Atom   lhs = this.reclassifyName(location, scope, identifiers, n - 1);
        String rhs = identifiers[n - 1];

        // 6.5.2.2.1
        UnitCompiler.LOGGER.log(Level.FINE, "lhs={0}", lhs);
        if (lhs instanceof Package) {
            String className = ((Package) lhs).name + '.' + rhs;
            IClass result    = this.findTypeByName(location, className);
            if (result != null) return new SimpleType(location, result);

            return new Package(location, className);
        }

        // 6.5.2.2.3.2 EXPRESSION.length
        if ("length".equals(rhs) && this.getType(lhs).isArray()) {
            ArrayLength al = new ArrayLength(location, this.toRvalueOrCompileException(lhs));
            if (!(scope instanceof BlockStatement)) {
                this.com.hazelcast.com.ileError("\".length\" only allowed in expression context");
                return al;
            }
            al.setEnclosingScope(scope);
            return al;
        }

        IClass lhsType = this.getType(lhs);

        // Notice: Don't need to check for 6.5.2.2.2.1 TYPE.METHOD and 6.5.2.2.3.1 EXPRESSION.METHOD here because that
        // has been done before.

        {
            IClass.IField field = this.findIField(lhsType, rhs, location);
            if (field != null) {
                // 6.5.2.2.2.2 TYPE.FIELD
                // 6.5.2.2.3.2 EXPRESSION.FIELD
                FieldAccess fa = new FieldAccess(
                    location,
                    lhs,
                    field
                );
                fa.setEnclosingScope(scope);
                return fa;
            }
        }

        IClass[] classes = lhsType.getDeclaredIClasses();
        for (final IClass memberType : classes) {
            String name = Descriptor.toClassName(memberType.getDescriptor());
            name = name.substring(name.lastIndexOf('$') + 1);
            if (name.equals(rhs)) {

                // 6.5.2.2.2.3 TYPE.TYPE
                // 6.5.2.2.3.3 EXPRESSION.TYPE
                return new SimpleType(location, memberType);
            }
        }

        this.com.hazelcast.com.ileError(
            "\"" + rhs + "\" is neither a method, a field, nor a member class of \"" + lhsType + "\"",
            location
        );
        return new Atom(location) {

            @Override @Nullable public <R, EX extends Throwable> R
            accept(AtomVisitor<R, EX> visitor) { return null; }

            @Override public String
            toString() { return Java.join(identifiers, "."); }
        };
    }

    /**
     * Finds the named {@link IClass} in this com.hazelcast.com.ilation unit, or through the {@link #iClassLoader}.
     *
     * @param className         Fully qualified class name, e.g. "pkg1.pkg2.Outer$Inner"
     * @return                  {@code null} iff an {@link IClass} with that name could not be loaded
     * @throws CompileException An exception was raised while loading the {@link IClass}
     */
    @Nullable private IClass
    findTypeByName(Location location, String className) throws CompileException {

        // Is the type defined in the same com.hazelcast.com.ilation unit?
        IClass res = this.findClass(className);
        if (res != null) return res;

        try {
            return this.iClassLoader.loadIClass(Descriptor.fromClassName(className));
        } catch (ClassNotFoundException ex) {
            // SUPPRESS CHECKSTYLE AvoidHidingCause
            if (ex.getException() instanceof CompileException) throw (CompileException) ex.getException();
            throw new CompileException(className, location, ex);
        }
    }

    /**
     * JLS7 6.5.2.1
     */
    private Atom
    reclassifyName(Location location, Scope scope, final String identifier) throws CompileException {

        // Determine scope block statement, type body declaration, type and com.hazelcast.com.ilation unit.
        TypeBodyDeclaration     scopeTbd             = null;
        AbstractTypeDeclaration scopeTypeDeclaration = null;
        CompilationUnit         scopeCompilationUnit;
        {
            Scope s = scope;
            while (
                (s instanceof BlockStatement || s instanceof CatchClause)
                && !(s instanceof TypeBodyDeclaration)
            ) s = s.getEnclosingScope();
            if (s instanceof TypeBodyDeclaration) {
                scopeTbd = (TypeBodyDeclaration) s;
                s        = s.getEnclosingScope();
            }
            if (s instanceof TypeDeclaration) {
                scopeTypeDeclaration = (AbstractTypeDeclaration) s;
                s                    = s.getEnclosingScope();
            }
            while (!(s instanceof CompilationUnit)) s = s.getEnclosingScope();
            scopeCompilationUnit = (CompilationUnit) s;
        }

        // 6.5.2.1.BL1

        // 6.5.2.BL1.B1.B1.1 (JLS7: 6.5.2.BL1.B1.B1.1) / 6.5.6.1.1 Local variable.
        // 6.5.2.BL1.B1.B1.2 (JLS7: 6.5.2.BL1.B1.B1.2) / 6.5.6.1.1 Parameter.
        {
            Scope s = scope;
            if (s instanceof BlockStatement) {
                BlockStatement bs = (BlockStatement) s;
                LocalVariable  lv = bs.findLocalVariable(identifier);
                if (lv != null) {
                    LocalVariableAccess lva = new LocalVariableAccess(location, lv);
                    lva.setEnclosingScope(bs);
                    return lva;
                }
                s = s.getEnclosingScope();
            }
            while (s instanceof BlockStatement || s instanceof CatchClause) s = s.getEnclosingScope();
            if (s instanceof FunctionDeclarator) {
                s = s.getEnclosingScope();
            }
            if (s instanceof InnerClassDeclaration) {
                InnerClassDeclaration icd = (InnerClassDeclaration) s; // SUPPRESS CHECKSTYLE UsageDistance

                s = s.getEnclosingScope();
                if (s instanceof AnonymousClassDeclaration) {
                    s = s.getEnclosingScope();
                } else
                if (s instanceof FieldDeclaration) {
                    s = s.getEnclosingScope().getEnclosingScope();
                }
                while (s instanceof BlockStatement) {
                    LocalVariable lv = ((BlockStatement) s).findLocalVariable(identifier);
                    if (lv != null) {
                        if (!lv.finaL) {
                            this.com.hazelcast.com.ileError(
                                "Cannot access non-final local variable \""
                                + identifier
                                + "\" from inner class"
                            );
                        }
                        final IClass  lvType = lv.type;
                        IClass.IField iField = new SimpleIField(
                            this.resolve(icd),
                            "val$" + identifier,
                            lvType
                        );
                        icd.defineSyntheticField(iField);
                        FieldAccess fa = new FieldAccess(
                            location,                                   // location
                            new QualifiedThisReference(                 // lhs
                                location,                                   // location
                                new SimpleType(location, this.resolve(icd)) // qualification
                            ),
                            iField                                      // field
                        );
                        fa.setEnclosingScope(scope);
                        return fa;
                    }
                    s = s.getEnclosingScope();
                    while (s instanceof BlockStatement) s = s.getEnclosingScope();
                    if (!(s instanceof FunctionDeclarator)) break;
                    s = s.getEnclosingScope();
                    if (!(s instanceof InnerClassDeclaration)) break;
                    icd = (InnerClassDeclaration) s;
                    s   = s.getEnclosingScope();
                }
            }
        }

        // 6.5.2.BL1.B1.B1.3 (JLS7: 6.5.2.BL1.B1.B1.3) / 6.5.6.1.2.1 Field.
        BlockStatement enclosingBlockStatement = null;
        for (Scope s = scope; !(s instanceof CompilationUnit); s = s.getEnclosingScope()) {
            if (s instanceof BlockStatement && enclosingBlockStatement == null) {
                enclosingBlockStatement = (BlockStatement) s;
            }
            if (s instanceof TypeDeclaration) {
                final AbstractTypeDeclaration enclosingTypeDecl = (AbstractTypeDeclaration) s;
                final IClass                  etd               = this.resolve(enclosingTypeDecl);
                final IClass.IField           f                 = this.findIField(etd, identifier, location);
                if (f != null) {
                    if (f.isStatic()) {
                        this.warning("IASF", (
                            "Implicit access to static field \""
                            + identifier
                            + "\" of declaring class (better write \""
                            + f.getDeclaringIClass()
                            + '.'
                            + f.getName()
                            + "\")"
                        ), location);
                    } else
                    if (f.getDeclaringIClass() == etd) {
                        this.warning("IANSF", (
                            "Implicit access to non-static field \""
                            + identifier
                            + "\" of declaring class (better write \"this."
                            + f.getName()
                            + "\")"
                        ), location);
                    } else {
                        this.warning("IANSFEI", (
                            "Implicit access to non-static field \""
                            + identifier
                            + "\" of enclosing instance (better write \""
                            + f.getDeclaringIClass()
                            + ".this."
                            + f.getName()
                            + "\")"
                        ), location);
                    }

                    assert scopeTypeDeclaration != null;
                    SimpleType ct = new SimpleType(scopeTypeDeclaration.getLocation(), etd);

                    Atom lhs;
                    if (scopeTbd == null) {

                        // Field access in top-level type declaration context (member annotation).
                        lhs = ct;
                    } else
                    if (scopeTbd.isStatic()) {

                        // Field access in static method context.
                        lhs = ct;
                    } else
                    {

                        // Field access in non-static method context.
                        if (f.isStatic()) {

                            // Access to static field.
                            lhs = ct;
                        } else {

                            // Access to non-static field.
                            lhs = new QualifiedThisReference(location, ct);
                        }
                    }

                    Rvalue res = new FieldAccess(location, lhs, f);
                    res.setEnclosingScope(scope);

                    return res;
                }
            }
        }

        // JLS7 6.5.2.BL1.B2.1 Static field imported through single static import.
        {
            List<Object/*IField+IMethod+IClass*/> l = (List<Object>) this.singleStaticImports.get(identifier);
            if (l != null) {
                for (Object o : l) {
                    if (o instanceof IField) {
                        FieldAccess fieldAccess = new FieldAccess(
                            location,
                            new SimpleType(location, ((IField) o).getDeclaringIClass()),
                            (IField) o
                        );
                        fieldAccess.setEnclosingScope(scope);
                        return fieldAccess;
                    }
                }
            }
        }

        // JLS7 6.5.2.BL1.B2.2 Static field imported through static-import-on-demand.
        {
            IField importedField = null;
            for (IClass iClass : this.staticImportsOnDemand) {

                IField f = iClass.getDeclaredIField(identifier);
                if (f != null) {

                    // JLS7 7.5.4 Static-Import-on-Demand Declaration
                    if (!this.isAccessible(f, scope)) continue;

                    if (importedField != null) {
                        this.com.hazelcast.com.ileError(
                            "Ambiguous static field import: \""
                            + importedField.toString()
                            + "\" vs. \""
                            + f.toString()
                            + "\""
                        );
                    }
                    importedField = f;
                }
            }
            if (importedField != null) {
                if (!importedField.isStatic()) this.com.hazelcast.com.ileError("Cannot static-import non-static field");
                FieldAccess fieldAccess = new FieldAccess(
                    location,
                    new SimpleType(location, importedField.getDeclaringIClass()),
                    importedField
                );
                fieldAccess.setEnclosingScope(scope);
                return fieldAccess;
            }
        }

        // Hack: "java" MUST be a package, not a class.
        if ("java".equals(identifier)) return new Package(location, identifier);

        // JLS7: 6.5.2.BL1.B3.1 Unnamed package class
        // JLS7: 6.5.2.BL1.B3.2 Unnamed package interface
        // JLS7: 7.4.2
        {
            IClass unnamedPackageType = this.findTypeByName(location, identifier);
            if (unnamedPackageType != null) return new SimpleType(location, unnamedPackageType);
        }

        // 6.5.2.BL1.B1.B2.1 (JLS7: 6.5.2.BL1.B3.3) Local class.
        {
            LocalClassDeclaration lcd = UnitCompiler.findLocalClassDeclaration(scope, identifier);
            if (lcd != null) return new SimpleType(location, this.resolve(lcd));
        }

        // 6.5.2.BL1.B1.B2.2 (JLS7: 6.5.2.BL1.B3.4) Member type.
        if (scopeTypeDeclaration != null) {
            IClass memberType = this.findMemberType(
                this.resolve(scopeTypeDeclaration),
                identifier,
                location
            );
            if (memberType != null) return new SimpleType(location, memberType);
        }

        // 6.5.2.BL1.B1.B3.1 (JLS7: 6.5.2.BL1.B1.B4.1) Single type import.
        {
            IClass iClass = this.importSingleType(identifier, location);
            if (iClass != null) return new SimpleType(location, iClass);
        }

        // 6.5.2.BL1.B1.B3.2 (JLS7: 6.5.2.BL1.B1.B3.1) Package member class/interface declared in this com.hazelcast.com.ilation
        // unit.
        // Notice that JLS2 looks this up AFTER local class, member type, single type import, while JLS3 looks this up
        // BEFORE local class, member type, single type import.
        {
            PackageMemberTypeDeclaration pmtd = scopeCompilationUnit.getPackageMemberTypeDeclaration(identifier);
            if (pmtd != null) return new SimpleType(location, this.resolve(pmtd));
        }

        // 6.5.2.BL1.B1.B4 Class or interface declared in same package.
        // Notice: Why is this missing in JLS3?
        {
            PackageDeclaration opd = scopeCompilationUnit.optionalPackageDeclaration;

            String className = opd == null ? identifier : opd.packageName + '.' + identifier;

            IClass result = this.findTypeByName(location, className);
            if (result != null) return new SimpleType(location, result);
        }

        // 6.5.2.BL1.B1.B5 (JLS7: 6.5.2.BL1.B1.B4.2), 6.5.2.BL1.B1.B6 Type-import-on-demand.
        {
            IClass importedClass = this.importTypeOnDemand(identifier, location);
            if (importedClass != null) {
                return new SimpleType(location, importedClass);
            }
        }

        // JLS7 6.5.2.BL1.B1.B4.3 Type imported through single static import.
        {
            List<Object/*IField+IMethod+IClass*/> l = (List<Object>) this.singleStaticImports.get(identifier);
            if (l != null) {
                for (Object o : l) {
                    if (o instanceof IClass) return new SimpleType(location, (IClass) o);
                }
            }
        }

        // JLS7 6.5.2.BL1.B1.B4.4 Type imported through static-import-on-demand.
        {
            IClass importedType = null;
            for (IClass ic : this.staticImportsOnDemand) {
                IClass[] memberTypes = ic.getDeclaredIClasses();
                for (IClass memberType : memberTypes) {
                    if (!this.isAccessible(memberType, scope)) continue;
                    if (memberType.getDescriptor().endsWith('$' + identifier + ';')) {
                        if (importedType != null) {
                            this.com.hazelcast.com.ileError(
                                "Ambiguous static type import: \""
                                + importedType.toString()
                                + "\" vs. \""
                                + memberType.toString()
                                + "\""
                            );
                        }
                        importedType = memberType;
                    }
                }
            }
            if (importedType != null) return new SimpleType(location, importedType);
        }

        // 6.5.2.BL1.B1.B7 Package name
        return new Package(location, identifier);
    }

    private Rvalue
    determineValue(FieldAccessExpression fae) throws CompileException {
        if (fae.value != null) return fae.value;

        IClass lhsType = this.getType(fae.lhs);

        Rvalue value;
        if (fae.fieldName.equals("length") && lhsType.isArray()) {
            value = new ArrayLength(
                fae.getLocation(),
                this.toRvalueOrCompileException(fae.lhs)
            );
        } else {
            IClass.IField iField = this.findIField(lhsType, fae.fieldName, fae.getLocation());
            if (iField == null) {
                this.com.hazelcast.com.ileError(
                    "\"" + this.getType(fae.lhs).toString() + "\" has no field \"" + fae.fieldName + "\"",
                    fae.getLocation()
                );
                value = new Rvalue(fae.getLocation()) {

                    @Override @Nullable public <R, EX extends Throwable> R
                    accept(RvalueVisitor<R, EX> visitor) { return null; }

                    @Override public String
                    toString() { return "???"; }
                };
            } else {
                value = new FieldAccess(
                    fae.getLocation(),
                    fae.lhs,
                    iField
                );
            }
        }

        value.setEnclosingScope(fae.getEnclosingScope());

        return (fae.value = value);
    }

    /**
     * "super.fld", "Type.super.fld"
     */
    private Rvalue
    determineValue(SuperclassFieldAccessExpression scfae) throws CompileException {
        if (scfae.value != null) return scfae.value;

        Rvalue lhs;
        {
            ThisReference tr = new ThisReference(scfae.getLocation());
            tr.setEnclosingScope(scfae.getEnclosingScope());
            IClass type;
            if (scfae.optionalQualification != null) {
                type = this.getType(scfae.optionalQualification);
            } else
            {
                type = this.getType(tr);
            }

            IClass superclass = type.getSuperclass();
            if (superclass == null) {
                throw new CompileException("Cannot use \"super\" on \"" + type + "\"", scfae.getLocation());
            }

            lhs = new Cast(scfae.getLocation(), new SimpleType(scfae.getLocation(), superclass), tr);
        }

        Rvalue value;

        IClass.IField iField = this.findIField(this.getType(lhs), scfae.fieldName, scfae.getLocation());
        if (iField == null) {
            this.com.hazelcast.com.ileError("Class has no field \"" + scfae.fieldName + "\"", scfae.getLocation());
            value = new Rvalue(scfae.getLocation()) {

                @Override @Nullable public <R, EX extends Throwable> R
                accept(RvalueVisitor<R, EX> visitor) { return null; }

                @Override public String
                toString() { return "???"; }
            };
        } else {
            value = new FieldAccess(
                scfae.getLocation(),
                lhs,
                iField
            );
        }
        value.setEnclosingScope(scfae.getEnclosingScope());

        return (scfae.value = value);
    }

    /**
     * Finds methods of the <var>mi</var>{@code .}{@link MethodInvocation#optionalTarget
     * optionalTarget} named <var>mi</var>{@code .}{@link Invocation#methodName methodName},
     * examines the argument types and chooses the most specific method. Checks that only the
     * allowed exceptions are thrown.
     * <p>
     *   Notice that the returned {@link IClass.IMethod} may be declared in an enclosing type.
     * </p>
     *
     * @return The selected {@link IClass.IMethod}
     */
    public IClass.IMethod
    findIMethod(MethodInvocation mi) throws CompileException {
        IClass.IMethod iMethod;
        FIND_METHOD: {

            Atom ot = mi.optionalTarget;
            if (ot == null) {

                // Method invocation by simple method name... method must be declared by an enclosing type declaration.
                for (
                    Scope s = mi.getEnclosingScope();
                    !(s instanceof CompilationUnit);
                    s = s.getEnclosingScope()
                ) {
                    if (s instanceof TypeDeclaration) {
                        TypeDeclaration td = (TypeDeclaration) s;

                        // Find methods with specified name.
                        iMethod = this.findIMethod(
                            this.resolve(td),  // targetType
                            mi                 // invocation
                        );
                        if (iMethod != null) break FIND_METHOD;
                    }
                }
            } else
            {

                // Method invocation by "target": "expr.meth(arguments)" -- method must be declared by the target's
                // type.
                iMethod = this.findIMethod(
                    this.getType(ot), // targetType
                    mi                // invocable
                );
                if (iMethod != null) break FIND_METHOD;
            }

            // Static method declared through single static import?
            {
                List<Object/*IField+IMethod+IClass*/> l = (List<Object>) this.singleStaticImports.get(mi.methodName);
                if (l != null) {
                    iMethod = null;
                    for (Object o : l) {
                        if (o instanceof IMethod) {
                            IClass  declaringIClass = ((IMethod) o).getDeclaringIClass();
                            IMethod im              = this.findIMethod(
                                declaringIClass, // targetType
                                mi               // invocable
                            );
                            if (im != null) {
                                if (iMethod != null && iMethod != im) {
                                    this.com.hazelcast.com.ileError(
                                        "Ambiguous static method import: \""
                                        + iMethod.toString()
                                        + "\" vs. \""
                                        + im.toString()
                                        + "\""
                                    );
                                }
                                iMethod = im;
                            }
                        }
                    }
                    if (iMethod != null) break FIND_METHOD;
                }
            }

            // Static method declared through static-import-on-demand?
            iMethod = null;
            for (IClass iClass : this.staticImportsOnDemand) {
                IMethod im = this.findIMethod(
                    iClass, // targetType
                    mi      // invocation
                );
                if (im != null) {
                    if (iMethod != null) {
                        this.com.hazelcast.com.ileError(
                            "Ambiguous static method import: \""
                            + iMethod.toString()
                            + "\" vs. \""
                            + im.toString()
                            + "\""
                        );
                    }
                    iMethod = im;
                }
            }
            if (iMethod != null) break FIND_METHOD;

            this.com.hazelcast.com.ileError((
                "A method named \""
                + mi.methodName
                + "\" is not declared in any enclosing class nor any supertype, nor through a static import"
            ), mi.getLocation());
            return this.fakeIMethod(this.iClassLoader.TYPE_java_lang_Object, mi.methodName, mi.arguments);
        }

        assert iMethod != null; // Don't know why JAVAC thinks "iMethod" could be null here!?

        this.checkThrownExceptions(mi, iMethod);
        return iMethod;
    }

    /**
     * Finds a {@link IClass.IMethod} in the given <var>targetType</var>, its superclasses or superinterfaces which is
     * applicable with the given <var>invocation</var>. If more than one such method exists, chooses the most
     * specific one (JLS7 15.11.2).
     *
     * @return {@code null} if no appropriate method could be found
     */
    @Nullable private IClass.IMethod
    findIMethod(IClass targetType, Invocation invocation) throws CompileException {

        // Get all methods.
        List<IClass.IMethod> ms = new ArrayList<IClass.IMethod>();
        this.getIMethods(targetType, invocation.methodName, ms);

        // Interfaces inherit the methods declared in 'Object'.
        if (targetType.isInterface()) {
            IClass.IMethod[] oms = this.iClassLoader.TYPE_java_lang_Object.getDeclaredIMethods(invocation.methodName);
            for (IMethod om : oms) {
                if (!om.isStatic() && om.getAccess() == Access.PUBLIC) ms.add(om);
            }
        }

        if (ms.size() == 0) return null;

        // Determine arguments' types, choose the most specific method.
        return (IClass.IMethod) this.findMostSpecificIInvocable(
            invocation,                                                   // locatable
            (IClass.IMethod[]) ms.toArray(new IClass.IMethod[ms.size()]), // iInvocables
            invocation.arguments,                                         // arguments
            invocation.getEnclosingScope()                                // contextScope
        );
    }

    private IMethod
    fakeIMethod(IClass targetType, final String name, Rvalue[] arguments) throws CompileException {
        final IClass[] pts = new IClass[arguments.length];
        for (int i = 0; i < arguments.length; ++i) pts[i] = this.getType(arguments[i]);
        return targetType.new IMethod() {
            @Override public String        getName()              { return name; }
            @Override public IClass        getReturnType()        { return IClass.INT; }
            @Override public boolean       isStatic()             { return false; }
            @Override public boolean       isAbstract()           { return false; }
            @Override public boolean       isVarargs()            { return false; }
            @Override public IClass[]      getParameterTypes2()   { return pts; }
            @Override public IClass[]      getThrownExceptions2() { return new IClass[0]; }
            @Override public Access        getAccess()            { return Access.PUBLIC; }
            @Override public IAnnotation[] getAnnotations()       { return new IAnnotation[0]; }
        };
    }

    /**
     * Adds all methods with the given <var>methodName</var> that are declared by the <var>type</var>, its superclasses
     * and all their superinterfaces to the result list <var>v</var>.
     */
    public void
    getIMethods(IClass type, String methodName, List<IMethod> v) throws CompileException {

        // Check methods declared by this type.
        {
            IClass.IMethod[] ims = type.getDeclaredIMethods(methodName);
            for (IMethod im : ims) v.add(im);
        }

        // Check superclass.
        IClass superclass = type.getSuperclass();
        if (superclass != null) this.getIMethods(superclass, methodName, v);

        // Check superinterfaces.
        IClass[] interfaces = type.getInterfaces();
        for (IClass interfacE : interfaces) this.getIMethods(interfacE, methodName, v);
    }

    /**
     * @return The {@link IClass.IMethod} that implements the <var>superclassMethodInvocation</var>
     */
    public IClass.IMethod
    findIMethod(SuperclassMethodInvocation superclassMethodInvocation) throws CompileException {
        AbstractClassDeclaration declaringClass;
        for (Scope s = superclassMethodInvocation.getEnclosingScope();; s = s.getEnclosingScope()) {
            if (s instanceof FunctionDeclarator) {
                FunctionDeclarator fd = (FunctionDeclarator) s;
                if (Mod.isStatic(fd.modifiers.accessFlags)) {
                    this.com.hazelcast.com.ileError(
                        "Superclass method cannot be invoked in static context",
                        superclassMethodInvocation.getLocation()
                    );
                }
            }
            if (s instanceof AbstractClassDeclaration) {
                declaringClass = (AbstractClassDeclaration) s;
                break;
            }
        }

        IClass superclass = this.resolve(declaringClass).getSuperclass();
        if (superclass == null) {
            throw new CompileException(
                "\"" + declaringClass + "\" has no superclass",
                superclassMethodInvocation.getLocation()
            );
        }

        IMethod iMethod = this.findIMethod(
            superclass,                // targetType
            superclassMethodInvocation // invocation
        );
        if (iMethod == null) {
            this.com.hazelcast.com.ileError(
                "Class \"" + superclass + "\" has no method named \"" + superclassMethodInvocation.methodName + "\"",
                superclassMethodInvocation.getLocation()
            );
            return this.fakeIMethod(
                superclass,
                superclassMethodInvocation.methodName,
                superclassMethodInvocation.arguments
            );
        }
        this.checkThrownExceptions(superclassMethodInvocation, iMethod);
        return iMethod;
    }

    /**
     * Determines the arguments' types, determine the applicable invocables and choose the most specific invocable
     * and adjust arguments as needed (for varargs case).
     *
     * @param iInvocables       Length must be greater than zero
     * @return                  The selected {@link IClass.IInvocable}
     */
    private IClass.IInvocable
    findMostSpecificIInvocable(
        Locatable          locatable,
        final IInvocable[] iInvocables,
        final Rvalue[]     arguments,
        Scope              contextScope
    ) throws CompileException {

        // Determine arguments' types.
        final IClass[] argumentTypes = new IClass[arguments.length];
        for (int i = 0; i < arguments.length; ++i) {
            argumentTypes[i] = this.getType(arguments[i]);
        }

        // Determine most specific invocable WITHOUT boxing.
        IInvocable ii = this.findMostSpecificIInvocable(locatable, iInvocables, argumentTypes, false, contextScope);
        if (ii != null) return ii;

        // Determine most specific invocable WITH boxing.
        ii = this.findMostSpecificIInvocable(locatable, iInvocables, argumentTypes, true, contextScope);
        if (ii != null) return ii;

        // Report a nice com.hazelcast.com.ile error.
        StringBuilder sb = new StringBuilder("No applicable constructor/method found for ");
        if (argumentTypes.length == 0) {
            sb.append("zero actual parameters");
        } else {
            sb.append("actual parameters \"").append(argumentTypes[0]);
            for (int i = 1; i < argumentTypes.length; ++i) {
                sb.append(", ").append(argumentTypes[i]);
            }
            sb.append("\"");
        }
        sb.append("; candidates are: \"").append(iInvocables[0]).append('"');
        for (int i = 1; i < iInvocables.length; ++i) {
            sb.append(", \"").append(iInvocables[i]).append('"');
        }
        this.com.hazelcast.com.ileError(sb.toString(), locatable.getLocation());

        // Well, returning a "fake" IInvocable is a bit tricky, because the iInvocables can be of different types.
        if (iInvocables[0] instanceof IClass.IConstructor) {
            return iInvocables[0].getDeclaringIClass().new IConstructor() {
                @Override public boolean       isVarargs()            { return false; }
                @Override public IClass[]      getParameterTypes2()   { return argumentTypes; }
                @Override public Access        getAccess()            { return Access.PUBLIC; }
                @Override public IClass[]      getThrownExceptions2() { return new IClass[0]; }
                @Override public IAnnotation[] getAnnotations()       { return new IAnnotation[0]; }
            };
        } else
        if (iInvocables[0] instanceof IClass.IMethod) {
            final String methodName = ((IClass.IMethod) iInvocables[0]).getName();
            return iInvocables[0].getDeclaringIClass().new IMethod() {
                @Override public boolean       isStatic()             { return true; }
                @Override public boolean       isAbstract()           { return false; }
                @Override public IClass        getReturnType()        { return IClass.INT; }
                @Override public String        getName()              { return methodName; }
                @Override public Access        getAccess()            { return Access.PUBLIC; }
                @Override public boolean       isVarargs()            { return false; }
                @Override public IClass[]      getParameterTypes2()   { return argumentTypes; }
                @Override public IClass[]      getThrownExceptions2() { return new IClass[0]; }
                @Override public IAnnotation[] getAnnotations()       { return new IAnnotation[0]; }
            };
        } else
        {
            return iInvocables[0];
        }
    }

    /**
     * Determines the applicable invocables and choose the most specific invocable.
     *
     * @return The maximally specific {@link IClass.IInvocable} or {@code null} if no {@link IClass.IInvocable} is
     *         applicable
     */
    @Nullable public IClass.IInvocable
    findMostSpecificIInvocable(
        Locatable          locatable,
        final IInvocable[] iInvocables,
        IClass[]           argumentTypes,
        boolean            boxingPermitted,
        Scope              contextScope
    ) throws CompileException {
        if (UnitCompiler.LOGGER.isLoggable(Level.FINER)) {
            UnitCompiler.LOGGER.entering(null, "findMostSpecificIInvocable", new Object[] {
                locatable, Arrays.toString(iInvocables), Arrays.toString(argumentTypes), boxingPermitted, contextScope
            });
        }

        // Select applicable methods (15.12.2.1).
        List<IClass.IInvocable> applicableIInvocables = new ArrayList<IClass.IInvocable>();
        List<IClass.IInvocable> varargApplicables     = new ArrayList<IClass.IInvocable>();

        NEXT_METHOD:
        for (IClass.IInvocable ii : iInvocables) {
            boolean argsNeedAdjust = false;

            // Ignore inaccessible invocables.
            if (!this.isAccessible(ii, contextScope)) continue;

            // Check parameter count.
            final IClass[] parameterTypes   = ii.getParameterTypes();
            int            formalParamCount = parameterTypes.length;
            int            nUncheckedArg    = argumentTypes.length;
            final boolean  isVarargs        = ii.isVarargs();

            // Match the last formal parameter with all args starting from that index (or none).
            VARARGS:
            if (isVarargs) {

                // Decrement the count to get the index.
                formalParamCount--;
                final IClass lastParamType = parameterTypes[formalParamCount].getComponentType();
                assert lastParamType != null;

                final int lastActualArg = nUncheckedArg - 1;

                // If the two have the same argCount and the last actual arg is an array of the same type accept it
                // (e.g. "void foo(int a, double...b) VS foo(1, new double[0]").
                if (
                    formalParamCount == lastActualArg
                    && argumentTypes[lastActualArg].isArray()
                    && this.isMethodInvocationConvertible(
                        (IClass) UnitCompiler.assertNonNull(argumentTypes[lastActualArg].getComponentType()),
                        lastParamType,
                        boxingPermitted
                    )
                ) {
                    nUncheckedArg--;
                } else {
                    for (int idx = lastActualArg; idx >= formalParamCount; --idx) {

                        // Is method invocation conversion possible (5.3)?
                        UnitCompiler.LOGGER.log(
                            Level.FINE,
                            "{0} <=> {1}",
                            new Object[] { lastParamType, argumentTypes[idx] }
                        );
                        if (!this.isMethodInvocationConvertible(argumentTypes[idx], lastParamType, boxingPermitted)) {
                            formalParamCount++;
                            break VARARGS;
                        }

                        nUncheckedArg--;
                    }
                    argsNeedAdjust = true;
                }
            }

            if (formalParamCount == nUncheckedArg) {
                for (int j = 0; j < nUncheckedArg; ++j) {
                    UnitCompiler.LOGGER.log(
                        Level.FINE,
                        "{0}: {1} <=> {2}",
                        new Object[] { j, parameterTypes[j], argumentTypes[j] }
                    );

                    // Is method invocation conversion possible (5.3)?
                    if (!this.isMethodInvocationConvertible(argumentTypes[j], parameterTypes[j], boxingPermitted)) {
                        continue NEXT_METHOD;
                    }
                }

                // Applicable!
                UnitCompiler.LOGGER.fine("Applicable!");

                // Varargs has lower priority.
                if (isVarargs) {
                    ii.setArgsNeedAdjust(argsNeedAdjust);
                    varargApplicables.add(ii);
                } else {
                    applicableIInvocables.add(ii);
                }
            }
        }

        // Choose the most specific invocable (15.12.2.2).
        if (applicableIInvocables.size() == 1) {
            return (IInvocable) applicableIInvocables.get(0);
        }

        // No method found by previous phase(s).
        if (applicableIInvocables.size() == 0 && !varargApplicables.isEmpty()) {
            //TODO: 15.12.2.3 (type-conversion?)

            // 15.12.2.4 : Phase 3: Identify Applicable Variable Arity Methods
            applicableIInvocables = varargApplicables;
            if (applicableIInvocables.size() == 1) {
                return (IInvocable) applicableIInvocables.get(0);
            }
        }

        if (applicableIInvocables.size() == 0) return null;

        // 15.12.2.5. Determine the "maximally specific invocables".
        List<IClass.IInvocable> maximallySpecificIInvocables = new ArrayList<IClass.IInvocable>();
        for (IClass.IInvocable applicableIInvocable : applicableIInvocables) {
            int moreSpecific = 0, lessSpecific = 0;
            for (IClass.IInvocable mostSpecificIInvocable : maximallySpecificIInvocables) {
                if (applicableIInvocable.isMoreSpecificThan(mostSpecificIInvocable)) {
                    ++moreSpecific;
                } else
                if (applicableIInvocable.isLessSpecificThan(mostSpecificIInvocable)) {
                    ++lessSpecific;
                }
            }
            if (moreSpecific == maximallySpecificIInvocables.size()) {
                maximallySpecificIInvocables.clear();
                maximallySpecificIInvocables.add(applicableIInvocable);
            } else
            if (lessSpecific < maximallySpecificIInvocables.size()) {
                maximallySpecificIInvocables.add(applicableIInvocable);
            } else
            {
                ;
            }
            UnitCompiler.LOGGER.log(Level.FINE, "maximallySpecificIInvocables={0}", maximallySpecificIInvocables);
        }

        if (maximallySpecificIInvocables.size() == 1) return (IInvocable) maximallySpecificIInvocables.get(0);

        ONE_NON_ABSTRACT_INVOCABLE:
        if (maximallySpecificIInvocables.size() > 1 && iInvocables[0] instanceof IClass.IMethod) {

            // Check if all methods have the same signature (i.e. the types of all their parameters are identical) and
            // exactly one of the methods is non-abstract (JLS7 15.12.2.2.BL2.B1).
            IClass.IMethod theNonAbstractMethod = null;
            {
                Iterator<IClass.IInvocable> it                          = maximallySpecificIInvocables.iterator();
                IClass.IMethod              m                           = (IClass.IMethod) it.next();
                final IClass[]              parameterTypesOfFirstMethod = m.getParameterTypes();
                for (;;) {
                    if (!m.isAbstract() && !m.getDeclaringIClass().isInterface()) {
                        if (theNonAbstractMethod == null) {
                            theNonAbstractMethod = m;
                        } else {
                            IClass declaringIClass                     = m.getDeclaringIClass();
                            IClass theNonAbstractMethodDeclaringIClass = theNonAbstractMethod.getDeclaringIClass();
                            if (declaringIClass == theNonAbstractMethodDeclaringIClass) {
                                if (m.getReturnType() == theNonAbstractMethod.getReturnType()) {

                                    // JLS8 15.12.2.5.B9: "Otherwise, the method invocation is ambiguous, and a
                                    // com.hazelcast.com.ile-time error occurs."
                                    throw new InternalCompilerException(
                                        "Two non-abstract methods \"" + m + "\" have the same parameter types, "
                                        + "declaring type and return type"
                                    );
                                } else
                                if (m.getReturnType().isAssignableFrom(theNonAbstractMethod.getReturnType())) {
                                    ;
                                } else
                                if (theNonAbstractMethod.getReturnType().isAssignableFrom(m.getReturnType())) {
                                    theNonAbstractMethod = m;
                                } else
                                {
                                    throw new InternalCompilerException("Incompatible return types");
                                }
                            } else
                            if (declaringIClass.isAssignableFrom(theNonAbstractMethodDeclaringIClass)) {
                                ;
                            } else
                            if (theNonAbstractMethodDeclaringIClass.isAssignableFrom(declaringIClass)) {
                                theNonAbstractMethod = m;
                            } else
                            {
                                throw new InternalCompilerException(
                                    "SNO: Types declaring \""
                                    + theNonAbstractMethod
                                    + "\" are not assignable"
                                );
                            }
                        }
                    }
                    if (!it.hasNext()) break;

                    m = (IClass.IMethod) it.next();
                    IClass[] pts = m.getParameterTypes();
                    for (int i = 0; i < pts.length; ++i) {
                        if (pts[i] != parameterTypesOfFirstMethod[i]) break ONE_NON_ABSTRACT_INVOCABLE;
                    }
                }
            }

            // JLS7 15.12.2.2.BL2.B1.B1
            if (theNonAbstractMethod != null) return theNonAbstractMethod;

            // JLS7 15.12.2.2.BL2.B1.B2
            // Check "that exception [te1] is declared in the THROWS clause of each of the maximally specific methods".
            Set<IClass> s = new HashSet<IClass>();
            {
                IClass[][]                  tes = new IClass[maximallySpecificIInvocables.size()][];
                Iterator<IClass.IInvocable> it  = maximallySpecificIInvocables.iterator();
                for (int i = 0; i < tes.length; ++i) {
                    tes[i] = ((IClass.IMethod) it.next()).getThrownExceptions();
                }
                for (int i = 0; i < tes.length; ++i) {
                    EACH_EXCEPTION:
                    for (IClass te1 : tes[i]) {
                        EACH_METHOD:
                        for (int k = 0; k < tes.length; ++k) {
                            if (k == i) continue;
                            for (IClass te2 : tes[k]) {
                                if (te2.isAssignableFrom(te1)) continue EACH_METHOD;
                            }
                            continue EACH_EXCEPTION;
                        }
                        s.add(te1);
                    }
                }
            }

            // JLS7 1.12.2.5.BL2.B1.BL.B2: "... the most specific method is chosen arbitrarily among the subset of the
            // maximally specific methods that have the most specific return type".
            final IClass.IMethod im;
            {
                Iterator<IInvocable> it2                              = maximallySpecificIInvocables.iterator();
                IClass.IMethod       methodWithMostSpecificReturnType = (IMethod) it2.next();

                while (it2.hasNext()) {
                    IMethod im2 = (IMethod) it2.next();
                    if (methodWithMostSpecificReturnType.getReturnType().isAssignableFrom(im2.getReturnType())) {
                        methodWithMostSpecificReturnType = im2;
                    }
                }
                im = methodWithMostSpecificReturnType;
            }

            // Return a "dummy" method.
            final IClass[]       tes = (IClass[]) s.toArray(new IClass[s.size()]);
            return im.getDeclaringIClass().new IMethod() {

                // SUPPRESS CHECKSTYLE LineLength:10
                @Override public String        getName()                                    { return im.getName();           }
                @Override public IClass        getReturnType()      throws CompileException { return im.getReturnType();     }
                // JLS8 15.12.2.5.B8.B2: "In this case, the most specific method is considered to be abstract"
                @Override public boolean       isAbstract()                                 { return true;                   }
                @Override public boolean       isStatic()                                   { return im.isStatic();          }
                @Override public Access        getAccess()                                  { return im.getAccess();         }
                @Override public boolean       isVarargs()                                  { return im.isVarargs();         }
                @Override public IClass[]      getParameterTypes2() throws CompileException { return im.getParameterTypes(); }
                @Override public IClass[]      getThrownExceptions2()                       { return tes;                    }
                @Override public IAnnotation[] getAnnotations()                             { return im.getAnnotations();    }
            };
        }

        if (!boxingPermitted) return null; // To try again.

        // JLS7 15.12.2.2.BL2.B2
        {
            StringBuilder sb = new StringBuilder("Invocation of constructor/method with argument type(s) \"");
            for (int i = 0; i < argumentTypes.length; ++i) {
                if (i > 0) sb.append(", ");
                sb.append(Descriptor.toString(argumentTypes[i].getDescriptor()));
            }
            sb.append("\" is ambiguous: ");
            for (int i = 0; i < maximallySpecificIInvocables.size(); ++i) {
                if (i > 0) sb.append(" vs. ");
                sb.append("\"" + maximallySpecificIInvocables.get(i) + "\"");
            }
            this.com.hazelcast.com.ileError(sb.toString(), locatable.getLocation());
        }

        return iInvocables[0];
    }

    private static <T> T
    assertNonNull(@Nullable T subject) {
        assert subject != null;
        return subject;
    }

    /**
     * Checks if "method invocation conversion" (5.3) is possible.
     */
    private boolean
    isMethodInvocationConvertible(
        IClass  sourceType,
        IClass  targetType,
        boolean boxingPermitted
    ) throws CompileException {

        // 5.3 Identity conversion.
        if (sourceType == targetType) return true;

        // 5.3 Widening primitive conversion.
        if (this.isWideningPrimitiveConvertible(sourceType, targetType)) return true;

        // 5.3 Widening reference conversion.
        if (this.isWideningReferenceConvertible(sourceType, targetType)) return true;

        // JLS7 5.3 A boxing conversion (JLS7 5.1.7) optionally followed by widening reference conversion.
        if (boxingPermitted) {
            IClass boxedType = this.isBoxingConvertible(sourceType);
            if (boxedType != null) {
                return (
                    this.isIdentityConvertible(boxedType, targetType)
                    || this.isWideningReferenceConvertible(boxedType, targetType)
                );
            }
        }

        // JLS7 5.3 An unboxing conversion (JLS7 5.1.8) optionally followed by a widening primitive conversion.
        if (boxingPermitted) {
            IClass unboxedType = this.isUnboxingConvertible(sourceType);
            if (unboxedType != null) {
                return (
                    this.isIdentityConvertible(unboxedType, targetType)
                    || this.isWideningPrimitiveConvertible(unboxedType, targetType)
                );
            }
        }

        // 5.3 TODO: FLOAT or DOUBLE value set conversion

        return false;
    }

    /**
     * @throws CompileException if the {@link Invocation} throws exceptions that are disallowed in the given scope
     */
    private void
    checkThrownExceptions(Invocation in, IMethod iMethod) throws CompileException {
        IClass[] thrownExceptions = iMethod.getThrownExceptions();
        for (IClass thrownException : thrownExceptions) {
            this.checkThrownException(
                in,                    // locatable
                thrownException,       // type
                in.getEnclosingScope() // scope
            );
        }
    }

    /**
     * @throws CompileException The exception with the given <var>type</var> must not be thrown in the given
     *                          <var>scope</var>
     */
    private void
    checkThrownException(Locatable locatable, IClass type, Scope scope) throws CompileException {

        // Thrown object must be assignable to "Throwable".
        if (!this.iClassLoader.TYPE_java_lang_Throwable.isAssignableFrom(type)) {
            this.com.hazelcast.com.ileError(
                "Thrown object of type \"" + type + "\" is not assignable to \"Throwable\"",
                locatable.getLocation()
            );
        }

        // "RuntimeException" and "Error" are never checked.
        if (
            this.iClassLoader.TYPE_java_lang_RuntimeException.isAssignableFrom(type)
            || this.iClassLoader.TYPE_java_lang_Error.isAssignableFrom(type)
        ) return;

        for (;; scope = scope.getEnclosingScope()) {

            // Match against enclosing "try...catch" blocks.
            if (scope instanceof TryStatement) {
                TryStatement ts = (TryStatement) scope;
                for (int i = 0; i < ts.catchClauses.size(); ++i) {
                    CatchClause cc         = (CatchClause) ts.catchClauses.get(i);
                    IClass      caughtType = this.getType(cc.caughtException.type);
                    if (caughtType.isAssignableFrom(type)) {

                        // This catch clause definitely catches the exception.
                        cc.reachable = true;
                        return;
                    }

                    CATCH_SUBTYPE:
                    if (type.isAssignableFrom(caughtType)) {

                        // This catch clause catches only a subtype of the exception type.
                        for (int j = 0; j < i; ++j) {
                            if (this.getType(
                                ((CatchClause) ts.catchClauses.get(j)).caughtException.type
                            ).isAssignableFrom(caughtType)) {

                                // A preceding catch clause is more general than this catch clause.
                                break CATCH_SUBTYPE;
                            }
                        }

                        // This catch clause catches PART OF the actual exceptions.
                        cc.reachable = true;
                    }
                }
            } else

            // Match against "throws" clause of declaring function.
            if (scope instanceof FunctionDeclarator) {
                FunctionDeclarator fd = (FunctionDeclarator) scope;
                for (Type thrownException : fd.thrownExceptions) {
                    if (this.getType(thrownException).isAssignableFrom(type)) return;
                }
                break;
            } else

            if (scope instanceof TypeBodyDeclaration) {
                break;
            }
        }

        this.com.hazelcast.com.ileError((
            "Thrown exception of type \""
            + type
            + "\" is neither caught by a \"try...catch\" block "
            + "nor declared in the \"throws\" clause of the declaring function"
        ), locatable.getLocation());
    }

    private IClass
    getTargetIClass(QualifiedThisReference qtr) throws CompileException {

        if (qtr.targetIClass != null) return qtr.targetIClass;

        // Determine target type.
        return (qtr.targetIClass = this.getType(qtr.qualification));
    }

    /**
     * Checks whether the operand is an integer-like local variable.
     *
     * @return {@code null} iff <var>c</var> is <em>not</em> an integer-like local variable
     */
    @Nullable LocalVariable
    isIntLv(Crement c) throws CompileException {
        if (!(c.operand instanceof AmbiguousName)) return null;
        AmbiguousName an = (AmbiguousName) c.operand;

        Atom rec = this.reclassify(an);
        if (!(rec instanceof LocalVariableAccess)) return null;
        LocalVariableAccess lva = (LocalVariableAccess) rec;

        LocalVariable lv = lva.localVariable;
        if (lv.finaL) this.com.hazelcast.com.ileError("Must not increment or decrement \"final\" local variable", lva.getLocation());
        if (
            lv.type == IClass.BYTE
            || lv.type == IClass.SHORT
            || lv.type == IClass.INT
            || lv.type == IClass.CHAR
        ) return lv;
        return null;
    }

    private IClass
    resolve(final TypeDeclaration td) {

        final AbstractTypeDeclaration atd = (AbstractTypeDeclaration) td;

        if (atd.resolvedType != null) return atd.resolvedType;

        return (atd.resolvedType = new IClass() {

//            final TypeParameter[] optionalTypeParameters = (
//                atd instanceof NamedTypeDeclaration
//                ? ((NamedTypeDeclaration) atd).getOptionalTypeParameters()
//                : null
//            );

            @Override protected IClass.IMethod[]
            getDeclaredIMethods2() {
                List<IClass.IMethod> res = new ArrayList<IClass.IMethod>(atd.getMethodDeclarations().size());
                for (MethodDeclarator md : atd.getMethodDeclarations()) {
                    res.add(UnitCompiler.this.toIMethod(md));
                }

                if (td instanceof Java.EnumDeclaration) {

                    res.add(new IMethod() {

                        @Override public IAnnotation[] getAnnotations()       { return new IAnnotation[0]; }
                        @Override public boolean       isStatic()             { return true; }
                        @Override public boolean       isAbstract()           { return false; }
                        @Override public String        getName()              { return "values"; }
                        @Override public boolean       isVarargs()            { return false; }
                        @Override public Access        getAccess()            { return Access.PUBLIC; }
                        @Override public IClass[]      getParameterTypes2()   { return new IClass[0]; }
                        @Override public IClass[]      getThrownExceptions2() { return new IClass[0]; }

                        @Override public IClass
                        getReturnType() {
                            IClass rt = atd.resolvedType;
                            assert rt != null;
                            return rt.getArrayIClass(UnitCompiler.this.iClassLoader.TYPE_java_lang_Object);
                        }
                    });

                    res.add(new IMethod() {

                        @Override public IAnnotation[] getAnnotations()       { return new IAnnotation[0]; }
                        @Override public boolean       isStatic()             { return true; }
                        @Override public boolean       isAbstract()           { return false; }
                        @Override public String        getName()              { return "valueOf"; }
                        @Override public boolean       isVarargs()            { return false; }
                        @Override public Access        getAccess()            { return Access.PUBLIC; }
                        @Override public IClass[]      getParameterTypes2()   { return new IClass[] { UnitCompiler.this.iClassLoader.TYPE_java_lang_String }; } // SUPPRESS CHECKSTYLE LineLength
                        @Override public IClass[]      getThrownExceptions2() { return new IClass[0]; }

                        @Override public IClass
                        getReturnType() {
                            IClass rt = atd.resolvedType;
                            assert rt != null;
                            return rt;
                        }
                    });
                }

                return (IClass.IMethod[]) res.toArray(new IClass.IMethod[res.size()]);
            }

            @Nullable private IClass[] declaredClasses;

            @Override protected IClass[]
            getDeclaredIClasses2() {

                if (this.declaredClasses != null) return this.declaredClasses;

                Collection<MemberTypeDeclaration> mtds = td.getMemberTypeDeclarations();
                IClass[]                          mts  = new IClass[mtds.size()];
                int                               i    = 0;
                for (MemberTypeDeclaration mtd : mtds) {
                    mts[i++] = UnitCompiler.this.resolve(mtd);
                }
                return (this.declaredClasses = mts);
            }

            @Override @Nullable protected IClass
            getDeclaringIClass2() {
                Scope s = atd;
                for (; !(s instanceof TypeBodyDeclaration); s = s.getEnclosingScope()) {
                    if (s instanceof CompilationUnit) return null;
                }
                return UnitCompiler.this.resolve((AbstractTypeDeclaration) s.getEnclosingScope());
            }

            @Override @Nullable protected IClass
            getOuterIClass2() {
                AbstractTypeDeclaration oc = (AbstractTypeDeclaration) UnitCompiler.getOuterClass(atd);
                if (oc == null) return null;
                return UnitCompiler.this.resolve(oc);
            }

            @Override protected String
            getDescriptor2() { return Descriptor.fromClassName(atd.getClassName()); }

            @Override public boolean
            isArray() { return false; }

            @Override protected IClass
            getComponentType2() { throw new InternalCompilerException("SNO: Non-array type has no com.hazelcast.com.onent type"); }

            @Override public boolean
            isPrimitive() { return false; }

            @Override public boolean
            isPrimitiveNumeric() { return false; }

            @Override protected IConstructor[]
            getDeclaredIConstructors2() {
                if (atd instanceof AbstractClassDeclaration) {
                    ConstructorDeclarator[] cs = ((AbstractClassDeclaration) atd).getConstructors();

                    IClass.IConstructor[] res = new IClass.IConstructor[cs.length];
                    for (int i = 0; i < cs.length; ++i) res[i] = UnitCompiler.this.toIConstructor(cs[i]);
                    return res;
                }
                return new IClass.IConstructor[0];
            }

            @Override protected IField[]
            getDeclaredIFields2() {
                if (atd instanceof AbstractClassDeclaration) {
                    AbstractClassDeclaration cd = (AbstractClassDeclaration) atd;
                    List<IClass.IField>      l  = new ArrayList<IClass.IField>();

                    // Determine variable declarators of type declaration.
                    for (BlockStatement vdoi : cd.variableDeclaratorsAndInitializers) {
                        if (vdoi instanceof FieldDeclaration) {
                            FieldDeclaration fd   = (FieldDeclaration) vdoi;
                            IClass.IField[]  flds = UnitCompiler.this.com.hazelcast.com.ileFields(fd);
                            for (IField fld : flds) l.add(fld);
                        }
                    }

                    if (atd instanceof EnumDeclaration) {
                        EnumDeclaration ed = (EnumDeclaration) atd;

                        for (EnumConstant ec : ed.getConstants()) {
                            l.add(UnitCompiler.this.com.hazelcast.com.ileField(
                                ed,                                                              // declaringType
                                new Modifiers((short) (Mod.PUBLIC | Mod.FINAL | Mod.STATIC)),    // modifiers
                                new SimpleType(ed.getLocation(), UnitCompiler.this.resolve(ed)), // type
                                new VariableDeclarator(                                          // variableDeclarator
                                    ec.getLocation(),  // location
                                    ec.name,           // name
                                    0,                 // brackets
                                    null               // optionalInitializer
                                )
                            ));
                        }
                    }

                    return (IField[]) l.toArray(new IClass.IField[l.size()]);
                } else
                if (atd instanceof InterfaceDeclaration) {
                    InterfaceDeclaration id = (InterfaceDeclaration) atd;
                    List<IClass.IField>  l  = new ArrayList<IClass.IField>();

                    // Determine static fields.
                    for (BlockStatement bs : id.constantDeclarations) {
                        if (bs instanceof FieldDeclaration) {
                            FieldDeclaration fd   = (FieldDeclaration) bs;
                            IClass.IField[]  flds = UnitCompiler.this.com.hazelcast.com.ileFields(fd);
                            for (IField fld : flds) l.add(fld);
                        }
                    }
                    return (IClass.IField[]) l.toArray(new IClass.IField[l.size()]);
                } else
                {
                    throw new InternalCompilerException(
                        "SNO: AbstractTypeDeclaration is neither ClassDeclaration nor InterfaceDeclaration"
                    );
                }
            }

            @Override public IField[]
            getSyntheticIFields() {
                if (atd instanceof AbstractClassDeclaration) {
                    Collection<IClass.IField> c = ((AbstractClassDeclaration) atd).syntheticFields.values();
                    return (IField[]) c.toArray(new IField[c.size()]);
                }
                return new IField[0];
            }

            @Override @Nullable protected IClass
            getSuperclass2() throws CompileException {

                if (atd instanceof EnumDeclaration) return UnitCompiler.this.iClassLoader.TYPE_java_lang_Enum;

                if (atd instanceof AnonymousClassDeclaration) {
                    IClass bt = UnitCompiler.this.getType(((AnonymousClassDeclaration) atd).baseType);
                    return bt.isInterface() ? UnitCompiler.this.iClassLoader.TYPE_java_lang_Object : bt;
                }

                if (atd instanceof NamedClassDeclaration) {
                    NamedClassDeclaration ncd = (NamedClassDeclaration) atd;
                    Type                  oet = ncd.optionalExtendedType;
                    if (oet == null) return UnitCompiler.this.iClassLoader.TYPE_java_lang_Object;
                    IClass superclass = UnitCompiler.this.getType(oet);
                    if (superclass.isInterface()) {
                        UnitCompiler.this.com.hazelcast.com.ileError(
                            "\"" + superclass.toString() + "\" is an interface; classes can only extend a class",
                            td.getLocation()
                        );
                    }
                    return superclass;
                }

                return null;
            }

            @Override public Access
            getAccess() { return UnitCompiler.modifiers2Access(atd.getModifierFlags()); }

            @Override public boolean
            isFinal() { return Mod.isFinal(atd.getModifierFlags());  }

            @Override protected IClass[]
            getInterfaces2() throws CompileException {
                if (atd instanceof AnonymousClassDeclaration) {
                    IClass bt = UnitCompiler.this.getType(((AnonymousClassDeclaration) atd).baseType);
                    return bt.isInterface() ? new IClass[] { bt } : new IClass[0];
                } else
                if (atd instanceof NamedClassDeclaration) {
                    NamedClassDeclaration ncd = (NamedClassDeclaration) atd;
                    IClass[]              res = new IClass[ncd.implementedTypes.length];
                    for (int i = 0; i < res.length; ++i) {
                        res[i] = UnitCompiler.this.getType(ncd.implementedTypes[i]);
                        if (!res[i].isInterface()) {
                            UnitCompiler.this.com.hazelcast.com.ileError((
                                "\""
                                + res[i].toString()
                                + "\" is not an interface; classes can only implement interfaces"
                            ), td.getLocation());
                        }
                    }
                    return res;
                } else
                if (atd instanceof InterfaceDeclaration) {
                    InterfaceDeclaration id  = (InterfaceDeclaration) atd;
                    IClass[]             res = new IClass[id.extendedTypes.length];
                    for (int i = 0; i < res.length; ++i) {
                        res[i] = UnitCompiler.this.getType(id.extendedTypes[i]);
                        if (!res[i].isInterface()) {
                            UnitCompiler.this.com.hazelcast.com.ileError((
                                "\""
                                + res[i].toString()
                                + "\" is not an interface; interfaces can only extend interfaces"
                            ), td.getLocation());
                        }
                    }
                    return res;
                } else {
                    throw new InternalCompilerException(
                        "SNO: AbstractTypeDeclaration is neither ClassDeclaration nor InterfaceDeclaration"
                    );
                }
            }

            @Override protected IAnnotation[]
            getIAnnotations2() { return UnitCompiler.this.toIAnnotations(td.getAnnotations()); }

            @Override public boolean
            isAbstract() { return atd instanceof InterfaceDeclaration || Mod.isAbstract(atd.getModifierFlags()); }

            @Override public boolean
            isEnum() { return atd instanceof EnumDeclaration; }

            @Override public boolean
            isInterface() { return atd instanceof InterfaceDeclaration; }
        });
    }

    private void
    referenceThis(
        Locatable                locatable,
        AbstractClassDeclaration declaringClass,
        TypeBodyDeclaration      declaringTypeBodyDeclaration,
        IClass                   targetIClass
    ) throws CompileException {
        List<TypeDeclaration> path = UnitCompiler.getOuterClasses(declaringClass);

        if (declaringTypeBodyDeclaration.isStatic()) {
            this.com.hazelcast.com.ileError("No current instance available in static context", locatable.getLocation());
        }

        int j;
        TARGET_FOUND: {
            for (j = 0; j < path.size(); ++j) {

                // Notice: JLS7 15.9.2.BL1.B3.B1.B2 seems to be wrong: Obviously, JAVAC does not only allow
                //
                //    O is the nth lexically enclosing class
                //
                // , but also
                //
                //    O is assignable from the nth lexically enclosing class
                //
                // However, this strategy bears the risk of ambiguities, because "O" may be assignable from more than
                // one enclosing class.
                if (targetIClass.isAssignableFrom(this.resolve((TypeDeclaration) path.get(j)))) {
                    break TARGET_FOUND;
                }
            }
            this.com.hazelcast.com.ileError(
                "\"" + declaringClass + "\" is not enclosed by \"" + targetIClass + "\"",
                locatable.getLocation()
            );
        }

        int i;
        if (declaringTypeBodyDeclaration instanceof ConstructorDeclarator) {
            if (j == 0) {
                this.writeOpcode(locatable, Opcode.ALOAD_0);
                return;
            }

            ConstructorDeclarator constructorDeclarator = (
                (ConstructorDeclarator) declaringTypeBodyDeclaration
            );
            String        spn                = "this$" + (path.size() - 2);
            LocalVariable syntheticParameter = (LocalVariable) constructorDeclarator.syntheticParameters.get(spn);
            if (syntheticParameter == null) {
                throw new InternalCompilerException("SNO: Synthetic parameter \"" + spn + "\" not found");
            }
            this.load(locatable, syntheticParameter);
            i = 1;
        } else {
            this.writeOpcode(locatable, Opcode.ALOAD_0);
            i = 0;
        }
        for (; i < j; ++i) {
            final InnerClassDeclaration inner     = (InnerClassDeclaration) path.get(i);
            final TypeDeclaration       outer     = (TypeDeclaration) path.get(i + 1);

            SimpleIField sf = new SimpleIField(
                this.resolve(inner),             // declaringIClass
                "this$" + (path.size() - i - 2), // name
                this.resolve(outer)              // type
            );
            inner.defineSyntheticField(sf);
            this.getfield(locatable, sf);
        }
    }

    /**
     * Returns a list consisting of the given <var>inner</var> class and all its enclosing (outer) classes.
     *
     * @return {@link List} of {@link TypeDeclaration}; has length one iff <var>inner</var> has no enclosing instance
     */
    private static List<TypeDeclaration>
    getOuterClasses(TypeDeclaration inner) {
        List<TypeDeclaration> path = new ArrayList<TypeDeclaration>();
        for (TypeDeclaration ic = inner; ic != null; ic = UnitCompiler.getOuterClass(ic)) path.add(ic);
        return path;
    }

    /**
     * @return The {@link TypeDeclaration} that immediately encloses the <var>typeDeclaration</var>, or {@code null}
     */
    @Nullable static TypeDeclaration
    getOuterClass(TypeDeclaration typeDeclaration) {

        // Package member class declaration.
        if (typeDeclaration instanceof PackageMemberClassDeclaration) return null;

        // Member enum declaration is always implicitly static (JLS8 8.9).
        if (typeDeclaration instanceof MemberEnumDeclaration) return null;

        // Local class declaration.
        if (typeDeclaration instanceof LocalClassDeclaration) {
            Scope s = typeDeclaration.getEnclosingScope();
            for (; !(s instanceof FunctionDeclarator); s = s.getEnclosingScope());
            if (
                (s instanceof MethodDeclarator)
                && Mod.isStatic(((FunctionDeclarator) s).modifiers.accessFlags)
            ) return null;
            for (; !(s instanceof TypeDeclaration); s = s.getEnclosingScope());
            TypeDeclaration immediatelyEnclosingTypeDeclaration = (TypeDeclaration) s;
            return (
                immediatelyEnclosingTypeDeclaration instanceof AbstractClassDeclaration
            ) ? immediatelyEnclosingTypeDeclaration : null;
        }

        // Member class declaration.
        if (
            typeDeclaration instanceof MemberClassDeclaration
            && Mod.isStatic(((MemberClassDeclaration) typeDeclaration).getModifierFlags())
        ) return null;

        // Anonymous class declaration, interface declaration.
        Scope s = typeDeclaration;
        for (; !(s instanceof TypeBodyDeclaration); s = s.getEnclosingScope()) {
            if (s instanceof ConstructorInvocation) return null;
            if (s instanceof CompilationUnit) return null;
        }
        //if (!(s instanceof ClassDeclaration)) return null;
        if (((TypeBodyDeclaration) s).isStatic()) return null;
        return (AbstractTypeDeclaration) s.getEnclosingScope();
    }

    private IClass
    getIClass(ThisReference tr) throws CompileException {

        if (tr.iClass != null) return tr.iClass;

        // Compile error if in static function context.
        Scope s;
        for (
            s = tr.getEnclosingScope();
            s instanceof Statement || s instanceof CatchClause;
            s = s.getEnclosingScope()
        );
        if (s instanceof FunctionDeclarator) {
            FunctionDeclarator function = (FunctionDeclarator) s;
            if (Mod.isStatic(function.modifiers.accessFlags)) {
                this.com.hazelcast.com.ileError("No current instance available in static method", tr.getLocation());
            }
        }

        // Determine declaring type.
        while (!(s instanceof TypeDeclaration)) {
            s = s.getEnclosingScope();
        }
        if (!(s instanceof AbstractClassDeclaration)) {
            this.com.hazelcast.com.ileError("Only methods of classes can have a current instance", tr.getLocation());
        }

        return (tr.iClass = this.resolve((AbstractClassDeclaration) s));
    }

    private IClass
    getReturnType(FunctionDeclarator fd) throws CompileException {

        if (fd.returnType != null) return fd.returnType;

        return (fd.returnType = this.getType(fd.type));
    }

    /**
     * @return the {@link IConstructor} that implements the <var>constructorDeclarator</var>
     */
    IClass.IConstructor
    toIConstructor(final ConstructorDeclarator constructorDeclarator) {
        if (constructorDeclarator.iConstructor != null) return constructorDeclarator.iConstructor;

        constructorDeclarator.iConstructor = this.resolve(constructorDeclarator.getDeclaringType()).new IConstructor() {

            @Nullable private IAnnotation[] ias;

            // Implement IMember.
            @Override public Access
            getAccess() {
                switch (constructorDeclarator.modifiers.accessFlags & Mod.PPP) {
                case Mod.PRIVATE:
                    return Access.PRIVATE;
                case Mod.PROTECTED:
                    return Access.PROTECTED;
                case Mod.PACKAGE:
                    return Access.DEFAULT;
                case Mod.PUBLIC:
                    return Access.PUBLIC;
                default:
                    throw new InternalCompilerException("Invalid access");
                }
            }

            @Override public IAnnotation[]
            getAnnotations() {

                if (this.ias != null) return this.ias;

                return (this.ias = UnitCompiler.this.toIAnnotations(constructorDeclarator.getAnnotations()));
            }

            // Implement IInvocable.

            @Override public MethodDescriptor
            getDescriptor2() throws CompileException {

                if (!(constructorDeclarator.getDeclaringClass() instanceof InnerClassDeclaration)) {
                    return super.getDescriptor2();
                }

                if (constructorDeclarator.getDeclaringClass() instanceof MemberEnumDeclaration) {
                    return super.getDescriptor2();
                }

                List<String> parameterFds = new ArrayList<String>();

                // Convert enclosing instance reference into prepended constructor parameters.
                IClass outerClass = UnitCompiler.this.resolve(
                    constructorDeclarator.getDeclaringClass()
                ).getOuterIClass();
                if (outerClass != null) parameterFds.add(outerClass.getDescriptor());

                // Convert synthetic fields into prepended constructor parameters.
                for (IField sf : constructorDeclarator.getDeclaringClass().syntheticFields.values()) {
                    if (sf.getName().startsWith("val$")) parameterFds.add(sf.getType().getDescriptor());
                }

                // Process the 'normal' (declared) function parameters.
                for (IClass pt : this.getParameterTypes2()) parameterFds.add(pt.getDescriptor());

                return new MethodDescriptor(
                    Descriptor.VOID,                                                 // returnFd
                    (String[]) parameterFds.toArray(new String[parameterFds.size()]) // parameterFds
                );
            }

            @Override public boolean
            isVarargs() { return Mod.isVarargs(constructorDeclarator.modifiers.accessFlags); }

            @Override public IClass[]
            getParameterTypes2() throws CompileException {

                FormalParameter[] parameters = constructorDeclarator.formalParameters.parameters;
                IClass[]          res        = new IClass[parameters.length];

                for (int i = 0; i < parameters.length; ++i) {
                    IClass parameterType = UnitCompiler.this.getType(parameters[i].type);
                    if (i == parameters.length - 1 && constructorDeclarator.formalParameters.variableArity) {
                        parameterType = parameterType.getArrayIClass(
                            UnitCompiler.this.iClassLoader.TYPE_java_lang_Object
                        );
                    }
                    res[i] = parameterType;
                }

                return res;
            }

            @Override public IClass[]
            getThrownExceptions2() throws CompileException {
                IClass[] res = new IClass[constructorDeclarator.thrownExceptions.length];
                for (int i = 0; i < res.length; ++i) {
                    res[i] = UnitCompiler.this.getType(constructorDeclarator.thrownExceptions[i]);
                }
                return res;
            }

            @Override public String
            toString() {
                StringBuilder sb = new StringBuilder().append(
                    constructorDeclarator.getDeclaringType().getClassName()
                ).append('(');

                FormalParameter[] parameters = constructorDeclarator.formalParameters.parameters;
                for (int i = 0; i < parameters.length; ++i) {
                    if (i != 0) sb.append(", ");
                    sb.append(parameters[i].toString(
                        i == parameters.length - 1
                        && constructorDeclarator.formalParameters.variableArity
                    ));
                }
                return sb.append(')').toString();
            }
        };
        return constructorDeclarator.iConstructor;
    }

    /**
     * @return The {@link IMethod} that implements the <var>methodDeclarator</var>
     */
    public IClass.IMethod
    toIMethod(final MethodDeclarator methodDeclarator) {

        if (methodDeclarator.iMethod != null) return methodDeclarator.iMethod;

        methodDeclarator.iMethod = this.resolve(methodDeclarator.getDeclaringType()).new IMethod() {

            @Nullable IAnnotation[] ias;

            // Implement IMember.
            @Override public Access
            getAccess() {
                switch (methodDeclarator.modifiers.accessFlags & Mod.PPP) {
                case Mod.PRIVATE:
                    return Access.PRIVATE;
                case Mod.PROTECTED:
                    return Access.PROTECTED;
                case Mod.PACKAGE:
                    return Access.DEFAULT;
                case Mod.PUBLIC:
                    return Access.PUBLIC;
                default:
                    throw new InternalCompilerException("Invalid access");
                }
            }

            @Override public IAnnotation[]
            getAnnotations() {

                if (this.ias != null) return this.ias;

                return (this.ias = UnitCompiler.this.toIAnnotations(methodDeclarator.getAnnotations()));
            }

            // Implement IInvocable.

            @Override public boolean
            isVarargs() { return Mod.isVarargs(methodDeclarator.modifiers.accessFlags); }

            @Override public IClass[]
            getParameterTypes2() throws CompileException {
                FormalParameter[] parameters = methodDeclarator.formalParameters.parameters;
                IClass[]          res        = new IClass[parameters.length];
                for (int i = 0; i < parameters.length; ++i) {
                    IClass parameterType = UnitCompiler.this.getType(parameters[i].type);
                    if (i == parameters.length - 1 && methodDeclarator.formalParameters.variableArity) {
                        parameterType = parameterType.getArrayIClass(
                            UnitCompiler.this.iClassLoader.TYPE_java_lang_Object
                        );
                    }
                    res[i] = parameterType;
                }
                return res;
            }

            @Override public IClass[]
            getThrownExceptions2() throws CompileException {

                List<IClass> result = new ArrayList<IClass>();
                for (Type ti : methodDeclarator.thrownExceptions) {

                    // KLUDGE: Iff the exception type in the THROWS clause sounds like a type parameter, then
                    // ignore it. Otherwise we'd have to put
                    //    try { ... } catch (Throwable t) { throw new AssertionError(t); }
                    // around all invocations of methods that use a type parameter for declaring their exception.
                    if (ti instanceof ReferenceType) {
                        String[] identifiers = ((ReferenceType) ti).identifiers;
                        if (
                            identifiers.length == 1
                            && UnitCompiler.LOOKS_LIKE_TYPE_PARAMETER.matcher(identifiers[0]).matches()
                        ) continue;
                    }

                    result.add(UnitCompiler.this.getType(ti));
                }

                return (IClass[]) result.toArray(new IClass[result.size()]);
            }

            // Implement IMethod.

            @Override public boolean
            isStatic() { return Mod.isStatic(methodDeclarator.modifiers.accessFlags); }

            @Override public boolean
            isAbstract() {
                return (
                    (methodDeclarator.getDeclaringType() instanceof InterfaceDeclaration)
                    || Mod.isAbstract(methodDeclarator.modifiers.accessFlags)
                );
            }

            @Override public IClass
            getReturnType() throws CompileException { return UnitCompiler.this.getReturnType(methodDeclarator); }

            @Override public String
            getName() { return methodDeclarator.name; }
        };
        return methodDeclarator.iMethod;
    }

    private IClass.IInvocable
    toIInvocable(final FunctionDeclarator fd) {
        IClass.IInvocable result = (IClass.IInvocable) fd.accept(
            new Visitor.FunctionDeclaratorVisitor<IInvocable, RuntimeException>() {

                // SUPPRESS CHECKSTYLE LineLength:2
                @Override public IInvocable visitMethodDeclarator(MethodDeclarator md)           { return UnitCompiler.this.toIMethod((MethodDeclarator) fd);           }
                @Override public IInvocable visitConstructorDeclarator(ConstructorDeclarator cd) { return UnitCompiler.this.toIConstructor((ConstructorDeclarator) fd); }
            }
        );

        assert result != null;
        return result;
    }

    /**
     * If the given name was declared in a simple type import, load that class.
     */
    @Nullable private IClass
    importSingleType(String simpleTypeName, Location location) throws CompileException {
        String[] ss = this.getSingleTypeImport(simpleTypeName, location);
        if (ss == null) return null;

        IClass iClass = this.findTypeByFullyQualifiedName(location, ss);
        if (iClass == null) {
            this.com.hazelcast.com.ileError("Imported class \"" + Java.join(ss, ".") + "\" could not be loaded", location);
            return this.iClassLoader.TYPE_java_lang_Object;
        }
        return iClass;
    }

    /**
     * Checks if the given simple name was imported through a single type import.
     *
     * @param name The simple type name, e.g. {@code "Inner"}
     * @return     The fully qualified name, e.g. {@code { "pkg", "Outer", "Inner" }}, or {@code null}
     */
    @Nullable public String[]
    getSingleTypeImport(String name, Location location) throws CompileException {

        // Resolve all single type imports (if not already done).
        Map<String, String[]> stis = this.singleTypeImports;
        if (stis == null) {

            // Collect all single type import declarations.
            final List<SingleTypeImportDeclaration> stids = new ArrayList<SingleTypeImportDeclaration>();
            for (ImportDeclaration id : this.com.hazelcast.com.ilationUnit.importDeclarations) {
                id.accept(new ImportVisitor<Void, RuntimeException>() {

                    // SUPPRESS CHECKSTYLE LineLength:4
                    @Override @Nullable public Void visitSingleTypeImportDeclaration(SingleTypeImportDeclaration stid)          { stids.add(stid); return null; }
                    @Override @Nullable public Void visitTypeImportOnDemandDeclaration(TypeImportOnDemandDeclaration tiodd)     { return null; }
                    @Override @Nullable public Void visitSingleStaticImportDeclaration(SingleStaticImportDeclaration ssid)      { return null; }
                    @Override @Nullable public Void visitStaticImportOnDemandDeclaration(StaticImportOnDemandDeclaration siodd) { return null; }
                });
            }

            // Resolve all single type imports.
            stis = new HashMap<String, String[]>();
            for (SingleTypeImportDeclaration stid : stids) {

                String[] ids        = stid.identifiers;
                String   simpleName = UnitCompiler.last(ids);

                // Check for re-import of same simple name.
                String[] prev = (String[]) stis.put(simpleName, ids);
                if (prev != null && !Arrays.equals(prev, ids)) {
                    UnitCompiler.this.com.hazelcast.com.ileError((
                        "Class \"" + simpleName + "\" was previously imported as "
                        + "\"" + Java.join(prev, ".") + "\", now as \"" + Java.join(ids, ".") + "\""
                    ), stid.getLocation());
                }

                if (this.findTypeByFullyQualifiedName(location, ids) == null) {
                    UnitCompiler.this.com.hazelcast.com.ileError(
                        "A class \"" + Java.join(ids, ".") + "\" could not be found",
                        stid.getLocation()
                    );
                }
            }

            this.singleTypeImports = stis;
        }

        return (String[]) stis.get(name);
    }

    /**
     * To be used only by {@link #getSingleTypeImport(String, Location)}; {@code null} means "not yet initialized"
     */
    @Nullable private Map<String /*simpleTypeName*/, String[] /*fullyQualifiedTypeName*/> singleTypeImports;

    /**
     * 6.5.2.BL1.B1.B5, 6.5.2.BL1.B1.B6 Type-import-on-demand.<br>
     * 6.5.5.1.6 Type-import-on-demand declaration.
     *
     * @return {@code null} if the given <var>simpleTypeName</var> cannot be resolved through any of the
     *         import-on-demand directives
     */
    @Nullable public IClass
    importTypeOnDemand(String simpleTypeName, Location location) throws CompileException {

        // Check cache. (A cache for unimportable types is not required, because the class is importable 99.9%.)
        {
            IClass importedClass = (IClass) this.onDemandImportableTypes.get(simpleTypeName);
            if (importedClass != null) return importedClass;
        }
        // Cache miss...

        // Compile all import-on-demand declarations (done here as late as possible).
        Collection<String[]> tiods = this.typeImportsOnDemand;
        if (tiods == null) {
            tiods = new ArrayList<String[]>();
            tiods.add(new String[] { "java", "lang" });
            for (ImportDeclaration id : this.com.hazelcast.com.ilationUnit.importDeclarations) {
                final Collection<String[]> finalTiods = tiods;
                id.accept(new ImportVisitor<Void, RuntimeException>() {

                    // SUPPRESS CHECKSTYLE LineLength:4
                    @Override @Nullable public Void visitTypeImportOnDemandDeclaration(TypeImportOnDemandDeclaration tiodd)     { finalTiods.add(tiodd.identifiers); return null; }
                    @Override @Nullable public Void visitSingleTypeImportDeclaration(SingleTypeImportDeclaration stid)          { return null; }
                    @Override @Nullable public Void visitSingleStaticImportDeclaration(SingleStaticImportDeclaration ssid)      { return null; }
                    @Override @Nullable public Void visitStaticImportOnDemandDeclaration(StaticImportOnDemandDeclaration siodd) { return null; }
                });
            }

            this.typeImportsOnDemand = tiods;
        }

        IClass importedClass = null;
        for (String[] packageComponents : tiods) {
            String[] typeComponents = UnitCompiler.concat(packageComponents, simpleTypeName);
            IClass   iClass         = this.findTypeByFullyQualifiedName(location, typeComponents);
            if (iClass != null) {
                if (importedClass != null && importedClass != iClass) {
                    this.com.hazelcast.com.ileError(
                        "Ambiguous class name: \"" + importedClass + "\" vs. \"" + iClass + "\"",
                        location
                    );
                }
                importedClass = iClass;
            }
        }
        if (importedClass == null) return null;

        // Put in cache and return.
        this.onDemandImportableTypes.put(simpleTypeName, importedClass);
        return importedClass;
    }

    /**
     * To be used only by {@link #importTypeOnDemand(String, Location)}; {@code null} means "not yet initialized.
     */
    @Nullable private Collection<String[]> typeImportsOnDemand;

    /**
     * To be used only by {@link #importTypeOnDemand(String, Location)}; cache for on-demand-imported types.
     */
    private final Map<String /*simpleTypeName*/, IClass> onDemandImportableTypes = new HashMap<String, IClass>();

    private void
    declareClassDollarMethod(ClassLiteral cl) {

        // Method "class$" is not yet declared; declare it like
        //
        //   static java.lang.Class class$(java.lang.String className) {
        //       try {
        //           return java.lang.Class.forName(className);
        //       } catch (java.lang.ClassNotFoundException e) {
        //           throw new java.lang.NoClassDefFoundError(e.getMessage());
        //       }
        //   }
        //
        Location                loc = cl.getLocation();
        AbstractTypeDeclaration declaringType;
        for (Scope s = cl.getEnclosingScope();; s = s.getEnclosingScope()) {
            if (s instanceof AbstractTypeDeclaration) {
                declaringType = (AbstractTypeDeclaration) s;
                break;
            }
        }

        // try {
        // return Class.forName(className);
        final MethodInvocation mi = new MethodInvocation(
            loc,                                                         // location
            new SimpleType(loc, this.iClassLoader.TYPE_java_lang_Class), // optionalTarget
            "forName",                                                   // methodName
            new Rvalue[] {                                               // arguments
                new AmbiguousName(loc, new String[] { "className" })
            }
        );

        IClass classNotFoundExceptionIClass;
        try {
            classNotFoundExceptionIClass = this.iClassLoader.loadIClass("Ljava/lang/ClassNotFoundException;");
        } catch (ClassNotFoundException ex) {
            throw new InternalCompilerException("Loading class \"ClassNotFoundException\": " + ex.getMessage(), ex);
        }
        if (classNotFoundExceptionIClass == null) {
            throw new InternalCompilerException("SNO: Cannot load \"ClassNotFoundException\"");
        }

        IClass noClassDefFoundErrorIClass;
        try {
            noClassDefFoundErrorIClass = this.iClassLoader.loadIClass("Ljava/lang/NoClassDefFoundError;");
        } catch (ClassNotFoundException ex) {
            throw new InternalCompilerException("Loading class \"NoClassDefFoundError\": " + ex.getMessage(), ex);
        }
        if (noClassDefFoundErrorIClass == null) {
            throw new InternalCompilerException("SNO: Cannot load \"NoClassFoundError\"");
        }

        // catch (ClassNotFoundException e) {
        Block b = new Block(loc);
        // throw new NoClassDefFoundError(e.getMessage());
        b.addStatement(new ThrowStatement(loc, new NewClassInstance(
            loc,                                             // location
            (Rvalue) null,                                   // optionalQualification
            new SimpleType(loc, noClassDefFoundErrorIClass), // type
            new Rvalue[] {                                   // arguments
                new MethodInvocation(
                    loc,                                           // location
                    new AmbiguousName(loc, new String[] { "ex" }), // optionalTarget
                    "getMessage",                                  // methodName
                    new Rvalue[0]                                  // arguments
                )
            }
        )));

        CatchClause cc = new CatchClause(
            loc,                 // location
            new FormalParameter( // caughtException
                loc,                                               // location
                true,                                              // finaL
                new SimpleType(loc, classNotFoundExceptionIClass), // type
                "ex"                                               // name
            ),
            b                    // body
        );
        TryStatement ts = new TryStatement(
            loc,                          // location
            new ReturnStatement(loc, mi), // body
            Collections.singletonList(cc) // catchClauses
        );

        List<BlockStatement> statements = new ArrayList<BlockStatement>();
        statements.add(ts);

        // Class class$(String className)
        FormalParameter parameter = new FormalParameter(
            loc,                                                          // location
            false,                                                        // finaL
            new SimpleType(loc, this.iClassLoader.TYPE_java_lang_String), // type
            "className"                                                   // name
        );
        MethodDeclarator cdmd = new MethodDeclarator(
            loc,                                                                   // location
            null,                                                                  // optionalDocComment
            new Modifiers(Mod.STATIC),                                             // modifiers
            null,                                                                  // optionalTypeParameters
            new SimpleType(loc, this.iClassLoader.TYPE_java_lang_Class),           // type
            "class$",                                                              // name
            new FormalParameters(loc, new FormalParameter[] { parameter }, false), // parameters
            new ReferenceType[0],                                                  // thrownExceptions
            null,                                                                  // defaultValue
            statements                                                             // optionalStatements
        );

        declaringType.addDeclaredMethod(cdmd);
        declaringType.invalidateMethodCaches();
    }

    /**
     * @param value A {@link Character}, {@link Byte}, {@link Short}, {@link Integer}, {@link Long}, {@link Float},
     *              {@link Double}, {@link String}, {@link Boolean} or {@code null}
     */
    private IClass
    pushConstant(Locatable locatable, @Nullable Object value) throws CompileException {

        PUSH_INTEGER_CONSTANT: {
            int iv;
            if (value instanceof Character) {
                iv = ((Character) value).charValue();
            } else
            if (value instanceof Byte) {
                iv = ((Byte) value).intValue();
            } else
            if (value instanceof Short) {
                iv = ((Short) value).intValue();
            } else
            if (value instanceof Integer) {
                iv = ((Integer) value).intValue();
            } else
            {
                break PUSH_INTEGER_CONSTANT;
            }

            if (iv >= -1 && iv <= 5) {
                this.writeOpcode(locatable, Opcode.ICONST_0 + iv);
            } else
            if (iv >= Byte.MIN_VALUE && iv <= Byte.MAX_VALUE) {
                this.writeOpcode(locatable, Opcode.BIPUSH);
                this.writeByte((byte) iv);
            } else
            if (iv >= Short.MIN_VALUE && iv <= Short.MAX_VALUE) {
                this.writeOpcode(locatable, Opcode.SIPUSH);
                this.writeShort(iv);
            } else
            {
                this.writeLdc(locatable, this.addConstantIntegerInfo(iv));
            }
            return IClass.INT;
        }

        if (value instanceof Long) {
            long lv = ((Long) value).longValue();

            if (lv == 0L) {
                this.writeOpcode(locatable, Opcode.LCONST_0);
            } else
            if (lv == 1L) {
                this.writeOpcode(locatable, Opcode.LCONST_1);
            } else
            {
                this.writeOpcode(locatable, Opcode.LDC2_W);
                this.writeConstantLongInfo(lv);
            }
            return IClass.LONG;
        }

        if (value instanceof Float) {
            float fv = ((Float) value).floatValue();

            if (
                Float.floatToIntBits(fv) == Float.floatToIntBits(0.0F) // POSITIVE zero!
                || fv == 1.0F
                || fv == 2.0F
            ) {
                this.writeOpcode(locatable, Opcode.FCONST_0 + (int) fv);
            } else
            {
                this.writeLdc(locatable, this.addConstantFloatInfo(fv));
            }
            return IClass.FLOAT;
        }

        if (value instanceof Double) {
            double dv = ((Double) value).doubleValue();

            if (
                Double.doubleToLongBits(dv) == Double.doubleToLongBits(0.0D) // POSITIVE zero!
                || dv == 1.0D
            ) {
                this.writeOpcode(locatable, Opcode.DCONST_0 + (int) dv);
            } else
            {
                this.writeOpcode(locatable, Opcode.LDC2_W);
                this.writeConstantDoubleInfo(dv);
            }
            return IClass.DOUBLE;
        }

        if (value instanceof String) {
            String   s  = (String) value;
            String[] ss = UnitCompiler.makeUtf8Able(s);
            this.writeLdc(locatable, this.addConstantStringInfo(ss[0]));
            for (int i = 1; i < ss.length; ++i) {
                this.writeLdc(locatable, this.addConstantStringInfo(ss[i]));
                this.invoke(locatable, this.iClassLoader.METH_java_lang_String__concat__java_lang_String);
            }
            return this.iClassLoader.TYPE_java_lang_String;
        }

        if (Boolean.TRUE.equals(value)) {
            this.writeOpcode(locatable, Opcode.ICONST_1);
            return IClass.BOOLEAN;
        }

        if (Boolean.FALSE.equals(value)) {
            this.writeOpcode(locatable, Opcode.ICONST_0);
            return IClass.BOOLEAN;
        }

        if (value == null) {
            this.writeOpcode(locatable, Opcode.ACONST_NULL);
            return IClass.VOID;
        }

        throw new InternalCompilerException("Unknown literal \"" + value + "\"");
    }

    /**
     * @return                  0 through 2<sup>32</sup> - 1
     * @throws CompileException The <var>value</var> is larger than {@code ffffffff} (2<sup>32</sup> - 1), or, in
     *                          other words, is longer than eight characters
     * @throws AssertionError   The <var>value</var> contains characters that are <em>not</em> hexadecimal digits
     *                          (0-9, A-F, a-f)
     */
    private static int
    hex2UnsignedInt(Locatable locatable, String value) throws CompileException {

        // Cannot use "Integer.parseInt(String, 16)", because that throws a NumberFormatException
        // if the value is between 2^31 and 2^32-1.

        int result = 0;
        for (int i = 0; i < value.length(); ++i) {
            if ((result & 0xf0000000) != 0) {
                throw UnitCompiler.com.hazelcast.com.ileException(
                    locatable,
                    "Value of hexadecimal integer literal \"" + value + "\" is out of range"
                );
            }
            int digitValue = Character.digit(value.charAt(i), 16);
            assert digitValue >= 0;
            result = (result << 4) + digitValue;
        }
        return result;
    }

    /**
     * @return                  0 through 2<sup>32</sup> - 1
     * @throws CompileException The <var>value</var> is larger than {@code 37777777777} (2<sup>32</sup> - 1)
     * @throws AssertionError   The <var>value</var> contains characters that are <em>not</em> octal digits
     *                          (0-7)
     */
    private static int
    oct2UnsignedInt(Locatable locatable, String value) throws CompileException {

        // Cannot use "Integer.parseInt(String, 8)", because that throws a NumberFormatException if
        // the value is between 2^31 and 2^32-1.

        int result = 0;
        for (int i = 0; i < value.length(); ++i) {
            if ((result & 0xe0000000) != 0) {
                throw UnitCompiler.com.hazelcast.com.ileException(
                    locatable,
                    "Value of octal integer literal \"" + value + "\" is out of range"
                );
            }
            int digitValue = Character.digit(value.charAt(i), 8);
            assert digitValue >= 0;
            result = (result << 3) + digitValue;
        }
        return result;
    }

    /**
     * @return                  0 through 2<sup>32</sup> - 1
     * @throws CompileException The <var>value</var> is larger than {@code 11111111111111111111111111111111}
     *                          (2<sup>32</sup> - 1), or, in other words, is longer than 32 characters
     * @throws AssertionError   The <var>value</var> contains characters other than {@code '0'} and {@code '1'}
     */
    private static int
    bin2UnsignedInt(Locatable locatable, String value) throws CompileException {

        // Cannot use "Integer.parseInt(String, 2)", because that throws a NumberFormatException if
        // the value is between 2^31 and 2^32-1.

        int result = 0;
        for (int i = 0; i < value.length(); ++i) {
            if ((result & 0x80000000) != 0) {
                throw UnitCompiler.com.hazelcast.com.ileException(
                    locatable,
                    "Value of binary integer literal \"" + value + "\" is out of range"
                );
            }
            int digitValue = Character.digit(value.charAt(i), 2);
            assert digitValue >= 0;
            result = (result << 1) + digitValue;
        }
        return result;
    }

    /**
     * @return                  0 through 2<sup>64</sup> - 1
     * @throws CompileException The <var>value</var> is larger than {@code ffffffffffffffff} (2<sup>64</sup> - 1), or,
     *                          in other words, is longer than sixteen characters
     * @throws AssertionError   The <var>value</var> contains characters that are <em>not</em> hexadecimal digits
     *                          (0-9, A-F, a-f)
     */
    private static long
    hex2UnsignedLong(Locatable locatable, String value) throws CompileException {

        // Cannot use "Long.parseLong(String, 16)", because that throws a NumberFormatException
        // if the value is between 2^63 and 2^64-1.

        long result = 0L;
        for (int i = 0; i < value.length(); ++i) {
            if ((result & 0xf000000000000000L) != 0L) {
                throw UnitCompiler.com.hazelcast.com.ileException(
                    locatable,
                    "Value of hexadecimal long literal \"" + value + "\" is out of range"
                );
            }
            int digitValue = Character.digit(value.charAt(i), 16);
            assert digitValue >= 0;
            result = (result << 4) + digitValue;
        }
        return result;
    }

    /**
     * @return                  0 through 2<sup>64</sup> - 1
     * @throws CompileException The <var>value</var> is larger than {@code 1777777777777777777777} (2<sup>64</sup> -
     *                          1)
     * @throws AssertionError   The <var>value</var> contains characters that are <em>not</em> octal digits
     *                          (0-7)
     */
    private static long
    oct2UnsignedLong(Locatable locatable, String value) throws CompileException {

        // Cannot use "Long.parseLong(String, 8)", because that throws a NumberFormatException
        // if the value is between 2^63 and 2^64-1.

        long result = 0L;
        for (int i = 0; i < value.length(); ++i) {
            if ((result & 0xe000000000000000L) != 0) {
                throw UnitCompiler.com.hazelcast.com.ileException(
                    locatable,
                    "Value of octal long literal \"" + value + "\" is out of range"
                );
            }
            int digitValue = Character.digit(value.charAt(i), 8);
            assert digitValue >= 0;
            result = (result << 3) + digitValue;
        }
        return result;
    }

    /**
     * @return                  0 through 2<sup>64</sup> - 1
     * @throws CompileException The <var>value</var> is larger than {@code
     *                          1111111111111111111111111111111111111111111111111111111111111111} (2<sup>64</sup> -
     *                          1), or, in other words, is longer than 64 characters
     * @throws AssertionError   The <var>value</var> contains characters other than {@code '0'} and {@code '1'}
     */
    private static long
    bin2UnsignedLong(Locatable locatable, String value) throws CompileException {

        // Cannot use "Long.parseLong(String, 2)", because that throws a NumberFormatException
        // if the value is between 2^63 and 2^64-1.

        long result = 0L;
        for (int i = 0; i < value.length(); ++i) {
            if ((result & 0x8000000000000000L) != 0) {
                throw UnitCompiler.com.hazelcast.com.ileException(
                    locatable,
                    "Value of binary long literal \"" + value + "\" is out of range"
                );
            }
            int digitValue = Character.digit(value.charAt(i), 2);
            assert digitValue >= 0;
            result = (result << 1) + digitValue;
        }
        return result;
    }

    /**
     * Only strings that can be UTF8-encoded into 65535 bytes can be stored as a constant string info.
     *
     * @param s The string to split into suitable chunks
     * @return  Strings that can be UTF8-encoded into 65535 bytes
     */
    private static String[]
    makeUtf8Able(String s) {
        if (s.length() < (65536 / 3)) return new String[] { s };

        int          sLength = s.length(), utfLength = 0;
        int          from    = 0;
        List<String> l       = new ArrayList<String>();
        for (int i = 0;; i++) {
            if (i == sLength) {
                l.add(s.substring(from));
                break;
            }
            if (utfLength >= 65532) {
                l.add(s.substring(from, i));
                if (i + (65536 / 3) > sLength) {
                    l.add(s.substring(i));
                    break;
                }
                from      = i;
                utfLength = 0;
            }
            int c = s.charAt(i);
            if (c >= 0x0001 && c <= 0x007F) {
                ++utfLength;
            } else
            if (c > 0x07FF) {
                utfLength += 3;
            } else
            {
                utfLength += 2;
            }
        }
        return (String[]) l.toArray(new String[l.size()]);

    }

    private void
    writeLdc(Locatable locatable, short index) {
        if (0 <= index && index <= 255) {
            this.writeOpcode(locatable, Opcode.LDC);
            this.writeByte((byte) index);
        } else {
            this.writeOpcode(locatable, Opcode.LDC_W);
            this.writeShort(index);
        }
    }

    /**
     * Implements "assignment conversion" (JLS7 5.2).
     */
    private void
    assignmentConversion(
        Locatable        locatable,
        IClass           sourceType,
        IClass           targetType,
        @Nullable Object optionalConstantValue
    ) throws CompileException {
        if (!this.tryAssignmentConversion(locatable, sourceType, targetType, optionalConstantValue)) {
            this.com.hazelcast.com.ileError(
                "Assignment conversion not possible from type \"" + sourceType + "\" to type \"" + targetType + "\"",
                locatable.getLocation()
            );
        }
    }

    private boolean
    tryAssignmentConversion(
        Locatable        locatable,
        IClass           sourceType,
        IClass           targetType,
        @Nullable Object optionalConstantValue
    ) throws CompileException {
        UnitCompiler.LOGGER.entering(
            null,
            "tryAssignmentConversion",
            new Object[] { locatable, sourceType, targetType, optionalConstantValue }
        );

        // JLS7 5.1.1 Identity conversion.
        if (this.tryIdentityConversion(sourceType, targetType)) return true;

        // JLS7 5.1.2 Widening primitive conversion.
        if (this.tryWideningPrimitiveConversion(locatable, sourceType, targetType)) return true;

        // JLS7 5.1.4 Widening reference conversion.
        if (this.isWideningReferenceConvertible(sourceType, targetType)) return true;

        // A boxing conversion (JLS7 5.1.7) optionally followed by a widening reference conversion.
        {
            IClass boxedType = this.isBoxingConvertible(sourceType);
            if (boxedType != null) {
                if (this.tryIdentityConversion(boxedType, targetType)) {
                    this.boxingConversion(locatable, sourceType, boxedType);
                    return true;
                }
                if (this.isWideningReferenceConvertible(boxedType, targetType)) {
                    this.boxingConversion(locatable, sourceType, boxedType);
                    return true;
                }
            }
        }

        // An unboxing conversion (JLS7 5.1.8) optionally followed by a widening primitive conversion.
        {
            IClass unboxedType = this.isUnboxingConvertible(sourceType);
            if (unboxedType != null) {
                if (this.tryIdentityConversion(unboxedType, targetType)) {
                    this.unboxingConversion(locatable, sourceType, unboxedType);
                    return true;
                }
                if (this.isWideningPrimitiveConvertible(unboxedType, targetType)) {
                    this.unboxingConversion(locatable, sourceType, unboxedType);
                    this.tryWideningPrimitiveConversion(locatable, unboxedType, targetType);
                    return true;
                }
            }
        }

        // 5.2 Special narrowing primitive conversion.
        if (optionalConstantValue != UnitCompiler.NOT_CONSTANT) {
            if (this.tryConstantAssignmentConversion(
                locatable,
                optionalConstantValue, // constantValue
                targetType             // targetType
            )) return true;
        }

        return false;
    }

    /**
     * Implements "assignment conversion" (JLS7 5.2) on a constant value.
     *
     * @param value Must be a {@link Boolean}, {@link String}, {@link Byte}, {@link Short}, {@link Integer}, {@link
     *              Character}, {@link Long}, {@link Float}, {@link Double} or {@code null}
     * @return      A {@link Boolean}, {@link String}, {@link Byte}, {@link Short}, {@link Integer}, {@link Character},
     *              {@link Long}, {@link Float}, {@link Double} or {@code null}
     */
    @Nullable private Object
    assignmentConversion(Locatable locatable, @Nullable Object value, IClass targetType) throws CompileException {
        if (targetType == IClass.BOOLEAN) {
            if (value instanceof Boolean) return value;
        } else
        if (targetType == this.iClassLoader.TYPE_java_lang_String) {
            if (value instanceof String) return value;
        } else
        if (targetType == IClass.BYTE) {
            if (value instanceof Byte) {
                return value;
            } else
            if (value instanceof Short || value instanceof Integer) {
                assert value != null;
                int x = ((Number) value).intValue();
                if (x >= Byte.MIN_VALUE && x <= Byte.MAX_VALUE) return new Byte((byte) x);
            } else
            if (value instanceof Character) {
                int x = ((Character) value).charValue();
                if (x >= Byte.MIN_VALUE && x <= Byte.MAX_VALUE) return new Byte((byte) x);
            }
        } else
        if (targetType == IClass.SHORT) {
            if (value instanceof Byte) {
                return new Short(((Number) value).shortValue());
            } else
            if (value instanceof Short) {
                return value;
            } else
            if (value instanceof Character) {
                int x = ((Character) value).charValue();
                if (x >= Short.MIN_VALUE && x <= Short.MAX_VALUE) return new Short((short) x);
            } else
            if (value instanceof Integer) {
                int x = ((Integer) value).intValue();
                if (x >= Short.MIN_VALUE && x <= Short.MAX_VALUE) return new Short((short) x);
            }
        } else
        if (targetType == IClass.CHAR) {
            if (value instanceof Short) {
                return value;
            } else
            if (value instanceof Byte || value instanceof Short || value instanceof Integer) {
                assert value != null;
                int x = ((Number) value).intValue();
                if (x >= Character.MIN_VALUE && x <= Character.MAX_VALUE) return new Character((char) x);
            }
        } else
        if (targetType == IClass.INT) {
            if (value instanceof Integer) {
                return value;
            } else
            if (value instanceof Byte || value instanceof Short) {
                assert value != null;
                return new Integer(((Number) value).intValue());
            } else
            if (value instanceof Character) {
                return new Integer(((Character) value).charValue());
            }
        } else
        if (targetType == IClass.LONG) {
            if (value instanceof Long) {
                return value;
            } else
            if (value instanceof Byte || value instanceof Short || value instanceof Integer) {
                assert value != null;
                return new Long(((Number) value).longValue());
            } else
            if (value instanceof Character) {
                return new Long(((Character) value).charValue());
            }
        } else
        if (targetType == IClass.FLOAT) {
            if (value instanceof Float) {
                return value;
            } else
            if (value instanceof Byte || value instanceof Short || value instanceof Integer || value instanceof Long) {
                assert value != null;
                return new Float(((Number) value).floatValue());
            } else
            if (value instanceof Character) {
                return new Float(((Character) value).charValue());
            }
        } else
        if (targetType == IClass.DOUBLE) {
            if (value instanceof Double) {
                return value;
            } else
            if (
                value instanceof Byte
                || value instanceof Short
                || value instanceof Integer
                || value instanceof Long
                || value instanceof Float
            ) {
                assert value != null;
                return new Double(((Number) value).doubleValue());
            } else
            if (value instanceof Character) {
                return new Double(((Character) value).charValue());
            }
        } else
        if (value == null && !targetType.isPrimitive()) {
            return null;
        }

        if (value == null) {
            this.com.hazelcast.com.ileError(
                "Cannot convert 'null' to type \"" + targetType.toString() + "\"",
                locatable.getLocation()
            );
        } else
        {
            this.com.hazelcast.com.ileError((
                "Cannot convert constant of type \""
                + value.getClass().getName()
                + "\" to type \""
                + targetType.toString()
                + "\""
            ), locatable.getLocation());
        }
        return value;
    }

    /**
     * Implements "unary numeric promotion" (JLS7 5.6.1).
     *
     * @return The promoted type
     */
    private IClass
    unaryNumericPromotion(Locatable locatable, IClass type) throws CompileException {
        type = this.convertToPrimitiveNumericType(locatable, type);

        IClass promotedType = this.unaryNumericPromotionType(locatable, type);

        this.numericPromotion(locatable, type, promotedType);
        return promotedType;
    }

    private void
    reverseUnaryNumericPromotion(Locatable locatable, IClass sourceType, IClass targetType) throws CompileException {
        IClass unboxedType = this.isUnboxingConvertible(targetType);
        IClass pt          = unboxedType != null ? unboxedType : targetType;
        if (
            !this.tryIdentityConversion(sourceType, pt)
            && !this.tryNarrowingPrimitiveConversion(
                locatable,  // locatable
                sourceType, // sourceType
                pt          // targetType
            )
        ) throw new InternalCompilerException("SNO: reverse unary numeric promotion failed");
        if (unboxedType != null) this.boxingConversion(locatable, unboxedType, targetType);
    }

    /**
     * If the given type is a primitive type, return that type. If the given type is a primitive wrapper class, unbox
     * the operand on top of the operand stack and return the primitive type. Otherwise, issue a com.hazelcast.com.ile error.
     */
    private IClass
    convertToPrimitiveNumericType(Locatable locatable, IClass type) throws CompileException {
        if (type.isPrimitiveNumeric()) return type;
        IClass unboxedType = this.isUnboxingConvertible(type);
        if (unboxedType != null) {
            this.unboxingConversion(locatable, type, unboxedType);
            return unboxedType;
        }
        this.com.hazelcast.com.ileError(
            "Object of type \"" + type.toString() + "\" cannot be converted to a numeric type",
            locatable.getLocation()
        );
        return type;
    }

    private void
    numericPromotion(Locatable locatable, IClass sourceType, IClass targetType) {
        if (
            !this.tryIdentityConversion(sourceType, targetType)
            && !this.tryWideningPrimitiveConversion(
                locatable,  // locatable
                sourceType, // sourceType
                targetType  // targetType
            )
        ) throw new InternalCompilerException("SNO: Conversion failed");
    }

    private IClass
    unaryNumericPromotionType(Locatable locatable, IClass type) throws CompileException {
        if (!type.isPrimitiveNumeric()) {
            this.com.hazelcast.com.ileError(
                "Unary numeric promotion not possible on non-numeric-primitive type \"" + type + "\"",
                locatable.getLocation()
            );
        }

        return (
            type == IClass.DOUBLE ? IClass.DOUBLE :
            type == IClass.FLOAT  ? IClass.FLOAT  :
            type == IClass.LONG   ? IClass.LONG   :
            IClass.INT
        );
    }

    /**
     * Implements "binary numeric promotion" (5.6.2)
     *
     * @return The promoted type
     */
    private IClass
    binaryNumericPromotion(
        Locatable            locatable,
        IClass               type1,
        CodeContext.Inserter convertInserter1,
        IClass               type2
    ) throws CompileException {
        return this.binaryNumericPromotion(
            locatable,
            type1,
            convertInserter1,
            type2,
            this.getCodeContext().currentInserter()
        );
    }

    /**
     * Implements "binary numeric promotion" (5.6.2), which may perform unboxing conversion.
     *
     * @return The promoted type
     */
    private IClass
    binaryNumericPromotion(
        Locatable                      locatable,
        IClass                         type1,
        @Nullable CodeContext.Inserter convertInserter1,
        IClass                         type2,
        @Nullable CodeContext.Inserter convertInserter2
    ) throws CompileException {
        IClass promotedType;
        {
            IClass c1 = this.isUnboxingConvertible(type1);
            IClass c2 = this.isUnboxingConvertible(type2);
            promotedType = this.binaryNumericPromotionType(
                locatable,
                c1 != null ? c1 : type1,
                c2 != null ? c2 : type2
            );
        }

        if (convertInserter1 != null) {
            this.getCodeContext().pushInserter(convertInserter1);
            try {
                this.numericPromotion(locatable, this.convertToPrimitiveNumericType(locatable, type1), promotedType);
            } finally {
                this.getCodeContext().popInserter();
            }
        }

        if (convertInserter2 != null) {
            this.getCodeContext().pushInserter(convertInserter2);
            try {
                this.numericPromotion(locatable, this.convertToPrimitiveNumericType(locatable, type2), promotedType);
            } finally {
                this.getCodeContext().popInserter();
            }
        }

        return promotedType;
    }

    private IClass
    binaryNumericPromotionType(Locatable locatable, IClass type1, IClass type2) throws CompileException {
        if (!type1.isPrimitiveNumeric() || !type2.isPrimitiveNumeric()) {
            this.com.hazelcast.com.ileError(
                "Binary numeric promotion not possible on types \"" + type1 + "\" and \"" + type2 + "\"",
                locatable.getLocation()
            );
        }

        return (
            type1 == IClass.DOUBLE || type2 == IClass.DOUBLE ? IClass.DOUBLE :
            type1 == IClass.FLOAT  || type2 == IClass.FLOAT  ? IClass.FLOAT  :
            type1 == IClass.LONG   || type2 == IClass.LONG   ? IClass.LONG   :
            IClass.INT
        );
    }

    /**
     * Checks whether "identity conversion" (5.1.1) is possible.
     *
     * @return Whether the conversion is possible
     */
    @SuppressWarnings("static-method") private boolean
    isIdentityConvertible(IClass sourceType, IClass targetType) { return sourceType == targetType; }

    /**
     * Implements "identity conversion" (5.1.1).
     *
     * @return Whether the conversion was possible
     */
    @SuppressWarnings("static-method") private boolean
    tryIdentityConversion(IClass sourceType, IClass targetType) { return sourceType == targetType; }

    @SuppressWarnings("static-method") private boolean
    isWideningPrimitiveConvertible(IClass sourceType, IClass targetType) {
        return UnitCompiler.PRIMITIVE_WIDENING_CONVERSIONS.get(
            sourceType.getDescriptor() + targetType.getDescriptor()
        ) != null;
    }

    /**
     * Implements "widening primitive conversion" (5.1.2).
     *
     * @return Whether the conversion succeeded
     */
    private boolean
    tryWideningPrimitiveConversion(Locatable locatable, IClass sourceType, IClass targetType) {
        byte[] opcodes = (byte[]) UnitCompiler.PRIMITIVE_WIDENING_CONVERSIONS.get(
            sourceType.getDescriptor() + targetType.getDescriptor()
        );
        if (opcodes != null) {
            this.writeOpcodes(locatable, opcodes);
            return true;
        }
        return false;
    }

    private static final Map<String /*descriptor*/, byte[] /*opcodes*/>
    PRIMITIVE_WIDENING_CONVERSIONS = new HashMap<String, byte[]>();

    static { UnitCompiler.fillConversionMap(new Object[] {
        new byte[0],
        Descriptor.BYTE  + Descriptor.SHORT,

        Descriptor.BYTE  + Descriptor.INT,
        Descriptor.SHORT + Descriptor.INT,
        Descriptor.CHAR  + Descriptor.INT,

        new byte[] { Opcode.I2L },
        Descriptor.BYTE  + Descriptor.LONG,
        Descriptor.SHORT + Descriptor.LONG,
        Descriptor.CHAR  + Descriptor.LONG,
        Descriptor.INT   + Descriptor.LONG,

        new byte[] { Opcode.I2F },
        Descriptor.BYTE  + Descriptor.FLOAT,
        Descriptor.SHORT + Descriptor.FLOAT,
        Descriptor.CHAR  + Descriptor.FLOAT,
        Descriptor.INT   + Descriptor.FLOAT,

        new byte[] { Opcode.L2F },
        Descriptor.LONG  + Descriptor.FLOAT,

        new byte[] { Opcode.I2D },
        Descriptor.BYTE  + Descriptor.DOUBLE,
        Descriptor.SHORT + Descriptor.DOUBLE,
        Descriptor.CHAR  + Descriptor.DOUBLE,
        Descriptor.INT   + Descriptor.DOUBLE,

        new byte[] { Opcode.L2D },
        Descriptor.LONG  + Descriptor.DOUBLE,

        new byte[] { Opcode.F2D },
        Descriptor.FLOAT + Descriptor.DOUBLE,
    }, UnitCompiler.PRIMITIVE_WIDENING_CONVERSIONS); }
    private static void
    fillConversionMap(Object[] array, Map<String /*descriptor*/, byte[] /*opcodes*/> map) {
        byte[] opcodes = null;
        for (Object o : array) {
            if (o instanceof byte[]) {
                opcodes = (byte[]) o;
            } else {
                map.put((String) o, opcodes);
            }
        }
    }

    /**
     * Checks if "widening reference conversion" (5.1.4) is possible.
     *
     * @return Whether the conversion is possible
     */
    @SuppressWarnings("static-method") private boolean
    isWideningReferenceConvertible(IClass sourceType, IClass targetType) throws CompileException {
        if (targetType.isPrimitive() || sourceType == targetType) return false;

        return targetType.isAssignableFrom(sourceType);
    }

    /**
     * Performs "widening reference conversion" (5.1.4) if possible.
     *
     * @return Whether the conversion was possible
     */
    @SuppressWarnings("static-method") private boolean
    tryWideningReferenceConversion(IClass sourceType, IClass targetType) throws CompileException {
        if (targetType.isPrimitive() || sourceType == targetType) return false;

        return targetType.isAssignableFrom(sourceType);
    }

    /**
     * Checks whether "narrowing primitive conversion" (JLS7 5.1.3) is possible.
     */
    @SuppressWarnings("static-method") private boolean
    isNarrowingPrimitiveConvertible(IClass sourceType, IClass targetType) {
        return UnitCompiler.PRIMITIVE_NARROWING_CONVERSIONS.containsKey(
            sourceType.getDescriptor() + targetType.getDescriptor()
        );
    }

    /**
     * Implements "narrowing primitive conversion" (JLS7 5.1.3).
     *
     * @return Whether the conversion succeeded
     */
    private boolean
    tryNarrowingPrimitiveConversion(Locatable locatable, IClass sourceType, IClass targetType) {
        byte[] opcodes = (byte[]) UnitCompiler.PRIMITIVE_NARROWING_CONVERSIONS.get(
            sourceType.getDescriptor() + targetType.getDescriptor()
        );
        if (opcodes != null) {
            this.writeOpcodes(locatable, opcodes);
            return true;
        }
        return false;
    }

    private static final Map<String /*descriptor*/, byte[] /*opcodes*/>
    PRIMITIVE_NARROWING_CONVERSIONS = new HashMap<String, byte[]>();

    static { UnitCompiler.fillConversionMap(new Object[] {
        new byte[0],
        Descriptor.BYTE + Descriptor.CHAR,
        Descriptor.SHORT + Descriptor.CHAR,
        Descriptor.CHAR + Descriptor.SHORT,

        new byte[] { Opcode.I2B },
        Descriptor.SHORT + Descriptor.BYTE,
        Descriptor.CHAR + Descriptor.BYTE,
        Descriptor.INT + Descriptor.BYTE,

        new byte[] { Opcode.I2S },
        Descriptor.INT + Descriptor.SHORT,
        Descriptor.INT + Descriptor.CHAR,

        new byte[] { Opcode.L2I, Opcode.I2B },
        Descriptor.LONG + Descriptor.BYTE,

        new byte[] { Opcode.L2I, Opcode.I2S },
        Descriptor.LONG + Descriptor.SHORT,
        Descriptor.LONG + Descriptor.CHAR,

        new byte[] { Opcode.L2I },
        Descriptor.LONG + Descriptor.INT,

        new byte[] { Opcode.F2I, Opcode.I2B },
        Descriptor.FLOAT + Descriptor.BYTE,

        new byte[] { Opcode.F2I, Opcode.I2S },
        Descriptor.FLOAT + Descriptor.SHORT,
        Descriptor.FLOAT + Descriptor.CHAR,

        new byte[] { Opcode.F2I },
        Descriptor.FLOAT + Descriptor.INT,

        new byte[] { Opcode.F2L },
        Descriptor.FLOAT + Descriptor.LONG,

        new byte[] { Opcode.D2I, Opcode.I2B },
        Descriptor.DOUBLE + Descriptor.BYTE,

        new byte[] { Opcode.D2I, Opcode.I2S },
        Descriptor.DOUBLE + Descriptor.SHORT,
        Descriptor.DOUBLE + Descriptor.CHAR,

        new byte[] { Opcode.D2I },
        Descriptor.DOUBLE + Descriptor.INT,

        new byte[] { Opcode.D2L },
        Descriptor.DOUBLE + Descriptor.LONG,

        new byte[] { Opcode.D2F },
        Descriptor.DOUBLE + Descriptor.FLOAT,
    }, UnitCompiler.PRIMITIVE_NARROWING_CONVERSIONS); }

    /**
     * Checks if "constant assignment conversion" (JLS7 5.2, paragraph 1) is possible.
     *
     * @param constantValue The constant value that is to be converted
     * @param targetType    The type to convert to
     */
    private boolean
    tryConstantAssignmentConversion(Locatable locatable, @Nullable Object constantValue, IClass targetType)
    throws CompileException {
        UnitCompiler.LOGGER.entering(
            null,
            "tryConstantAssignmentConversion",
            new Object[] { locatable, constantValue, targetType }
        );

        int cv;
        if (constantValue instanceof Byte) {
            cv = ((Byte) constantValue).byteValue();
        } else
        if (constantValue instanceof Short) {
            cv = ((Short) constantValue).shortValue();
        } else
        if (constantValue instanceof Integer) {
            cv = ((Integer) constantValue).intValue();
        } else
        if (constantValue instanceof Character) {
            cv = ((Character) constantValue).charValue();
        } else
        {
            return false;
        }

        if (targetType == IClass.BYTE)  return cv >= Byte.MIN_VALUE && cv <= Byte.MAX_VALUE;
        if (targetType == IClass.SHORT) return cv >= Short.MIN_VALUE && cv <= Short.MAX_VALUE;
        if (targetType == IClass.CHAR)  return cv >= Character.MIN_VALUE && cv <= Character.MAX_VALUE;

        IClassLoader icl = this.iClassLoader;
        if (targetType == icl.TYPE_java_lang_Byte && cv >= Byte.MIN_VALUE && cv <= Byte.MAX_VALUE) {
            this.boxingConversion(locatable, IClass.BYTE, targetType);
            return true;
        }
        if (targetType == icl.TYPE_java_lang_Short && cv >= Short.MIN_VALUE && cv <= Short.MAX_VALUE) {
            this.boxingConversion(locatable, IClass.SHORT, targetType);
            return true;
        }
        if (targetType == icl.TYPE_java_lang_Character && cv >= Character.MIN_VALUE && cv <= Character.MAX_VALUE) {
            this.boxingConversion(locatable, IClass.CHAR, targetType);
            return true;
        }

        return false;
    }

    /**
     * Checks whether "narrowing reference conversion" (JLS7 5.1.5) is possible.
     */
    private boolean
    isNarrowingReferenceConvertible(IClass sourceType, IClass targetType) throws CompileException {
        if (sourceType.isPrimitive()) return false;
        if (sourceType == targetType) return false;

        // 5.1.5.1
        if (sourceType.isAssignableFrom(targetType)) return true;

        // 5.1.5.2
        if (targetType.isInterface() && !sourceType.isFinal() && !targetType.isAssignableFrom(sourceType)) return true;

        // 5.1.5.3
        if (sourceType == this.iClassLoader.TYPE_java_lang_Object && targetType.isArray()) return true;

        // 5.1.5.4
        if (sourceType == this.iClassLoader.TYPE_java_lang_Object && targetType.isInterface()) return true;

        // 5.1.5.5
        if (sourceType.isInterface() && !targetType.isFinal()) return true;

        // 5.1.5.6
        if (sourceType.isInterface() && targetType.isFinal() && sourceType.isAssignableFrom(targetType)) return true;

        // 5.1.5.7
        // TODO: Check for redefinition of methods with same signature but different return type.
        if (sourceType.isInterface() && targetType.isInterface() && !targetType.isAssignableFrom(sourceType)) {
            return true;
        }

        // 5.1.5.8
        if (sourceType.isArray() && targetType.isArray()) {

            IClass st = sourceType.getComponentType();
            assert st != null;

            IClass tt = targetType.getComponentType();
            assert tt != null;

            if (this.isNarrowingPrimitiveConvertible(st, tt) || this.isNarrowingReferenceConvertible(st, tt)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Implements "narrowing reference conversion" (5.1.5).
     *
     * @return Whether the conversion succeeded
     */
    private boolean
    tryNarrowingReferenceConversion(Locatable locatable, IClass sourceType, IClass targetType) throws CompileException {
        if (!this.isNarrowingReferenceConvertible(sourceType, targetType)) return false;

        this.writeOpcode(locatable, Opcode.CHECKCAST);
        this.writeConstantClassInfo(targetType.getDescriptor());
        return true;
    }

    /**
     * JLS7 5.5
     */
    private boolean
    isCastReferenceConvertible(IClass sourceType, IClass targetType) throws CompileException {
        return (
            this.isIdentityConvertible(sourceType, targetType)
            || this.isWideningReferenceConvertible(sourceType, targetType)
            || this.isNarrowingReferenceConvertible(sourceType, targetType)
        );
    }

    /**
     * @return The boxed type or {@code null}
     */
    @Nullable private IClass
    isBoxingConvertible(IClass sourceType) {
        IClassLoader icl = this.iClassLoader;
        if (sourceType == IClass.BOOLEAN) return icl.TYPE_java_lang_Boolean;
        if (sourceType == IClass.BYTE)    return icl.TYPE_java_lang_Byte;
        if (sourceType == IClass.CHAR)    return icl.TYPE_java_lang_Character;
        if (sourceType == IClass.SHORT)   return icl.TYPE_java_lang_Short;
        if (sourceType == IClass.INT)     return icl.TYPE_java_lang_Integer;
        if (sourceType == IClass.LONG)    return icl.TYPE_java_lang_Long;
        if (sourceType == IClass.FLOAT)   return icl.TYPE_java_lang_Float;
        if (sourceType == IClass.DOUBLE)  return icl.TYPE_java_lang_Double;
        return null;
    }

    private boolean
    tryBoxingConversion(Locatable locatable, IClass sourceType, IClass targetType) throws CompileException {
        if (this.isBoxingConvertible(sourceType) == targetType) {
            this.boxingConversion(locatable, sourceType, targetType);
            return true;
        }
        return false;
    }

    /**
     * @param sourceType a primitive type (except VOID)
     * @param targetType the corresponding wrapper type
     */
    private void
    boxingConversion(Locatable locatable, IClass sourceType, IClass targetType) throws CompileException {

        // In some pre-1.5 JDKs, only some wrapper classes have the static "Target.valueOf(source)" method.
        if (targetType.hasIMethod("valueOf", new IClass[] { sourceType })) {
            this.writeOpcode(locatable, Opcode.INVOKESTATIC);
            this.writeConstantMethodrefInfo(
                targetType.getDescriptor(), // classFD
                "valueOf",                  // methodName
                new MethodDescriptor(       // methodFD
                    targetType.getDescriptor(), // returnFd
                    sourceType.getDescriptor()  // parameterFds...
                )
            );
            return;
        }
        // new Target(source)
        this.writeOpcode(locatable, Opcode.NEW);
        this.writeConstantClassInfo(targetType.getDescriptor());
        if (Descriptor.hasSize2(sourceType.getDescriptor())) {
            this.writeOpcode(locatable, Opcode.DUP_X2);
            this.writeOpcode(locatable, Opcode.DUP_X2);
            this.writeOpcode(locatable, Opcode.POP);
        } else
        {
            this.writeOpcode(locatable, Opcode.DUP_X1);
            this.writeOpcode(locatable, Opcode.SWAP);
        }
        this.writeOpcode(locatable, Opcode.INVOKESPECIAL);
        this.writeConstantMethodrefInfo(
            targetType.getDescriptor(), // classFd
            "<init>",                   // methodName
            new MethodDescriptor(       // methodFD
                Descriptor.VOID,           // returnFd
                sourceType.getDescriptor() // parameterFds...
            )
        );
    }

    /**
     * @return Iff <var>sourceType</var> is a primitive wrapper type, the unboxed type, otherwise {@code null}
     */
    @Nullable private IClass
    isUnboxingConvertible(IClass sourceType) {
        IClassLoader icl = this.iClassLoader;
        if (sourceType == icl.TYPE_java_lang_Boolean)   return IClass.BOOLEAN;
        if (sourceType == icl.TYPE_java_lang_Byte)      return IClass.BYTE;
        if (sourceType == icl.TYPE_java_lang_Character) return IClass.CHAR;
        if (sourceType == icl.TYPE_java_lang_Short)     return IClass.SHORT;
        if (sourceType == icl.TYPE_java_lang_Integer)   return IClass.INT;
        if (sourceType == icl.TYPE_java_lang_Long)      return IClass.LONG;
        if (sourceType == icl.TYPE_java_lang_Float)     return IClass.FLOAT;
        if (sourceType == icl.TYPE_java_lang_Double)    return IClass.DOUBLE;
        return null;
    }

    /**
     * @return Whether the <var>sourceType</var> is a primitive numeric type, or a wrapper type of a primitive numeric
     *         type
     */
    private boolean
    isConvertibleToPrimitiveNumeric(IClass sourceType) {
        if (sourceType.isPrimitiveNumeric()) return true;
        IClass unboxedType = this.isUnboxingConvertible(sourceType);
        return unboxedType != null && unboxedType.isPrimitiveNumeric();
    }

    private boolean
    tryUnboxingConversion(
        Locatable          locatable,
        IClass             sourceType,
        IClass             targetType,
        @Nullable Inserter optionalInserter
    ) {
        if (this.isUnboxingConvertible(sourceType) == targetType) {
            this.unboxingConversion(locatable, sourceType, targetType, optionalInserter);
            return true;
        }
        return false;
    }

    /**
     * @param targetType a primitive type (except VOID)
     * @param sourceType the corresponding wrapper type
     */
    private void
    unboxingConversion(Locatable locatable, IClass sourceType, IClass targetType, @Nullable Inserter optionalInserter) {
        if (optionalInserter == null) {
            this.unboxingConversion(locatable, sourceType, targetType);
        } else {
            this.getCodeContext().pushInserter(optionalInserter);
            try {
                this.unboxingConversion(locatable, sourceType, targetType);
            } finally {
                this.getCodeContext().popInserter();
            }
        }
    }

    /**
     * @param targetType a primitive type (except VOID)
     * @param sourceType the corresponding wrapper type
     */
    private void
    unboxingConversion(Locatable locatable, IClass sourceType, IClass targetType) {

        // "source.targetValue()"
        this.writeOpcode(locatable, Opcode.INVOKEVIRTUAL);
        this.writeConstantMethodrefInfo(
            sourceType.getDescriptor(),                                     // classFD
            targetType.toString() + "Value",                                // methodName
            new MethodDescriptor(targetType.getDescriptor(), new String[0]) // methodFD
        );
    }

    /**
     * Attempts to load an {@link IClass} by fully-qualified name through {@link #iClassLoader}.
     *
     * @param identifiers       The fully qualified type name, e.g. '{@code { "pkg", "Outer", "Inner" }}'
     * @return                  {@code null} if a class with the given name could not be loaded
     * @throws CompileException The type exists, but a problem occurred when it was loaded
     */
    @Nullable private IClass
    findTypeByFullyQualifiedName(Location location, String[] identifiers) throws CompileException {

        // Try all 'flavors', i.e. 'a.b.c', 'a.b$c', 'a$b$c'.
        String className = Java.join(identifiers, ".");
        for (;;) {
            IClass iClass = UnitCompiler.this.findTypeByName(location, className);
            if (iClass != null) return iClass;

            int idx = className.lastIndexOf('.');
            if (idx == -1) break;
            className = className.substring(0, idx) + '$' + className.substring(idx + 1);
        }

        return null;
    }

    // Load the value of a local variable onto the stack and return its type.
    private IClass
    load(Locatable locatable, LocalVariable localVariable) {
        this.load(locatable, localVariable.type, localVariable.getSlotIndex());
        return localVariable.type;
    }
    private void
    load(Locatable locatable, IClass type, int index) {
        if (index <= 3) {
            this.writeOpcode(locatable, Opcode.ILOAD_0 + 4 * UnitCompiler.ilfda(type) + index);
        } else
        if (index <= 255) {
            this.writeOpcode(locatable, Opcode.ILOAD + UnitCompiler.ilfda(type));
            this.writeByte(index);
        } else
        {
            this.writeOpcode(locatable, Opcode.WIDE);
            this.writeOpcode(locatable, Opcode.ILOAD + UnitCompiler.ilfda(type));
            this.writeShort(index);
        }
    }

    /**
     * Assigns top stack top value to the given local variable.
     */
    private void
    store(Locatable locatable, LocalVariable localVariable) {
        this.store(
            locatable,                   // locatable
            localVariable.type,          // lvType
            localVariable.getSlotIndex() // lvIndex
        );
    }
    private void
    store(Locatable locatable, IClass lvType, short lvIndex) {
        if (lvIndex <= 3) {
            this.writeOpcode(locatable, Opcode.ISTORE_0 + 4 * UnitCompiler.ilfda(lvType) + lvIndex);
        } else
        if (lvIndex <= 255) {
            this.writeOpcode(locatable, Opcode.ISTORE + UnitCompiler.ilfda(lvType));
            this.writeByte(lvIndex);
        } else
        {
            this.writeOpcode(locatable, Opcode.WIDE);
            this.writeOpcode(locatable, Opcode.ISTORE + UnitCompiler.ilfda(lvType));
            this.writeShort(lvIndex);
        }
    }

    private void
    getfield(Locatable locatable, IClass.IField iField) throws CompileException {
        this.writeOpcode(locatable, iField.isStatic() ? Opcode.GETSTATIC : Opcode.GETFIELD);
        this.writeConstantFieldrefInfo(
            iField.getDeclaringIClass().getDescriptor(), // classFD
            iField.getName(),                            // fieldName
            iField.getDescriptor()                       // fieldFD
        );
    }

    private void
    putfield(Locatable locatable, IField iField) throws CompileException {
        this.writeOpcode(locatable, iField.isStatic() ? Opcode.PUTSTATIC : Opcode.PUTFIELD);
        this.writeConstantFieldrefInfo(
            iField.getDeclaringIClass().getDescriptor(), // classFD
            iField.getName(),                            // fieldName
            iField.getDescriptor()                       // fieldFD
        );
    }

    private void
    dup(Locatable locatable, int n) {
        switch (n) {

        case 0:
            ;
            break;

        case 1:
            this.writeOpcode(locatable, Opcode.DUP);
            break;

        case 2:
            this.writeOpcode(locatable, Opcode.DUP2);
            break;

        default:
            throw new InternalCompilerException("dup(" + n + ")");
        }
    }
    private void
    dupx(Locatable locatable, IClass type, int x) {
        if (x < 0 || x > 2) throw new InternalCompilerException("SNO: x has value " + x);
        int dup  = Opcode.DUP  + x;
        int dup2 = Opcode.DUP2 + x;
        this.writeOpcode(locatable, type == IClass.LONG || type == IClass.DOUBLE ? dup2 : dup);
    }

    private void
    pop(Locatable locatable, IClass type) {
        if (type == IClass.VOID) return;
        this.writeOpcode(locatable, type == IClass.LONG || type == IClass.DOUBLE ? Opcode.POP2 : Opcode.POP);
    }

    private static int
    ilfd(final IClass t) {
        if (t == IClass.BYTE || t == IClass.CHAR || t == IClass.INT || t == IClass.SHORT || t == IClass.BOOLEAN) {
            return 0;
        }
        if (t == IClass.LONG)   return 1;
        if (t == IClass.FLOAT)  return 2;
        if (t == IClass.DOUBLE) return 3;
        throw new InternalCompilerException("Unexpected type \"" + t + "\"");
    }

    private static int
    ilfd(IClass t, int opcodeInt, int opcodeLong, int opcodeFloat, int opcodeDouble) {
        if (t == IClass.BYTE || t == IClass.CHAR || t == IClass.INT || t == IClass.SHORT || t == IClass.BOOLEAN) {
            return opcodeInt;
        }
        if (t == IClass.LONG)   return opcodeLong;
        if (t == IClass.FLOAT)  return opcodeFloat;
        if (t == IClass.DOUBLE) return opcodeDouble;
        throw new InternalCompilerException("Unexpected type \"" + t + "\"");
    }

    private static int
    ilfda(IClass t) { return !t.isPrimitive() ? 4 : UnitCompiler.ilfd(t); }

    private static int
    ilfdabcs(IClass t) {
        if (t == IClass.INT)     return 0;
        if (t == IClass.LONG)    return 1;
        if (t == IClass.FLOAT)   return 2;
        if (t == IClass.DOUBLE)  return 3;
        if (!t.isPrimitive())    return 4;
        if (t == IClass.BOOLEAN) return 5;
        if (t == IClass.BYTE)    return 5;
        if (t == IClass.CHAR)    return 6;
        if (t == IClass.SHORT)   return 7;
        throw new InternalCompilerException("Unexpected type \"" + t + "\"");
    }

    /**
     * Invokes the <var>iMethod</var>; assumes that {@code this} (unless <var>iMethod</var> is static) and the correct
     * number and types of arguments are on the operand stack.
     */
    private void
    invoke(Locatable locatable, IMethod iMethod) throws CompileException {
        if (iMethod.getDeclaringIClass().isInterface() && !iMethod.isStatic()) {
            this.writeOpcode(locatable, Opcode.INVOKEINTERFACE);
            this.writeConstantInterfaceMethodrefInfo(
                iMethod.getDeclaringIClass().getDescriptor(), // classFD
                iMethod.getName(),                            // methodName
                iMethod.getDescriptor()                       // methodMD
            );
            int count = 1;
            for (IClass pt : iMethod.getParameterTypes()) count += Descriptor.size(pt.getDescriptor());
            this.writeByte(count);
            this.writeByte(0);
        } else {
            this.writeOpcode(locatable, iMethod.isStatic() ? Opcode.INVOKESTATIC : Opcode.INVOKEVIRTUAL);
            this.writeConstantMethodrefInfo(
                iMethod.getDeclaringIClass().getDescriptor(), // classFD
                iMethod.getName(),                            // methodName
                iMethod.getDescriptor()                       // methodMD
            );
        }
    }

    /**
     * Invokes the <var>iConstructor</var>; assumes that {@code this} and the correct number and types of arguments are
     * on the operand stack.
     */
    private void
    invoke(Locatable locatable, IConstructor iConstructor) throws CompileException {
        this.writeOpcode(locatable, Opcode.INVOKESPECIAL);
        this.writeConstantMethodrefInfo(
            iConstructor.getDeclaringIClass().getDescriptor(), // classFD
            "<init>",                                          // methodName
            iConstructor.getDescriptor()                       // methodMD
        );
    }

    /**
     * Finds a named field in the given {@link IClass}. Honors superclasses and interfaces. See JLS7 8.3.
     *
     * @return {@code null} if no field is found
     */
    @Nullable private IClass.IField
    findIField(IClass iClass, String name, Location location) throws CompileException {

        // Search for a field with the given name in the current class.
        IClass.IField f = iClass.getDeclaredIField(name);
        if (f != null) return f;

        // Examine superclass.
        {
            IClass superclass = iClass.getSuperclass();
            if (superclass != null) f = this.findIField(superclass, name, location);
        }

        // Examine interfaces.
        IClass[] ifs = iClass.getInterfaces();
        for (IClass iF : ifs) {
            IClass.IField f2 = this.findIField(iF, name, location);
            if (f2 != null) {
                if (f != null) {
                    throw new CompileException((
                        "Access to field \""
                        + name
                        + "\" is ambiguous - both \""
                        + f.getDeclaringIClass()
                        + "\" and \""
                        + f2.getDeclaringIClass()
                        + "\" declare it"
                    ), location);
                }
                f = f2;
            }
        }
        return f;
    }

    /**
     * Finds a named type in the given {@link IClass}. Honors superclasses, interfaces and enclosing type declarations.
     *
     * @return {@code null} if no type with the given name is found
     */
    @Nullable private IClass
    findMemberType(IClass iClass, String name, Location location) throws CompileException {
        IClass[] types = iClass.findMemberType(name);
        if (types.length == 0) return null;
        if (types.length == 1) return types[0];

        StringBuilder sb = new StringBuilder("Type \"").append(name).append("\" is ambiguous: ").append(types[0]);
        for (int i = 1; i < types.length; ++i) sb.append(" vs. ").append(types[i].toString());
        this.com.hazelcast.com.ileError(sb.toString(), location);
        return types[0];
    }

    /**
     * Finds one class or interface declared in this com.hazelcast.com.ilation unit by name.
     *
     * @param className Fully qualified class name, e.g. "pkg1.pkg2.Outer$Inner"
     * @return          {@code null} if a class or an interface with that name is not declared in this com.hazelcast.com.ilation unit
     */
    @Nullable public IClass
    findClass(String className) {

        // Examine package name.
        PackageDeclaration opd = this.com.hazelcast.com.ilationUnit.optionalPackageDeclaration;
        if (opd != null) {
            String packageName = opd.packageName;
            if (!className.startsWith(packageName + '.')) return null;
            className = className.substring(packageName.length() + 1);
        }

        StringTokenizer st = new StringTokenizer(className, "$");
        TypeDeclaration td = this.com.hazelcast.com.ilationUnit.getPackageMemberTypeDeclaration(st.nextToken());
        if (td == null) return null;
        while (st.hasMoreTokens()) {
            td = td.getMemberTypeDeclaration(st.nextToken());
            if (td == null) return null;
        }
        return this.resolve(td);
    }

    /**
     * Equivalent with {@link #com.hazelcast.com.ileError(String, Location)} with a {@code null} location argument.
     */
    private void
    com.hazelcast.com.ileError(String message) throws CompileException { this.com.hazelcast.com.ileError(message, null); }

    /**
     * Issues a com.hazelcast.com.ile error with the given message. This is done through the {@link ErrorHandler} that was installed
     * through {@link #setCompileErrorHandler(ErrorHandler)}. Such a handler typically throws a {@link
     * CompileException}, but it may as well decide to return normally. Consequently, the calling code must be prepared
     * that {@link #com.hazelcast.com.ileError(String, Location)} returns normally, and must attempt to continue com.hazelcast.com.iling.
     *
     * @param message          The message to report
     * @param optionalLocation The location to report
     */
    private void
    com.hazelcast.com.ileError(String message, @Nullable Location optionalLocation) throws CompileException {
        ++this.com.hazelcast.com.ileErrorCount;
        if (this.optionalCompileErrorHandler != null) {
            this.optionalCompileErrorHandler.handleError(message, optionalLocation);
        } else {
            throw new CompileException(message, optionalLocation);
        }
    }

    /**
     * Issues a warning with the given message an location an returns. This is done through a {@link WarningHandler}
     * that was installed through {@link #setWarningHandler(WarningHandler)}.
     * <p>
     *   The <var>handle</var> argument qualifies the warning and is typically used by the {@link WarningHandler} to
     *   suppress individual warnings.
     * </p>
     */
    private void
    warning(String handle, String message, @Nullable Location optionalLocation) throws CompileException {
        if (this.optionalWarningHandler != null) {
            this.optionalWarningHandler.handleWarning(handle, message, optionalLocation);
        }
    }

    /**
     * By default, {@link CompileException}s are thrown on com.hazelcast.com.ile errors, but an application my install its own
     * (thread-local) {@link ErrorHandler}.
     * <p>
     *   Be aware that a single problem during com.hazelcast.com.ilation often causes a bunch of com.hazelcast.com.ile errors, so a good {@link
     *   ErrorHandler} counts errors and throws a {@link CompileException} when a limit is reached.
     * </p>
     * <p>
     *   If the given {@link ErrorHandler} does not throw {@link CompileException}s, then {@link
     *   #com.hazelcast.com.ileUnit(boolean, boolean, boolean)} will throw one when the com.hazelcast.com.ilation of the unit is finished, and
     *   errors had occurred. In other words: The {@link ErrorHandler} may throw a {@link CompileException} or not, but
     *   {@link #com.hazelcast.com.ileUnit(boolean, boolean, boolean)} will definitely throw a {@link CompileException} if one or
     *   more com.hazelcast.com.ile errors have occurred.
     * </p>
     *
     * @param optionalCompileErrorHandler {@code null} to restore the default behavior (throwing a {@link
     *                                    CompileException}
     */
    public void
    setCompileErrorHandler(@Nullable ErrorHandler optionalCompileErrorHandler) {
        this.optionalCompileErrorHandler = optionalCompileErrorHandler;
    }

    /**
     * By default, warnings are discarded, but an application my install a custom {@link WarningHandler}.
     *
     * @param optionalWarningHandler {@code null} to indicate that no warnings be issued
     */
    public void
    setWarningHandler(@Nullable WarningHandler optionalWarningHandler) {
        this.optionalWarningHandler = optionalWarningHandler;
    }

    @Nullable private CodeContext
    replaceCodeContext(@Nullable CodeContext newCodeContext) {
        CodeContext oldCodeContext = this.codeContext;
        this.codeContext = newCodeContext;
        return oldCodeContext;
    }

    private void
    writeByte(int v) {
        if (v > Byte.MAX_VALUE - Byte.MIN_VALUE) {
            throw new InternalCompilerException("Byte value out of legal range");
        }
        this.getCodeContext().write(-1, (byte) v);
    }
    private void
    writeShort(int v) {
        if (v > Short.MAX_VALUE - Short.MIN_VALUE) {
            throw new InternalCompilerException("Short value out of legal range");
        }
        this.getCodeContext().write(-1, (byte) (v >> 8), (byte) v);
    }
    private void
    writeInt(int v) {
        this.getCodeContext().write(-1, (byte) (v >> 24), (byte) (v >> 16), (byte) (v >> 8), (byte) v);
    }

    private void
    writeOpcode(Locatable locatable, int opcode) {
        this.getCodeContext().write(locatable.getLocation().getLineNumber(), (byte) opcode);
    }

    private void
    writeOpcodes(Locatable locatable, byte[] opcodes) {
        this.getCodeContext().write(locatable.getLocation().getLineNumber(), opcodes);
    }

    private void
    writeBranch(Locatable locatable, int opcode, final CodeContext.Offset dst) {
        this.getCodeContext().writeBranch(locatable.getLocation().getLineNumber(), opcode, dst);
    }

    private void
    writeOffset(CodeContext.Offset src, final CodeContext.Offset dst) {
        this.getCodeContext().writeOffset(-1, src, dst);
    }

    // Wrappers for "ClassFile.addConstant...Info()". Saves us some coding overhead.

    private void
    writeConstantClassInfo(String descriptor) {
        CodeContext ca = this.getCodeContext();
        ca.writeShort(ca.getClassFile().addConstantClassInfo(descriptor));
    }
    private void
    writeConstantFieldrefInfo(String classFd, String fieldName, String fieldFd) {
        CodeContext ca = this.getCodeContext();
        ca.writeShort(ca.getClassFile().addConstantFieldrefInfo(classFd, fieldName, fieldFd));
    }
    private void
    writeConstantMethodrefInfo(String classFd, String methodName, MethodDescriptor methodMd) {
        CodeContext ca = this.getCodeContext();
        ca.writeShort(ca.getClassFile().addConstantMethodrefInfo(classFd, methodName, methodMd.toString()));
    }
    private void
    writeConstantInterfaceMethodrefInfo(String classFd, String methodName, MethodDescriptor methodMd) {
        CodeContext ca = this.getCodeContext();
        ca.writeShort(ca.getClassFile().addConstantInterfaceMethodrefInfo(classFd, methodName, methodMd.toString()));
    }
/* UNUSED
    private void writeConstantStringInfo(String value) {
        this.codeContext.writeShort(-1, this.addConstantStringInfo(value));
    }
*/
    private short
    addConstantStringInfo(String value) {
        return this.getCodeContext().getClassFile().addConstantStringInfo(value);
    }
/* UNUSED
    private void writeConstantIntegerInfo(int value) {
        this.codeContext.writeShort(-1, this.addConstantIntegerInfo(value));
    }
*/
    private short
    addConstantIntegerInfo(int value) {
        return this.getCodeContext().getClassFile().addConstantIntegerInfo(value);
    }
/* UNUSED
    private void writeConstantFloatInfo(float value) {
        this.codeContext.writeShort(-1, this.addConstantFloatInfo(value));
    }
*/
    private short
    addConstantFloatInfo(float value) {
        return this.getCodeContext().getClassFile().addConstantFloatInfo(value);
    }
    private void
    writeConstantLongInfo(long value) {
        CodeContext ca = this.getCodeContext();
        ca.writeShort(ca.getClassFile().addConstantLongInfo(value));
    }
    private void
    writeConstantDoubleInfo(double value) {
        CodeContext ca = this.getCodeContext();
        ca.writeShort(ca.getClassFile().addConstantDoubleInfo(value));
    }

    private CodeContext.Offset
    getWhereToBreak(BreakableStatement bs) {

        if (bs.whereToBreak != null) return bs.whereToBreak;

        return (bs.whereToBreak = this.getCodeContext().new Offset());
    }

    private TypeBodyDeclaration
    getDeclaringTypeBodyDeclaration(QualifiedThisReference qtr) throws CompileException {

        if (qtr.declaringTypeBodyDeclaration != null) return qtr.declaringTypeBodyDeclaration;

        // Compile error if in static function context.
        Scope s;
        for (
            s = qtr.getEnclosingScope();
            !(s instanceof TypeBodyDeclaration);
            s = s.getEnclosingScope()
        );
        TypeBodyDeclaration result = (TypeBodyDeclaration) s;

        if (result.isStatic()) {
            this.com.hazelcast.com.ileError("No current instance available in static method", qtr.getLocation());
        }

        // Determine declaring type.
        qtr.declaringClass = (AbstractClassDeclaration) result.getDeclaringType();

        return (qtr.declaringTypeBodyDeclaration = result);
    }

    private AbstractClassDeclaration
    getDeclaringClass(QualifiedThisReference qtr) throws CompileException {

        if (qtr.declaringClass != null) return qtr.declaringClass;

        this.getDeclaringTypeBodyDeclaration(qtr);
        assert qtr.declaringClass != null;
        return qtr.declaringClass;
    }

    private void
    referenceThis(Locatable locatable) { this.writeOpcode(locatable, Opcode.ALOAD_0); }

    /**
     * Expects <var>dimExprCount</var> values of type {@code int} on the operand stack. Creates an array of
     * <var>dimExprCount</var> {@code +} <var>dims</var> dimensions of <var>com.hazelcast.com.onentType</var>.
     *
     * @return The type of the created array
     */
    private IClass
    newArray(Locatable locatable, int dimExprCount, int dims, IClass com.hazelcast.com.onentType) {
        if (dimExprCount == 1 && dims == 0 && com.hazelcast.com.onentType.isPrimitive()) {

            // "new <primitive>[<size>]"
            this.writeOpcode(locatable, Opcode.NEWARRAY);
            this.writeByte((
                com.hazelcast.com.onentType == IClass.BOOLEAN ? 4 :
                com.hazelcast.com.onentType == IClass.CHAR    ? 5 :
                com.hazelcast.com.onentType == IClass.FLOAT   ? 6 :
                com.hazelcast.com.onentType == IClass.DOUBLE  ? 7 :
                com.hazelcast.com.onentType == IClass.BYTE    ? 8 :
                com.hazelcast.com.onentType == IClass.SHORT   ? 9 :
                com.hazelcast.com.onentType == IClass.INT     ? 10 :
                com.hazelcast.com.onentType == IClass.LONG    ? 11 : -1
            ));
            return com.hazelcast.com.onentType.getArrayIClass(this.iClassLoader.TYPE_java_lang_Object);
        }

        if (dimExprCount == 1) {
            IClass at = com.hazelcast.com.onentType.getArrayIClass(dims, this.iClassLoader.TYPE_java_lang_Object);

            // "new <class-or-interface>[<size>]"
            // "new <anything>[<size>][]..."
            this.writeOpcode(locatable, Opcode.ANEWARRAY);
            this.writeConstantClassInfo(at.getDescriptor());
            return at.getArrayIClass(this.iClassLoader.TYPE_java_lang_Object);
        } else {
            IClass at = com.hazelcast.com.onentType.getArrayIClass(dimExprCount + dims, this.iClassLoader.TYPE_java_lang_Object);

            // "new <anything>[]..."
            // "new <anything>[<size1>][<size2>]..."
            // "new <anything>[<size1>][<size2>]...[]..."
            this.writeOpcode(locatable, Opcode.MULTIANEWARRAY);
            this.writeConstantClassInfo(at.getDescriptor());
            this.writeByte(dimExprCount);
            return at;
        }
    }

    /**
     * Short-hand implementation of {@link IClass.IField} that implements a non-constant, non-static,
     * package-accessible field.
     */
    public static
    class SimpleIField extends IClass.IField {
        private final String name;
        private final IClass type;

        public
        SimpleIField(IClass declaringIClass, String name, IClass type) {
            declaringIClass.super();
            this.name = name;
            this.type = type;
        }
        @Override public Object        getConstantValue() { return UnitCompiler.NOT_CONSTANT; }
        @Override public String        getName()          { return this.name; }
        @Override public IClass        getType()          { return this.type; }
        @Override public boolean       isStatic()         { return false; }
        @Override public Access        getAccess()        { return Access.DEFAULT; }
        @Override public IAnnotation[] getAnnotations()   { return new IAnnotation[0]; }
    }

    private static Access
    modifiers2Access(short modifiers) {
        return (
            Mod.isPublicAccess(modifiers)    ? Access.PUBLIC    :
            Mod.isProtectedAccess(modifiers) ? Access.PROTECTED :
            Mod.isPrivateAccess(modifiers)   ? Access.PRIVATE   :
            Access.DEFAULT
        );
    }

    private static String
    last(String[] sa) {
        if (sa.length == 0) throw new IllegalArgumentException("SNO: Empty string array");
        return sa[sa.length - 1];
    }

    private static String[]
    allButLast(String[] sa) {
        if (sa.length == 0) throw new IllegalArgumentException("SNO: Empty string array");
        String[] tmp = new String[sa.length - 1];
        System.arraycopy(sa, 0, tmp, 0, tmp.length);
        return tmp;
    }

    private static String[]
    concat(String[] sa, String s) {
        String[] tmp = new String[sa.length + 1];
        System.arraycopy(sa, 0, tmp, 0, sa.length);
        tmp[sa.length] = s;
        return tmp;
    }

    private static CompileException
    com.hazelcast.com.ileException(Locatable locatable, String message) {
        return new CompileException(message, locatable.getLocation());
    }

    /**
     * Decodes any escape sequences like {@code \n}, or {@code \377}, but <em>not</em> {@code &#92;uxxxx}.
     *
     * @return                           <var>s</var>, with all escape sequences replaced with their literal values
     * @throws CompileException          <var>s</var> contains an invalid escape sequence
     * @throws IndexOutOfBoundsException <var>s</var>ends with a backslash
     * @see                              JLS8, section 3.10.6, "Escape Sequences for Character and String Literals"
     */
    private static String
    unescape(String s, @Nullable Location optionalLocation) throws CompileException {

        // Find the first backslash.
        int i = s.indexOf('\\');

        if (i == -1) {

            // Subject string contains no backslash and thus no escape sequences; so return the original string.
            return s;
        }

        StringBuilder sb = new StringBuilder().append(s, 0, i);

        while (i < s.length()) {

            char c = s.charAt(i++);
            if (c != '\\') {
                sb.append(c);
                continue;
            }

            c = s.charAt(i++);
            {
                int idx = "btnfr\"'\\".indexOf(c);
                if (idx != -1) {
                    sb.append("\b\t\n\f\r\"'\\".charAt(idx));
                    continue;
                }
            }

            // Must be an an OctalEscape (JLS8, section 3.10.6).
            int x = Character.digit(c, 8);
            if (x == -1) throw new CompileException("Invalid escape sequence \"\\" + c + "\"", optionalLocation);

            if (i < s.length()) {
                c = s.charAt(i);
                int secondDigit = Character.digit(c, 8);
                if (secondDigit != -1) {
                    x = 8 * x + secondDigit;
                    i++;
                    if (i < s.length() && x <= 037) {
                        c = s.charAt(i);
                        int thirdDigit = Character.digit(c, 8);
                        if (thirdDigit != -1) {
                            x = 8 * x + thirdDigit;
                            i++;
                        }
                    }
                }
            }
            sb.append((char) x);
        }

        return sb.toString();
    }

    // Used to write byte code while com.hazelcast.com.iling one constructor/method.
    @Nullable private CodeContext codeContext;

    // Used for elaborate com.hazelcast.com.ile error handling.
    @Nullable private ErrorHandler optionalCompileErrorHandler;
    private int                    com.hazelcast.com.ileErrorCount;

    // Used for elaborate warning handling.
    @Nullable private WarningHandler optionalWarningHandler;

    private final CompilationUnit com.hazelcast.com.ilationUnit;

    private final IClassLoader iClassLoader;

    /**
     * Non-{@code null} while {@link #com.hazelcast.com.ileUnit(boolean, boolean, boolean)} is executing.
     */
    @Nullable private List<ClassFile> generatedClassFiles;

    private boolean debugSource;
    private boolean debugLines;
    private boolean debugVars;

    private final Map<String /*staticMemberName*/, List<Object /*IField+IMethod+IClass*/>>
    singleStaticImports = new HashMap<String, List<Object>>();

    private final Collection<IClass> staticImportsOnDemand = new ArrayList<IClass>();

    /**
     * Loads a "{@code de.unkrig.jdisasm.Disassembler}" through reflection (to avoid a com.hazelcast.com.ile-time dependency) and
     * uses it to disassemble the given bytes to {@code System.out}.
     */
    public static void
    disassembleToStdout(byte[] contents) {
        try {
            Class<?> disassemblerClass = Class.forName("de.unkrig.jdisasm.Disassembler");
            disassemblerClass.getMethod("disasm", InputStream.class).invoke(
                disassemblerClass.newInstance(),
                new ByteArrayInputStream(contents)
            );
        } catch (Exception e) {
            UnitCompiler.LOGGER.log(Level.FINEST, (
                "Notice: Could not disassemble class file for logging because "
                + "\"de.unkrig.jdisasm.Disassembler\" is not on the classpath. "
                + "If you are interested in disassemblies of class files generated by JANINO, "
                + "get de.unkrig.jdisasm and put it on the classpath."
            ), e);
        }
    }
}
