/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package com.hazelcast.org.apache.hc.client5.http.impl.async;

import java.io.Closeable;
import java.util.List;
import java.util.concurrent.ThreadFactory;

import com.hazelcast.org.apache.hc.client5.http.HttpRoute;
import com.hazelcast.org.apache.hc.client5.http.async.AsyncExecRuntime;
import com.hazelcast.org.apache.hc.client5.http.auth.AuthSchemeFactory;
import com.hazelcast.org.apache.hc.client5.http.auth.CredentialsProvider;
import com.hazelcast.org.apache.hc.client5.http.config.RequestConfig;
import com.hazelcast.org.apache.hc.client5.http.cookie.CookieSpecFactory;
import com.hazelcast.org.apache.hc.client5.http.cookie.CookieStore;
import com.hazelcast.org.apache.hc.client5.http.nio.AsyncClientConnectionManager;
import com.hazelcast.org.apache.hc.client5.http.protocol.HttpClientContext;
import com.hazelcast.org.apache.hc.client5.http.routing.HttpRoutePlanner;
import com.hazelcast.org.apache.hc.core5.annotation.Contract;
import com.hazelcast.org.apache.hc.core5.annotation.Internal;
import com.hazelcast.org.apache.hc.core5.annotation.ThreadingBehavior;
import com.hazelcast.org.apache.hc.core5.http.HttpException;
import com.hazelcast.org.apache.hc.core5.http.HttpHost;
import com.hazelcast.org.apache.hc.core5.http.config.Lookup;
import com.hazelcast.org.apache.hc.core5.http.nio.AsyncPushConsumer;
import com.hazelcast.org.apache.hc.core5.http.nio.HandlerFactory;
import com.hazelcast.org.apache.hc.core5.http2.HttpVersionPolicy;
import com.hazelcast.org.apache.hc.core5.reactor.DefaultConnectingIOReactor;
import com.hazelcast.org.slf4j.Logger;
import com.hazelcast.org.slf4j.LoggerFactory;

/**
 * Internal implementation of {@link CloseableHttpAsyncClient} that can negotiate
 * the most optimal HTTP protocol version during during the {@code TLS} handshake
 * with {@code ALPN} extension if supported by the Java runtime.
 * <p>
 * Concurrent message exchanges executed by this client will get assigned to
 * separate connections leased from the connection pool.
 * </p>
 *
 * @since 5.0
 */
@Contract(threading = ThreadingBehavior.SAFE_CONDITIONAL)
@Internal
public final class InternalHttpAsyncClient extends InternalAbstractHttpAsyncClient {

    private static final Logger LOG = LoggerFactory.getLogger(InternalHttpAsyncClient.class);
    private final AsyncClientConnectionManager manager;
    private final HttpRoutePlanner routePlanner;
    private final HttpVersionPolicy versionPolicy;

    InternalHttpAsyncClient(
            final DefaultConnectingIOReactor ioReactor,
            final AsyncExecChainElement execChain,
            final AsyncPushConsumerRegistry pushConsumerRegistry,
            final ThreadFactory threadFactory,
            final AsyncClientConnectionManager manager,
            final HttpRoutePlanner routePlanner,
            final HttpVersionPolicy versionPolicy,
            final Lookup<CookieSpecFactory> cookieSpecRegistry,
            final Lookup<AuthSchemeFactory> authSchemeRegistry,
            final CookieStore cookieStore,
            final CredentialsProvider credentialsProvider,
            final RequestConfig defaultConfig,
            final List<Closeable> closeables) {
        super(ioReactor, pushConsumerRegistry, threadFactory, execChain,
                cookieSpecRegistry, authSchemeRegistry, cookieStore, credentialsProvider, defaultConfig, closeables);
        this.manager = manager;
        this.routePlanner = routePlanner;
        this.versionPolicy = versionPolicy;
    }

    @Override
    AsyncExecRuntime createAsyncExecRuntime(final HandlerFactory<AsyncPushConsumer> pushHandlerFactory,
                                            final HttpRoute route) {
        // Automatically set protocol policy to force HTTP/1.1
        // when executing requests via proxy tunnel
        final HttpVersionPolicy actualVersionPolicy;
        if (route.isTunnelled() && versionPolicy == HttpVersionPolicy.NEGOTIATE) {
            actualVersionPolicy = HttpVersionPolicy.FORCE_HTTP_1;
        } else {
            actualVersionPolicy = versionPolicy;
        }
        return new InternalHttpAsyncExecRuntime(LOG, manager, getConnectionInitiator(), pushHandlerFactory, actualVersionPolicy);
    }

    @Override
    HttpRoute determineRoute(final HttpHost httpHost, final HttpClientContext clientContext) throws HttpException {
        final HttpRoute route = routePlanner.determineRoute(httpHost, clientContext);
        if (route.isTunnelled() && versionPolicy == HttpVersionPolicy.FORCE_HTTP_2) {
            throw new HttpException("HTTP/2 tunneling not supported");
        }
        return route;
    }

}
