/*
 * Copyright (c) 2008-2018, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.client.impl.protocol.task;

import com.hazelcast.client.impl.protocol.ClientMessage;
import com.hazelcast.client.impl.protocol.codec.ClientDeployClassesCodec;
import com.hazelcast.core.Member;
import com.hazelcast.core.MemberLeftException;
import com.hazelcast.instance.Node;
import com.hazelcast.internal.cluster.Versions;
import com.hazelcast.internal.usercodedeployment.UserCodeDeploymentService;
import com.hazelcast.internal.usercodedeployment.impl.operation.DeployClassesOperation;
import com.hazelcast.nio.Connection;
import com.hazelcast.security.permission.ActionConstants;
import com.hazelcast.security.permission.UserCodeDeploymentPermission;
import com.hazelcast.spi.Operation;
import com.hazelcast.util.function.Supplier;

import java.security.Permission;
import java.util.Collection;
import java.util.Map;

/**
 * Message task handling user code deployment from client to cluster
 */
public class DeployClassesMessageTask extends AbstractMultiTargetMessageTask<ClientDeployClassesCodec.RequestParameters>
        implements Supplier<Operation> {

    public DeployClassesMessageTask(ClientMessage clientMessage, Node node, Connection connection) {
        super(clientMessage, node, connection);
    }

    @Override
    public Operation get() {
        return new DeployClassesOperation(parameters.classDefinitions);
    }

    @Override
    protected Supplier<Operation> createOperationSupplier() {
        if (nodeEngine.getClusterService().getClusterVersion().isLessThan(Versions.V3_9)) {
            throw new UnsupportedOperationException("Client user code deployment available in 3.9");
        }
        return this;
    }

    @Override
    protected Object reduce(Map<Member, Object> map) throws Throwable {
        for (Object result : map.values()) {
            if (result instanceof Throwable && !(result instanceof MemberLeftException)) {
                throw (Throwable) result;
            }
        }
        return null;
    }

    @Override
    public Collection<Member> getTargets() {
        return nodeEngine.getClusterService().getMembers();
    }

    @Override
    protected ClientDeployClassesCodec.RequestParameters decodeClientMessage(ClientMessage clientMessage) {
        return ClientDeployClassesCodec.decodeRequest(clientMessage);
    }

    @Override
    protected ClientMessage encodeResponse(Object response) {
        return ClientDeployClassesCodec.encodeResponse();
    }

    @Override
    public String getServiceName() {
        return UserCodeDeploymentService.SERVICE_NAME;
    }

    @Override
    public String getDistributedObjectName() {
        return null;
    }

    @Override
    public String getMethodName() {
        return null;
    }

    @Override
    public Permission getRequiredPermission() {
        return new UserCodeDeploymentPermission(ActionConstants.ACTION_USER_CODE_DEPLOY);
    }

    @Override
    public Object[] getParameters() {
        return null;
    }
}
