/*
 * Copyright (c) 2008-2021, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.sql.impl.calcite;

import com.hazelcast.sql.impl.calcite.validate.types.HazelcastIntegerType;
import com.hazelcast.sql.impl.calcite.validate.types.HazelcastTypeFactory;
import com.hazelcast.sql.impl.calcite.validate.types.HazelcastTypeUtils;
import com.hazelcast.sql.impl.type.QueryDataTypeFamily;
import com.hazelcast.sql.impl.type.converter.Converter;
import com.hazelcast.sql.impl.type.converter.Converters;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rex.RexBuilder;
import org.apache.calcite.rex.RexNode;
import org.apache.calcite.sql.type.SqlTypeName;

import static org.apache.calcite.sql.type.SqlTypeName.ANY;

/**
 * Custom Hazelcast expression builder.
 * <p>
 * Currently, this custom expression builder is used just to workaround quirks
 * of the default Calcite expression builder.
 */
public final class HazelcastRexBuilder extends RexBuilder {
    public HazelcastRexBuilder(HazelcastTypeFactory typeFactory) {
        super(typeFactory);
    }

    @Override
    public RexNode makeLiteral(Object value, RelDataType type, boolean allowCast) {
        // Make sure that numeric literals get a correct return type during the conversion.
        // Without this code, Apache Calcite may assign incorrect types to some literals during conversion.
        // For example, new BigDecimal(Long.MAX_VALUE + "1") will receive the BIGINT type.
        // To see the problem in action, you may comment out this code and run CastFunctionIntegrationTest.
        // Some conversions will fail due to precision loss.

        if (type.getSqlTypeName() == ANY && value instanceof Number) {
            Converter converter = Converters.getConverter(value.getClass());

            if (converter != null) {
                QueryDataTypeFamily typeFamily = converter.getTypeFamily();

                if (typeFamily.isNumericInteger()) {
                    int bitWidth = HazelcastIntegerType.bitWidthOf(((Number) value).longValue());
                    type = HazelcastIntegerType.create(bitWidth, false);
                } else {
                    SqlTypeName typeName = HazelcastTypeUtils.toCalciteType(typeFamily);

                    type = HazelcastTypeFactory.INSTANCE.createSqlType(typeName);
                }
            }
        }

        return super.makeLiteral(value, type, allowCast);
    }

}
