/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.jet.mongodb.dataconnection;

import com.hazelcast.config.DataConnectionConfig;
import com.hazelcast.core.HazelcastException;
import com.hazelcast.dataconnection.DataConnectionBase;
import com.hazelcast.dataconnection.DataConnectionResource;
import com.hazelcast.internal.util.Preconditions;
import com.hazelcast.jet.impl.util.ConcurrentMemoizingSupplier;
import com.hazelcast.jet.mongodb.dataconnection.CloseableMongoClient;
import com.hazelcast.jet.mongodb.dataconnection.SslConf;
import com.hazelcast.jet.mongodb.impl.Mappers;
import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.MongoCredential;
import com.mongodb.ServerAddress;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import com.mongodb.client.MongoDatabase;
import com.mongodb.connection.ConnectionPoolSettings;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import javax.annotation.Nonnull;

public class MongoDataConnection
extends DataConnectionBase {
    public static final String CONNECTION_STRING_PROPERTY = "connectionString";
    public static final String DATABASE_PROPERTY = "database";
    public static final String USERNAME_PROPERTY = "username";
    public static final String PASSWORD_PROPERTY = "password";
    public static final String HOST_PROPERTY = "host";
    public static final String AUTH_DB_PROPERTY = "authDb";
    public static final String CONNECTION_POOL_MIN = "connectionPoolMinSize";
    public static final String CONNECTION_POOL_MAX = "connectionPoolMaxSize";
    private volatile ConcurrentMemoizingSupplier<MongoClient> mongoClientSup;
    private final String name;
    private final String connectionString;
    private final String databaseName;
    private final String username;
    private final String password;
    private final String host;
    private final String authDb;
    private final int connectionPoolMinSize;
    private final int connectionPoolMaxSize;
    private final SslConf sslConf;

    public MongoDataConnection(DataConnectionConfig config) {
        super(config);
        this.name = config.getName();
        this.connectionString = config.getProperty(CONNECTION_STRING_PROPERTY);
        this.databaseName = config.getProperty(DATABASE_PROPERTY);
        this.username = config.getProperty(USERNAME_PROPERTY);
        this.password = config.getProperty(PASSWORD_PROPERTY);
        this.host = config.getProperty(HOST_PROPERTY);
        this.authDb = config.getProperty(AUTH_DB_PROPERTY, "admin");
        this.connectionPoolMinSize = Integer.parseInt(config.getProperty(CONNECTION_POOL_MIN, "10"));
        this.connectionPoolMaxSize = Integer.parseInt(config.getProperty(CONNECTION_POOL_MAX, "10"));
        this.sslConf = new SslConf(config);
        Preconditions.checkState((this.connectionPoolMinSize <= this.connectionPoolMaxSize ? 1 : 0) != 0, (String)"connection pool max size cannot be lower than min size");
        Preconditions.checkState((boolean)MongoDataConnection.allSame(this.username == null, this.password == null, this.host == null), (String)"You have to provide connectionString property or combination of username, password and host");
        if (config.isShared()) {
            this.mongoClientSup = new ConcurrentMemoizingSupplier(() -> new CloseableMongoClient(this.createClient(), () -> ((MongoDataConnection)this).release()));
        }
    }

    static boolean allSame(boolean ... booleans) {
        if (booleans.length == 0) {
            return true;
        }
        boolean first = booleans[0];
        for (boolean aBoolean : booleans) {
            if (first == aBoolean) continue;
            return false;
        }
        return true;
    }

    private MongoClient createClient() {
        try {
            MongoClientSettings.Builder builder = MongoClientSettings.builder().codecRegistry(Mappers.defaultCodecRegistry()).applyToConnectionPoolSettings(this::connectionPoolSettings).applyToSslSettings(this.sslConf::apply);
            if (this.connectionString != null) {
                builder.applyConnectionString(new ConnectionString(this.connectionString));
            } else {
                ServerAddress serverAddress = new ServerAddress(this.host);
                MongoCredential credential = MongoCredential.createCredential((String)this.username, (String)this.authDb, (char[])this.password.toCharArray());
                builder.applyToClusterSettings(s -> s.hosts(Collections.singletonList(serverAddress))).credential(credential);
            }
            return MongoClients.create((MongoClientSettings)builder.build());
        }
        catch (Exception e) {
            throw new HazelcastException("Unable to create Mongo client for data connection '" + this.name + "': " + e.getMessage(), (Throwable)e);
        }
    }

    private void connectionPoolSettings(ConnectionPoolSettings.Builder builder) {
        builder.minSize(this.connectionPoolMinSize).maxSize(this.connectionPoolMaxSize);
    }

    @Nonnull
    public MongoClient getClient() {
        if (this.getConfig().isShared()) {
            this.retain();
            ConcurrentMemoizingSupplier<MongoClient> supplier = this.mongoClientSup;
            Preconditions.checkState((supplier != null ? 1 : 0) != 0, (String)"Mongo client should not be closed at this point");
            return (MongoClient)supplier.get();
        }
        MongoClient client = this.createClient();
        return new CloseableMongoClient(client, () -> ((MongoClient)client).close());
    }

    public String getDatabaseName() {
        return this.databaseName;
    }

    @Nonnull
    public List<DataConnectionResource> listResources() {
        ArrayList<DataConnectionResource> resources = new ArrayList<DataConnectionResource>();
        try (MongoClient client = this.getClient();){
            if (this.databaseName != null) {
                MongoDatabase mongoDatabase = client.getDatabase(this.databaseName);
                MongoDataConnection.addResources(resources, mongoDatabase);
            } else {
                for (String databaseName : client.listDatabaseNames()) {
                    MongoDatabase database = client.getDatabase(databaseName);
                    MongoDataConnection.addResources(resources, database);
                }
            }
        }
        return resources;
    }

    @Nonnull
    public Collection<String> resourceTypes() {
        return Arrays.asList("Collection", "ChangeStream");
    }

    private static void addResources(List<DataConnectionResource> resources, MongoDatabase database) {
        for (String collectionName : database.listCollectionNames()) {
            resources.add(new DataConnectionResource("Collection", new String[]{database.getName(), collectionName}));
            resources.add(new DataConnectionResource("ChangeStream", new String[]{database.getName(), collectionName}));
        }
    }

    public void destroy() {
        ConcurrentMemoizingSupplier<MongoClient> supplier = this.mongoClientSup;
        if (supplier != null) {
            this.mongoClientSup = null;
            MongoClient mongoClient = (MongoClient)supplier.remembered();
            if (mongoClient != null) {
                ((CloseableMongoClient)mongoClient).unwrap().close();
            }
        }
    }

    @Nonnull
    public static DataConnectionConfig mongoDataConnectionConf(String name, String connectionString) {
        DataConnectionConfig dataConnectionConfig = new DataConnectionConfig();
        dataConnectionConfig.setName(name);
        dataConnectionConfig.setShared(true);
        dataConnectionConfig.setProperty(CONNECTION_STRING_PROPERTY, connectionString);
        dataConnectionConfig.setType("Mongo");
        return dataConnectionConfig;
    }
}

