/*
 * This file is part of ClassGraph.
 *
 * Author: Luke Hutchison
 *
 * Hosted at: https://github.com/classgraph/classgraph
 *
 * --
 *
 * The MIT License (MIT)
 *
 * Copyright (c) 2019 Luke Hutchison
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
 * documentation files (the "Software"), to deal in the Software without restriction, including without
 * limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
 * LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
 * EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
 * OR OTHER DEALINGS IN THE SOFTWARE.
 */
package nonapi.com.hazelcast.jet.io.github.classgraph.classloaderhandler;

import nonapi.com.hazelcast.jet.io.github.classgraph.classpath.ClassLoaderOrder;
import nonapi.com.hazelcast.jet.io.github.classgraph.classpath.ClasspathOrder;
import nonapi.com.hazelcast.jet.io.github.classgraph.scanspec.ScanSpec;
import nonapi.com.hazelcast.jet.io.github.classgraph.utils.LogNode;
import nonapi.com.hazelcast.jet.io.github.classgraph.utils.ReflectionUtils;

/** Extract classpath entries from the Weblogic ClassLoaders. */
class WeblogicClassLoaderHandler implements ClassLoaderHandler {
    /** Class cannot be constructed. */
    private WeblogicClassLoaderHandler() {
    }

    /**
     * Check whether this {@link ClassLoaderHandler} can handle a given {@link ClassLoader}.
     *
     * @param classLoaderClass
     *            the {@link ClassLoader} class or one of its superclasses.
     * @param log
     *            the log
     * @return true if this {@link ClassLoaderHandler} can handle the {@link ClassLoader}.
     */
    public static boolean canHandle(final Class<?> classLoaderClass, final LogNode log) {
        return "weblogic.utils.classloaders.ChangeAwareClassLoader".equals(classLoaderClass.getName())
                || "weblogic.utils.classloaders.GenericClassLoader".equals(classLoaderClass.getName())
                || "weblogic.utils.classloaders.FilteringClassLoader".equals(classLoaderClass.getName())
                // TODO: The following two known classloader names have not been tested, and the fields/methods
                // may not match those of the above classloaders.
                || "weblogic.servlet.jsp.JspClassLoader".equals(classLoaderClass.getName())
                || "weblogic.servlet.jsp.TagFileClassLoader".equals(classLoaderClass.getName());
    }

    /**
     * Find the {@link ClassLoader} delegation order for a {@link ClassLoader}.
     *
     * @param classLoader
     *            the {@link ClassLoader} to find the order for.
     * @param classLoaderOrder
     *            a {@link ClassLoaderOrder} object to update.
     * @param log
     *            the log
     */
    public static void findClassLoaderOrder(final ClassLoader classLoader, final ClassLoaderOrder classLoaderOrder,
            final LogNode log) {
        classLoaderOrder.delegateTo(classLoader.getParent(), /* isParent = */ true, log);
        classLoaderOrder.add(classLoader, log);
    }

    /**
     * Find the classpath entries for the associated {@link ClassLoader}.
     *
     * @param classLoader
     *            the {@link ClassLoader} to find the classpath entries order for.
     * @param classpathOrder
     *            a {@link ClasspathOrder} object to update.
     * @param scanSpec
     *            the {@link ScanSpec}.
     * @param log
     *            the log.
     */
    public static void findClasspathOrder(final ClassLoader classLoader, final ClasspathOrder classpathOrder,
            final ScanSpec scanSpec, final LogNode log) {
        classpathOrder.addClasspathPathStr( //
                (String) ReflectionUtils.invokeMethod(classLoader, "getFinderClassPath", false), classLoader,
                scanSpec, log);
        classpathOrder.addClasspathPathStr( //
                (String) ReflectionUtils.invokeMethod(classLoader, "getClassPath", false), classLoader, scanSpec,
                log);
    }
}
