package com.hccake.ballcat.system.controller;

import com.hccake.ballcat.common.core.validation.group.CreateGroup;
import com.hccake.ballcat.common.core.validation.group.UpdateGroup;
import com.hccake.ballcat.common.log.operation.annotation.CreateOperationLogging;
import com.hccake.ballcat.common.log.operation.annotation.DeleteOperationLogging;
import com.hccake.ballcat.common.log.operation.annotation.UpdateOperationLogging;
import com.hccake.ballcat.common.model.domain.PageParam;
import com.hccake.ballcat.common.model.domain.PageResult;
import com.hccake.ballcat.common.model.result.BaseResultCode;
import com.hccake.ballcat.common.model.result.R;
import com.hccake.ballcat.system.manager.SysDictManager;
import com.hccake.ballcat.system.model.dto.SysDictItemDTO;
import com.hccake.ballcat.system.model.entity.SysDict;
import com.hccake.ballcat.system.model.qo.SysDictQO;
import com.hccake.ballcat.system.model.vo.DictDataVO;
import com.hccake.ballcat.system.model.vo.SysDictItemPageVO;
import com.hccake.ballcat.system.model.vo.SysDictPageVO;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.groups.Default;
import java.util.List;
import java.util.Map;

/**
 * 字典表
 *
 * @author hccake 2020-03-26 18:40:20
 */
@Validated
@RestController
@RequiredArgsConstructor
@RequestMapping("/system/dict")
@Tag(name = "字典表管理")
public class SysDictController {

	private final SysDictManager sysDictManager;

	/**
	 * 通过字典标识查找对应字典项
	 * @param dictCodes 字典标识列表
	 * @return 同类型字典
	 */
	@GetMapping("/data")
	public R<List<DictDataVO>> getDictData(@RequestParam("dictCodes") String[] dictCodes) {
		return R.ok(sysDictManager.queryDictDataAndHashVO(dictCodes));
	}

	/**
	 * 通过字典标识查找对应字典项
	 * @param dictHashCode 字典标识
	 * @return 同类型字典
	 */
	@PostMapping("/invalid-hash")
	public R<List<String>> invalidDictHash(@RequestBody Map<String, String> dictHashCode) {
		return R.ok(sysDictManager.invalidDictHash(dictHashCode));
	}

	/**
	 * 分页查询
	 * @param pageParam 分页参数
	 * @param sysDictQO 字典查询参数
	 * @return R<PageResult<SysDictVO>>
	 */
	@GetMapping("/page")
	@PreAuthorize("@per.hasPermission('system:dict:read')")
	@Operation(summary = "分页查询", description = "分页查询")
	public R<PageResult<SysDictPageVO>> getSysDictPage(@Validated PageParam pageParam, SysDictQO sysDictQO) {
		return R.ok(sysDictManager.dictPage(pageParam, sysDictQO));
	}

	/**
	 * 新增字典表
	 * @param sysDict 字典表
	 * @return R
	 */
	@CreateOperationLogging(msg = "新增字典表")
	@PostMapping
	@PreAuthorize("@per.hasPermission('system:dict:add')")
	@Operation(summary = "新增字典表", description = "新增字典表")
	public R<Void> save(@RequestBody SysDict sysDict) {
		return sysDictManager.dictSave(sysDict) ? R.ok() : R.failed(BaseResultCode.UPDATE_DATABASE_ERROR, "新增字典表失败");
	}

	/**
	 * 修改字典表
	 * @param sysDict 字典表
	 * @return R
	 */
	@UpdateOperationLogging(msg = "修改字典表")
	@PutMapping
	@PreAuthorize("@per.hasPermission('system:dict:edit')")
	@Operation(summary = "修改字典表", description = "修改字典表")
	public R<Void> updateById(@RequestBody SysDict sysDict) {
		return sysDictManager.updateDictById(sysDict) ? R.ok()
				: R.failed(BaseResultCode.UPDATE_DATABASE_ERROR, "修改字典表失败");
	}

	/**
	 * 通过id删除字典表
	 * @param id id
	 * @return R
	 */
	@DeleteOperationLogging(msg = "通过id删除字典表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@per.hasPermission('system:dict:del')")
	@Operation(summary = "通过id删除字典表", description = "通过id删除字典表")
	public R<Void> removeById(@PathVariable("id") Long id) {
		sysDictManager.removeDictById(id);
		return R.ok();
	}

	/**
	 * 分页查询
	 * @param pageParam 分页参数
	 * @param dictCode 字典标识
	 * @return R
	 */
	@GetMapping("/item/page")
	@PreAuthorize("@per.hasPermission('system:dict:read')")
	@Operation(summary = "分页查询", description = "分页查询")
	public R<PageResult<SysDictItemPageVO>> getSysDictItemPage(PageParam pageParam,
			@RequestParam("dictCode") String dictCode) {
		return R.ok(sysDictManager.dictItemPage(pageParam, dictCode));
	}

	/**
	 * 新增字典项
	 * @param sysDictItemDTO 字典项
	 * @return R
	 */
	@CreateOperationLogging(msg = "新增字典项")
	@PostMapping("item")
	@PreAuthorize("@per.hasPermission('system:dict:add')")
	@Operation(summary = "新增字典项", description = "新增字典项")
	public R<Void> saveItem(
			@Validated({ Default.class, CreateGroup.class }) @RequestBody SysDictItemDTO sysDictItemDTO) {
		return sysDictManager.saveDictItem(sysDictItemDTO) ? R.ok()
				: R.failed(BaseResultCode.UPDATE_DATABASE_ERROR, "新增字典项失败");
	}

	/**
	 * 修改字典项
	 * @param sysDictItemDTO 字典项
	 * @return R
	 */
	@UpdateOperationLogging(msg = "修改字典项")
	@PutMapping("item")
	@PreAuthorize("@per.hasPermission('system:dict:edit')")
	@Operation(summary = "修改字典项", description = "修改字典项")
	public R<Void> updateItemById(
			@Validated({ Default.class, UpdateGroup.class }) @RequestBody SysDictItemDTO sysDictItemDTO) {
		return sysDictManager.updateDictItemById(sysDictItemDTO) ? R.ok()
				: R.failed(BaseResultCode.UPDATE_DATABASE_ERROR, "修改字典项失败");
	}

	/**
	 * 通过id删除字典项
	 * @param id id
	 * @return R
	 */
	@DeleteOperationLogging(msg = "通过id删除字典项")
	@DeleteMapping("/item/{id}")
	@PreAuthorize("@per.hasPermission('system:dict:del')")
	@Operation(summary = "通过id删除字典项", description = "通过id删除字典项")
	public R<Void> removeItemById(@PathVariable("id") Long id) {
		return sysDictManager.removeDictItemById(id) ? R.ok()
				: R.failed(BaseResultCode.UPDATE_DATABASE_ERROR, "通过id删除字典项失败");
	}

	/**
	 * 通过id修改字典项状态
	 * @param id id
	 * @return R
	 */
	@UpdateOperationLogging(msg = "通过id修改字典项状态")
	@PatchMapping("/item/{id}")
	@PreAuthorize("@per.hasPermission('system:dict:edit')")
	@Operation(summary = "通过id修改字典项状态", description = "通过id修改字典项状态")
	public R<Void> updateDictItemStatusById(@PathVariable("id") Long id, @RequestParam("status") Integer status) {
		sysDictManager.updateDictItemStatusById(id, status);
		return R.ok();
	}

}
