/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.hashgraph.sdk;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonIOException;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import com.google.gson.stream.JsonWriter;
import com.hedera.hashgraph.sdk.BadKeyException;
import com.hedera.hashgraph.sdk.Crypto;
import com.hedera.hashgraph.sdk.PrivateKey;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.util.encoders.Hex;

final class Keystore {
    private static final Gson gson = new Gson();
    private static final JsonParser jsonParser = new JsonParser();
    private byte[] keyBytes;

    private Keystore(byte[] keyBytes) {
        this.keyBytes = keyBytes;
    }

    public Keystore(PrivateKey privateKey) {
        this.keyBytes = privateKey.toBytes();
    }

    public static Keystore fromStream(InputStream stream, String passphrase) throws IOException {
        try {
            JsonObject jsonObject = jsonParser.parse((Reader)new InputStreamReader(stream, StandardCharsets.UTF_8)).getAsJsonObject();
            return Keystore.fromJson(jsonObject, passphrase);
        }
        catch (IllegalStateException e) {
            throw new BadKeyException(Optional.ofNullable(e.getMessage()).orElse("failed to parse Keystore"));
        }
        catch (JsonIOException e) {
            throw (IOException)Objects.requireNonNull(e.getCause());
        }
        catch (JsonSyntaxException e) {
            throw new BadKeyException(e);
        }
    }

    public PrivateKey getEd25519() {
        return PrivateKey.fromBytes(this.keyBytes);
    }

    private static Keystore fromJson(JsonObject object, String passphrase) {
        int version = Keystore.expectInt(object, "version");
        switch (version) {
            case 1: {
                return Keystore.parseKeystoreV1(Keystore.expectObject(object, "crypto"), passphrase);
            }
        }
        throw new BadKeyException("unsupported keystore version: " + version);
    }

    public void export(OutputStream outputStream, String passphrase) throws IOException {
        JsonWriter writer = new JsonWriter((Writer)new OutputStreamWriter(outputStream, StandardCharsets.UTF_8));
        gson.toJson((JsonElement)this.exportJson(passphrase), writer);
        writer.flush();
    }

    private JsonObject exportJson(String passphrase) {
        JsonObject object = new JsonObject();
        object.addProperty("version", (Number)1);
        JsonObject crypto = new JsonObject();
        crypto.addProperty("cipher", "aes-128-ctr");
        crypto.addProperty("kdf", "pbkdf2");
        byte[] salt = Crypto.randomBytes(32);
        KeyParameter cipherKey = Crypto.deriveKeySha256(passphrase, salt, 262144, 32);
        byte[] iv = Crypto.randomBytes(16);
        byte[] cipherBytes = Crypto.encryptAesCtr128(cipherKey, iv, this.keyBytes);
        byte[] mac = Crypto.calcHmacSha384(cipherKey, cipherBytes);
        JsonObject cipherParams = new JsonObject();
        cipherParams.addProperty("iv", Hex.toHexString((byte[])iv));
        JsonObject kdfParams = new JsonObject();
        kdfParams.addProperty("dkLen", (Number)32);
        kdfParams.addProperty("salt", Hex.toHexString((byte[])salt));
        kdfParams.addProperty("c", (Number)262144);
        kdfParams.addProperty("prf", "hmac-sha256");
        crypto.add("cipherparams", (JsonElement)cipherParams);
        crypto.addProperty("ciphertext", Hex.toHexString((byte[])cipherBytes));
        crypto.add("kdfparams", (JsonElement)kdfParams);
        crypto.addProperty("mac", Hex.toHexString((byte[])mac));
        object.add("crypto", (JsonElement)crypto);
        return object;
    }

    private static Keystore parseKeystoreV1(JsonObject crypto, String passphrase) {
        byte[] salt;
        KeyParameter cipherKey;
        byte[] testHmac;
        String ciphertext = Keystore.expectString(crypto, "ciphertext");
        String ivString = Keystore.expectString(Keystore.expectObject(crypto, "cipherparams"), "iv");
        String cipher = Keystore.expectString(crypto, "cipher");
        String kdf = Keystore.expectString(crypto, "kdf");
        JsonObject kdfParams = Keystore.expectObject(crypto, "kdfparams");
        String macString = Keystore.expectString(crypto, "mac");
        if (!cipher.equals("aes-128-ctr")) {
            throw new BadKeyException("unsupported keystore cipher: " + cipher);
        }
        if (!kdf.equals("pbkdf2")) {
            throw new BadKeyException("unsuppported KDF: " + kdf);
        }
        int dkLen = Keystore.expectInt(kdfParams, "dkLen");
        String saltStr = Keystore.expectString(kdfParams, "salt");
        int count = Keystore.expectInt(kdfParams, "c");
        String prf = Keystore.expectString(kdfParams, "prf");
        if (!prf.equals("hmac-sha256")) {
            throw new BadKeyException("unsupported KDF hash function: " + prf);
        }
        byte[] cipherBytes = Hex.decode((String)ciphertext);
        byte[] iv = Hex.decode((String)ivString);
        byte[] mac = Hex.decode((String)macString);
        if (!Arrays.equals(mac, testHmac = Crypto.calcHmacSha384(cipherKey = Crypto.deriveKeySha256(passphrase, salt = Hex.decode((String)saltStr), count, dkLen), cipherBytes))) {
            throw new BadKeyException("HMAC mismatch; passphrase is incorrect");
        }
        return new Keystore(Crypto.decryptAesCtr128(cipherKey, iv, cipherBytes));
    }

    private static JsonObject expectObject(JsonObject object, String key) {
        try {
            return object.get(key).getAsJsonObject();
        }
        catch (ClassCastException | NullPointerException e) {
            throw new Error("expected key '" + key + "' to be an object", e);
        }
    }

    private static int expectInt(JsonObject object, String key) {
        try {
            return object.get(key).getAsInt();
        }
        catch (ClassCastException | NullPointerException e) {
            throw new Error("expected key '" + key + "' to be an integer", e);
        }
    }

    private static String expectString(JsonObject object, String key) {
        try {
            return object.get(key).getAsString();
        }
        catch (ClassCastException | NullPointerException e) {
            throw new Error("expected key '" + key + "' to be a string", e);
        }
    }
}

