/*
 * Copyright 2017-2022 Philip Helger, pgcc@helger.com
 *
 * Copyright 2011 Google Inc. All Rights Reserved.
 * Author: sreeni@google.com (Sreeni Viswanadha)
 *
 * Copyright (c) 2006, Sun Microsystems, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the Sun Microsystems, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived from
 *       this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.helger.pgcc.jjtree;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.Writer;

import com.helger.commons.io.file.FileHelper;
import com.helger.commons.io.stream.NonBlockingBufferedReader;
import com.helger.commons.io.stream.StreamHelper;
import com.helger.commons.string.StringHelper;
import com.helger.pgcc.PGPrinter;
import com.helger.pgcc.parser.JavaCCGlobals;
import com.helger.pgcc.parser.Options;

public final class JJTreeIO
{
  private String m_sInputFilename = "<uninitialized input>";
  private String m_sOutputFilename;
  private Reader m_in;
  private PrintWriter m_out;

  JJTreeIO ()
  {}

  String getInputFilename ()
  {
    return m_sInputFilename;
  }

  Reader getIn ()
  {
    return m_in;
  }

  String getOutputFilename ()
  {
    return m_sOutputFilename;
  }

  PrintWriter getOut ()
  {
    return m_out;
  }

  void print (final String s)
  {
    m_out.print (s);
  }

  public void println (final String s)
  {
    m_out.println (s);
  }

  public void println ()
  {
    m_out.println ();
  }

  void closeAll ()
  {
    StreamHelper.close (m_out);
    PGPrinter.flush ();
  }

  private static String _createOutputFilename (final String sInput)
  {
    String sOutput = JJTreeOptions.getOutputFile ();

    if (StringHelper.hasNoText (sOutput))
    {
      String sRealIn = sInput;
      final int nLastSep = sRealIn.lastIndexOf (File.separatorChar);
      if (nLastSep >= 0)
      {
        sRealIn = sRealIn.substring (nLastSep + 1);
      }

      final int nLastDot = sRealIn.lastIndexOf ('.');
      if (nLastDot == -1)
      {
        // No dot
        sOutput = sRealIn + ".jj";
      }
      else
      {
        // Cut dot
        final String suffix = sRealIn.substring (nLastDot);
        if (suffix.equals (".jj"))
        {
          sOutput = sRealIn + ".jj";
        }
        else
        {
          sOutput = sRealIn.substring (0, nLastDot) + ".jj";
        }
      }
    }

    return sOutput;
  }

  void setInput (final String fn) throws IOException
  {
    try
    {
      final File fp = new File (fn);
      if (!fp.exists ())
        throw new IOException ("File " + fn + " not found.");
      if (fp.isDirectory ())
        throw new IOException (fn + " is a directory. Please use a valid file name.");
      if (JavaCCGlobals.isGeneratedBy ("JJTree", fn))
        throw new IOException (fn + " was generated by jjtree.  Cannot run jjtree again.");
      m_sInputFilename = fp.getPath ();

      m_in = new NonBlockingBufferedReader (new InputStreamReader (new FileInputStream (m_sInputFilename), Options.getGrammarEncoding ()));

    }
    catch (final SecurityException se)
    {
      throw new IOException ("Security violation while trying to open " + fn, se);
    }
  }

  void setOutput () throws IOException
  {
    JavaCCGlobals.createOutputDir (JJTreeOptions.getJJTreeOutputDirectory ());
    final File ofile = new File (JJTreeOptions.getJJTreeOutputDirectory (), _createOutputFilename (m_sInputFilename));
    m_sOutputFilename = ofile.toString ();
    final Writer w = FileHelper.getBufferedWriter (ofile, Options.getOutputEncoding ());
    if (w == null)
      throw new IOException ("Failed to open file " + ofile);
    m_out = new PrintWriter (w);
  }
}
