/**
 * Copyright (C) 2014 Philip Helger (www.helger.com)
 * philip[at]helger[dot]com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.helger.commons.io.file.filter;

import java.io.File;
import java.io.FilenameFilter;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;

import com.helger.commons.ValueEnforcer;
import com.helger.commons.annotations.Nonempty;
import com.helger.commons.annotations.ReturnsMutableCopy;
import com.helger.commons.collections.ArrayHelper;
import com.helger.commons.io.file.FilenameHelper;
import com.helger.commons.regex.RegExHelper;
import com.helger.commons.string.ToStringGenerator;

/**
 * A filter that only accepts certain file names, based on a regular expression.
 * If at least one regular expressions is fulfilled, the file is accepted. The
 * filter is applied on directories and files!
 * 
 * @author Philip Helger
 */
@ThreadSafe
public final class FilenameFilterMatchAnyRegEx implements FilenameFilter
{
  private final String [] m_aRegExs;

  public FilenameFilterMatchAnyRegEx (@Nonnull @Nonempty final String sRegEx)
  {
    this (new String [] { sRegEx });
  }

  public FilenameFilterMatchAnyRegEx (@Nonnull @Nonempty final String... aRegExs)
  {
    m_aRegExs = ArrayHelper.getCopy (ValueEnforcer.notEmpty (aRegExs, "RegularExpressions"));
  }

  @Nonnull
  @ReturnsMutableCopy
  public String [] getRegularExpressions ()
  {
    return ArrayHelper.getCopy (m_aRegExs);
  }

  public boolean accept (@Nonnull final File aDir, @Nonnull final String sName)
  {
    final String sRealName = FilenameHelper.getSecureFilename (sName);
    if (sRealName != null)
      for (final String sRegEx : m_aRegExs)
        if (RegExHelper.stringMatchesPattern (sRegEx, sRealName))
          return true;
    return false;
  }

  @Override
  public String toString ()
  {
    return new ToStringGenerator (this).append ("regExs", m_aRegExs).toString ();
  }
}
