/*
 * Decompiled with CFR 0.152.
 */
package com.helger.commons.io.file;

import com.helger.commons.CGlobal;
import com.helger.commons.SystemProperties;
import com.helger.commons.ValueEnforcer;
import com.helger.commons.annotations.ReturnsMutableCopy;
import com.helger.commons.collections.ContainerHelper;
import com.helger.commons.io.EAppend;
import com.helger.commons.io.file.ERandomAccessFileMode;
import com.helger.commons.io.file.FileIOError;
import com.helger.commons.io.file.FileOperations;
import com.helger.commons.io.file.FilenameHelper;
import com.helger.commons.io.misc.SizeHelper;
import com.helger.commons.io.streams.ByteBufferInputStream;
import com.helger.commons.io.streams.ByteBufferOutputStream;
import com.helger.commons.io.streams.CountingFileInputStream;
import com.helger.commons.io.streams.CountingFileOutputStream;
import com.helger.commons.io.streams.StreamUtils;
import com.helger.commons.state.EChange;
import com.helger.commons.state.EValidity;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.List;
import javax.annotation.CheckForSigned;
import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.WillNotClose;
import javax.annotation.concurrent.Immutable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Immutable
public final class FileUtils {
    private static final boolean USE_MEMORY_MAPPED_FILES = false;
    private static final Logger s_aLogger = LoggerFactory.getLogger(FileUtils.class);
    private static final FileUtils s_aInstance = new FileUtils();

    private FileUtils() {
    }

    public static boolean existsFile(@Nonnull File file) {
        ValueEnforcer.notNull(file, "File");
        return file.isFile();
    }

    public static boolean existsDir(@Nonnull File file) {
        ValueEnforcer.notNull(file, "Directory");
        return file.isDirectory();
    }

    public static boolean canRead(@Nonnull File file) {
        ValueEnforcer.notNull(file, "File");
        return file.canRead();
    }

    public static boolean canWrite(@Nonnull File file) {
        ValueEnforcer.notNull(file, "File");
        return file.canWrite();
    }

    public static boolean canExecute(@Nonnull File file) {
        ValueEnforcer.notNull(file, "File");
        return file.canExecute();
    }

    public static boolean canReadAndWriteFile(@Nonnull File file) {
        File file2;
        ValueEnforcer.notNull(file, "File");
        return !(FileUtils.existsFile(file) ? !FileUtils.canRead(file) || !FileUtils.canWrite(file) : (file2 = file.getParentFile()) != null && file2.isDirectory() && (!FileUtils.canRead(file2) || !FileUtils.canWrite(file2)));
    }

    @Nonnull
    public static EChange ensureParentDirectoryIsPresent(@Nonnull File file) {
        ValueEnforcer.notNull(file, "File");
        File file2 = file.getParentFile();
        if (file2 == null || file2.exists()) {
            if (file2 != null && !file2.isDirectory()) {
                s_aLogger.warn("Parent object specified is not a directory: '" + file2 + "'");
            }
            return EChange.UNCHANGED;
        }
        FileIOError fileIOError = FileOperations.createDirRecursive(file2);
        if (fileIOError.isFailure()) {
            throw new IllegalStateException("Failed to create parent of " + file.getAbsolutePath() + ": " + fileIOError);
        }
        if (!file2.exists()) {
            throw new IllegalStateException("Parent of " + file.getAbsolutePath() + " is still not existing!");
        }
        return EChange.CHANGED;
    }

    @Nullable
    public static File getCanonicalFile(@Nullable File file) throws IOException {
        return file == null ? null : file.getCanonicalFile();
    }

    @Nullable
    public static File getCanonicalFileOrNull(@Nullable File file) {
        try {
            return FileUtils.getCanonicalFile(file);
        }
        catch (IOException iOException) {
            return null;
        }
    }

    @Nullable
    public static String getCanonicalPath(@Nullable File file) throws IOException {
        return file == null ? null : file.getCanonicalPath();
    }

    @Nullable
    public static String getCanonicalPathOrNull(@Nullable File file) {
        try {
            return FileUtils.getCanonicalPath(file);
        }
        catch (IOException iOException) {
            return null;
        }
    }

    @SuppressFBWarnings(value={"IL_INFINITE_LOOP"})
    public static boolean isParentDirectory(@Nonnull File file, @Nonnull File file2) {
        ValueEnforcer.notNull(file, "SearchDirectory");
        ValueEnforcer.notNull(file2, "StartDirectory");
        File file3 = file.getAbsoluteFile();
        File file4 = file2.getAbsoluteFile();
        try {
            file3 = FileUtils.getCanonicalFile(file3);
            file4 = FileUtils.getCanonicalFile(file4);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        if (!file3.isDirectory()) {
            return false;
        }
        for (File file5 = file4; file5 != null; file5 = file5.getParentFile()) {
            if (!file5.equals(file3)) continue;
            return true;
        }
        return false;
    }

    @CheckForSigned
    public static long getFileSize(@Nonnull @WillNotClose FileChannel fileChannel) {
        if (fileChannel != null) {
            try {
                return fileChannel.size();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return -1L;
    }

    @Nullable
    private static FileInputStream _getFileInputStream(@Nonnull File file) {
        try {
            return new CountingFileInputStream(file);
        }
        catch (FileNotFoundException fileNotFoundException) {
            return null;
        }
    }

    @Nullable
    private static InputStream _getMappedInputStream(@Nonnull @WillNotClose FileChannel fileChannel, @Nonnull File file) {
        try {
            MappedByteBuffer mappedByteBuffer = fileChannel.map(FileChannel.MapMode.READ_ONLY, 0L, fileChannel.size());
            s_aLogger.info("Created memory mapped input stream for " + file);
            return new ByteBufferInputStream(mappedByteBuffer);
        }
        catch (IOException iOException) {
            s_aLogger.warn("Failed to create memory mapped input stream for " + file + ": " + iOException.getMessage());
            return null;
        }
    }

    @Nullable
    public static FileChannel getFileReadChannel(@Nonnull String string) {
        return FileUtils.getFileReadChannel(new File(string));
    }

    @Nullable
    public static FileChannel getFileReadChannel(@Nonnull File file) {
        ValueEnforcer.notNull(file, "File");
        FileInputStream fileInputStream = FileUtils._getFileInputStream(file);
        return fileInputStream == null ? null : fileInputStream.getChannel();
    }

    @Nullable
    public static InputStream getInputStream(@Nonnull String string) {
        return FileUtils.getInputStream(new File(string));
    }

    @Nullable
    public static InputStream getInputStream(@Nonnull File file) {
        ValueEnforcer.notNull(file, "File");
        FileInputStream fileInputStream = FileUtils._getFileInputStream(file);
        return fileInputStream;
    }

    @Nullable
    public static InputStream getMappedInputStream(@Nonnull File file) {
        ValueEnforcer.notNull(file, "File");
        FileInputStream fileInputStream = FileUtils._getFileInputStream(file);
        if (fileInputStream == null) {
            return null;
        }
        InputStream inputStream = FileUtils._getMappedInputStream(fileInputStream.getChannel(), file);
        if (inputStream != null) {
            return inputStream;
        }
        s_aLogger.warn("Failed to map file " + file + ". Falling though to regular FileInputStream");
        return fileInputStream;
    }

    @Nullable
    private static FileOutputStream _getFileOutputStream(@Nonnull File file, @Nonnull EAppend eAppend) {
        try {
            return new CountingFileOutputStream(file, eAppend);
        }
        catch (FileNotFoundException fileNotFoundException) {
            s_aLogger.warn("Failed to create output stream for '" + file + "'; append: " + eAppend + ": " + fileNotFoundException.getClass().getName() + " - " + fileNotFoundException.getMessage());
            return null;
        }
    }

    @Nullable
    private static OutputStream _getMappedOutputStream(@Nonnull @WillNotClose FileChannel fileChannel, @Nonnull File file) {
        try {
            MappedByteBuffer mappedByteBuffer = fileChannel.map(FileChannel.MapMode.READ_WRITE, 0L, Integer.MAX_VALUE);
            s_aLogger.info("Created memory mapped output stream for " + file);
            return new ByteBufferOutputStream(mappedByteBuffer, false);
        }
        catch (IOException iOException) {
            s_aLogger.warn("Failed to create memory mapped output stream for " + file + ": " + iOException.getMessage());
            return null;
        }
    }

    @Nullable
    public static FileChannel getFileWriteChannel(@Nonnull String string) {
        return FileUtils.getFileWriteChannel(string, EAppend.DEFAULT);
    }

    @Nullable
    public static FileChannel getFileWriteChannel(@Nonnull String string, @Nonnull EAppend eAppend) {
        return FileUtils.getFileWriteChannel(new File(string), eAppend);
    }

    @Nullable
    public static FileChannel getFileWriteChannel(@Nonnull File file) {
        return FileUtils.getFileWriteChannel(file, EAppend.DEFAULT);
    }

    @Nullable
    public static FileChannel getFileWriteChannel(@Nonnull File file, @Nonnull EAppend eAppend) {
        ValueEnforcer.notNull(file, "File");
        ValueEnforcer.notNull(eAppend, "Append");
        FileOutputStream fileOutputStream = FileUtils._getFileOutputStream(file, eAppend);
        return fileOutputStream == null ? null : fileOutputStream.getChannel();
    }

    @Nullable
    public static OutputStream getOutputStream(@Nonnull String string) {
        return FileUtils.getOutputStream(string, EAppend.DEFAULT);
    }

    @Nullable
    public static OutputStream getOutputStream(@Nonnull String string, @Nonnull EAppend eAppend) {
        return FileUtils.getOutputStream(new File(string), eAppend);
    }

    @Nullable
    public static OutputStream getOutputStream(@Nonnull File file) {
        return FileUtils.getOutputStream(file, EAppend.DEFAULT);
    }

    @Nonnull
    private static EValidity _checkParentDirectoryExistanceAndAccess(@Nonnull File file) {
        try {
            FileUtils.ensureParentDirectoryIsPresent(file);
        }
        catch (IllegalStateException illegalStateException) {
            s_aLogger.warn("Failed to create parent directory of '" + file + "'", (Throwable)illegalStateException);
            return EValidity.INVALID;
        }
        File file2 = file.getParentFile();
        if (file2 != null && !FileUtils.canWrite(file2)) {
            s_aLogger.warn("Parent directory '" + file2 + "' of '" + file + "' is not writable for current user '" + SystemProperties.getUserName() + "'");
            return EValidity.INVALID;
        }
        return EValidity.VALID;
    }

    @Nullable
    public static OutputStream getOutputStream(@Nonnull File file, @Nonnull EAppend eAppend) {
        ValueEnforcer.notNull(file, "File");
        ValueEnforcer.notNull(eAppend, "Append");
        if (FileUtils._checkParentDirectoryExistanceAndAccess(file).isInvalid()) {
            return null;
        }
        return FileUtils._getFileOutputStream(file, eAppend);
    }

    @Nullable
    public static OutputStream getMappedOutputStream(@Nonnull String string) {
        return FileUtils.getMappedOutputStream(new File(string));
    }

    @Nullable
    public static OutputStream getMappedOutputStream(@Nonnull String string, @Nonnull EAppend eAppend) {
        return FileUtils.getMappedOutputStream(new File(string), eAppend);
    }

    @Nullable
    public static OutputStream getMappedOutputStream(@Nonnull File file) {
        return FileUtils.getMappedOutputStream(file, EAppend.DEFAULT);
    }

    @Nullable
    public static OutputStream getMappedOutputStream(@Nonnull File file, @Nonnull EAppend eAppend) {
        ValueEnforcer.notNull(file, "File");
        ValueEnforcer.notNull(eAppend, "Append");
        if (FileUtils._checkParentDirectoryExistanceAndAccess(file).isInvalid()) {
            return null;
        }
        RandomAccessFile randomAccessFile = FileUtils.getRandomAccessFile(file, ERandomAccessFileMode.READ_WRITE);
        if (randomAccessFile == null) {
            s_aLogger.error("Failed to open random access file " + file);
            return null;
        }
        OutputStream outputStream = FileUtils._getMappedOutputStream(randomAccessFile.getChannel(), file);
        if (outputStream != null) {
            return outputStream;
        }
        StreamUtils.close(randomAccessFile);
        s_aLogger.warn("Failed to map file " + file + ". Falling though to regular FileOutputStream");
        return FileUtils._getFileOutputStream(file, eAppend);
    }

    @Nullable
    public static RandomAccessFile getRandomAccessFile(@Nonnull String string, @Nonnull ERandomAccessFileMode eRandomAccessFileMode) {
        return FileUtils.getRandomAccessFile(new File(string), eRandomAccessFileMode);
    }

    @Nullable
    public static RandomAccessFile getRandomAccessFile(@Nonnull File file, @Nonnull ERandomAccessFileMode eRandomAccessFileMode) {
        ValueEnforcer.notNull(file, "File");
        ValueEnforcer.notNull(eRandomAccessFileMode, "Mode");
        try {
            return new RandomAccessFile(file, eRandomAccessFileMode.getMode());
        }
        catch (FileNotFoundException fileNotFoundException) {
            return null;
        }
    }

    public static boolean isFileNewer(@Nonnull File file, @Nonnull File file2) {
        ValueEnforcer.notNull(file, "File1");
        ValueEnforcer.notNull(file2, "aFile2");
        if (file.equals(file2)) {
            return false;
        }
        if (!file.exists()) {
            return false;
        }
        if (!file2.exists()) {
            return true;
        }
        return file.lastModified() > file2.lastModified();
    }

    @Nonnull
    public static String getFileSizeDisplay(@Nullable File file) {
        return FileUtils.getFileSizeDisplay(file, 0);
    }

    @Nonnull
    public static String getFileSizeDisplay(@Nullable File file, @Nonnegative int n) {
        if (file != null && file.exists()) {
            return FileUtils.getFileSizeDisplay(file.length(), n);
        }
        return "";
    }

    @Nonnull
    public static String getFileSizeDisplay(long l) {
        return FileUtils.getFileSizeDisplay(l, 0);
    }

    @Nonnull
    public static String getFileSizeDisplay(@Nonnegative long l, @Nonnegative int n) {
        ValueEnforcer.isGE0(l, "FileSize");
        ValueEnforcer.isGE0(n, "Decimals");
        return SizeHelper.getSizeHelperOfLocale(CGlobal.LOCALE_FIXED_NUMBER_FORMAT).getAsMatching(l, n);
    }

    @Nullable
    public static File getSecureFile(@Nullable File file) {
        if (file == null) {
            return null;
        }
        String string = null;
        for (File file2 = new File(FilenameHelper.getCleanPath(file)); file2 != null; file2 = file2.getParentFile()) {
            String string2 = file2.getParent() == null ? file2.getPath() : FilenameHelper.getAsSecureValidFilename(file2.getName());
            string = string == null ? string2 : string2 + "/" + string;
        }
        return new File(string);
    }

    @Nonnegative
    public static int getDirectoryObjectCount(@Nonnull File file) {
        ValueEnforcer.notNull(file, "Directory");
        if (!file.isDirectory()) {
            throw new IllegalArgumentException("Passed object is not a directory: " + file);
        }
        int n = 0;
        for (File file2 : FileUtils.getDirectoryContent(file)) {
            if (FilenameHelper.isSystemInternalDirectory(file2.getName())) continue;
            ++n;
        }
        return n;
    }

    @Nonnull
    @ReturnsMutableCopy
    private static List<File> _getDirectoryContent(@Nonnull File file, @Nullable File[] fileArray) {
        if (fileArray == null) {
            if (!file.isDirectory()) {
                s_aLogger.warn("Cannot list non-directory: " + file.getAbsolutePath());
            } else if (!FileUtils.canExecute(file)) {
                s_aLogger.warn("Existing directory is missing the listing permission: " + file.getAbsolutePath());
            } else if (!FileUtils.canRead(file)) {
                s_aLogger.warn("Cannot list directory because of no read-rights: " + file.getAbsolutePath());
            } else {
                s_aLogger.warn("Directory listing failed because of IO error: " + file.getAbsolutePath());
            }
        } else if (!FileUtils.canExecute(file)) {
            s_aLogger.warn("Directory is missing the listing permission: " + file.getAbsolutePath());
        }
        return ContainerHelper.newList(fileArray);
    }

    @Nonnull
    @ReturnsMutableCopy
    public static List<File> getDirectoryContent(@Nonnull File file) {
        ValueEnforcer.notNull(file, "Directory");
        return FileUtils._getDirectoryContent(file, file.listFiles());
    }

    @Nonnull
    @ReturnsMutableCopy
    public static List<File> getDirectoryContent(@Nonnull File file, @Nonnull FilenameFilter filenameFilter) {
        ValueEnforcer.notNull(file, "Directory");
        ValueEnforcer.notNull(filenameFilter, "FilenameFilter");
        return FileUtils._getDirectoryContent(file, file.listFiles(filenameFilter));
    }

    @Nonnull
    @ReturnsMutableCopy
    public static List<File> getDirectoryContent(@Nonnull File file, @Nonnull FileFilter fileFilter) {
        ValueEnforcer.notNull(file, "Directory");
        ValueEnforcer.notNull(fileFilter, "FileFilter");
        return FileUtils._getDirectoryContent(file, file.listFiles(fileFilter));
    }
}

