/**
 * Copyright (C) 2014-2015 Philip Helger (www.helger.com)
 * philip[at]helger[dot]com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.helger.commons.collections.multimap;

import java.util.Map;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.helger.commons.state.EChange;

/**
 * Base interface for a multi map (one key with several values).
 *
 * @author Philip Helger
 * @param <KEYTYPE1>
 *        Key type of outer map
 * @param <KEYTYPE2>
 *        Key type of inner map
 * @param <VALUETYPE>
 *        Element type
 */
public interface IMultiMapMapBased <KEYTYPE1, KEYTYPE2, VALUETYPE> extends Map <KEYTYPE1, Map <KEYTYPE2, VALUETYPE>>
{
  /**
   * Get or create the collection of the specified key.
   *
   * @param aKey
   *        The key to use. May not be <code>null</code>.
   * @return {@link EChange}
   */
  @Nonnull
  Map <KEYTYPE2, VALUETYPE> getOrCreate (@Nonnull KEYTYPE1 aKey);

  /**
   * Add a single value into the container identified by the passed key.
   *
   * @param aKey
   *        The key to use. May not be <code>null</code>.
   * @param aInnerKey
   *        The key for the inner map to use. May not be <code>null</code>.
   * @param aValue
   *        The value to be added. May be <code>null</code>.
   * @return {@link EChange}
   */
  @Nonnull
  EChange putSingle (@Nonnull KEYTYPE1 aKey, @Nonnull KEYTYPE2 aInnerKey, @Nullable VALUETYPE aValue);

  /**
   * Add all values into the container identified by the passed key-value-map.
   *
   * @param aMap
   *        The key-value-map to use. May not be <code>null</code>.
   * @return {@link EChange}
   */
  @Nonnull
  EChange putAllIn (@Nonnull Map <? extends KEYTYPE1, ? extends Map <KEYTYPE2, VALUETYPE>> aMap);

  /**
   * Remove a single element from the container identified by the passed key.
   *
   * @param aKey
   *        The key to use. May not be <code>null</code>.
   * @param aInnerKey
   *        The key for the inner map to be removed. May be <code>null</code>.
   * @return {@link EChange}
   */
  @Nonnull
  EChange removeSingle (@Nonnull KEYTYPE1 aKey, @Nonnull KEYTYPE2 aInnerKey);

  /**
   * Check a single element from the container identified by the passed key is
   * present.
   *
   * @param aKey
   *        The key to use. May not be <code>null</code>.
   * @param aInnerKey
   *        The key of the inner map to be checked. May be <code>null</code>.
   * @return <code>true</code> if contained, <code>false</code> otherwise.
   */
  boolean containsSingle (@Nonnull KEYTYPE1 aKey, @Nonnull KEYTYPE2 aInnerKey);

  /**
   * @return The total number of contained values recursively over all contained
   *         maps. Always &ge; 0.
   */
  @Nonnegative
  long getTotalValueCount ();
}
