/*
 * Decompiled with CFR 0.152.
 */
package com.helger.commons.charset;

import com.helger.commons.ValueEnforcer;
import com.helger.commons.annotation.CodingStyleguideUnaware;
import com.helger.commons.annotation.Nonempty;
import com.helger.commons.annotation.ReturnsMutableCopy;
import com.helger.commons.charset.EUnicodeBOM;
import com.helger.commons.collection.ArrayHelper;
import com.helger.commons.collection.impl.CommonsLinkedHashMap;
import com.helger.commons.collection.impl.ICommonsOrderedMap;
import com.helger.commons.io.IHasInputStream;
import com.helger.commons.io.stream.NonBlockingPushbackInputStream;
import com.helger.commons.io.stream.StreamHelper;
import com.helger.commons.string.StringHelper;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.Map;
import java.util.SortedMap;
import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.WillNotClose;
import javax.annotation.concurrent.Immutable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Immutable
public final class CharsetHelper {
    private static final Logger s_aLogger = LoggerFactory.getLogger(CharsetHelper.class);
    @CodingStyleguideUnaware
    private static final SortedMap<String, Charset> s_aAllCharsets = Charset.availableCharsets();
    private static final CharsetHelper s_aInstance = new CharsetHelper();

    private CharsetHelper() {
    }

    @Nonnull
    public static Charset getCharsetFromName(@Nonnull @Nonempty String string) {
        ValueEnforcer.notNull(string, "CharsetName");
        try {
            return Charset.forName(string);
        }
        catch (IllegalCharsetNameException illegalCharsetNameException) {
            throw new IllegalArgumentException("Charset '" + string + "' unsupported in Java", illegalCharsetNameException);
        }
        catch (UnsupportedCharsetException unsupportedCharsetException) {
            throw new IllegalArgumentException("Charset '" + string + "' unsupported on this platform", unsupportedCharsetException);
        }
    }

    @Nullable
    public static Charset getCharsetFromNameOrNull(@Nullable String string) {
        return CharsetHelper.getCharsetFromNameOrNull(string, null);
    }

    @Nullable
    public static Charset getCharsetFromNameOrNull(@Nullable String string, @Nullable Charset charset) {
        if (StringHelper.hasText(string)) {
            try {
                return CharsetHelper.getCharsetFromName(string);
            }
            catch (IllegalArgumentException illegalArgumentException) {
                // empty catch block
            }
        }
        return charset;
    }

    @Nonnull
    @ReturnsMutableCopy
    public static ICommonsOrderedMap<String, Charset> getAllCharsets() {
        return new CommonsLinkedHashMap<String, Charset>((Map<String, Charset>)s_aAllCharsets);
    }

    @Nullable
    public static String getAsStringInOtherCharset(@Nullable String string, @Nonnull Charset charset, @Nonnull Charset charset2) {
        ValueEnforcer.notNull(charset, "CurrentCharset");
        ValueEnforcer.notNull(charset2, "NewCharset");
        if (string == null || charset.equals(charset2)) {
            return string;
        }
        return new String(string.getBytes(charset), charset2);
    }

    @Nonnegative
    public static int getUTF8ByteCount(@Nullable String string) {
        return string == null ? 0 : CharsetHelper.getUTF8ByteCount(string.toCharArray());
    }

    @Nonnegative
    public static int getUTF8ByteCount(@Nullable char[] cArray) {
        int n = 0;
        if (cArray != null) {
            for (char c : cArray) {
                n += CharsetHelper.getUTF8ByteCount(c);
            }
        }
        return n;
    }

    @Nonnegative
    public static int getUTF8ByteCount(char c) {
        return CharsetHelper.getUTF8ByteCount((int)c);
    }

    @Nonnegative
    public static int getUTF8ByteCount(@Nonnegative int n) {
        ValueEnforcer.isBetweenInclusive(n, "c", 0, 65535);
        if (n == 0) {
            return 2;
        }
        if (n <= 127) {
            return 1;
        }
        if (n <= 2047) {
            return 2;
        }
        if (n <= 55295) {
            return 3;
        }
        return 0;
    }

    @Nonnull
    public static InputStreamAndCharset getInputStreamAndCharsetFromBOM(@Nonnull @WillNotClose InputStream inputStream) {
        ValueEnforcer.notNull(inputStream, "InputStream");
        int n = EUnicodeBOM.getMaximumByteCount();
        NonBlockingPushbackInputStream nonBlockingPushbackInputStream = new NonBlockingPushbackInputStream(StreamHelper.getBuffered(inputStream), n);
        try {
            byte[] byArray = new byte[n];
            int n2 = nonBlockingPushbackInputStream.read(byArray);
            EUnicodeBOM eUnicodeBOM = null;
            Charset charset = null;
            if (n2 > 0) {
                eUnicodeBOM = EUnicodeBOM.getFromBytesOrNull(ArrayHelper.getCopy(byArray, 0, n2));
                if (eUnicodeBOM == null) {
                    nonBlockingPushbackInputStream.unread(byArray, 0, n2);
                } else {
                    int n3;
                    if (s_aLogger.isDebugEnabled()) {
                        s_aLogger.debug("Found " + (Object)((Object)eUnicodeBOM) + " on " + inputStream.getClass().getName());
                    }
                    if ((n3 = eUnicodeBOM.getByteCount()) < n2) {
                        nonBlockingPushbackInputStream.unread(byArray, n3, n2 - n3);
                    }
                    charset = eUnicodeBOM.getCharset();
                }
            }
            return new InputStreamAndCharset(nonBlockingPushbackInputStream, eUnicodeBOM, charset);
        }
        catch (IOException iOException) {
            s_aLogger.error("Failed to determine BOM", (Throwable)iOException);
            throw new UncheckedIOException(iOException);
        }
    }

    @Nonnull
    public static InputStreamReader getReaderByBOM(@Nonnull InputStream inputStream, @Nonnull Charset charset) {
        ValueEnforcer.notNull(inputStream, "InputStream");
        ValueEnforcer.notNull(charset, "FallbackCharset");
        InputStreamAndCharset inputStreamAndCharset = CharsetHelper.getInputStreamAndCharsetFromBOM(inputStream);
        Charset charset2 = inputStreamAndCharset.getCharset(charset);
        return StreamHelper.createReader(inputStreamAndCharset.getInputStream(), charset2);
    }

    public static final class InputStreamAndCharset
    implements IHasInputStream {
        private final InputStream m_aIS;
        private final EUnicodeBOM m_eBOM;
        private final Charset m_aCharset;

        public InputStreamAndCharset(@Nonnull InputStream inputStream, @Nullable EUnicodeBOM eUnicodeBOM, @Nullable Charset charset) {
            this.m_aIS = inputStream;
            this.m_eBOM = eUnicodeBOM;
            this.m_aCharset = charset;
        }

        @Override
        @Nonnull
        public InputStream getInputStream() {
            return this.m_aIS;
        }

        @Nullable
        public EUnicodeBOM getBOM() {
            return this.m_eBOM;
        }

        public boolean hasBOM() {
            return this.m_eBOM != null;
        }

        @Nullable
        public Charset getCharset() {
            return this.m_aCharset;
        }

        public boolean hasCharset() {
            return this.m_aCharset != null;
        }

        @Nullable
        public Charset getCharset(@Nullable Charset charset) {
            return this.m_aCharset != null ? this.m_aCharset : charset;
        }
    }
}

