/*
 * Decompiled with CFR 0.152.
 */
package com.here.account.auth;

import com.here.account.auth.SignatureMethod;
import com.here.account.util.OAuthConstants;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

public class SignatureCalculator {
    private static final Logger LOG = Logger.getLogger(SignatureCalculator.class.getName());
    private final String consumerKey;
    private final String consumerSecret;
    public static final String ELLIPTIC_CURVE_ALGORITHM = "EC";

    public SignatureCalculator(String consumerKey, String consumerSecret) {
        this.consumerKey = consumerKey;
        this.consumerSecret = consumerSecret;
    }

    public String calculateSignature(String method, String baseURL, long oauthTimestamp, String nonce, SignatureMethod signatureMethod, Map<String, List<String>> formParams, Map<String, List<String>> queryParams) {
        return this.calculateSignature(method, baseURL, oauthTimestamp, nonce, signatureMethod, "1.0", formParams, queryParams);
    }

    public String calculateSignature(String method, String baseURL, long oauthTimestamp, String nonce, SignatureMethod signatureMethod, String oauthVersion, Map<String, List<String>> formParams, Map<String, List<String>> queryParams) {
        String signatureBaseString = SignatureCalculator.computeSignatureBaseString(this.consumerKey, method, baseURL, oauthTimestamp, nonce, signatureMethod, oauthVersion, formParams, queryParams);
        return SignatureCalculator.generateSignature(signatureBaseString.toString(), this.consumerSecret, signatureMethod);
    }

    public String constructAuthHeader(String signature, String nonce, long oauthTimestamp, SignatureMethod signatureMethod) {
        return "OAuth " + "oauth_consumer_key" + "=\"" + SignatureCalculator.urlEncode(this.consumerKey) + "\", " + "oauth_signature_method" + "=\"" + signatureMethod.getOauth1SignatureMethod() + "\", " + "oauth_signature" + "=\"" + SignatureCalculator.urlEncode(signature) + "\", " + "oauth_timestamp" + "=\"" + oauthTimestamp + "\", " + "oauth_nonce" + "=\"" + SignatureCalculator.urlEncode(nonce) + "\", " + "oauth_version" + "=\"" + "1.0" + "\"";
    }

    public static boolean verifySignature(String consumerKey, String method, String baseURL, long oauthTimestamp, String nonce, SignatureMethod signatureMethod, Map<String, List<String>> formParams, Map<String, List<String>> queryParams, String signatureToVerify, String verificationKey) {
        String signatureBaseString = SignatureCalculator.computeSignatureBaseString(consumerKey, method, baseURL, oauthTimestamp, nonce, signatureMethod, "1.0", formParams, queryParams);
        return SignatureCalculator.verifySignature(signatureBaseString, signatureMethod, signatureToVerify, verificationKey);
    }

    protected static boolean verifySignature(String signedText, SignatureMethod signatureMethod, String signatureToVerify, String verificationKey) {
        if (signatureMethod.equals((Object)SignatureMethod.ES512)) {
            return SignatureCalculator.verifyECDSASignature(signedText, signatureToVerify, verificationKey, signatureMethod);
        }
        return SignatureCalculator.generateSignature(signedText, verificationKey, signatureMethod).equals(signatureToVerify);
    }

    private static String computeSignatureBaseString(String consumerKey, String method, String baseURL, long oauthTimestamp, String nonce, SignatureMethod signatureMethod, String oauthVersion, Map<String, List<String>> formParams, Map<String, List<String>> queryParams) {
        List<String> values;
        StringBuilder signatureBaseString = new StringBuilder(100);
        signatureBaseString.append(method.toUpperCase());
        signatureBaseString.append('&');
        signatureBaseString.append(SignatureCalculator.urlEncode(SignatureCalculator.normalizeBaseURL(baseURL)));
        OAuthParameterSet parameterSet = new OAuthParameterSet();
        parameterSet.add("oauth_consumer_key", consumerKey);
        parameterSet.add("oauth_nonce", nonce);
        parameterSet.add("oauth_signature_method", signatureMethod.getOauth1SignatureMethod());
        parameterSet.add("oauth_timestamp", String.valueOf(oauthTimestamp));
        if (null != oauthVersion) {
            parameterSet.add("oauth_version", oauthVersion);
        }
        if (formParams != null && !formParams.isEmpty()) {
            for (String key : formParams.keySet()) {
                values = formParams.get(key);
                for (String value : values) {
                    parameterSet.add(key, value);
                }
            }
        }
        if (queryParams != null && !queryParams.isEmpty()) {
            for (String key : queryParams.keySet()) {
                values = queryParams.get(key);
                for (String value : values) {
                    parameterSet.add(key, value);
                }
            }
        }
        String parameterString = parameterSet.sortAndConcat();
        signatureBaseString.append('&');
        signatureBaseString.append(SignatureCalculator.urlEncode(parameterString));
        if (LOG.isLoggable(Level.FINE)) {
            LOG.fine("signatureBaseString=" + signatureBaseString);
        }
        return signatureBaseString.toString();
    }

    private static String generateSignature(String signatureBaseString, String key, SignatureMethod signatureMethod) {
        byte[] bytesToSign = signatureBaseString.getBytes(OAuthConstants.UTF_8_CHARSET);
        try {
            if (signatureMethod.equals((Object)SignatureMethod.ES512)) {
                return SignatureCalculator.computeECDSASignature(bytesToSign, key, signatureMethod.getAlgorithm());
            }
            return SignatureCalculator.computeHMACSignature(bytesToSign, key, signatureMethod.getAlgorithm());
        }
        catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    static String urlEncode(String s) {
        try {
            return URLEncoder.encode(s, "UTF-8").replaceAll("\\+", "%20");
        }
        catch (UnsupportedEncodingException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static String computeECDSASignature(byte[] bytesToSign, String key, String algorithm) {
        try {
            Signature s = Signature.getInstance(algorithm);
            s.initSign(SignatureCalculator.consumerSecretToEllipticCurvePrivateKey(key));
            s.update(bytesToSign);
            return Base64.getEncoder().encodeToString(s.sign());
        }
        catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static String computeHMACSignature(byte[] bytesToSign, String key, String algorithm) {
        try {
            byte[] keyBytes = (SignatureCalculator.urlEncode(key) + "&").getBytes(OAuthConstants.UTF_8_CHARSET);
            SecretKeySpec signingKey = new SecretKeySpec(keyBytes, algorithm);
            Mac mac = Mac.getInstance(algorithm);
            mac.init(signingKey);
            byte[] signedBytes = mac.doFinal(bytesToSign);
            return Base64.getEncoder().encodeToString(signedBytes);
        }
        catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static PrivateKey consumerSecretToEllipticCurvePrivateKey(String key) {
        try {
            byte[] keyBytes = Base64.getDecoder().decode(key);
            PKCS8EncodedKeySpec privateSpec = new PKCS8EncodedKeySpec(keyBytes);
            KeyFactory kf = KeyFactory.getInstance(ELLIPTIC_CURVE_ALGORITHM);
            return kf.generatePrivate(privateSpec);
        }
        catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static String normalizeBaseURL(String baseURL) {
        int index;
        if (baseURL.startsWith("http:")) {
            int index2 = baseURL.indexOf(":80/", 4);
            if (index2 > 0) {
                baseURL = baseURL.substring(0, index2) + baseURL.substring(index2 + 3);
            }
        } else if (baseURL.startsWith("https:") && (index = baseURL.indexOf(":443/", 5)) > 0) {
            baseURL = baseURL.substring(0, index) + baseURL.substring(index + 4);
        }
        return baseURL;
    }

    private static boolean verifyECDSASignature(String cipherText, String signature, String verificationKey, SignatureMethod signatureMethod) {
        try {
            byte[] keyBytes = Base64.getDecoder().decode(verificationKey);
            X509EncodedKeySpec publicSpec = new X509EncodedKeySpec(keyBytes);
            KeyFactory kf = KeyFactory.getInstance(ELLIPTIC_CURVE_ALGORITHM);
            PublicKey pubKey = kf.generatePublic(publicSpec);
            byte[] signatureBytes = Base64.getDecoder().decode(signature.getBytes("UTF-8"));
            Signature s = Signature.getInstance(signatureMethod.getAlgorithm());
            s.initVerify(pubKey);
            s.update(cipherText.getBytes("UTF-8"));
            return s.verify(signatureBytes);
        }
        catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static final class Parameter
    implements Comparable<Parameter> {
        private final String key;
        private final String value;

        private Parameter(String key, String value) {
            this.key = key;
            this.value = value;
        }

        private String getKey() {
            return this.key;
        }

        private String getValue() {
            return this.value;
        }

        @Override
        public int compareTo(Parameter other) {
            int diff = this.key.compareTo(other.key);
            if (diff == 0) {
                diff = this.value.compareTo(other.value);
            }
            return diff;
        }
    }

    private static final class OAuthParameterSet {
        private final List<Parameter> allParameters = new ArrayList<Parameter>();

        private OAuthParameterSet() {
        }

        private List<Parameter> add(String key, String value) {
            this.allParameters.add(new Parameter(SignatureCalculator.urlEncode(key), SignatureCalculator.urlEncode(value)));
            return this.allParameters;
        }

        private String sortAndConcat() {
            Object[] params = new Parameter[this.allParameters.size()];
            this.allParameters.toArray(params);
            Arrays.sort(params);
            StringBuilder encodedParams = new StringBuilder(100);
            for (Object param : params) {
                if (encodedParams.length() > 0) {
                    encodedParams.append('&');
                }
                encodedParams.append(((Parameter)param).getKey()).append('=').append(((Parameter)param).getValue());
            }
            return encodedParams.toString();
        }
    }
}

