/*
 * Decompiled with CFR 0.152.
 */
package com.hsbc.cranker.mucranker;

import com.hsbc.cranker.mucranker.DarkHost;
import com.hsbc.cranker.mucranker.RouteResolver;
import com.hsbc.cranker.mucranker.RouterSocket;
import io.muserver.MuRequest;
import io.muserver.MuResponse;
import io.muserver.Mutils;
import io.netty.util.HashedWheelTimer;
import io.netty.util.Timeout;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.ObjLongConsumer;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class WebSocketFarm {
    private static final Logger log = LoggerFactory.getLogger(WebSocketFarm.class);
    private static final String MU_ID = "muid";
    private final RouteResolver routeResolver;
    private final Map<String, Queue<RouterSocket>> sockets = new ConcurrentHashMap<String, Queue<RouterSocket>>();
    private final Map<String, Queue<WaitingSocketTask>> waitingTasks = new ConcurrentHashMap<String, Queue<WaitingSocketTask>>();
    private final Map<String, Long> routeLastRemovalTimes = new ConcurrentHashMap<String, Long>();
    private final AtomicInteger idleCount = new AtomicInteger(0);
    private final AtomicInteger waitingTaskCount = new AtomicInteger(0);
    private final ConcurrentHashMap.KeySetView<DarkHost, Boolean> darkHosts = ConcurrentHashMap.newKeySet();
    private volatile boolean hasCatchAll = false;
    private final long maxWaitInMillis;
    private final ExecutorService executor = Executors.newSingleThreadExecutor(runnable -> new Thread(runnable, "websocket-farm-execution"));
    private final HashedWheelTimer timer = new HashedWheelTimer(runnable -> new Thread(runnable, "websocket-farm-timer"));

    public WebSocketFarm(RouteResolver routeResolver, long maxWaitInMillis) {
        this.routeResolver = routeResolver;
        this.maxWaitInMillis = maxWaitInMillis;
    }

    public void start() {
        this.timer.start();
    }

    public void stop() {
        this.timer.stop();
        this.executor.shutdown();
        for (Queue<RouterSocket> queue : this.sockets.values()) {
            for (RouterSocket routerSocket : queue) {
                routerSocket.socketSessionClose();
            }
        }
        this.sockets.clear();
        this.waitingTasks.clear();
    }

    public void cleanRoutes(long routesKeepTimeMillis) {
        long cutoffTime = System.currentTimeMillis() - routesKeepTimeMillis;
        this.sockets.entrySet().stream().filter(entry -> entry.getValue() != null && ((Queue)entry.getValue()).size() == 0 && this.routeLastRemovalTimes.containsKey(entry.getKey()) && this.routeLastRemovalTimes.get(entry.getKey()) < cutoffTime).forEach(entry -> {
            log.info("removing registration info for {}, consequence requests to {} will receive 404", entry.getKey(), entry.getKey());
            this.sockets.remove(entry.getKey());
            this.routeLastRemovalTimes.remove(entry.getKey());
        });
    }

    public boolean canHandle(String target, boolean useCatchAll) {
        String routeKey = this.resolveRouteKey(target, useCatchAll);
        if (routeKey == null) {
            return false;
        }
        Queue<RouterSocket> routerSockets = this.sockets.get(routeKey);
        return routerSockets != null && routerSockets.size() > 0;
    }

    public int idleCount() {
        return this.idleCount.get();
    }

    public void removeWebSocketAsync(String route, RouterSocket socket, Runnable onRemoveSuccess) {
        this.executor.submit(() -> ThrowingFunction.logIfFail(() -> {
            this.routeLastRemovalTimes.put(route, System.currentTimeMillis());
            boolean removed = false;
            Queue<RouterSocket> routerSockets = this.sockets.get(route);
            if (routerSockets != null) {
                removed = routerSockets.remove(socket);
            }
            if (removed) {
                this.idleCount.decrementAndGet();
                onRemoveSuccess.run();
            }
        }));
    }

    public void addWebSocketAsync(String route, RouterSocket socket) {
        this.executor.submit(() -> ThrowingFunction.logIfFail(() -> this.addWebSocketSync(route, socket)));
    }

    private void addWebSocketSync(String route, RouterSocket socket) {
        WaitingSocketTask waitTask;
        Queue<WaitingSocketTask> waiting;
        if (socket.isCatchAll()) {
            this.hasCatchAll = true;
        }
        if ((waiting = this.waitingTasks.get(route)) != null && waiting.size() > 0 && (waitTask = waiting.poll()) != null) {
            waitTask.notifySuccess(socket);
            return;
        }
        this.sockets.putIfAbsent(route, new ConcurrentLinkedQueue());
        Queue<RouterSocket> queue = this.sockets.get(route);
        if (queue.offer(socket)) {
            this.idleCount.incrementAndGet();
        }
    }

    private RouterSocket getRouterSocket(String routeKey) {
        Queue<RouterSocket> routerSockets = this.sockets.get(routeKey);
        RouterSocket socket = this.darkHosts.isEmpty() ? routerSockets.poll() : this.getNonDarkSocket(routerSockets, this.darkHosts);
        return socket;
    }

    private long peekTime(long start) {
        return System.currentTimeMillis() - start;
    }

    public void acquireSocket(String target, boolean useCatchall, MuRequest clientRequest, MuResponse clientResponse, ObjLongConsumer<RouterSocket> onSuccess, SocketAcquireFailedListener onFailure) {
        if (clientResponse.responseState().endState()) {
            log.info("client response state is {}, skip processing, muid={}", (Object)clientResponse.responseState(), clientRequest.attribute(MU_ID));
            return;
        }
        String routeKey = this.resolveRouteKey(target, useCatchall);
        if (routeKey == null) {
            onFailure.accept(404, 0L, "404 Not Found", "Page not found");
            return;
        }
        if ("*".equals(routeKey) && !this.hasCatchAll) {
            onFailure.accept(404, 0L, "404 Not Found", "Page not found");
            return;
        }
        long[] startTime = new long[]{System.currentTimeMillis()};
        this.executor.submit(() -> ThrowingFunction.logIfFail(() -> {
            RouterSocket routerSocket = this.getRouterSocket(routeKey);
            if (routerSocket != null) {
                this.idleCount.decrementAndGet();
                this.routeLastRemovalTimes.put(routeKey, System.currentTimeMillis());
                onSuccess.accept(routerSocket, this.peekTime(startTime[0]));
                return;
            }
            this.waitingTasks.putIfAbsent(routeKey, new ConcurrentLinkedQueue());
            Queue<WaitingSocketTask> waiting = this.waitingTasks.get(routeKey);
            WaitingSocketTask waitingSocketTask = new WaitingSocketTask(target);
            Timeout timeoutHandle = this.timer.newTimeout(timeout -> this.executor.submit(() -> ThrowingFunction.logIfFail(() -> {
                if (timeout.isCancelled()) {
                    return;
                }
                waiting.remove(waitingSocketTask);
                this.waitingTaskCount.decrementAndGet();
                onFailure.accept(503, this.peekTime(startTime[0]), "503 Service Unavailable", String.format("No cranker connectors available within %s ms", this.maxWaitInMillis));
            })), this.maxWaitInMillis, TimeUnit.MILLISECONDS);
            waitingSocketTask.onSuccess(socket -> {
                waiting.remove(waitingSocketTask);
                this.waitingTaskCount.decrementAndGet();
                timeoutHandle.cancel();
                if (clientResponse.responseState().endState()) {
                    log.info("Connector available, but client response state is {}, skip processing, muid={}", (Object)clientResponse.responseState(), clientRequest.attribute(MU_ID));
                    this.addWebSocketSync(routeKey, (RouterSocket)socket);
                } else {
                    this.routeLastRemovalTimes.put(routeKey, System.currentTimeMillis());
                    onSuccess.accept((RouterSocket)socket, this.peekTime(startTime[0]));
                }
            });
            if (waiting.offer(waitingSocketTask)) {
                this.waitingTaskCount.incrementAndGet();
            }
        }));
    }

    private String resolveRouteKey(String target, boolean useCatchAll) {
        String resolved = this.routeResolver.resolve(this.sockets.keySet(), target);
        if (!useCatchAll && (resolved == null || "*".equals(resolved))) {
            return null;
        }
        return Objects.requireNonNullElse(resolved, "*");
    }

    private RouterSocket getNonDarkSocket(Queue<RouterSocket> routerSockets, ConcurrentHashMap.KeySetView<DarkHost, Boolean> darkHosts) {
        for (RouterSocket candidate : routerSockets) {
            boolean removed;
            if (candidate.isDarkModeOn(darkHosts) || !(removed = routerSockets.remove(candidate))) continue;
            return candidate;
        }
        return null;
    }

    public void deRegisterSocket(String target, String remoteAddr, String connectorInstanceID) {
        log.info("Going to deregister targetName=" + target + " and the targetAddr=" + remoteAddr + " and the connectorInstanceID=" + connectorInstanceID);
        Queue<RouterSocket> routerSockets = this.sockets.get(target);
        if (routerSockets != null) {
            routerSockets.forEach(a -> this.removeSockets(connectorInstanceID, (RouterSocket)a));
        }
    }

    private void removeSockets(String connectorInstanceID, RouterSocket routerSocket) {
        String currentConnectorInstanceID = routerSocket.connectorInstanceID();
        if (currentConnectorInstanceID.equals(connectorInstanceID)) {
            this.removeWebSocketAsync(routerSocket.route, routerSocket, routerSocket::socketSessionClose);
        }
    }

    Map<String, Queue<RouterSocket>> getSockets() {
        return this.sockets;
    }

    Map<String, List<String>> getWaitingTasks() {
        HashMap<String, List<String>> result = new HashMap<String, List<String>>();
        this.waitingTasks.forEach((key, value) -> result.put((String)key, value.stream().map(WaitingSocketTask::getTarget).collect(Collectors.toList())));
        return result;
    }

    void enableDarkMode(DarkHost darkHost) {
        Mutils.notNull((String)"darkHost", (Object)darkHost);
        boolean added = this.darkHosts.add(darkHost);
        if (added) {
            log.info("Enabled dark mode for " + darkHost);
        } else {
            log.info("Requested dark mode for " + darkHost + " but it was already in dark mode, so doing nothing.");
        }
    }

    void disableDarkMode(DarkHost darkHost) {
        Mutils.notNull((String)"darkHost", (Object)darkHost);
        boolean removed = this.darkHosts.remove(darkHost);
        if (removed) {
            log.info("Disabled dark mode for " + darkHost);
        } else {
            log.info("Requested to disable dark mode for " + darkHost + " but it was not in dark mode, so doing nothing.");
        }
    }

    Set<DarkHost> getDarkHosts() {
        return Set.copyOf(this.darkHosts);
    }

    static interface SocketAcquireFailedListener {
        public void accept(int var1, long var2, String var4, String var5);
    }

    private class WaitingSocketTask {
        private final String target;
        private Consumer<RouterSocket> successListener;

        public WaitingSocketTask(String target) {
            this.target = target;
        }

        public void notifySuccess(RouterSocket socket) {
            if (this.successListener != null) {
                this.successListener.accept(socket);
            }
        }

        public WaitingSocketTask onSuccess(Consumer<RouterSocket> successListener) {
            this.successListener = successListener;
            return this;
        }

        public String getTarget() {
            return this.target;
        }
    }

    @FunctionalInterface
    public static interface ThrowingFunction {
        public void run() throws Throwable;

        public static void logIfFail(ThrowingFunction f) {
            try {
                f.run();
            }
            catch (Throwable e) {
                log.warn("Exception", e);
            }
        }
    }
}

