package com.huawei.dli.jdbc;

import static com.huawei.dli.sdk.common.SqlDialect.SPARK;
import static com.huawei.dli.sdk.common.SqlDialect.HETU_ENGINE;

import com.huawei.dli.jdbc.model.DliException;
import com.huawei.dli.jdbc.utils.ConnectionResource;
import com.huawei.dli.jdbc.utils.ExceptionReporter;
import com.huawei.dli.jdbc.utils.LoggerFactory;
import com.huawei.dli.jdbc.utils.Reporter;
import com.huawei.dli.sdk.util.ApiTemplate;

import org.slf4j.Logger;

import java.sql.Array;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.NClob;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLClientInfoException;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Savepoint;
import java.sql.Statement;
import java.sql.Struct;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Executor;

public class DliConnection implements java.sql.Connection {
    private final static String JDBC_VERSION = "dli-jdbc/2.1.3";

    private final ConnectionResource connRes;

    private final Map<String, String> sessConfMap = new LinkedHashMap<>();

    private final List<Statement> stmtHandles;

    private boolean isClosed = false;

    protected final Logger log;

    public DliConnection(Properties info, String url) throws DliException {
        ApiTemplate.setVersion(JDBC_VERSION);
        connRes = ConnectionResource.constructConnectionResource(info, url);
        parseSqlProperties(info);
        stmtHandles = new ArrayList<>();

        log = LoggerFactory.getLogger(connRes.getLogConfFile(), getClass().getName());
        log.info("DLI.JDBC driver, Version " + ConnectionResource.SERVICE_VERSION);
        log.info(String.format("endpoint=%s, project=%s", connRes.getEndpoint(), connRes.getProjectId()));

        Reporter logRepoter = new Reporter() {
            @Override
            public void report(Throwable t) {
                log.error(t.getMessage());
            }
        };
        ExceptionReporter.setExceptionReporter(logRepoter);
    }

    private void parseSqlProperties(Properties info) {
        if (info == null) {
            return;
        }
        String v1 = info.getProperty(ConnectionResource.DLI_PARTITION_TABLE_FORCE_FILTER_KEY);
        if (v1 != null && !v1.isEmpty()) {
            sessConfMap.put(ConnectionResource.DLI_PARTITION_TABLE_FORCE_FILTER_KEY, v1);
        }

        String v2 = info.getProperty(ConnectionResource.DLI_SMALL_MERGE_KEY);
        if (v2 != null && !v2.isEmpty()) {
            sessConfMap.put(ConnectionResource.DLI_SMALL_MERGE_KEY, v2);
        }
    }

    public ConnectionResource getConnRes() {
        return connRes;
    }

    public Map<String, String> getSessConfMap() {
        return sessConfMap;
    }

    /**
     * Creates a Statement object for sending SQL statements to the database.
     *
     * @throws SQLException if a database access error occurs.
     */
    @Override
    public Statement createStatement() throws SQLException {
        checkClosed();
        DliStatement stmt = new DliStatement(this, false);
        stmtHandles.add(stmt);
        return stmt;
    }

    @Override
    public PreparedStatement prepareStatement(String sql) throws SQLException {
        return new DliPreparedStatement(this, sql);
    }

    @Override
    public CallableStatement prepareCall(String sql) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public String nativeSQL(String sql) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setAutoCommit(boolean autoCommit) throws SQLException {
        if (isClosed) {
            throw new SQLException("Connection is closed");
        }
        if (!autoCommit) {
            throw new SQLFeatureNotSupportedException("Enabling autocommit is not supported");
        }
    }

    @Override
    public boolean getAutoCommit() throws SQLException {
        return true;
    }

    @Override
    public void commit() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void rollback() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void close() throws SQLException {
        if (!isClosed) {
            for (Statement stmt : stmtHandles) {
                if (stmt != null) {
                    stmt.close();
                }
            }
        }
        isClosed = true;
        log.info("connection closed");
    }

    @Override
    public boolean isClosed() throws SQLException {
        return isClosed;
    }

    @Override
    public DatabaseMetaData getMetaData() throws SQLException {
        String sqlEngine = SPARK.getValue();
        if (connRes != null && HETU_ENGINE.getValue().equalsIgnoreCase(connRes.getEngineType())) {
            sqlEngine = HETU_ENGINE.getValue();
        }
        return new DliDatabaseMetaData(this, sqlEngine);
    }

    @Override
    public void setReadOnly(boolean readOnly) throws SQLException {
        if (readOnly) {
            throw new SQLFeatureNotSupportedException();
        }
    }

    @Override
    public boolean isReadOnly() throws SQLException {
        return false;
    }

    @Override
    public void setCatalog(String catalog) throws SQLException {

    }

    @Override
    public String getCatalog() throws SQLException {
        return null;
    }

    @Override
    public void setTransactionIsolation(int level) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int getTransactionIsolation() throws SQLException {
        return Connection.TRANSACTION_NONE;
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        return null;
    }

    @Override
    public void clearWarnings() throws SQLException {

    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        // This method only for compatible of SQL Workbench tools
        if (resultSetConcurrency != ResultSet.CONCUR_READ_ONLY) {
            throw new SQLException("Statement with resultset concurrency "
                + resultSetConcurrency + " is not supported"); // Optional feature not implemented
        }

        if (resultSetType != ResultSet.TYPE_FORWARD_ONLY) {
            throw new SQLException("Statement with resultset type " + resultSetType
                + " is not supported"); // Optional feature not implemented
        }
        return createStatement();
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency)
        throws SQLException {
        return new DliPreparedStatement(this, sql);
    }

    @Override
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Map<String, Class<?>> getTypeMap() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setHoldability(int holdability) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int getHoldability() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Savepoint setSavepoint() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Savepoint setSavepoint(String name) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void rollback(Savepoint savepoint) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void releaseSavepoint(Savepoint savepoint) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability)
        throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency,
        int resultSetHoldability) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency,
        int resultSetHoldability) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        return new DliPreparedStatement(this, sql);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Clob createClob() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Blob createBlob() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public NClob createNClob() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public SQLXML createSQLXML() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean isValid(int timeout) throws SQLException {
        return true;
    }

    @Override
    public void setClientInfo(String name, String value) throws SQLClientInfoException {

    }

    @Override
    public void setClientInfo(Properties properties) throws SQLClientInfoException {

    }

    @Override
    public String getClientInfo(String name) throws SQLException {
        return null;
    }

    @Override
    public Properties getClientInfo() throws SQLException {
        return null;
    }

    @Override
    public Array createArrayOf(String typeName, Object[] elements) throws SQLException {
        return null;
    }

    @Override
    public Struct createStruct(String typeName, Object[] attributes) throws SQLException {
        return null;
    }

    @Override
    public void setSchema(String schema) throws SQLException {

    }

    @Override
    public String getSchema() throws SQLException {
        return null;
    }

    @Override
    public void abort(Executor executor) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setNetworkTimeout(Executor executor, int milliseconds) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int getNetworkTimeout() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        throw new SQLException("Cannot unwrap to " + iface);
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return false;
    }

    private void checkClosed() throws SQLException {
        if (isClosed) {
            throw new SQLException("The connection has already been closed");
        }
    }
}
