package com.huawei.dli.jdbc;

import com.huawei.dli.jdbc.model.DliColumn;
import com.huawei.dli.jdbc.model.DliException;
import com.huawei.dli.jdbc.utils.type.DliType;
import com.huawei.dli.sdk.DLIClient;
import com.huawei.dli.sdk.exception.DLIException;
import com.huawei.dli.sdk.meta.Table;
import com.huawei.dli.sdk.meta.types.Column;
import com.huawei.dli.sdk.meta.types.DataType;
import com.huawei.dli.sdk.meta.types.PrimitiveType;

import org.apache.commons.lang3.StringUtils;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.RowIdLifetime;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

public class DliDatabaseMetaData implements DatabaseMetaData {
    private static final String PRODUCT_NAME = "Data Lake Insight";

    private static final String PRODUCT_VERSION = "1.1.1";

    private static final String DRIVER_NAME = "DLI";

    private static final int DRIVER_MAJOR_VERSION = 1;

    private static final int DRIVER_MINOR_VERSION = 1;

    private static final int MAX_TABLE_NAME_LENGTH = 128;

    private static final String CATALOG_SEPARATOR = ".";

    private static final char SEARCH_STRING_ESCAPE = '\\';

    private final DliConnection connection;

    private final String sqlEngine;

    private volatile DLIClient dliClient;

    public DliDatabaseMetaData(DliConnection connection, String sqlEngine) {
        this.connection = connection;
        this.sqlEngine = sqlEngine;
    }

    void setDliClient(DLIClient dliClient) {
        this.dliClient = dliClient;
        this.dliClient.setDialect(sqlEngine);
    }

    private synchronized DLIClient getDliClient() {
        if (dliClient == null) {
            dliClient = new DLIClient(connection.getConnRes().toDliInfo());
            dliClient.setDialect(sqlEngine);
        }
        return dliClient;
    }

    public boolean allProceduresAreCallable() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean allTablesAreSelectable() throws SQLException {
        return true;
    }

    public boolean autoCommitFailureClosesAllResultSets() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean dataDefinitionCausesTransactionCommit() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean dataDefinitionIgnoredInTransactions() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean deletesAreDetected(int type) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean doesMaxRowSizeIncludeBlobs() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public ResultSet getAttributes(String catalog, String schemaPattern,
        String typeNamePattern, String attributeNamePattern) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public ResultSet getBestRowIdentifier(String catalog, String schema,
        String table, int scope, boolean nullable) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public String getCatalogSeparator() throws SQLException {
        return CATALOG_SEPARATOR;
    }

    public String getCatalogTerm() throws SQLException {
        return "instance";
    }

    /**
     * In UQuery, we list all the databases in the service
     *
     * @return ResultSet
     * @throws SQLException If fail to call the API, e.g. server error or other reason.
     */
    public ResultSet getCatalogs() throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(
            Collections.singletonList(new Column("TABLE_CAT", new PrimitiveType(DataType.TypeName.STRING))));
        try {
            List<Object[]> catalogs = getDliClient().listCatalogs().stream()
                .map(v -> new Object[] {v})
                .collect(Collectors.toList());
            return new DliStaticResultSet(connection, rsMeta, catalogs.iterator());
        } catch (DLIException e) {
            throw new SQLException("Failed to get catalogs", e);
        }
    }

    public ResultSet getClientInfoProperties() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public ResultSet getColumnPrivileges(String catalog, String schema,
        String table, String columnNamePattern) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public ResultSet getPseudoColumns(String catalog, String schemaPattern,
        String tableNamePattern, String columnNamePattern) throws SQLException {
        // JDK 1.7
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean generatedKeyAlwaysReturned() throws SQLException {
        // JDK 1.7
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public ResultSet getColumns(String catalog, String schemaPattern, String tableNamePattern, String columnNamePattern)
        throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("TABLE_CAT", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TABLE_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TABLE_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("COLUMN_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("DATA_TYPE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("TYPE_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("COLUMN_SIZE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("BUFFER_LENGTH", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("DECIMAL_DIGITS", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("NUM_PERC_RADIX", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("NULLABLE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("REMARKS", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("COLUMN_DEF", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("SQL_DATA_TYPE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("SQL_DATETIME_SUB", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("CHAR_OCTET_LENGTH", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("ORDINAL_POSITION", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("IS_NULLABLE", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("SCOPE_CATALOG", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("SCOPE_SCHEMA", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("SCOPE_TABLE", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("SOURCE_DATA_TYPE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("IS_AUTO_INCREMENT", new PrimitiveType(DataType.TypeName.STRING))));
        try {
            String dbPattern = updatePattern(schemaPattern);
            String tblPattern = updatePattern(tableNamePattern);
            List<Table> tables = getDliClient().listTables(getCatalog(catalog), dbPattern, tblPattern);
            List<DliColumn> dliColumns = new ArrayList<>();
            for (Table tbl : tables) {
                dliColumns.addAll(toDliColumns(tbl));
            }
            List<Object[]> rows = toResultRows(dliColumns, columnNamePattern);
            return new DliStaticResultSet(connection, rsMeta, rows.iterator());
        } catch (DLIException e) {
            throw new SQLException("Failed to list tables with detail:" + e.getMessage(), e);
        }
    }

    private List<Object[]> toResultRows(List<DliColumn> dliColumns, String colNamePattern)
        throws SQLException {
        List<Object[]> rows = new ArrayList<>();
        for (DliColumn column : dliColumns) {
            String colName = column.getColumnName();
            if (colNamePattern != null && !colNamePattern.isEmpty()) {
                String regex = colNamePattern.replace("%", ".*");
                if (!colName.matches(regex)) {
                    continue;
                }
            }
            Object[] rowVals = {
                null, column.getTableSchema(), column.getTableName(), column.getColumnName(),
                (long) column.getType(), column.getTypeName(), null, null,
                (long) column.getDecimalDigits(), (long) column.getNumPercRaidx(),
                (long) column.getIsNullable(), column.getComment(), null, null, null, null,
                (long) column.getOrdinalPos(), column.getIsNullableString(), null, null, null,
                null, "NO"
            };
            rows.add(rowVals);
        }
        return rows;
    }

    private List<DliColumn> toDliColumns(Table tbl) throws DliException {
        List<DliColumn> cols = new ArrayList<>();
        int i = 1;
        for (Column col : tbl.getColumns()) {
            cols.add(new DliColumn(
                col.getName(),
                tbl.getTableName(),
                tbl.getDatabase().getDatabaseName(),
                DliType.of(col.getType().getName()),
                col.getType().getName(),
                col.getDescription() == null ? "" : col.getDescription(),
                i++));
        }
        return cols;
    }

    public Connection getConnection() throws SQLException {
        return this.connection;
    }

    public ResultSet getCrossReference(String primaryCatalog,
        String primarySchema, String primaryTable, String foreignCatalog,
        String foreignSchema, String foreignTable) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    public int getDatabaseMajorVersion() throws SQLException {
        return 0;
    }

    public int getDatabaseMinorVersion() throws SQLException {
        return 0;
    }

    public String getDatabaseProductName() throws SQLException {
        return PRODUCT_NAME;
    }

    public String getDatabaseProductVersion() throws SQLException {
        return PRODUCT_VERSION;
    }

    public int getDefaultTransactionIsolation() throws SQLException {
        return Connection.TRANSACTION_NONE;
    }

    public int getDriverMajorVersion() {
        return DRIVER_MAJOR_VERSION;
    }

    public int getDriverMinorVersion() {
        return DRIVER_MINOR_VERSION;
    }

    public String getDriverName() throws SQLException {
        return DRIVER_NAME;
    }

    public String getDriverVersion() throws SQLException {
        return DRIVER_MAJOR_VERSION + "." + DRIVER_MINOR_VERSION;
    }

    public ResultSet getExportedKeys(String catalog, String schema, String table)
        throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public String getExtraNameCharacters() throws SQLException {
        // TODO: verify that this is correct
        return "";
    }

    public ResultSet getFunctionColumns(String arg0, String arg1, String arg2,
        String arg3) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public ResultSet getFunctions(String catalog, String schemaPattern, String functionNamePattern)
        throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("FUNCTION_CAT", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("FUNCTION_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("FUNCTION_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("REMARKS", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("FUNCTION_TYPE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("SPECIFIC_NAME", new PrimitiveType(DataType.TypeName.BIGINT))));
        try {
            String dbPattern = updatePattern(schemaPattern);
            String funcPattern = updatePattern(functionNamePattern);
            List<Object[]> functions = getDliClient().listFunctions(getCatalog(catalog), dbPattern, funcPattern).stream()
                .map(funcIdent -> new Object[] {
                    getCatalog(catalog), funcIdent.getDatabaseName(), funcIdent.getFunctionName(),
                    null, "JAVA", funcIdent.getFunctionName()
                })
                .collect(Collectors.toList());
            return new DliStaticResultSet(connection, rsMeta, functions.iterator());
        } catch (DLIException e) {
            throw new SQLException("Failed to get functions", e);
        }
    }

    public String getIdentifierQuoteString() throws SQLException {
        return " ";
    }

    public ResultSet getImportedKeys(String catalog, String schema, String table)
        throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("PKTABLE_CAT", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("PKTABLE_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("PKTABLE_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("PKCOLUMN_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("FKTABLE_CAT", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("FKTABLE_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("FKTABLE_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("FKCOLUMN_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("KEY_SEQ", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("UPDATE_RULE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("DELETE_RULE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("FK_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("PK_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("DEFERRABILITY", new PrimitiveType(DataType.TypeName.STRING))));
        return new DliStaticResultSet(connection, rsMeta);
    }

    public ResultSet getIndexInfo(String catalog, String schema, String table,
        boolean unique, boolean approximate) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    public int getJDBCMajorVersion() throws SQLException {
        return 3;
    }

    public int getJDBCMinorVersion() throws SQLException {
        return 0;
    }

    public int getMaxBinaryLiteralLength() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxCatalogNameLength() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxCharLiteralLength() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    /**
     * Returns the value of maxColumnNameLength.
     */
    public int getMaxColumnNameLength() throws SQLException {
        return 128;
    }

    public int getMaxColumnsInGroupBy() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxColumnsInIndex() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxColumnsInOrderBy() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxColumnsInSelect() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxColumnsInTable() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxConnections() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxCursorNameLength() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxIndexLength() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxProcedureNameLength() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxRowSize() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxSchemaNameLength() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxStatementLength() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxStatements() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxTableNameLength() throws SQLException {
        return MAX_TABLE_NAME_LENGTH;
    }

    public int getMaxTablesInSelect() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getMaxUserNameLength() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public String getNumericFunctions() throws SQLException {
        return "";
    }

    public ResultSet getPrimaryKeys(String catalog, String schema, String table) throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("TABLE_CAT", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TABLE_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TABLE_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("COLUMN_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("KEY_SEQ", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("PK_NAME", new PrimitiveType(DataType.TypeName.STRING))));
        return new DliStaticResultSet(connection, rsMeta);
    }

    public ResultSet getProcedureColumns(
        String catalog,
        String schemaPattern,
        String procedureNamePattern,
        String columnNamePattern) throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("STUPID_PLACEHOLDERS", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("USELESS_PLACEHOLDER", new PrimitiveType(DataType.TypeName.STRING))));
        return new DliStaticResultSet(connection, rsMeta);
    }

    public String getProcedureTerm() throws SQLException {
        return "UDF";
    }

    public ResultSet getProcedures(String catalog, String schemaPattern, String procedureNamePattern)
        throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("PROCEDURE_CAT", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("PROCEDURE_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("PROCEDURE_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("RESERVERD", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("RESERVERD", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("RESERVERD", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("REMARKS", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("PROCEDURE_TYPE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("SPECIFIC_NAME", new PrimitiveType(DataType.TypeName.STRING))));
        return new DliStaticResultSet(connection, rsMeta);
    }

    public int getResultSetHoldability() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public RowIdLifetime getRowIdLifetime() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public String getSQLKeywords() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public int getSQLStateType() throws SQLException {
        return DatabaseMetaData.sqlStateSQL99;
    }

    public String getSchemaTerm() throws SQLException {
        return "database";
    }

    public ResultSet getSchemas() throws SQLException {
        return getSchemas(null, null);
    }

    /**
     * In UQuery, we list all matched databases as schema
     *
     * @param catalog       the catalog to get schema
     * @param schemaPattern the reg pattern
     * @return ResultSet
     * @throws SQLException If fail to call the API, e.g. server error or other reason.
     */
    @Override
    public ResultSet getSchemas(String catalog, String schemaPattern) throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("TABLE_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TABLE_CATALOG", new PrimitiveType(DataType.TypeName.STRING))));
        try {
            String dbPattern = updatePattern(schemaPattern);
            List<Object[]> schemas = getDliClient().listDatabases(getCatalog(catalog), dbPattern).stream()
                .map(dbName -> new Object[] {dbName, null})
                .collect(Collectors.toList());
            return new DliStaticResultSet(connection, rsMeta, schemas.iterator());
        } catch (DLIException e) {
            throw new SQLException("Failed to get schemas", e);
        }
    }

    public String getSearchStringEscape() throws SQLException {
        return String.valueOf(SEARCH_STRING_ESCAPE);
    }

    public String getStringFunctions() throws SQLException {
        return "";
    }

    public ResultSet getSuperTables(String catalog, String schemaPattern,
        String tableNamePattern) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public ResultSet getSuperTypes(String catalog, String schemaPattern,
        String typeNamePattern) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public String getSystemFunctions() throws SQLException {
        return "";
    }

    public ResultSet getTablePrivileges(String catalog, String schemaPattern,
        String tableNamePattern) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public ResultSet getTableTypes() throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Collections.singletonList(
            new Column("TABLE_TYPE", new PrimitiveType(DataType.TypeName.STRING))));
        List<Object[]> rows = Arrays.asList(new Object[] {"TABLE"}, new Object[] {"VIEW"});
        return new DliStaticResultSet(connection, rsMeta, rows.iterator());
    }

    public ResultSet getTables(String catalog, String schemaPattern, String tableNamePattern, String[] types)
        throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("TABLE_CAT", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TABLE_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TABLE_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TABLE_TYPE", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("REMARKS", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TYPE_CAT", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TYPE_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TYPE_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("SELF_REFERENCING_COL_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("REF_GENERATION", new PrimitiveType(DataType.TypeName.STRING))));
        try {
            String dbPattern = updatePattern(schemaPattern);
            String tblPattern = updatePattern(tableNamePattern);
            List<Object[]> tables = getDliClient().listTables(getCatalog(catalog), dbPattern, tblPattern, types).stream()
                .map(tblIdent -> new Object[] {
                    getCatalog(catalog), tblIdent.getDatabaseName(),
                    tblIdent.getTableName(), "TABLE", tblIdent.getTableName(),
                    null, null, null, null, "USER"
                })
                .collect(Collectors.toList());
            return new DliStaticResultSet(connection, rsMeta, tables.iterator());
        } catch (DLIException e) {
            throw new SQLException("Failed to get tables", e);
        }
    }

    public String getTimeDateFunctions() throws SQLException {
        return "";
    }

    public ResultSet getTypeInfo() throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("STUPID_PLACEHOLDERS", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("USELESS_PLACEHOLDER", new PrimitiveType(DataType.TypeName.STRING))));
        return new DliStaticResultSet(connection, rsMeta);
    }

    public ResultSet getUDTs(String catalog, String schemaPattern, String typeNamePattern, int[] types)
        throws SQLException {
        DliResultSetMetaData rsMeta = new DliResultSetMetaData(Arrays.asList(
            new Column("TYPE_CAT", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TYPE_SCHEM", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("TYPE_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("CLASS_NAME", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("DATA_TYPE", new PrimitiveType(DataType.TypeName.BIGINT)),
            new Column("REMARKS", new PrimitiveType(DataType.TypeName.STRING)),
            new Column("BASE_TYPE", new PrimitiveType(DataType.TypeName.BIGINT))));
        return new DliStaticResultSet(connection, rsMeta);
    }

    public String getURL() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public String getUserName() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public ResultSet getVersionColumns(String catalog, String schema, String table)
        throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean insertsAreDetected(int type) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean isCatalogAtStart() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean isReadOnly() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean locatorsUpdateCopy() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean nullPlusNonNullIsNull() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean nullsAreSortedAtEnd() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean nullsAreSortedAtStart() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean nullsAreSortedHigh() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean nullsAreSortedLow() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean othersDeletesAreVisible(int type) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean othersInsertsAreVisible(int type) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean othersUpdatesAreVisible(int type) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean ownDeletesAreVisible(int type) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean ownInsertsAreVisible(int type) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean ownUpdatesAreVisible(int type) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean storesLowerCaseIdentifiers() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean storesLowerCaseQuotedIdentifiers() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean storesMixedCaseIdentifiers() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean storesMixedCaseQuotedIdentifiers() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean storesUpperCaseIdentifiers() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean storesUpperCaseQuotedIdentifiers() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsANSI92EntryLevelSQL() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsANSI92FullSQL() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsANSI92IntermediateSQL() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsAlterTableWithAddColumn() throws SQLException {
        return true;
    }

    public boolean supportsAlterTableWithDropColumn() throws SQLException {
        return false;
    }

    public boolean supportsBatchUpdates() throws SQLException {
        return false;
    }

    public boolean supportsCatalogsInDataManipulation() throws SQLException {
        return false;
    }

    public boolean supportsCatalogsInIndexDefinitions() throws SQLException {
        return false;
    }

    public boolean supportsCatalogsInPrivilegeDefinitions() throws SQLException {
        return false;
    }

    public boolean supportsCatalogsInProcedureCalls() throws SQLException {
        return false;
    }

    public boolean supportsCatalogsInTableDefinitions() throws SQLException {
        return false;
    }

    public boolean supportsColumnAliasing() throws SQLException {
        return true;
    }

    public boolean supportsConvert() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsConvert(int fromType, int toType) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsCoreSQLGrammar() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsCorrelatedSubqueries() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsDataDefinitionAndDataManipulationTransactions()
        throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsDataManipulationTransactionsOnly() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsDifferentTableCorrelationNames() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsExpressionsInOrderBy() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsExtendedSQLGrammar() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsFullOuterJoins() throws SQLException {
        return true;
    }

    public boolean supportsGetGeneratedKeys() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsGroupBy() throws SQLException {
        return true;
    }

    public boolean supportsGroupByBeyondSelect() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsGroupByUnrelated() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsIntegrityEnhancementFacility() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsLikeEscapeClause() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsLimitedOuterJoins() throws SQLException {
        return true;
    }

    public boolean supportsMinimumSQLGrammar() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsMixedCaseIdentifiers() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsMixedCaseQuotedIdentifiers() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsMultipleOpenResults() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsMultipleResultSets() throws SQLException {
        return false;
    }

    public boolean supportsMultipleTransactions() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsNamedParameters() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsNonNullableColumns() throws SQLException {
        return false;
    }

    public boolean supportsOpenCursorsAcrossCommit() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsOpenCursorsAcrossRollback() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsOpenStatementsAcrossCommit() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsOpenStatementsAcrossRollback() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsOrderByUnrelated() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsOuterJoins() throws SQLException {
        return true;
    }

    public boolean supportsPositionedDelete() throws SQLException {
        return false;
    }

    public boolean supportsPositionedUpdate() throws SQLException {
        return false;
    }

    public boolean supportsResultSetConcurrency(int type, int concurrency)
        throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsResultSetHoldability(int holdability)
        throws SQLException {
        return false;
    }

    public boolean supportsResultSetType(int type) throws SQLException {
        return true;
    }

    public boolean supportsSavepoints() throws SQLException {
        return false;
    }

    public boolean supportsSchemasInDataManipulation() throws SQLException {
        return true;
    }

    public boolean supportsSchemasInIndexDefinitions() throws SQLException {
        return false;
    }

    public boolean supportsSchemasInPrivilegeDefinitions() throws SQLException {
        return false;
    }

    public boolean supportsSchemasInProcedureCalls() throws SQLException {
        return false;
    }

    public boolean supportsSchemasInTableDefinitions() throws SQLException {
        return true;
    }

    public boolean supportsSelectForUpdate() throws SQLException {
        return false;
    }

    public boolean supportsStatementPooling() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsStoredFunctionsUsingCallSyntax() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsStoredProcedures() throws SQLException {
        return false;
    }

    public boolean supportsSubqueriesInComparisons() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsSubqueriesInExists() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsSubqueriesInIns() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsSubqueriesInQuantifieds() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsTableCorrelationNames() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsTransactionIsolationLevel(int level)
        throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean supportsTransactions() throws SQLException {
        return false;
    }

    public boolean supportsUnion() throws SQLException {
        return false;
    }

    public boolean supportsUnionAll() throws SQLException {
        return true;
    }

    public boolean updatesAreDetected(int type) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean usesLocalFilePerTable() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean usesLocalFiles() throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    public <T> T unwrap(Class<T> iface) throws SQLException {
        throw new SQLFeatureNotSupportedException("Method not supported");
    }

    private String updatePattern(String sqlPattern) {
        if (StringUtils.isEmpty(sqlPattern)) {
            return sqlPattern;
        }
        return sqlPattern.replace("%", "*");
    }

    private String getCatalog(String catalog) {
        if (StringUtils.isNoneBlank(catalog)) {
            return catalog;
        }
        if (!StringUtils.isEmpty(connection.getConnRes().getCatalog())) {
            return connection.getConnRes().getCatalog();
        }
        return catalog;
    }
}
