package com.huawei.dli.jdbc;

import com.huawei.dli.sdk.SQLJob;
import com.huawei.dli.sdk.exception.DLIException;
import com.huawei.dli.sdk.meta.types.Column;
import com.huawei.dli.sdk.read.ResultSet;

import java.sql.SQLException;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

public class DliForwardResultSet extends DliResultSet {
    private final AtomicInteger fetchNum = new AtomicInteger();

    private final ResultSet sqlResultSet;

    private final long expectNum;

    private boolean isClosed = false;

    private Object[] currentRow;

    public DliForwardResultSet(DliStatement stmt, SQLJob sqlJob, long expectNum) throws SQLException {
        super(stmt.getConnection(), stmt, new DliResultSetMetaData(sqlJob.getResultSchema()));
        try {
            this.sqlResultSet = sqlJob.getResultSet();
            this.expectNum = expectNum;
        } catch (DLIException e) {
            throw new SQLException("Failed to init result set", e);
        }
    }

    public DliForwardResultSet(DliStatement stmt, List<Column> resultSchema, ResultSet sqlResultSet, long expectNum)
        throws SQLException {
        super(stmt.getConnection(), stmt, new DliResultSetMetaData(resultSchema));
        this.sqlResultSet = sqlResultSet;
        this.expectNum = expectNum;
    }

    @Override
    public int getRow() throws SQLException {
        checkClosed();
        try {
            if (!sqlResultSet.hasNext() || (expectNum > 0 && fetchNum.get() >= expectNum)) {
                return -1;
            }
            return fetchNum.get();
        } catch (DLIException e) {
            throw new SQLException("Failed to get row", e);
        }
    }

    @Override
    protected Object[] rowAtCursor() throws SQLException {
        if (currentRow == null) {
            throw new SQLException("the row should be not-null");
        }

        if (currentRow.length == 0) {
            throw new SQLException("the row should have more than 1 column");
        }
        return currentRow;
    }

    @Override
    public boolean next() throws SQLException {
        checkClosed();
        try {
            if (!sqlResultSet.hasNext() || (expectNum > 0 && fetchNum.get() >= expectNum)) {
                close();
                return false;
            }
            currentRow = sqlResultSet.read().getRecord().toArray();
            fetchNum.incrementAndGet();
            return true;
        } catch (DLIException e) {
            throw new SQLException("Failed to read record", e);
        }
    }

    @Override
    public boolean isClosed() throws SQLException {
        return isClosed;
    }

    @Override
    public void close() throws SQLException {
        if (isClosed) {
            return;
        }
        currentRow = null;
        isClosed = true;
        try {
            sqlResultSet.close();
        } catch (DLIException e) {
            throw new SQLException("Failed to close result set", e);
        }
    }

    private void checkClosed() throws SQLException {
        if (isClosed) {
            throw new SQLException("The result set has been closed");
        }
    }
}
