package com.huawei.dli.jdbc;

import com.huawei.dli.jdbc.model.DliException;
import com.huawei.dli.jdbc.utils.type.DliType;
import com.huawei.dli.sdk.meta.types.Column;
import com.huawei.dli.sdk.meta.types.DataType;

import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import lombok.Setter;

public class DliResultSetMetaData implements ResultSetMetaData {
    @Setter
    private String catalogName = " ";

    @Setter
    private String schemeName = " ";

    @Setter
    private String tableName = " ";

    private final List<Column> columns;

    private final List<DliType> columnTypes = new ArrayList<>();

    private final Map<String, Integer> colNameIndexMap = new HashMap<>();

    public DliResultSetMetaData(List<Column> columns) {
        this.columns = columns;
        try {
            int i = 1;
            for (Column col : columns) {
                columnTypes.add(DliType.of(col.getType().getName()));
                colNameIndexMap.put(col.getName().toLowerCase(Locale.ROOT), i++);
            }
        } catch (DliException e) {
            throw new IllegalArgumentException(e.getMessage());
        }
    }

    @Override
    public String getCatalogName(int column) throws SQLException {
        return catalogName;
    }

    @Override
    public String getSchemaName(int column) throws SQLException {
        return schemeName;
    }

    @Override
    public String getTableName(int column) throws SQLException {
        return tableName;
    }

    @Override
    public int getColumnCount() throws SQLException {
        return columns.size();
    }

    @Override
    public String getColumnClassName(int column) throws SQLException {
        return columnTypes.get(toZeroIndex(column)).getSqlTypeClass().getName();
    }

    @Override
    public String getColumnLabel(int column) throws SQLException {
        return getColumnName(column);
    }

    @Override
    public String getColumnName(int column) throws SQLException {
        return columns.get(toZeroIndex(column)).getName();
    }

    @Override
    public int getColumnType(int column) throws SQLException {
        return columnTypes.get(toZeroIndex(column)).getSqlType();
    }

    @Override
    public String getColumnTypeName(int column) throws SQLException {
        return columns.get(toZeroIndex(column)).getType().getName();
    }

    @Override
    public int getPrecision(int column) throws SQLException {
        int zeroIdx = toZeroIndex(column);
        DataType dataType = columns.get(zeroIdx).getType();
        return columnTypes.get(zeroIdx).precision(dataType);
    }

    @Override
    public int getScale(int column) throws SQLException {
        int zeroIdx = toZeroIndex(column);
        DataType dataType = columns.get(zeroIdx).getType();
        return columnTypes.get(zeroIdx).scale(dataType);
    }

    @Override
    public int getColumnDisplaySize(int column) throws SQLException {
        int zeroIdx = toZeroIndex(column);
        DataType dataType = columns.get(zeroIdx).getType();
        return columnTypes.get(zeroIdx).displaySize(dataType);
    }

    @Override
    public boolean isAutoIncrement(int column) throws SQLException {
        return false;
    }

    @Override
    public boolean isCaseSensitive(int column) throws SQLException {
        return columnTypes.get(toZeroIndex(column)).caseSensitive();
    }

    @Override
    public boolean isCurrency(int column) throws SQLException {
        return false;
    }

    @Override
    public boolean isDefinitelyWritable(int column) throws SQLException {
        return false;
    }

    // TODO: check
    @Override
    public int isNullable(int column) throws SQLException {
        return ResultSetMetaData.columnNullable;
    }

    @Override
    public boolean isReadOnly(int column) throws SQLException {
        return true;
    }

    @Override
    public boolean isSearchable(int column) throws SQLException {
        return true;
    }

    @Override
    public boolean isSigned(int column) throws SQLException {
        return columnTypes.get(toZeroIndex(column)).signed();
    }

    @Override
    public boolean isWritable(int column) throws SQLException {
        return false;
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        return null;
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return false;
    }

    private int toZeroIndex(int column) {
        if (column <= 0 || column > columns.size()) {
            throw new IllegalArgumentException();
        }
        return column - 1;
    }

    public int getColumnIndex(String columnLabel) {
        return colNameIndexMap.getOrDefault(columnLabel.toLowerCase(Locale.ROOT), -1);
    }
}
