package com.huawei.dli.jdbc;

import com.huawei.dli.jdbc.model.DliException;
import com.huawei.dli.jdbc.utils.ConnectionResource;
import com.huawei.dli.jdbc.utils.ErrorCode;
import com.huawei.dli.jdbc.utils.ExecutorUtils;
import com.huawei.dli.jdbc.utils.SqlUtils;
import com.huawei.dli.sdk.SQLJob;
import com.huawei.dli.sdk.common.DLIInfo;
import com.huawei.dli.sdk.util.V3ClientUtils;

import com.huaweicloud.sdk.dli.v1.DliClient;
import com.huaweicloud.sdk.dli.v1.model.CancelSqlJobRequest;
import com.huaweicloud.sdk.dli.v1.model.CancelSqlJobResponse;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Pattern;

public class DliStatement implements Statement {
    // ExecuteResult is used to store the execute result
    private class ExecuteResult {
        public boolean hasResultSet;

        public SQLException exception;
    }

    private DliConnection connHandle;

    private ResultSet resultSet = null;

    private boolean isClosed = false;

    private int resultSetMaxRows = Integer.MAX_VALUE;

    private long updateCount = -1;

    private String jobId = null;

    private int queryTimeout = 0;

    // This only for compatible of SQL Workbench tools
    private SQLWarning warning = null;

    protected Map<String, String> annotationConf = new HashMap<>();

    private static final Pattern SET_PATTERN = Pattern.compile(
        "(?i)^(SET)(\\s){1,10}(.){1,50}=(.){1,50};?(\\s){0,10}$");

    private static final Pattern USE_PATTERN = Pattern.compile("(?i)^(USE)(\\s){1,10}(.){0,200};?(\\s){0,10}$");

    public DliStatement(DliConnection conn) throws SQLException {
        this(conn, false);
    }

    DliStatement(DliConnection conn, boolean isResultSetScrollable) throws SQLException {
        this.connHandle = conn;
        // This only for compatible of SQL Workbench tools
        this.warning = new SQLWarning("successful completion");
    }

    @Override
    public ResultSet executeQuery(String sql) throws SQLException {
        if (connHandle.getConnRes().isCheckNoResultQuery()) {
            SqlUtils.checkForNoResult(sql);
        }
        return executeQueryInternal(sql);
    }

    private ResultSet executeQueryInternal(String sql) throws SQLException {
        if (!execute(sql)) {
            throw new SQLException("The query did not generate a result set!");
        }
        return resultSet;
    }

    @Override
    public int executeUpdate(String sql) throws SQLException {
        execute(sql);
        return (int) updateCount;
    }

    @Override
    public void close() throws SQLException {
        if (isClosed) {
            return;
        }
        if (resultSet != null) {
            resultSet.close();
            resultSet = null;
        }
        annotationConf.clear();
        jobId = null;
        connHandle = null;
        isClosed = true;
    }

    @Override
    public int getMaxFieldSize() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setMaxFieldSize(int max) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int getMaxRows() throws SQLException {
        return resultSetMaxRows;
    }

    @Override
    public void setMaxRows(int max) throws SQLException {
        if (max <= 0) {
            throw new SQLException("max must be greater than 0");
        }
        this.resultSetMaxRows = max;
    }

    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException {
        // This method only for compatible of SQL Workbench tools
    }

    @Override
    public int getQueryTimeout() throws SQLException {
        return this.queryTimeout;
    }

    @Override
    public void setQueryTimeout(int seconds) throws SQLException {
        this.queryTimeout = seconds;
    }

    @Override
    public void cancel() throws SQLException {
        connHandle.log.debug("execute cancel");
        if (jobId != null) {
            try {
                DliClient v3DliClient = V3ClientUtils.getDliClient(connHandle.getConnRes().toDliInfo());
                CancelSqlJobResponse resp = v3DliClient.cancelSqlJob(new CancelSqlJobRequest().withJobId(jobId));
                if (resp.getIsSuccess()) {
                    connHandle.log.warn(String.format("the job : %s is cancelled because of "
                        + "execution timeout.", jobId));
                } else {
                    connHandle.log.warn(
                        String.format("the job : %s is cancel failed, %s.", jobId, resp.getMessage()));
                }
            } catch (Exception e) {
                connHandle.log.error(String.format("Cancel failed, %s", e.getMessage()));
            }
        } else {
            connHandle.log.debug("jobId is null");
        }
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        // This method only for compatible of SQL Workbench tools
        return warning;
    }

    @Override
    public void clearWarnings() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setCursorName(String name) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    class ExecuteTask implements Runnable {

        ExecuteTask(String sql) {
            this.executeResult = new ExecuteResult();
            this.sql = sql;
        }

        ExecuteResult executeResult;

        String sql;

        @Override
        public void run() {
            try {
                checkClosed();
                beforeExecute();
                if (processSetClause(sql)) {
                    executeResult.hasResultSet = false;
                    return;
                }

                if (processUseClause(sql)) {
                    executeResult.hasResultSet = false;
                    return;
                }
                processSetConfInSql(sql);
                ResultSet directResult = directResult(getCurrentDatabase(), this.sql);
                if (directResult != null) {
                    connHandle.log.debug("Directory fetch this query sql result");
                    executeResult.hasResultSet = true;
                    resultSet = directResult;
                    return;
                }

                // need to submit job to uquery rest service
                connHandle.log.debug("submit sql request to DLI rest service.");
                DLIInfo dliInfo = connHandle.getConnRes().toDliInfo();
                SQLJob sqlJob = new SQLJob(dliInfo, getCurrentDatabase(), sql);
                sqlJob.setJobTimeout(connHandle.getConnRes().getJobTimeoutSeconds());
                Map<String, Object> confMap = new HashMap<>();
                confMap.putAll(connHandle.getSessConfMap());
                confMap.putAll(annotationConf);
                sqlJob.setConf(Collections.singletonList(confMap));
                sqlJob.setEngineType(connHandle.getConnRes().getEngineType());
                sqlJob.setCatalog(connHandle.getConnRes().getCatalog());
                sqlJob.submit();
                jobId = sqlJob.getJobId();

                if (SqlUtils.isQuery(sqlJob.getJobType().name(), sql)) {
                    initResultSet(sqlJob);
                    executeResult.hasResultSet = true;
                    updateCount = sqlJob.getResultCount();
                } else {
                    executeResult.hasResultSet = false;
                }
            } catch (Exception e) {
                connHandle.log.error(String.format("Fail to run sql: %s, %s", sql, e.getMessage()), e);
                jobId = null;
                executeResult.exception = new SQLException(e.getMessage());
            }
        }
    }

    public String getCurrentDatabase() {
        String db = annotationConf.get(ConnectionResource.SQL_CURRENT_DATABASE_KEY);
        if (db != null && !db.isEmpty()) {
            return db;
        }
        db = connHandle.getSessConfMap().get(ConnectionResource.SQL_CURRENT_DATABASE_KEY);
        if (db != null && !db.isEmpty()) {
            return db;
        }
        return connHandle.getConnRes().getDatabaseName();
    }

    @Override
    public boolean execute(final String sql) throws SQLException {
        connHandle.log.debug("Run SQL: " + sql);
        ExecuteTask task = new ExecuteTask(sql);
        CompletableFuture<Void> future = null;
        try {
            int timeOut = getQueryTimeout() > 0 ? getQueryTimeout() : Integer.MAX_VALUE;
            future = CompletableFuture.runAsync(task,
                ExecutorUtils.sqlSubmitExecutor(connHandle.getConnRes().getSqlSubmitThreadNum()));
            future.get(timeOut, TimeUnit.SECONDS);
            if (task.executeResult.exception == null) {
                return task.executeResult.hasResultSet;
            } else {
                throw task.executeResult.exception;
            }
        } catch (InterruptedException e) {
            throw new DliException("The sql request is interrupt", ErrorCode.JDBC_SYSTEM_ERROR.toString());
        } catch (TimeoutException e) {
            if (future != null) {
                boolean cancel = future.cancel(true);
                connHandle.log.info("Sql cancel result: " + cancel);
            }
            throw new DliException("The sql request is timeout", ErrorCode.JDBC_DLI_TIMEOUT_ERROR.toString());
        } catch (ExecutionException e) {
            connHandle.log.error("Run SQL ExecutionException ", e);
            throw new DliException("The sql request is execute failed", ErrorCode.JDBC_SYSTEM_ERROR.toString());
        }
    }

    public Map<String, String> getAnnotationConf() {
        return annotationConf;
    }

    // hook for inheritor skip execute, direct return result
    protected ResultSet directResult(String db, String sql) {
        return null;
    }

    private void processSetConfInSql(String sql) throws SQLException {
        if (!sql.contains(SqlUtils.SET_CONF_ANNOTATION_KEY.toUpperCase(Locale.ENGLISH))
            && !sql.contains(SqlUtils.SET_CONF_ANNOTATION_KEY)) {
            return;
        }
        List<String> lines = new ArrayList<>(Arrays.asList(sql.split("\\r|\\n|\\r\\n")));
        if (lines.size() == 1) {
            return;
        }
        annotationConf.putAll(parseSetConfInSql(lines));
    }

    private Map<String, String> parseSetConfInSql(List<String> sqlLines) {
        Map<String, String> conf = new HashMap<>();
        for (String item : sqlLines) {
            String setSql = item.trim().replace("\\t", "").replace(";", "");
            if (setSql.isEmpty() || !setSql.startsWith("--")) {
                continue;
            }
            String keyValueStr = setSql.replaceAll(SqlUtils.SET_CONF_ANNOTATION_REGEX, "$1,$2");
            if (setSql.equals(keyValueStr)) {
                continue;
            }
            String[] keyValues = keyValueStr.split(",");
            if (processSpecialKeys(keyValues[0].trim(), keyValues[1].trim())) {
                continue;
            }
            conf.put(keyValues[0].trim(), keyValues[1].trim());
        }
        return conf;
    }

    private boolean processSetClause(String sql) {
        sql = Normalizer.normalize(sql, Normalizer.Form.NFKC);
        if (SET_PATTERN.matcher(sql).matches()) {
            if (sql.contains(";")) {
                sql = sql.replace(';', ' ');
            }
            int i = sql.toLowerCase(Locale.US).indexOf("set");
            String pairString = sql.substring(i + 3);
            String[] pair = pairString.split("=");
            if (pair.length == 2) {
                String key = pair[0].trim();
                String value = getTrimedString(pair[1]);
                updateCount = 0;
                if (validSqlConfKey(key)) {
                    connHandle.getSessConfMap().put(key, value);
                    connHandle.log.info("set statement property: " + key + "=" + value);
                } else if (key.equals("*") && value.toLowerCase(Locale.US).equals("null")) {
                    connHandle.getSessConfMap().clear();
                    connHandle.log.info("clear all the statement properties");
                } else {
                    connHandle.log.warn(String.format("the property name : '%s' should start with '%s'",
                        key, ConnectionResource.DLI_CONF_PREFIX));
                }

                return true;
            } else {
                return false;
            }
        }
        return false;
    }

    private boolean validSqlConfKey(String key) {
        String lowerKey = key.toLowerCase(Locale.US);
        return lowerKey.startsWith(ConnectionResource.DLI_CONF_PREFIX)
            || lowerKey.startsWith(ConnectionResource.SPAKR_CONF_PREFIX);
    }

    private boolean processSpecialKeys(String key, String value) {
        if (key.equalsIgnoreCase(ConnectionResource.RESULT_DATA_LINE_NUM_KEY)) {
            connHandle.getConnRes().setResultDataLineNum(Integer.parseInt(value));
            return true;
        }
        return false;
    }

    private boolean processUseClause(String sql) throws SQLException {
        sql = Normalizer.normalize(sql, Normalizer.Form.NFKC);
        if (USE_PATTERN.matcher(sql).matches()) {
            if (sql.contains(";")) {
                sql = sql.replace(';', ' ');
            }
            int i = sql.toLowerCase(Locale.US).indexOf("use");
            String databaseName = getTrimedString(sql.substring(i + 3)).replace("`", "");
            if (databaseName.length() > 0) {
                connHandle.getConnRes().setDatabaseName(databaseName);
                connHandle.log.debug("set database to " + databaseName);
            } else {
                throw new DliException(String.format("database '%s' name can not be empty", databaseName),
                    ErrorCode.JDBC_NO_SUCH_OBJECT_ERROR.toString());
            }
            updateCount = 0;
            return true;
        }
        return false;
    }

    /**
     * trim empty space and "'"
     *
     * @param str
     * @return
     */
    private String getTrimedString(String str) {
        String trimedStr = str.trim();
        if (trimedStr.startsWith("'")) {
            trimedStr = trimedStr.substring(1);
        }
        if (trimedStr.endsWith("'")) {
            trimedStr = trimedStr.substring(0, trimedStr.lastIndexOf("'"));
        }
        return trimedStr;
    }

    private void initResultSet(SQLJob sqlJob) throws SQLException {
        long expectNum = getExpectResultNum(sqlJob.getResultCount());
        this.resultSet = initResultSet(sqlJob, expectNum);
    }

    private long getExpectResultNum(long resultCount) {
        long expectNum = connHandle.getConnRes().getResultDataLineNum() <= 0
            ? resultCount
            : Math.min(connHandle.getConnRes().getResultDataLineNum(), resultCount);
        return resultSetMaxRows > 0 ? Math.min(resultSetMaxRows, expectNum) : expectNum;
    }

    // hook for inheritor decorate resultset
    protected DliForwardResultSet initResultSet(SQLJob sqlJob, long expectNum) throws SQLException {
        return new DliForwardResultSet(this, sqlJob, expectNum);
    }

    @Override
    public ResultSet getResultSet() throws SQLException {
        return resultSet;
    }

    @Override
    public synchronized int getUpdateCount() throws SQLException {
        checkClosed();
        return (int) updateCount;
    }

    @Override
    public boolean getMoreResults() throws SQLException {
        return false;
    }

    @Override
    public void setFetchDirection(int direction) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int getFetchDirection() throws SQLException {
        return ResultSet.FETCH_FORWARD;
    }

    @Override
    public void setFetchSize(int rows) throws SQLException {
        connHandle.log.warn("setFetchSize unsupported");
    }

    @Override
    public int getFetchSize() throws SQLException {
        return 0;
    }

    @Override
    public int getResultSetConcurrency() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int getResultSetType() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void addBatch(String sql) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void clearBatch() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int[] executeBatch() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public DliConnection getConnection() throws SQLException {
        checkClosed();
        return connHandle;
    }

    @Override
    public boolean getMoreResults(int current) throws SQLException {
        return false;
    }

    @Override
    public ResultSet getGeneratedKeys() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int getResultSetHoldability() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean isClosed() throws SQLException {
        return isClosed;
    }

    @Override
    public void setPoolable(boolean poolable) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean isPoolable() throws SQLException {
        return false;
    }

    @Override
    public void closeOnCompletion() throws SQLException {
    }

    @Override
    public boolean isCloseOnCompletion() throws SQLException {
        return false;
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        return null;
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return false;
    }

    private void beforeExecute() throws SQLException {
        // If the statement re-executes another query, the previously-generated resultSet
        // will be implicit closed. And the corresponding temp table will be dropped as well.
        if (resultSet != null) {
            resultSet.close();
            resultSet = null;
        }
        updateCount = -1;
        isClosed = false;
    }

    protected void checkClosed() throws SQLException {
        if (isClosed) {
            throw new DliException("The statement has been closed",
                ErrorCode.JDBC_CONNECTION_ALREADY_CLOSED_ERROR.toString());
        }
    }
}
