package com.huawei.dli.jdbc.utils.type;

import com.huawei.dli.jdbc.model.DliException;
import com.huawei.dli.sdk.meta.types.DataType;
import com.huawei.dli.sdk.meta.types.DecimalType;

import java.math.BigDecimal;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Arrays;

import lombok.SneakyThrows;

public enum DliType {
    BOOLEAN {
        @Override
        public Class<?> getSqlTypeClass() {
            return Boolean.class;
        }

        @Override
        public int getSqlType() {
            return Types.BOOLEAN;
        }

        @Override
        public int precision(DataType dataType) {
            return 5;
        }
    },
    TINYINT {
        @Override
        public Class<?> getSqlTypeClass() {
            return Byte.class;
        }

        @Override
        public int getSqlType() {
            return Types.TINYINT;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 3;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 4;
        }
    },
    BYTE {
        @Override
        public Class<?> getSqlTypeClass() {
            return Byte.class;
        }

        @Override
        public int getSqlType() {
            return Types.TINYINT;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 3;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 4;
        }
    },
    SMALLINT {
        @Override
        public Class<?> getSqlTypeClass() {
            return Short.class;
        }

        @Override
        public int getSqlType() {
            return Types.SMALLINT;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 5;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 6;
        }
    },
    SHORT {
        @Override
        public Class<?> getSqlTypeClass() {
            return Short.class;
        }

        @Override
        public int getSqlType() {
            return Types.SMALLINT;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 5;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 6;
        }
    },
    INT {
        @Override
        public Class<?> getSqlTypeClass() {
            return Integer.class;
        }

        @Override
        public int getSqlType() {
            return Types.INTEGER;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 10;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 11;
        }
    },
    INTEGER {
        @Override
        public Class<?> getSqlTypeClass() {
            return Integer.class;
        }

        @Override
        public int getSqlType() {
            return Types.INTEGER;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 10;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 11;
        }
    },
    BIGINT {
        @Override
        public Class<?> getSqlTypeClass() {
            return Long.class;
        }

        @Override
        public int getSqlType() {
            return Types.BIGINT;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 19;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 20;
        }
    },
    LONG {
        @Override
        public Class<?> getSqlTypeClass() {
            return Long.class;
        }

        @Override
        public int getSqlType() {
            return Types.BIGINT;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 19;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 20;
        }
    },
    FLOAT {
        @Override
        public Class<?> getSqlTypeClass() {
            return Float.class;
        }

        @Override
        public int getSqlType() {
            return Types.FLOAT;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 7;
        }

        @Override
        public int scale(DataType dataType) {
            return 7;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 24;
        }
    },
    REAL {
        @Override
        public Class<?> getSqlTypeClass() {
            return Float.class;
        }

        @Override
        public int getSqlType() {
            return Types.REAL;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 7;
        }

        @Override
        public int scale(DataType dataType) {
            return 7;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 24;
        }
    },
    DOUBLE {
        @Override
        public Class<?> getSqlTypeClass() {
            return Double.class;
        }

        @Override
        public int getSqlType() {
            return Types.DOUBLE;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @Override
        public int precision(DataType dataType) {
            return 15;
        }

        @Override
        public int scale(DataType dataType) {
            return 15;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 25;
        }
    },
    DECIMAL {
        @Override
        public Class<?> getSqlTypeClass() {
            return BigDecimal.class;
        }

        @Override
        public int getSqlType() {
            return Types.DECIMAL;
        }

        @Override
        public boolean signed() {
            return true;
        }

        @SneakyThrows
        @Override
        public int precision(DataType dataType) {
            if (dataType instanceof DecimalType) {
                return  ((DecimalType) dataType).getPrecision();
            }
            throw new SQLException("Invalid DLI decimal type");
        }

        @SneakyThrows
        @Override
        public int scale(DataType dataType) {
            if (dataType instanceof DecimalType) {
                return  ((DecimalType) dataType).getScale();
            }
            throw new SQLException("Invalid DLI decimal type");
        }

        @Override
        public int displaySize(DataType dataType) {
            return precision(dataType) + 2;
        }
    },
    DATE {
        @Override
        public Class<?> getSqlTypeClass() {
            return Date.class;
        }

        @Override
        public int getSqlType() {
            return Types.DATE;
        }

        @Override
        public int precision(DataType dataType) {
            return 10;
        }
    },
    TIME {
        @Override
        public Class<?> getSqlTypeClass() {
            return Time.class;
        }

        @Override
        public int getSqlType() {
            return Types.TIME;
        }

        @Override
        public int precision(DataType dataType) {
            return 12;
        }
    },
    TIMESTAMP {
        @Override
        public Class<?> getSqlTypeClass() {
            return Timestamp.class;
        }

        @Override
        public int getSqlType() {
            return Types.TIMESTAMP;
        }

        @Override
        public int precision(DataType dataType) {
            return 29;
        }

        @Override
        public int scale(DataType dataType) {
            return 9;
        }
    },
    STRING {
        @Override
        public int getSqlType() {
            return Types.VARCHAR;
        }

        @Override
        public boolean caseSensitive() {
            return true;
        }
    },
    CHAR {
        @Override
        public int getSqlType() {
            return Types.CHAR;
        }

        @Override
        public boolean caseSensitive() {
            return true;
        }
    },
    VARCHAR {
        @Override
        public int getSqlType() {
            return Types.VARCHAR;
        }

        @Override
        public boolean caseSensitive() {
            return true;
        }
    },
    BINARY {
        @Override
        public Class<?> getSqlTypeClass() {
            return byte[].class;
        }

        @Override
        public int getSqlType() {
            return Types.BINARY;
        }
    },
    ARRAY {
        @Override
        public int getSqlType() {
            return Types.ARRAY;
        }
    },
    MAP {
        @Override
        public int getSqlType() {
            return Types.VARCHAR;
        }

        @Override
        public boolean caseSensitive() {
            return true;
        }
    },
    STRUCT {
        @Override
        public int getSqlType() {
            return Types.STRUCT;
        }
    },
    NULL {
        @Override
        public int getSqlType() {
            return Types.NULL;
        }

        @Override
        public int precision(DataType dataType) {
            return 0;
        }

        @Override
        public int displaySize(DataType dataType) {
            return 4;
        }
    };

    public static DliType of(String typeName) throws DliException {
        return of(typeName, true);
    }

    public static DliType of(final String typeName, final boolean notExistException) throws DliException {
        if (typeName == null || typeName.isEmpty()) {
            return null;
        }
        if ("void".equalsIgnoreCase(typeName)) {
            return DliType.NULL;
        }

        DliType v = Arrays.stream(DliType.values())
            .filter(p -> p.name().equalsIgnoreCase(typeName)).findFirst().orElse(null);
        if (v != null) {
            return v;
        }
        if (typeName.trim().matches("(?i)^char\\(\\d{1,10}\\)$")) {
            return DliType.CHAR;
        }
        if (typeName.trim().matches("(?i)^varchar\\(\\d{1,10}\\)$")) {
            return DliType.VARCHAR;
        }
        if (typeName.trim().matches("(?i)^decimal\\(\\d{1,2}(,\\d{1,2})?\\)$")) {
            return DliType.DECIMAL;
        }
        if (typeName.trim().matches("(?i)^(map|array|struct)<.+$")) {
            int complexIndex = typeName.indexOf("<");
            StringBuilder type = new StringBuilder(typeName);
            if (complexIndex > 0) {
                type.substring(0, complexIndex);
            }
            return Arrays.stream(DliType.values()).filter(
                p -> p.name().equalsIgnoreCase(complexIndex > 0 ? typeName.substring(0, complexIndex) : typeName))
                .findFirst().get();
        }

        if (notExistException) {
            throw new DliException("Invalid type name: " + typeName);
        }
        return null;
    }

    public abstract int getSqlType();

    public Class<?> getSqlTypeClass() {
        return String.class;
    }

    public int precision(DataType dataType) {
        return Integer.MAX_VALUE;
    }

    public int scale(DataType dataType) {
        return 0;
    }

    public boolean caseSensitive() {
        return false;
    }

    public boolean signed() {
        return false;
    }

    public int displaySize(DataType dataType) {
        return precision(dataType);
    }
}
