package com.huawei.dli.jdbc.utils.type;

import com.huawei.dli.sdk.meta.types.DataType;
import com.huawei.dli.sdk.meta.types.StructType;
import com.huawei.dli.sdk.util.DateFormatUtils;

import java.math.BigDecimal;
import java.sql.SQLException;

public class TypeUtils {
    public static int getIntField(Object fieldObj) throws SQLException {
        try {
            if (fieldObj instanceof Number) {
                return ((Number) fieldObj).intValue();
            } else if (fieldObj instanceof String) {
                return Integer.parseInt((String) fieldObj);
            } else if (fieldObj == null) {
                return 0;
            }
            throw new SQLException("Illegal conversion");
        } catch (Exception e) {
            throw new SQLException("Cannot convert the field " + e.toString(), e);
        }
    }

    public static long getLongField(Object obj) throws SQLException {
        try {
            if (obj instanceof Number) {
                return ((Number) obj).longValue();
            } else if (obj == null) {
                return 0;
            } else if (obj instanceof String) {
                return Long.parseLong((String) obj);
            }
            throw new SQLException("Illegal conversion");
        } catch (Exception e) {
            throw new SQLException("Cannot convert field" + e.toString(), e);
        }
    }

    public static short getShortField(Object obj) throws SQLException {
        try {
            if (obj instanceof Number) {
                return ((Number) obj).shortValue();
            } else if (obj == null) {
                return 0;
            } else if (obj instanceof String) {
                return Short.parseShort((String) obj);
            }
            throw new SQLException("Illegal conversion");
        } catch (Exception e) {
            throw new SQLException("Cannot convert field" + e.toString(), e);
        }
    }

    public static double getDoubleField(Object obj) throws SQLException {
        try {
            if (obj instanceof Number) {
                return ((Number) obj).doubleValue();
            } else if (obj == null) {
                return 0;
            } else if (obj instanceof String) {
                return Double.parseDouble((String) obj);
            }
            throw new SQLException("Illegal conversion");
        } catch (Exception e) {
            throw new SQLException("Cannot convert field" + e.toString(), e);
        }
    }

    public static String getStringField(Object obj) throws SQLException {
        if (obj == null) {
            return null;
        } else if (obj instanceof byte[]) {
            return new String((byte[]) obj);
        } else if (obj instanceof java.util.Date) {
            return DateFormatUtils.getDateTimeString(((java.util.Date) obj));
        } else {
            return obj.toString();
        }
    }

    public static boolean getBooleanField(Object obj) throws SQLException {
        if (obj instanceof Boolean) {
            return (Boolean) obj;
        } else if (obj == null) {
            return false;
        } else if (obj instanceof Number) {
            return ((Number) obj).intValue() != 0;
        } else if (obj instanceof String) {
            return !obj.equals("0");
        }
        throw new SQLException("Cannot convert column to boolean");
    }

    public static BigDecimal getBigDecimalField(Object obj) throws SQLException {
        return BigDecimal.valueOf(getLongField(obj));
    }

    public static Object getField(Object fieldObj, DataType dataType) throws SQLException {
        switch (DataType.TypeName.fromName(dataType.getName())) {
            case BOOLEAN:
                return TypeUtils.getBooleanField(fieldObj);
            case TINYINT:
            case SMALLINT:
                return TypeUtils.getShortField(fieldObj);
            case INT:
                return TypeUtils.getIntField(fieldObj);
            case BIGINT:
                return TypeUtils.getLongField(fieldObj);
            case FLOAT:
            case DOUBLE:
                return TypeUtils.getDoubleField(fieldObj);
            case DECIMAL:
                return TypeUtils.getBigDecimalField(fieldObj);
            case STRING:
            case CHAR:
            case VARCHAR:
                return TypeUtils.getStringField(fieldObj);
            case STRUCT:
                return new SimpleStruct((StructType) dataType, fieldObj);
            default:
                return fieldObj;
        }
    }
}
