/*
 * Decompiled with CFR 0.152.
 */
package com.hubspot.smtp.client;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.collect.Sets;
import com.hubspot.smtp.client.DotCrlfBuffer;
import com.hubspot.smtp.client.EhloResponse;
import com.hubspot.smtp.client.Extension;
import com.hubspot.smtp.client.MessageTooLargeException;
import com.hubspot.smtp.client.ResponseHandler;
import com.hubspot.smtp.client.SendInterceptor;
import com.hubspot.smtp.client.SmtpClientResponse;
import com.hubspot.smtp.client.SmtpSessionConfig;
import com.hubspot.smtp.client.SmtpSessionFactoryConfig;
import com.hubspot.smtp.messages.MessageContent;
import com.hubspot.smtp.messages.MessageContentEncoding;
import com.hubspot.smtp.utils.SmtpResponses;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.handler.codec.smtp.DefaultSmtpRequest;
import io.netty.handler.codec.smtp.SmtpCommand;
import io.netty.handler.codec.smtp.SmtpContent;
import io.netty.handler.codec.smtp.SmtpRequest;
import io.netty.handler.codec.smtp.SmtpRequests;
import io.netty.handler.codec.smtp.SmtpResponse;
import io.netty.handler.ssl.SslHandler;
import io.netty.handler.stream.ChunkedInput;
import io.netty.util.concurrent.GenericFutureListener;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLSession;

public class SmtpSession {
    private static final Set<SmtpCommand> VALID_ANYWHERE_PIPELINED_COMMANDS = Sets.newHashSet((Object[])new SmtpCommand[]{SmtpCommand.RSET, SmtpCommand.MAIL, SmtpCommand.RCPT});
    private static final Set<SmtpCommand> VALID_AT_END_PIPELINED_COMMANDS = Sets.newHashSet((Object[])new SmtpCommand[]{SmtpCommand.RSET, SmtpCommand.MAIL, SmtpCommand.RCPT, SmtpCommand.EHLO, SmtpCommand.DATA, SmtpCommand.VRFY, SmtpCommand.EXPN, SmtpCommand.QUIT, SmtpCommand.NOOP});
    private static final Joiner COMMA_JOINER = Joiner.on((String)", ");
    private static final SmtpCommand STARTTLS_COMMAND = SmtpCommand.valueOf((CharSequence)"STARTTLS");
    private static final SmtpCommand AUTH_COMMAND = SmtpCommand.valueOf((CharSequence)"AUTH");
    private static final SmtpCommand BDAT_COMMAND = SmtpCommand.valueOf((CharSequence)"BDAT");
    private static final String AUTH_PLAIN_MECHANISM = "PLAIN";
    private static final String AUTH_LOGIN_MECHANISM = "LOGIN";
    private static final String AUTH_XOAUTH2_MECHANISM = "XOAUTH2";
    private static final String CRLF = "\r\n";
    private final Channel channel;
    private final ResponseHandler responseHandler;
    private final SmtpSessionConfig config;
    private final Executor executor;
    private final Supplier<SSLEngine> sslEngineSupplier;
    private final CompletableFuture<Void> closeFuture;
    private final AtomicInteger chunkedBytesSent = new AtomicInteger(0);
    private volatile boolean requiresRset = false;
    private volatile EhloResponse ehloResponse = EhloResponse.EMPTY;

    SmtpSession(Channel channel, ResponseHandler responseHandler, SmtpSessionConfig config, Executor executor, Supplier<SSLEngine> sslEngineSupplier) {
        this.channel = channel;
        this.responseHandler = responseHandler;
        this.config = config;
        this.executor = executor;
        this.sslEngineSupplier = sslEngineSupplier;
        this.closeFuture = new CompletableFuture();
        this.channel.pipeline().addLast(new ChannelHandler[]{new ErrorHandler()});
    }

    public String getConnectionId() {
        return this.config.getConnectionId();
    }

    public CompletableFuture<Void> getCloseFuture() {
        return this.closeFuture;
    }

    public EhloResponse getEhloResponse() {
        return this.ehloResponse;
    }

    public CompletableFuture<Void> close() {
        this.channel.close();
        return this.closeFuture;
    }

    public CompletableFuture<SmtpClientResponse> startTls() {
        Preconditions.checkState((!this.isEncrypted() ? 1 : 0) != 0, (Object)"This connection is already using TLS");
        return this.send((SmtpRequest)new DefaultSmtpRequest(STARTTLS_COMMAND)).thenCompose(startTlsResponse -> {
            if (startTlsResponse.containsError()) {
                return CompletableFuture.completedFuture(startTlsResponse);
            }
            return this.performTlsHandshake((SmtpClientResponse)startTlsResponse).thenCompose(ignored -> this.send(SmtpRequests.ehlo((CharSequence)this.ehloResponse.getEhloDomain()))).thenApply(ignored -> startTlsResponse);
        });
    }

    private CompletionStage<SmtpClientResponse> performTlsHandshake(SmtpClientResponse r) {
        CompletableFuture<SmtpClientResponse> ourFuture = new CompletableFuture<SmtpClientResponse>();
        SslHandler sslHandler = new SslHandler(this.sslEngineSupplier.get());
        this.channel.pipeline().addFirst(new ChannelHandler[]{sslHandler});
        sslHandler.handshakeFuture().addListener(nettyFuture -> {
            if (nettyFuture.isSuccess()) {
                ourFuture.complete(r);
            } else {
                ourFuture.completeExceptionally(nettyFuture.cause());
                this.close();
            }
        });
        return ourFuture;
    }

    public boolean isEncrypted() {
        return this.channel.pipeline().get(SslHandler.class) != null;
    }

    public Optional<SSLSession> getSSLSession() {
        return Optional.ofNullable((SslHandler)this.channel.pipeline().get(SslHandler.class)).map(handler -> handler.engine().getSession());
    }

    public CompletableFuture<SmtpClientResponse> send(String from, String to, MessageContent content) {
        return this.send(from, Collections.singleton(to), content, Optional.empty());
    }

    public CompletableFuture<SmtpClientResponse> send(String from, String to, MessageContent content, SendInterceptor sendInterceptor) {
        return this.send(from, Collections.singleton(to), content, Optional.of(sendInterceptor));
    }

    public CompletableFuture<SmtpClientResponse> send(String from, Collection<String> recipients, MessageContent content) {
        return this.send(from, recipients, content, Optional.empty());
    }

    public CompletableFuture<SmtpClientResponse> send(String from, Collection<String> recipients, MessageContent content, SendInterceptor sendInterceptor) {
        return this.send(from, recipients, content, Optional.of(sendInterceptor));
    }

    private CompletableFuture<SmtpClientResponse> send(String from, Collection<String> recipients, MessageContent content, Optional<SendInterceptor> sequenceInterceptor) {
        return this.sendInternal(from, recipients, content, sequenceInterceptor);
    }

    private CompletableFuture<SmtpClientResponse> sendInternal(String from, Collection<String> recipients, MessageContent content, Optional<SendInterceptor> sequenceInterceptor) {
        Preconditions.checkNotNull((Object)from);
        Preconditions.checkNotNull(recipients);
        Preconditions.checkArgument((!recipients.isEmpty() ? 1 : 0) != 0, (Object)"recipients must be > 0");
        Preconditions.checkNotNull((Object)content);
        this.checkMessageSize(content.size());
        Preconditions.checkNotNull(sequenceInterceptor);
        if (this.ehloResponse.isSupported(Extension.CHUNKING)) {
            return this.sendAsChunked(from, recipients, content, sequenceInterceptor);
        }
        if (content.getEncoding() == MessageContentEncoding.SEVEN_BIT) {
            return this.sendAs7Bit(from, recipients, content, sequenceInterceptor);
        }
        if (this.ehloResponse.isSupported(Extension.EIGHT_BIT_MIME)) {
            return this.sendAs8BitMime(from, recipients, content, sequenceInterceptor);
        }
        if (content.get8bitCharacterProportion() == 0.0f) {
            return this.sendAs7Bit(from, recipients, content, sequenceInterceptor);
        }
        return this.sendAs7Bit(from, recipients, this.encodeContentAs7Bit(content), sequenceInterceptor);
    }

    private CompletableFuture<SmtpClientResponse> sendAsChunked(String from, Collection<String> recipients, MessageContent content, Optional<SendInterceptor> sequenceInterceptor) {
        if (this.ehloResponse.isSupported(Extension.PIPELINING)) {
            ArrayList objects = Lists.newArrayListWithExpectedSize((int)(3 + recipients.size()));
            objects.add(this.mailCommand(from, recipients));
            objects.addAll(this.rpctCommands(recipients));
            Iterator<ByteBuf> chunkIterator = content.getContentChunkIterator(this.channel.alloc());
            ByteBuf firstChunk = chunkIterator.next();
            if (firstChunk == null) {
                throw new IllegalArgumentException("The MessageContent was empty; size is " + (content.size().isPresent() ? Integer.toString(content.size().getAsInt()) : "not present"));
            }
            objects.add(this.getBdatRequestWithData(firstChunk, !chunkIterator.hasNext()));
            return this.beginSequence(sequenceInterceptor, objects.size(), objects.toArray()).thenSendInTurn(this.getBdatIterator(chunkIterator)).toResponses();
        }
        SendSequence sequence = this.beginSequence(sequenceInterceptor, 1, this.mailCommand(from, recipients));
        for (String recipient : recipients) {
            sequence.thenSend(SmtpRequests.rcpt((CharSequence)recipient, (CharSequence[])new CharSequence[0]));
        }
        return sequence.thenSendInTurn(this.getBdatIterator(content.getContentChunkIterator(this.channel.alloc()))).toResponses();
    }

    @SuppressFBWarnings(value={"VA_FORMAT_STRING_USES_NEWLINE"})
    private ByteBuf getBdatRequestWithData(ByteBuf data, boolean isLast) {
        String request = String.format("BDAT %d%s\r\n", data.readableBytes(), isLast ? " LAST" : "");
        ByteBuf requestBuf = this.channel.alloc().buffer(request.length());
        ByteBufUtil.writeAscii((ByteBuf)requestBuf, (CharSequence)request);
        return this.channel.alloc().compositeBuffer().addComponents(true, new ByteBuf[]{requestBuf, data});
    }

    private Iterator<Object> getBdatIterator(final Iterator<ByteBuf> chunkIterator) {
        return new Iterator<Object>(){

            @Override
            public boolean hasNext() {
                return chunkIterator.hasNext();
            }

            @Override
            public Object next() {
                ByteBuf buf = (ByteBuf)chunkIterator.next();
                return SmtpSession.this.getBdatRequestWithData(buf, !chunkIterator.hasNext());
            }
        };
    }

    private CompletableFuture<SmtpClientResponse> sendAs7Bit(String from, Collection<String> recipients, MessageContent content, Optional<SendInterceptor> sequenceInterceptor) {
        return this.sendPipelinedIfPossible(this.mailCommand(from, recipients), recipients, SmtpRequests.data(), sequenceInterceptor).thenSend(content.getDotStuffedContent(), DotCrlfBuffer.get()).toResponses();
    }

    private CompletableFuture<SmtpClientResponse> sendAs8BitMime(String from, Collection<String> recipients, MessageContent content, Optional<SendInterceptor> sequenceInterceptor) {
        return this.sendPipelinedIfPossible(this.mailCommandWith8BitMime(from, recipients), recipients, SmtpRequests.data(), sequenceInterceptor).thenSend(content.getDotStuffedContent(), DotCrlfBuffer.get()).toResponses();
    }

    private SendSequence sendPipelinedIfPossible(SmtpRequest mailRequest, Collection<String> recipients, SmtpRequest dataRequest, Optional<SendInterceptor> sequenceInterceptor) {
        ArrayList requests = Lists.newArrayListWithExpectedSize((int)(2 + recipients.size()));
        requests.add(mailRequest);
        requests.addAll(this.rpctCommands(recipients));
        requests.add(dataRequest);
        if (this.ehloResponse.isSupported(Extension.PIPELINING)) {
            return this.beginSequence(sequenceInterceptor, requests.size(), requests.toArray());
        }
        SendSequence s = this.beginSequence(sequenceInterceptor, 1, requests.get(0));
        for (int i = 1; i < requests.size(); ++i) {
            s.thenSend(requests.get(i));
        }
        return s;
    }

    private Collection<SmtpRequest> rpctCommands(Collection<String> recipients) {
        return recipients.stream().map(x$0 -> SmtpRequests.rcpt((CharSequence)x$0, (CharSequence[])new CharSequence[0])).collect(Collectors.toList());
    }

    private SmtpRequest mailCommand(String from, Collection<String> recipients) {
        if (!this.ehloResponse.isSupported(Extension.SMTPUTF8) || SmtpSession.isAllAscii(from) && SmtpSession.isAllAscii(recipients)) {
            return SmtpRequests.mail((CharSequence)from, (CharSequence[])new CharSequence[0]);
        }
        return SmtpRequests.mail((CharSequence)from, (CharSequence[])new CharSequence[]{"SMTPUTF8"});
    }

    private SmtpRequest mailCommandWith8BitMime(String from, Collection<String> recipients) {
        if (!this.ehloResponse.isSupported(Extension.SMTPUTF8) || SmtpSession.isAllAscii(from) && SmtpSession.isAllAscii(recipients)) {
            return SmtpRequests.mail((CharSequence)from, (CharSequence[])new CharSequence[]{"BODY=8BITMIME"});
        }
        return SmtpRequests.mail((CharSequence)from, (CharSequence[])new CharSequence[]{"BODY=8BITMIME", "SMTPUTF8"});
    }

    private static boolean isAllAscii(String s) {
        return CharMatcher.ascii().matchesAllOf((CharSequence)s);
    }

    private static boolean isAllAscii(Collection<String> strings) {
        return strings.stream().allMatch(SmtpSession::isAllAscii);
    }

    private SendSequence beginSequence(Optional<SendInterceptor> sequenceInterceptor, int expectedResponses, Object ... objects) {
        if (this.requiresRset) {
            if (this.ehloResponse.isSupported(Extension.PIPELINING)) {
                return new SendSequence(sequenceInterceptor, expectedResponses + 1, ObjectArrays.concat((Object)SmtpRequests.rset(), (Object[])objects));
            }
            return new SendSequence(sequenceInterceptor, 1, SmtpRequests.rset()).thenSend(objects);
        }
        this.requiresRset = true;
        return new SendSequence(sequenceInterceptor, expectedResponses, objects);
    }

    private MessageContent encodeContentAs7Bit(MessageContent content) {
        return content;
    }

    public CompletableFuture<SmtpClientResponse> send(SmtpRequest request) {
        Preconditions.checkNotNull((Object)request);
        return this.applyOnExecutor(this.executeRequestInterceptor(this.config.getSendInterceptor(), request, () -> {
            CompletionStage<List<Object>> responseFuture = this.responseHandler.createResponseFuture(1, () -> SmtpSession.createDebugString(request));
            this.writeAndFlush(request);
            if (request.command().equals((Object)SmtpCommand.EHLO)) {
                responseFuture = responseFuture.whenComplete((responses, ignored) -> {
                    if (responses != null) {
                        String ehloDomain = request.parameters().isEmpty() ? "" : ((CharSequence)request.parameters().get(0)).toString();
                        this.parseEhloResponse(ehloDomain, ((SmtpResponse)responses.get(0)).details());
                    }
                });
            }
            return responseFuture;
        }), this::wrapFirstResponse);
    }

    public CompletableFuture<SmtpClientResponse> send(MessageContent content) {
        Preconditions.checkNotNull((Object)content);
        this.checkMessageSize(content.size());
        return this.applyOnExecutor(this.executeDataInterceptor(this.config.getSendInterceptor(), () -> {
            CompletableFuture<List<SmtpResponse>> responseFuture = this.responseHandler.createResponseFuture(1, () -> "message contents");
            this.writeContent(content);
            this.channel.flush();
            return responseFuture;
        }), this::wrapFirstResponse);
    }

    public CompletableFuture<SmtpClientResponse> sendChunk(ByteBuf data, boolean isLast) {
        Preconditions.checkState((boolean)this.ehloResponse.isSupported(Extension.CHUNKING), (Object)"Chunking is not supported on this server");
        Preconditions.checkNotNull((Object)data);
        this.checkMessageSize(OptionalInt.of(this.chunkedBytesSent.addAndGet(data.readableBytes())));
        if (isLast) {
            this.chunkedBytesSent.set(0);
        }
        return this.applyOnExecutor(this.executeDataInterceptor(this.config.getSendInterceptor(), () -> {
            CompletableFuture<List<SmtpResponse>> responseFuture = this.responseHandler.createResponseFuture(1, () -> "BDAT message chunk");
            String size = Integer.toString(data.readableBytes());
            if (isLast) {
                this.write(new DefaultSmtpRequest(BDAT_COMMAND, new CharSequence[]{size, "LAST"}));
            } else {
                this.write(new DefaultSmtpRequest(BDAT_COMMAND, new CharSequence[]{size}));
            }
            this.write(data);
            this.channel.flush();
            return responseFuture;
        }), this::wrapFirstResponse);
    }

    public CompletableFuture<SmtpClientResponse> sendPipelined(SmtpRequest ... requests) {
        return this.sendPipelined((MessageContent)null, requests);
    }

    public CompletableFuture<SmtpClientResponse> sendPipelined(MessageContent content, SmtpRequest ... requests) {
        Preconditions.checkState((boolean)this.ehloResponse.isSupported(Extension.PIPELINING), (Object)"Pipelining is not supported on this server");
        Preconditions.checkNotNull((Object)requests);
        SmtpSession.checkValidPipelinedRequest(requests);
        this.checkMessageSize(content == null ? OptionalInt.empty() : content.size());
        return this.applyOnExecutor(this.executePipelineInterceptor(this.config.getSendInterceptor(), Lists.newArrayList((Object[])requests), () -> {
            int expectedResponses = requests.length + (content == null ? 0 : 1);
            CompletableFuture<List<SmtpResponse>> responseFuture = this.responseHandler.createResponseFuture(expectedResponses, () -> SmtpSession.createDebugString(requests));
            if (content != null) {
                this.writeContent(content);
            }
            for (SmtpRequest r : requests) {
                this.write(r);
            }
            this.channel.flush();
            return responseFuture;
        }), this::wrapResponses);
    }

    private SmtpClientResponse wrapResponses(List<SmtpResponse> responses) {
        return new SmtpClientResponse(this, responses);
    }

    private SmtpClientResponse wrapFirstResponse(List<SmtpResponse> responses) {
        return new SmtpClientResponse(this, responses.get(0));
    }

    public CompletableFuture<SmtpClientResponse> authPlain(String username, String password) {
        Preconditions.checkState((boolean)this.ehloResponse.isAuthPlainSupported(), (Object)"Auth plain is not supported on this server");
        String s = String.format("%s\u0000%s\u0000%s", username, username, password);
        return this.send((SmtpRequest)new DefaultSmtpRequest(AUTH_COMMAND, new CharSequence[]{AUTH_PLAIN_MECHANISM, this.encodeBase64(s)}));
    }

    public CompletableFuture<SmtpClientResponse> authXoauth2(String username, String accessToken) {
        Preconditions.checkState((boolean)this.ehloResponse.isAuthXoauth2Supported(), (Object)"Auth xoauth2 is not supported on this server");
        String s = String.format("user=%s\u0001auth=Bearer %s\u0001\u0001", username, accessToken);
        return this.send((SmtpRequest)new DefaultSmtpRequest(AUTH_COMMAND, new CharSequence[]{AUTH_XOAUTH2_MECHANISM, this.encodeBase64(s)}));
    }

    public CompletableFuture<SmtpClientResponse> authLogin(String username, String password) {
        Preconditions.checkState((boolean)this.ehloResponse.isAuthLoginSupported(), (Object)"Auth login is not supported on this server");
        return this.send((SmtpRequest)new DefaultSmtpRequest(AUTH_COMMAND, new CharSequence[]{AUTH_LOGIN_MECHANISM, this.encodeBase64(username)})).thenCompose(r -> {
            if (r.containsError()) {
                return CompletableFuture.completedFuture(r);
            }
            return this.sendAuthLoginPassword(password);
        });
    }

    private CompletionStage<SmtpClientResponse> sendAuthLoginPassword(String password) {
        return this.applyOnExecutor(this.executeRequestInterceptor(this.config.getSendInterceptor(), (SmtpRequest)new DefaultSmtpRequest(AUTH_COMMAND), () -> {
            CompletableFuture<List<SmtpResponse>> responseFuture = this.responseHandler.createResponseFuture(1, () -> "auth login password");
            String passwordResponse = this.encodeBase64(password) + CRLF;
            ByteBuf passwordBuffer = this.channel.alloc().buffer().writeBytes(passwordResponse.getBytes(StandardCharsets.UTF_8));
            this.writeAndFlush(passwordBuffer);
            return responseFuture;
        }), this::wrapFirstResponse);
    }

    private void checkMessageSize(OptionalInt size) {
        if (!this.ehloResponse.getMaxMessageSize().isPresent() || !size.isPresent()) {
            return;
        }
        long maximumSize = this.ehloResponse.getMaxMessageSize().get();
        if (maximumSize < (long)size.getAsInt()) {
            throw new MessageTooLargeException(this.config.getConnectionId(), maximumSize);
        }
    }

    private String encodeBase64(String s) {
        return Base64.getEncoder().encodeToString(s.getBytes(StandardCharsets.UTF_8));
    }

    private void writeContent(MessageContent content) {
        this.write(content.getDotStuffedContent());
        this.write(DotCrlfBuffer.get());
    }

    private void write(Object obj) {
        this.channel.write(obj).addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
    }

    private void writeAndFlush(Object obj) {
        this.channel.writeAndFlush(obj).addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
    }

    @VisibleForTesting
    void parseEhloResponse(String ehloDomain, Iterable<CharSequence> response) {
        this.ehloResponse = EhloResponse.parse(ehloDomain, response, this.config.getDisabledExtensions());
    }

    @VisibleForTesting
    static String createDebugString(Object ... objects) {
        return COMMA_JOINER.join((Iterable)Arrays.stream(objects).map(SmtpSession::objectToString).collect(Collectors.toList()));
    }

    private static String objectToString(Object o) {
        if (o instanceof SmtpRequest) {
            SmtpRequest request = (SmtpRequest)o;
            if (request.command().equals((Object)AUTH_COMMAND)) {
                return "<redacted-auth-command>";
            }
            return String.format("%s %s", request.command().name(), Joiner.on((String)" ").join((Iterable)request.parameters()));
        }
        if (o instanceof SmtpContent || o instanceof ByteBuf || o instanceof ChunkedInput) {
            return "[CONTENT]";
        }
        return o.toString();
    }

    private static void checkValidPipelinedRequest(SmtpRequest[] requests) {
        Preconditions.checkArgument((requests.length > 0 ? 1 : 0) != 0, (Object)"You must provide requests to pipeline");
        for (int i = 0; i < requests.length; ++i) {
            boolean isLastRequest;
            SmtpCommand command = requests[i].command();
            boolean bl = isLastRequest = i == requests.length - 1;
            if (isLastRequest) {
                Preconditions.checkArgument((boolean)VALID_AT_END_PIPELINED_COMMANDS.contains(command), (Object)(command.name() + " cannot be used in a pipelined request"));
                continue;
            }
            String errorMessage = VALID_AT_END_PIPELINED_COMMANDS.contains(command) ? " must appear last in a pipelined request" : " cannot be used in a pipelined request";
            Preconditions.checkArgument((boolean)VALID_ANYWHERE_PIPELINED_COMMANDS.contains(command), (Object)(command.name() + errorMessage));
        }
    }

    private <R, T> CompletableFuture<R> applyOnExecutor(CompletableFuture<T> eventLoopFuture, Function<T, R> mapper) {
        if (this.executor == SmtpSessionFactoryConfig.DIRECT_EXECUTOR) {
            return eventLoopFuture.thenApply(mapper);
        }
        return eventLoopFuture.handleAsync((rs, e) -> {
            if (e != null) {
                throw Throwables.propagate((Throwable)e);
            }
            return mapper.apply(rs);
        }, this.executor);
    }

    CompletableFuture<List<SmtpResponse>> executeRequestInterceptor(Optional<SendInterceptor> interceptor, SmtpRequest request, Supplier<CompletableFuture<List<SmtpResponse>>> supplier) {
        return interceptor.map(h -> h.aroundRequest(request, supplier)).orElseGet(supplier);
    }

    CompletableFuture<List<SmtpResponse>> executeDataInterceptor(Optional<SendInterceptor> interceptor, Supplier<CompletableFuture<List<SmtpResponse>>> supplier) {
        return interceptor.map(h -> h.aroundData(supplier)).orElseGet(supplier);
    }

    CompletableFuture<List<SmtpResponse>> executePipelineInterceptor(Optional<SendInterceptor> interceptor, List<SmtpRequest> requests, Supplier<CompletableFuture<List<SmtpResponse>>> supplier) {
        return interceptor.map(h -> h.aroundPipelinedSequence(requests, supplier)).orElseGet(supplier);
    }

    private class ErrorHandler
    extends ChannelInboundHandlerAdapter {
        private Throwable cause;

        private ErrorHandler() {
        }

        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
            this.cause = cause;
            ctx.close();
        }

        @SuppressFBWarnings(value={"NP_NONNULL_PARAM_VIOLATION"})
        public void channelInactive(ChannelHandlerContext ctx) throws Exception {
            if (this.cause != null) {
                SmtpSession.this.closeFuture.completeExceptionally(this.cause);
            } else {
                SmtpSession.this.closeFuture.complete(null);
            }
            super.channelInactive(ctx);
        }
    }

    private class SendSequence {
        final Optional<SendInterceptor> sequenceInterceptor;
        CompletableFuture<List<SmtpResponse>> responseFuture;

        SendSequence(Optional<SendInterceptor> sequenceInterceptor, int expectedResponses, Object ... objects) {
            this.sequenceInterceptor = sequenceInterceptor;
            this.responseFuture = this.writeObjectsAndCollectResponses(expectedResponses, objects);
        }

        SendSequence thenSend(Object ... objects) {
            this.responseFuture = this.responseFuture.thenCompose(responses -> {
                if (SmtpResponses.isError((SmtpResponse)responses.get(responses.size() - 1))) {
                    return CompletableFuture.completedFuture(responses);
                }
                return this.writeObjectsAndCollectResponses(1, objects).thenApply(this.mergeResponses((List<SmtpResponse>)responses));
            });
            return this;
        }

        SendSequence thenSendInTurn(Iterator<Object> iterator) {
            this.responseFuture = this.sendNext(this.responseFuture, iterator);
            return this;
        }

        private CompletableFuture<List<SmtpResponse>> sendNext(CompletableFuture<List<SmtpResponse>> prevFuture, Iterator<Object> iterator) {
            if (!iterator.hasNext()) {
                return prevFuture;
            }
            return prevFuture.thenCompose(responses -> {
                if (SmtpResponses.isError((SmtpResponse)responses.get(responses.size() - 1))) {
                    return CompletableFuture.completedFuture(responses);
                }
                Object nextObject = iterator.next();
                CompletionStage f = this.writeObjectsAndCollectResponses(1, nextObject).thenApply(this.mergeResponses((List<SmtpResponse>)responses));
                return this.sendNext((CompletableFuture<List<SmtpResponse>>)f, iterator);
            });
        }

        private Function<List<SmtpResponse>, List<SmtpResponse>> mergeResponses(List<SmtpResponse> existingResponses) {
            return newResponses -> {
                ArrayList newList = Lists.newArrayList((Iterable)existingResponses);
                newList.addAll(newResponses);
                return newList;
            };
        }

        private CompletableFuture<List<SmtpResponse>> writeObjectsAndCollectResponses(int expectedResponses, Object ... objects) {
            return this.executeInterceptor(expectedResponses, objects, () -> {
                CompletableFuture<List<SmtpResponse>> nextFuture = this.createFuture(expectedResponses, objects);
                this.writeObjects(objects);
                return nextFuture;
            });
        }

        private CompletableFuture<List<SmtpResponse>> executeInterceptor(int expectedResponses, Object[] objects, Supplier<CompletableFuture<List<SmtpResponse>>> supplier) {
            Optional<SendInterceptor> interceptor = Optional.ofNullable(this.sequenceInterceptor.orElse(SmtpSession.this.config.getSendInterceptor().orElse(null)));
            if (!interceptor.isPresent()) {
                return supplier.get();
            }
            if (expectedResponses > 1) {
                ArrayList requests = Lists.newArrayList();
                for (Object obj : objects) {
                    if (!(obj instanceof SmtpRequest)) continue;
                    requests.add((SmtpRequest)obj);
                }
                return SmtpSession.this.executePipelineInterceptor(interceptor, requests, supplier);
            }
            if (objects[0] instanceof SmtpRequest) {
                return SmtpSession.this.executeRequestInterceptor(interceptor, (SmtpRequest)objects[0], supplier);
            }
            return SmtpSession.this.executeDataInterceptor(interceptor, supplier);
        }

        CompletableFuture<SmtpClientResponse> toResponses() {
            return SmtpSession.this.applyOnExecutor(this.responseFuture, x$0 -> SmtpSession.this.wrapResponses((List<SmtpResponse>)x$0));
        }

        private void writeObjects(Object[] objects) {
            for (Object obj : objects) {
                SmtpSession.this.write(obj);
            }
            SmtpSession.this.channel.flush();
        }

        private CompletableFuture<List<SmtpResponse>> createFuture(int expectedResponses, Object[] objects) {
            return SmtpSession.this.responseHandler.createResponseFuture(expectedResponses, () -> SmtpSession.createDebugString(objects));
        }
    }
}

