package com.huifu.adapay.model;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.X509EncodedKeySpec;
import java.util.Map;

import com.huifu.adapay.APIUrlEnum;
import com.huifu.adapay.core.exception.BaseAdaPayException;
import com.huifu.adapay.core.util.AdapaySign;
import com.huifu.adapay.core.util.StreamUtil;
import com.huifu.adapay.model.AdapayRequest;
import org.apache.commons.codec.binary.Base64;

/**
 * @author yingyong.wang
 */
public class AdapayTools  {


    /**
     * 获取银联云闪付用户标识
     *
     * @param params 获取银联云闪付用户标识的请求参数，参见 AdaPay api
     * @param merchantKey merchantKey
     * @return 银联云闪付用户标识
     * @throws BaseAdaPayException 异常
     */
    public static Map<String, Object> unionUserId(Map<String, Object> params, String merchantKey) throws BaseAdaPayException {
        String uri = new StringBuilder(APIUrlEnum.UNION_USER_ID_V1.getCode()).toString();
        return AdapayRequest.requestAdapay(uri, params, merchantKey);
    }

    /**
     * 获取银联云闪付用户标识
     *
     * @param params 获取银联云闪付用户标识的请求参数，参见 AdaPay api
     * @return 获取银联云闪付用户标识
     * @throws BaseAdaPayException 异常
     */
    public static Map<String, Object> unionUserId(Map<String, Object> params) throws BaseAdaPayException {
        String uri = new StringBuilder(APIUrlEnum.UNION_USER_ID_V1.getCode()).toString();
        return AdapayRequest.requestAdapay( uri, params);
    }

    /**
     * 下载对账单
     *
     * @param params 建用户对象的请求参数，参见 AdaPay api
     * @param merchantKey merchantKey
     * @return 创建用户对象
     * @throws BaseAdaPayException 异常
     */
    public static Map<String, Object> downloadBill(Map<String, Object> params, String merchantKey) throws BaseAdaPayException {
        String uri = new StringBuilder(APIUrlEnum.BILL_DOWNLOAD_V1.getCode()).toString();
        return AdapayRequest.requestAdapay(uri, params, merchantKey);
    }

    /**
     * 下载对账单
     *
     * @param params 建用户对象的请求参数，参见 AdaPay api
     * @return 创建用户对象
     * @throws BaseAdaPayException 异常
     */
    public static Map<String, Object> downloadBill(Map<String, Object> params) throws BaseAdaPayException {
        String uri = new StringBuilder(APIUrlEnum.BILL_DOWNLOAD_V1.getCode()).toString();
        return AdapayRequest.requestAdapay( uri, params);
    }

    /**
     * 验证签名，固定UTF-8编码
     * @param content 待验证数据
     * @param sign 待验证签名
     * @param publicKey 公钥数据
     * @return 验证结果
     * @throws Exception 异常
     */
    public static boolean verifySign(String content, String sign, String publicKey) throws Exception {
        return AdapaySign.verifySign(content, sign, publicKey, "UTF-8");
    }

    /**
     * 验证签名，指定编码
     * @param content 待验证数据
     * @param sign 待验证签名
     * @param publicKey 公钥数据
     * @param charset 编码
     * @return 验证结果
     * @throws Exception 异常
     */
    public static boolean verifySign(String content, String sign, String publicKey, String charset) throws Exception {
        try {
            PublicKey pubKey = getPublicKeyFromX509("RSA", new ByteArrayInputStream(publicKey.getBytes()));
            Signature signature = Signature.getInstance("SHA1withRSA");
            signature.initVerify(pubKey);
            if (charset==null||charset.length()==0) {
                signature.update(content.getBytes());
            } else {
                signature.update(content.getBytes(charset));
            }

            return signature.verify(Base64.decodeBase64(sign.getBytes()));
        } catch (Exception var6) {
            throw new Exception("RSAcontent = " + content + ",sign=" + sign + ",charset = " + charset, var6);
        }
    }
    /**
     * 生成公钥
     * @param algorithm 签名算法
     * @param ins 公钥数据
     * @return 公钥
     * @throws Exception 异常
     */
    public static PublicKey getPublicKeyFromX509(String algorithm, InputStream ins) throws Exception {
        KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
        StringWriter writer = new StringWriter();
        StreamUtil.io(new InputStreamReader(ins), writer);
        byte[] encodedKey = writer.toString().getBytes();
        encodedKey = Base64.decodeBase64(encodedKey);
        return keyFactory.generatePublic(new X509EncodedKeySpec(encodedKey));
    }

}
