package com.hyperscience.saas.api;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.hyperscience.saas.api.enums.HttpVerb;
import com.hyperscience.saas.api.enums.SupportedMediaType;
import com.hyperscience.saas.api.utils.Constants;
import com.hyperscience.saas.auth.OauthService;
import com.hyperscience.saas.auth.model.Credentials;
import com.hyperscience.saas.config.Configuration;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import lombok.extern.slf4j.Slf4j;
import okhttp3.Call;
import okhttp3.FormBody;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.apache.commons.collections4.MultiValuedMap;
import org.apache.commons.collections4.multimap.ArrayListValuedHashMap;
import org.apache.tika.Tika;

/**
 * RequestHandler acts as interceptor for all requests sent to Hyperscience.
 */
@Slf4j
class RequestHandlerImpl implements RequestHandler {
  static final String USER_AGENT_HEADER_VALUE;
  static final Tika TIKA = new Tika();
  private static final ObjectMapper om = new ObjectMapper();

  static {
    String version = RequestHandler.class.getPackage().getImplementationVersion();
    USER_AGENT_HEADER_VALUE = String.format("hyperscience-saas-client-java/%s", version);
  }

  OauthService oauthService;
  Credentials credentials;
  Configuration configuration;
  String cookie;
  String csrfToken = "";
  private final OkHttpClient client;
  private boolean initialized = false;

  /**
   * Constructor for creating RequestHandler based on Credentials and Configuration.
   *  @param credentials   - Credentials
   * @param configuration - Configuration
   * @param oauthService - OauthService
   */
  public RequestHandlerImpl(Credentials credentials, Configuration configuration,
                            OauthService oauthService) {
    this.credentials = credentials;
    this.oauthService = oauthService;
    this.configuration = configuration;
    this.client = new OkHttpClient().newBuilder().followRedirects(false)
        .connectTimeout(
            configuration.getTimeOutConfiguration().getConnectionTimeout(),
            TimeUnit.SECONDS
        )
        .readTimeout(
            configuration.getTimeOutConfiguration().getReadTimeout(),
            TimeUnit.SECONDS
        ).build();
  }

  @Override
  public void init() throws IOException {
    initialized = true;
    this.cookie = oauthService.login(credentials);
    this.csrfToken = getCsrfToken(cookie);
  }

  @Override
  public Response handleRequest(
      HttpVerb httpVerb,
      String url,
      Map<String, Object> data,
      SupportedMediaType supportedMediaType
  ) throws IOException {
    return this.handleRequest(httpVerb, url, new ArrayListValuedHashMap<>(data),
        supportedMediaType);
  }

  @Override
  public Response handleRequest(
      HttpVerb httpVerb,
      String url,
      MultiValuedMap<String, Object> data,
      SupportedMediaType supportedMediaType
  ) throws IOException {
    if (!initialized) {
      init();
    }

    String preparedUrl = url.startsWith("/") ? url.substring(1) : url;
    Request.Builder requestBuilder =
        createRequest(httpVerb, preparedUrl, data, supportedMediaType);
    requestBuilder.header("Cookie", String.format("csrftoken=%s; %s", csrfToken, cookie));
    requestBuilder.header("User-Agent", USER_AGENT_HEADER_VALUE);
    requestBuilder.header("X-CSRFToken", csrfToken);
    requestBuilder
        .header("Referer", String.format("https://%s", configuration.getHyperscienceDomain()));
    Response response = callClient(requestBuilder);
    if (isSessionExpired(response)) {
      init();
      requestBuilder.header("Cookie", String.format("csrftoken=%s; %s", csrfToken, cookie));
      requestBuilder.header("X-CSRFToken", csrfToken);
      response = callClient(requestBuilder);
    }

    return response;
  }

  /**
   * Generate and send HTTP request.
   *
   * @param requestBuilder - Request builder to generate the request
   * @return HTTP response
   * @throws IOException - In case of error
   */
  public Response callClient(Request.Builder requestBuilder) throws IOException {
    Call call = client.newCall(requestBuilder.build());
    return call.execute();
  }

  String getCsrfToken(String cookie) throws JsonProcessingException {
    String csrfToken = "";
    Request.Builder requestBuilder =
        createRequest(HttpVerb.Get, "", new ArrayListValuedHashMap<>(Collections.emptyMap()), null);
    requestBuilder.header("Cookie", cookie);
    requestBuilder.header("User-Agent", USER_AGENT_HEADER_VALUE);
    Response response;
    try {
      response = callClient(requestBuilder);
      if (response.isSuccessful()) {
        String responseCookie = response.header("Set-Cookie", "");
        String[] csrfTokenParts = responseCookie.split(";");
        String csrfCookie =
            Arrays.stream(csrfTokenParts).filter(cookiePart -> cookiePart.startsWith("csrftoken="))
                .findFirst().orElse("csrftoken=");
        csrfToken = csrfCookie.split("=")[1];
      }
    } catch (IOException ioException) {
      log.error("Failed to get CSRF token, Error: {}", ioException.getMessage());
    }

    return csrfToken;
  }

  private boolean isSessionExpired(Response response) {
    boolean isCookieExpired = response.code() == 302
        && response.header("Location") != null
        && response.header("Location").contains(configuration.getAuthServer());
    boolean isCsrfTokenExpired = false;
    try {
      isCsrfTokenExpired = response.code() == 403
          && response.peekBody(512).string().contains("CSRF Failed");
    } catch (IOException ioException) {
      log.warn("Failed to read the response, request csrf token verification skipped.");
    }

    return isCookieExpired || isCsrfTokenExpired;
  }

  private Request.Builder createRequest(HttpVerb httpVerb, String url,
                                        MultiValuedMap<String, Object> data,
                                        SupportedMediaType contentType)
      throws JsonProcessingException {
    Request.Builder requestBuilder = new Request.Builder();
    HttpUrl.Builder urlBuilder = new HttpUrl.Builder().scheme("https")
        .host(this.configuration.getHyperscienceDomain()).addEncodedPathSegments(url);
    switch (httpVerb) {
      case Get:
        requestBuilder = createGetRequestBody(requestBuilder, urlBuilder, data);
        break;
      case Post:
        requestBuilder =
            createPostRequestBody(requestBuilder, urlBuilder, data, contentType);
        break;
      default:
        throw new IllegalStateException(httpVerb + "is not a supported HTTP verb!");
    }

    return requestBuilder;
  }

  private Request.Builder createGetRequestBody(Request.Builder requestBuilder,
                                               HttpUrl.Builder urlBuilder,
                                               MultiValuedMap<String, Object> data) {
    if (data != null) {
      data.asMap().forEach((key, values) -> values.forEach(value -> {
        urlBuilder.addQueryParameter(key, value.toString());
      }));
    }
    requestBuilder.get().url(urlBuilder.build().url());

    return requestBuilder;
  }

  private Request.Builder createPostRequestBody(Request.Builder requestBuilder,
                                                HttpUrl.Builder urlBuilder,
                                                MultiValuedMap<String, Object> data,
                                                SupportedMediaType contentType)
      throws JsonProcessingException {
    requestBuilder.url(urlBuilder.build().url());
    switch (contentType) {
      case FormUrlEncoded:
        FormBody.Builder formBodyBuilder = new FormBody.Builder();
        data.asMap().forEach((key, values) -> values.forEach(value -> {
          formBodyBuilder.add(key, value.toString());
        }));
        requestBuilder.addHeader("Content-Type", "application/x-www-form-urlencoded");
        requestBuilder.post(formBodyBuilder.build());
        break;
      case MultipartFormData:
        MultipartBody.Builder multiPartBuilder = new MultipartBody.Builder()
            .setType(MultipartBody.FORM);
        data.asMap().forEach((key, values) -> values.forEach(value -> {
          String stringValue = value.toString();
          if (isFile(stringValue)) {
            addFile(multiPartBuilder, key, stringValue);
          } else {
            multiPartBuilder.addFormDataPart(key, stringValue);
          }
        }));
        requestBuilder.post(multiPartBuilder.build());
        break;
      case ApplicationJson:
        Map<String, Object> jsonMap = new HashMap<>();
        data.asMap().forEach((key, value) -> {
          if (value.size() != 1) {
            throw new IllegalStateException("Not a valid json");
          }
          jsonMap.put(key, (new ArrayList<>(value)).get(0));
        });
        String jsonString = om.writeValueAsString(jsonMap);
        RequestBody body =
            RequestBody.create(jsonString, MediaType.parse("application/json; charset=utf-8"));
        requestBuilder.addHeader(Constants.CONTENT_TYPE_HEADER,
            SupportedMediaType.ApplicationJson.getValue());
        requestBuilder.post(body);
        break;
      default:
        throw new IllegalStateException(contentType + "is not supported contentType");
    }
    return requestBuilder;
  }

  private void addFile(MultipartBody.Builder multiPartBuilder, String key, String value) {
    try {
      Path path = Paths.get(value);
      File file = new File(value);
      String mimeType = TIKA.detect(file);
      multiPartBuilder.addFormDataPart(
          key, path.getFileName().toString(),
          RequestBody.create(file, MediaType.parse(mimeType)));
    } catch (IOException exception) {
      log.error("failed to detect mime-type for the file, ignoring {}", value);
    }
  }

  private boolean isFile(String value) {
    boolean fileExists;
    try {
      fileExists = Files.exists(Paths.get(value));
    } catch (InvalidPathException ex) {
      fileExists = false;
    }

    return fileExists;
  }
}
