package com.hyperscience.saas.auth;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.hyperscience.saas.api.enums.SupportedMediaType;
import com.hyperscience.saas.api.utils.Constants;
import com.hyperscience.saas.auth.model.Credentials;
import com.hyperscience.saas.auth.model.NonceResponse;
import com.hyperscience.saas.auth.model.OktaRequestDto;
import com.hyperscience.saas.auth.model.OktaResponseDto;
import com.hyperscience.saas.config.Configuration;
import com.hyperscience.saas.utils.Validator;
import java.io.IOException;
import java.net.URL;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import lombok.extern.slf4j.Slf4j;
import okhttp3.Call;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

/**
 * OAuth Service to authenticate with the SaaS instance.
 */
@Slf4j
public class OauthServiceImpl implements OauthService {
  Configuration configuration;
  static ObjectMapper objectMapper = new ObjectMapper();
  private final OkHttpClient client;
  static final RequestBody EMPTY_REQUEST =
      RequestBody.create("", MediaType.parse("application/json"));

  /**
   * Builds new OauthServiceImpl Object.
   * @param configuration to configure OauthService
   */
  public OauthServiceImpl(Configuration configuration) {
    this.configuration = configuration;
    this.client = new OkHttpClient().newBuilder().followRedirects(false)
        .connectTimeout(
            configuration.getTimeOutConfiguration().getConnectionTimeout(),
            TimeUnit.SECONDS
        )
        .readTimeout(
            configuration.getTimeOutConfiguration().getReadTimeout(),
            TimeUnit.SECONDS
        ).build();
  }

  @Override
  public String login(Credentials credentials) throws IOException {
    URL url = new HttpUrl.Builder().scheme("https").host(this.configuration.getAuthServer())
        .addPathSegments("api/v1/authn").build().url();
    OktaRequestDto oktaRequestDto = new OktaRequestDto(
        credentials.decodeClientId(), credentials.decodeClientSecret());
    RequestBody requestBody = RequestBody.create(
        objectMapper.writeValueAsString(oktaRequestDto), MediaType.parse("application/json"));
    Request.Builder oktaRequestBuilder = new Request.Builder().url(url)
        .header(Constants.CONTENT_TYPE_HEADER, SupportedMediaType.ApplicationJson.getValue())
        .header(Constants.ACCEPT_HEADER, SupportedMediaType.ApplicationJson.getValue())
        .post(requestBody);
    Response response = callClient(oktaRequestBuilder);
    OktaResponseDto oktaResponseDto = objectMapper.readValue(
        Objects.requireNonNull(response.body(), "illegal okta response!").byteStream(),
        OktaResponseDto.class);
    Validator.validateNotNull(oktaResponseDto, "illegal okta response!");
    Validator
        .validateNotEmpty(oktaResponseDto.getSessionToken(), "failed to authenticate with okta!");
    log.debug("Successfully logged in to OKTA.");

    url = new HttpUrl.Builder().scheme("https")
        .host(this.configuration.getHyperscienceDomain()).build().url();
    response = callClient(new Request.Builder().get().url(url));
    String location = response.header("Location");
    Validator.validateNotEmpty(location, "Failed to receive authentication location header.");
    log.debug("Successfully received redirection URL");
    URL locationUrl = new URL(Objects.requireNonNull(location));

    URL nonceUrl = new HttpUrl.Builder().scheme("https").host(locationUrl.getHost())
        .addPathSegments("api/v1/internal/device/nonce").build().url();
    response = callClient(new Request.Builder().post(EMPTY_REQUEST).url(nonceUrl));
    NonceResponse nonceResponse = objectMapper.readValue(
        Objects.requireNonNull(response.body(), "illegal nonce response!").byteStream(),
        NonceResponse.class
    );
    Validator.validateNotNull(nonceResponse, "illegal nonce response!");
    Validator.validateNotEmpty(nonceResponse.getNonce(), "illegal nonce response value!");
    log.debug("Successfully received nonce.");

    String queryString = locationUrl.getQuery() + "&sessionToken="
        + oktaResponseDto.getSessionToken() + "&nonce=" + nonceResponse.getNonce();
    url = new HttpUrl.Builder().scheme("https").host(locationUrl.getHost())
        .addEncodedPathSegments(locationUrl.getPath().substring(1))
            .encodedQuery(queryString).build().url();
    response = callClient(new Request.Builder().get().url(url));
    location = response.header("Location");
    Validator.validateNotEmpty(location, "Authentication has failed!");
    log.debug("Successfully authenticated!");

    locationUrl = new URL(Objects.requireNonNull(location));
    url = new HttpUrl.Builder().scheme("https").host(configuration.getHyperscienceDomain())
        .addEncodedPathSegments("oauth2/idpresponse").encodedQuery(locationUrl.getQuery())
        .build().url();
    response = callClient(new Request.Builder().get().url(url));
    String cookie = response.header("Set-Cookie");
    Validator.validateNotEmpty(cookie, "Failed to authenticate, cookie is not valid!");
    log.info("Successfully received cookie! Login is complete.");
    return cookie;
  }

  public Response callClient(Request.Builder requestBuilder) throws IOException {
    Call call = client.newCall(requestBuilder.build());
    return call.execute();
  }
}
