package com.hyperscience.saas.config;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.hyperscience.saas.config.model.ConfigurationDto;
import com.hyperscience.saas.config.model.TimeoutConfigurationDto;
import com.hyperscience.saas.errors.ConfigurationLoadException;
import com.hyperscience.saas.utils.Validator;
import java.io.File;
import java.io.IOException;
import lombok.Getter;

/**
 * Configuration servers to use with API calls.
 */
public class Configuration {
  private static final ObjectMapper om = new ObjectMapper();
  @Getter
  private final String hyperscienceDomain;
  @Getter
  private final String authServer;
  private TimeoutConfiguration timeOutConfiguration;

  /**
   * Builds new configuration Object.
   *
   * @param hyperscienceDomain uri
   */
  public Configuration(String hyperscienceDomain) {
    this.hyperscienceDomain = hyperscienceDomain;
    this.authServer = PropertiesSingleton.getSaasClientProperties().getProperty("auth_server");
    this.timeOutConfiguration = TimeoutConfiguration.getDefaultTimeoutConfiguration();
  }

  /**
   * Builds new configuration Object.
   *
   * @param hyperscienceDomain uri
   * @param authServer         uri
   */
  public Configuration(String hyperscienceDomain, String authServer) {
    this.hyperscienceDomain = hyperscienceDomain;
    this.authServer = authServer;
    this.timeOutConfiguration = TimeoutConfiguration.getDefaultTimeoutConfiguration();
  }

  /**
   * Builds new configuration Object.
   *
   * @param hyperscienceDomain   uri
   * @param timeoutConfiguration to configure requests timeout, will use default if null
   *                             (60 seconds for connectionTimeout and 60 seconds for readTimeout)
   */
  public Configuration(String hyperscienceDomain, TimeoutConfiguration timeoutConfiguration) {
    this.hyperscienceDomain = hyperscienceDomain;
    this.authServer = PropertiesSingleton.getSaasClientProperties().getProperty("auth_server");
    this.timeOutConfiguration = timeoutConfiguration != null ? timeoutConfiguration
        : TimeoutConfiguration.getDefaultTimeoutConfiguration();
  }

  /**
   * Builds new configuration Object.
   *
   * @param hyperscienceDomain   uri
   * @param authServer           uri
   * @param timeoutConfiguration to configure requests timeout, will use default if null
   *                             (60 seconds for connectionTimeout and 60 seconds for readTimeout)
   */
  public Configuration(String hyperscienceDomain, String authServer,
                       TimeoutConfiguration timeoutConfiguration) {
    this.hyperscienceDomain = hyperscienceDomain;
    this.authServer = authServer;
    this.timeOutConfiguration = timeoutConfiguration != null ? timeoutConfiguration
        : TimeoutConfiguration.getDefaultTimeoutConfiguration();
  }

  /**
   * Configuration constructor for loading configuration from file or string.
   *
   * @param configurationDto - ConfigurationDTO
   */
  public Configuration(ConfigurationDto configurationDto) {
    String authServer = configurationDto.getAuthServer();
    this.authServer =
        authServer != null ? authServer :
            PropertiesSingleton.getSaasClientProperties().getProperty("auth_server");
    this.hyperscienceDomain = configurationDto.getHyperscienceDomain();
    TimeoutConfigurationDto timeoutConfigurationDto = configurationDto.getTimeoutConfiguration();
    if (timeoutConfigurationDto != null) {
      this.timeOutConfiguration = new TimeoutConfiguration(
          timeoutConfigurationDto.getConnectionTimeout(),
          timeoutConfigurationDto.getReadTimeout()
      );
    }
    validate(this);
  }

  /**
   * Create Configuration from file.
   *
   * @param path - Absolute path to file
   * @return Configuration
   * @throws ConfigurationLoadException - in case loading from file fails
   */
  public static Configuration fromFile(String path) throws ConfigurationLoadException {
    File file = new File(path);
    ConfigurationDto configurationDto;
    try {
      configurationDto = om.readValue(file, ConfigurationDto.class);
    } catch (IOException ex) {
      throw new ConfigurationLoadException("Failed to load configuration from file", ex);
    }
    return new Configuration(configurationDto);
  }

  /**
   * Load configuration using json object from string.
   *
   * @param json - String JSON object
   * @return - Configuration
   * @throws ConfigurationLoadException - in case loading from file fails
   */
  public static Configuration fromJsonString(String json) throws ConfigurationLoadException {
    ConfigurationDto configurationDto;
    try {
      configurationDto = om.readValue(json, ConfigurationDto.class);
    } catch (JsonProcessingException ex) {
      throw new ConfigurationLoadException("Failed to load configuration from string", ex);
    }

    return new Configuration(configurationDto);
  }

  private static void validate(Configuration configuration) {
    Validator.validateNotEmpty(configuration.authServer, "auth_server is required!");
    Validator.validateNotEmpty(configuration.hyperscienceDomain, "hs_domain is required!");
    Validator.validateGreaterThanZero(
        configuration.getTimeOutConfiguration().getConnectionTimeout(),
        "connection_timeout should be greater than 0!"
    );
    Validator.validateGreaterThanZero(
        configuration.getTimeOutConfiguration().getReadTimeout(),
        "read_timeout should be greater than 0!"
    );
  }

  public TimeoutConfiguration getTimeOutConfiguration() {
    return this.timeOutConfiguration != null ? this.timeOutConfiguration
        : TimeoutConfiguration.getDefaultTimeoutConfiguration();
  }
}
