package com.hyprmx.android

import android.app.Activity
import android.content.Context
import android.util.DisplayMetrics
import android.view.WindowManager
import android.widget.LinearLayout
import androidx.annotation.Keep
import com.applovin.mediation.MaxAdFormat
import com.applovin.mediation.adapter.MaxAdViewAdapter
import com.applovin.mediation.adapter.MaxAdapter
import com.applovin.mediation.adapter.MaxAdapterError
import com.applovin.mediation.adapter.MaxInterstitialAdapter
import com.applovin.mediation.adapter.MaxRewardedAdapter
import com.applovin.mediation.adapter.listeners.MaxAdViewAdapterListener
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener
import com.applovin.mediation.adapter.listeners.MaxRewardedAdapterListener
import com.applovin.mediation.adapter.parameters.MaxAdapterInitializationParameters
import com.applovin.mediation.adapter.parameters.MaxAdapterParameters
import com.applovin.mediation.adapter.parameters.MaxAdapterResponseParameters
import com.applovin.mediation.adapters.MediationAdapterBase
import com.applovin.sdk.AppLovinSdk
import com.hyprmx.android.maxadapter.BuildConfig
import com.hyprmx.android.sdk.banner.HyprMXBannerListener
import com.hyprmx.android.sdk.banner.HyprMXBannerSize
import com.hyprmx.android.sdk.banner.HyprMXBannerSize.HyprMXAdSizeBanner
import com.hyprmx.android.sdk.banner.HyprMXBannerSize.HyprMXAdSizeLeaderboard
import com.hyprmx.android.sdk.banner.HyprMXBannerSize.HyprMXAdSizeMediumRectangle
import com.hyprmx.android.sdk.banner.HyprMXBannerView
import com.hyprmx.android.sdk.consent.ConsentStatus
import com.hyprmx.android.sdk.core.HyprMX
import com.hyprmx.android.sdk.core.HyprMXErrors
import com.hyprmx.android.sdk.core.HyprMXIf.HyprMXInitializationListener
import com.hyprmx.android.sdk.core.InitResult
import com.hyprmx.android.sdk.placement.HyprMXLoadAdListener
import com.hyprmx.android.sdk.placement.HyprMXRewardedShowListener
import com.hyprmx.android.sdk.placement.HyprMXShowListener
import com.hyprmx.android.sdk.placement.Placement
import com.hyprmx.android.sdk.utility.HyprMXLog
import java.util.concurrent.atomic.AtomicBoolean

@Keep
class HyprMXMaxAdapter(appLovinSdk: AppLovinSdk) :
  MediationAdapterBase(appLovinSdk),
  MaxAdViewAdapter,
  MaxInterstitialAdapter,
  MaxRewardedAdapter {
  private var adView: HyprMXBannerView? = null
  private var interstitialAd: Placement? = null
  private var rewardedAd: Placement? = null

  override fun getSdkVersion(): String {
    return BuildConfig.HYPRMX_SDK_VERSION
  }

  override fun getAdapterVersion(): String {
    return BuildConfig.HYPRMX_MAX_ADAPTER_VERSION
  }

  override fun onDestroy() {
    adView?.destroy()
    adView?.listener = null
    adView = null

    interstitialAd = null
    rewardedAd = null
  }

  override fun initialize(
    parameters: MaxAdapterInitializationParameters,
    activity: Activity?,
    onCompletionListener: MaxAdapter.OnCompletionListener,
  ) {
    HyprMXLog.i("Initializing HyprMX SDK with adapter version: $adapterVersion and SDK version: $sdkVersion and build number ${BuildConfig.HYPRMX_MAX_ADAPTER_BUILD_NUMBER}")
    if (activity == null) {
      HyprMXLog.d("HyprMX SDK failed to initialize.  Activity is null")
      initializationStatus = MaxAdapter.InitializationStatus.INITIALIZED_FAILURE
      onCompletionListener.onCompletion(
        initializationStatus,
        "HyprMX SDK failed to initialize.  Activity is null",
      )
      return
    }

    if (initialized.compareAndSet(false, true)) {
      initializationStatus = MaxAdapter.InitializationStatus.INITIALIZING

      val distributorId = if (enableTestMode) {
        HyprMXLog.enableDebugLogs(true)
        TEST_DIST_ID.also {
          HyprMXLog.d("Test mode is enabled. Replacing distributorId with test one $it")
        }
      } else {
        parameters.serverParameters.getString("app_id")
      }

      if (distributorId == null) {
        HyprMXLog.d("HyprMX SDK failed to initialize.  distributorId is null")
        initializationStatus = MaxAdapter.InitializationStatus.INITIALIZED_FAILURE
        onCompletionListener.onCompletion(
          initializationStatus,
          "HyprMX SDK failed to initialize.  distributorId is null",
        )
        return
      }

      HyprMXLog.d("Initializing HyprMX SDK with distributor id: $distributorId")

      HyprMX.setMediationProvider(mediator = "applovin_max_custom", mediatorSDKVersion = AppLovinSdk.VERSION, adapterVersion = "${adapterVersion}b${BuildConfig.HYPRMX_MAX_ADAPTER_BUILD_NUMBER}")

      updateUserConsent(parameters)

      HyprMX.initialize(
        activity.applicationContext,
        distributorId,
        object : HyprMXInitializationListener {
          override fun onInitialized(result: InitResult) {
            if (!result.success) {
              HyprMXLog.d("HyprMX SDK failed to initialize for distributorId: $distributorId")
              initializationStatus = MaxAdapter.InitializationStatus.INITIALIZED_FAILURE
              onCompletionListener.onCompletion(initializationStatus, result.message)
              return
            }
            HyprMXLog.d("HyprMX SDK initialized for distributorId: $distributorId")
            initializationStatus = MaxAdapter.InitializationStatus.INITIALIZED_SUCCESS
            onCompletionListener.onCompletion(initializationStatus, null)
          }
        },
      )
    } else {
      onCompletionListener.onCompletion(initializationStatus, null)
    }
  }

  override fun loadAdViewAd(
    parameters: MaxAdapterResponseParameters,
    adFormat: MaxAdFormat,
    activity: Activity?,
    listener: MaxAdViewAdapterListener,
  ) {
    val placementId = if (enableTestMode) {
      TEST_BANNER_UNIT_ID.also {
        HyprMXLog.d("Test mode is enabled. Replacing placementId with test one $it")
      }
    } else {
      parameters.thirdPartyAdPlacementId
    }

    HyprMXLog.d("Loading ${adFormat.label} AdView ad for placement: $placementId...")

    updateUserConsent(parameters)

    // NOTE: `activity` can only be null in 11.1.0+, and `getApplicationContext()` is introduced in 11.1.0
    if (activity == null) {
      listener.onAdViewAdLoadFailed(MaxAdapterError.NO_FILL)
      return
    }

    // runOnUiThread is necessary to fix a MAX internal crash where HyprMXBannerView is being called in a background thread
    activity.runOnUiThread {
      adView = HyprMXBannerView(activity, null, placementId, toAdSize(adFormat)).apply {
        this.listener = AdViewListener(listener)

        val displayMetrics = DisplayMetrics()
        val windowManager = activity.getSystemService(Context.WINDOW_SERVICE) as WindowManager
        val display = windowManager.defaultDisplay
        display.getMetrics(displayMetrics)

        val size = adFormat.size
        this.layoutParams = LinearLayout.LayoutParams(
          Math.round(size.width * displayMetrics.density),
          Math.round(size.height * displayMetrics.density),
        )

        val adViewAdLoadListener: HyprMXLoadAdListener =
          object : HyprMXLoadAdListener {
            override fun onAdLoaded(isAdAvailable: Boolean) {
              if (!isAdAvailable) {
                HyprMXLog.d("AdView failed to load for placement: $placementId")
                listener.onAdViewAdLoadFailed(MaxAdapterError.NO_FILL)
                return
              }
              HyprMXLog.d("AdView loaded for placement: $placementId")
              listener.onAdViewAdLoaded(adView)
            }
          }

        this.loadAd(adViewAdLoadListener)
      }
    }
  }

  override fun loadInterstitialAd(
    parameters: MaxAdapterResponseParameters,
    activity: Activity?,
    listener: MaxInterstitialAdapterListener,
  ) {
    val placementId = if (enableTestMode) {
      TEST_INTERSTITIAL_UNIT_ID.also {
        HyprMXLog.d("Test mode is enabled. Replacing placementId with test one $it")
      }
    } else {
      parameters.thirdPartyAdPlacementId
    }

    HyprMXLog.d("Loading interstitial ad for placement: $placementId")

    updateUserConsent(parameters)

    interstitialAd = HyprMX.getPlacement(placementId).apply {
      val interstitialAdLoadListener: HyprMXLoadAdListener =
        object : HyprMXLoadAdListener {
          override fun onAdLoaded(isAdAvailable: Boolean) {
            if (!isAdAvailable) {
              HyprMXLog.d("Interstitial failed to load for placement: $placementId")
              listener.onInterstitialAdLoadFailed(MaxAdapterError.NO_FILL)
              return
            }
            HyprMXLog.d("Interstitial ad loaded for placement: $placementId")
            listener.onInterstitialAdLoaded()
          }
        }
      this.loadAd(interstitialAdLoadListener)
    }
  }

  override fun showInterstitialAd(
    parameters: MaxAdapterResponseParameters,
    activity: Activity?,
    listener: MaxInterstitialAdapterListener,
  ) {
    val placementId = parameters.thirdPartyAdPlacementId
    HyprMXLog.d("Showing interstitial ad for placement: $placementId")

    if (interstitialAd == null || !interstitialAd!!.isAdAvailable()) {
      HyprMXLog.d("Interstitial ad not ready for placement: $placementId")
      listener.onInterstitialAdDisplayFailed(
        MaxAdapterError(
          MaxAdapterError.UNSPECIFIED.errorCode,
          "Ad Display Failed",
          0,
          "Interstitial ad not ready",
        ),
      )
      return
    }

    interstitialAd!!.showAd(InterstitialListener(listener))
  }

  override fun loadRewardedAd(
    parameters: MaxAdapterResponseParameters,
    activity: Activity?,
    listener: MaxRewardedAdapterListener,
  ) {
    val placementId = if (enableTestMode) {
      TEST_REWARDED_UNIT_ID.also {
        HyprMXLog.d("Test mode is enabled. Replacing placementId with test one $it")
      }
    } else {
      parameters.thirdPartyAdPlacementId
    }

    HyprMXLog.d("Loading rewarded ad for placement: $placementId")

    updateUserConsent(parameters)

    rewardedAd = HyprMX.getPlacement(placementId).apply {
      val rewardedAdLoadListener: HyprMXLoadAdListener =
        object : HyprMXLoadAdListener {
          override fun onAdLoaded(isAdAvailable: Boolean) {
            if (!isAdAvailable) {
              HyprMXLog.d("Rewarded ad failed to load for placement: $placementId")
              listener.onRewardedAdLoadFailed(MaxAdapterError.NO_FILL)
              return
            }
            HyprMXLog.d("Rewarded ad loaded for placement: $placementId")
            listener.onRewardedAdLoaded()
          }
        }
      this.loadAd(rewardedAdLoadListener)
    }
  }

  override fun showRewardedAd(
    parameters: MaxAdapterResponseParameters,
    activity: Activity?,
    listener: MaxRewardedAdapterListener,
  ) {
    val placementId = parameters.thirdPartyAdPlacementId
    HyprMXLog.d("Showing rewarded ad for placement: $placementId")

    rewardedAd?.let {
      if (it.isAdAvailable()) {
        configureReward(parameters)
        it.showAd(RewardedAdListener(listener))
      } else {
        val adapterError = MaxAdapterError(
          MaxAdapterError.UNSPECIFIED.errorCode,
          "Ad Display Failed",
          0,
          "Rewarded ad not ready",
        )
        HyprMXLog.d("Rewarded ad failed to display with error: $adapterError")
        listener.onRewardedAdDisplayFailed(adapterError)
      }
    } ?: run {
      val adapterError = MaxAdapterError(
        MaxAdapterError.UNSPECIFIED.errorCode,
        "Ad Display Failed",
        0,
        "Rewarded ad is null",
      )
      listener.onRewardedAdDisplayFailed(adapterError)
    }
  }

  //region Helper Methods
  private fun getConsentStatus(parameters: MaxAdapterParameters): ConsentStatus {
    val hasUserConsent = parameters.hasUserConsent()
    val isDoNotSell = parameters.isDoNotSell
    return getConsentStatus(hasUserConsent, isDoNotSell)
  }

  private fun updateUserConsent(parameters: MaxAdapterParameters) {
    // NOTE: HyprMX requested to always set GDPR regardless of region.
    HyprMX.setConsentStatus(getConsentStatus(parameters))
  }

  private fun toAdSize(adFormat: MaxAdFormat): HyprMXBannerSize =
    when (adFormat) {
      MaxAdFormat.BANNER -> HyprMXAdSizeBanner
      MaxAdFormat.MREC -> HyprMXAdSizeMediumRectangle
      MaxAdFormat.LEADER -> HyprMXAdSizeLeaderboard
      else -> throw IllegalArgumentException("Unsupported ad format: $adFormat")
    }

  //endregion
  private inner class AdViewListener
  (val listener: MaxAdViewAdapterListener) : HyprMXBannerListener {
    override fun onAdImpression(view: HyprMXBannerView) {
      HyprMXLog.d("AdView tracked impression for placement: " + view.placementName)
      listener.onAdViewAdDisplayed()
    }

    override fun onAdClicked(view: HyprMXBannerView) {
      HyprMXLog.d("AdView clicked for placement: " + view.placementName)
      listener.onAdViewAdClicked()
    }

    override fun onAdOpened(view: HyprMXBannerView) {
      HyprMXLog.d("AdView expanded for placement: " + view.placementName)
      listener.onAdViewAdExpanded()
    }

    override fun onAdClosed(view: HyprMXBannerView) {
      HyprMXLog.d("AdView collapsed for placement: " + view.placementName)
      listener.onAdViewAdCollapsed()
    }

    @Suppress("OVERRIDE_DEPRECATION")
    override fun onAdLeftApplication(view: HyprMXBannerView) {
      HyprMXLog.d("AdView will leave application for placement: " + view.placementName)
    }
  }

  private inner class InterstitialListener
  (val listener: MaxInterstitialAdapterListener) : HyprMXShowListener {
    override fun onAdStarted(placement: Placement) {
      HyprMXLog.d("Interstitial did show: " + placement.name)
    }

    override fun onAdImpression(placement: Placement) {
      HyprMXLog.d("Interstitial did track impression: " + placement.name)
      listener.onInterstitialAdDisplayed()
    }

    override fun onAdDisplayError(placement: Placement, hyprMXError: HyprMXErrors) {
      val adapterError = MaxAdapterError(
        MaxAdapterError.UNSPECIFIED.errorCode,
        "Ad Display Failed",
        hyprMXError.ordinal,
        hyprMXError.name,
      )
      HyprMXLog.d("Interstitial failed to display with error: " + adapterError + ", for placement: " + placement.name)

      listener.onInterstitialAdDisplayFailed(adapterError)
    }

    override fun onAdClosed(placement: Placement, finished: Boolean) {
      HyprMXLog.d("Interstitial ad hidden with finished state: " + finished + " for placement: " + placement.name)
      listener.onInterstitialAdHidden()
    }
  }

  private inner class RewardedAdListener
  (val listener: MaxRewardedAdapterListener) : HyprMXRewardedShowListener {
    private var hasGrantedReward = false

    override fun onAdStarted(placement: Placement) {
      HyprMXLog.d("Rewarded ad did show: " + placement.name)
    }

    override fun onAdImpression(placement: Placement) {
      HyprMXLog.d("Rewarded ad did track impression: " + placement.name)
      listener.onRewardedAdDisplayed()
    }

    override fun onAdDisplayError(placement: Placement, hyprMXError: HyprMXErrors) {
      val adapterError = MaxAdapterError(
        MaxAdapterError.UNSPECIFIED.errorCode,
        "Ad Display Failed",
        hyprMXError.ordinal,
        hyprMXError.name,
      )
      HyprMXLog.d("Rewarded ad failed to display with error: " + adapterError + ", for placement: " + placement.name)

      listener.onRewardedAdDisplayFailed(adapterError)
    }

    override fun onAdRewarded(placement: Placement, rewardName: String, rewardValue: Int) {
      HyprMXLog.d("Rewarded ad for placement: " + placement.name + " granted reward with rewardName: " + rewardName + " rewardValue: " + rewardValue)
      hasGrantedReward = true
    }

    override fun onAdClosed(placement: Placement, finished: Boolean) {
      if (hasGrantedReward || shouldAlwaysRewardUser()) {
        val reward = reward
        HyprMXLog.d("Rewarded user with reward: " + reward + " for placement: " + placement.name)
        listener.onUserRewarded(reward)
      }

      HyprMXLog.d("Rewarded ad hidden with finished state: " + finished + " for placement: " + placement.name)
      listener.onRewardedAdHidden()
    }
  }

  companion object {
    private val initialized = AtomicBoolean()
    private var initializationStatus: MaxAdapter.InitializationStatus? = null
    private var enableTestMode: Boolean = false
    private const val TEST_DIST_ID = "1000201301"
    private const val TEST_REWARDED_UNIT_ID = "Rewarded_POS1"
    private const val TEST_INTERSTITIAL_UNIT_ID = "Interstitial_POS1"
    private const val TEST_BANNER_UNIT_ID = "Test_Android_320x50"

    /**
     * Determines the consent status based on user consent and do-not-sell flags.
     * Handles null values for both inputs.
     *
     * Logic Summary:
     * hasUserConsent | isDoNotSell | Result
     * ---------------|-------------|---------------------------
     * true           | true        | ConsentStatus.CONSENT_DECLINED
     * true           | false       | ConsentStatus.CONSENT_GIVEN
     * true           | null        | ConsentStatus.CONSENT_STATUS_UNKNOWN
     * false          | true        | ConsentStatus.CONSENT_DECLINED
     * false          | false       | ConsentStatus.CONSENT_DECLINED
     * false          | null        | ConsentStatus.CONSENT_DECLINED
     * null           | true        | ConsentStatus.CONSENT_DECLINED
     * null           | false       | ConsentStatus.CONSENT_STATUS_UNKNOWN
     * null           | null        | ConsentStatus.CONSENT_STATUS_UNKNOWN
     *
     * @param hasUserConsent Boolean? - Represents the user's consent status (true, false, or null if unknown/unset).
     * @param isDoNotSell Boolean? - Represents the Do Not Sell status (true, false, or null if unknown/unset).
     * @return The calculated ConsentStatus based on the inputs. // Using simple @return for plain text
     */
    internal fun getConsentStatus(
      hasUserConsent: Boolean?,
      isDoNotSell: Boolean?,
    ): ConsentStatus {
      val consentStatus = when {
        isDoNotSell == true || hasUserConsent == false -> ConsentStatus.CONSENT_DECLINED
        isDoNotSell == false && hasUserConsent == true -> ConsentStatus.CONSENT_GIVEN
        else -> ConsentStatus.CONSENT_STATUS_UNKNOWN
      }

      HyprMXLog.d("Consent status: $consentStatus hasUserConsent: $hasUserConsent, isDoNotSell: $isDoNotSell")
      return consentStatus
    }

    @JvmStatic
    fun enableTestMode() {
      enableTestMode = true
    }
  }
}
