package com.ibm.ims.application;

import com.ibm.ims.base.DLIBaseSegment;
import com.ibm.ims.base.DLITypeInfo;
import com.ibm.ims.dli.DLIException;
import com.ibm.ims.dli.DLIWarning;

/**
 * <code>IMSFieldMessage</code> is an abstract base class for objects representing a
 * message either comming from and going to an IMS message queue.  A subclassed 
 * <code>IMSFieldMessage</code> provides a mapping between the data contained in
 * the message and access functions that operate on the class. User applications 
 * can reference individual fields of the message by either field index or field name
 * utilizing a wide range of data conversion functions provided by the base class
 * <code>DLIBaseSegment</code>.
 * <p>
 * To create this mapping, a subclass must provide an array of <code>DLITypeInfo</code> 
 * objects representing the fields of the message as the argument in the 
 * <code>IMSFieldMessage</code> constructor.  By doing so, the 
 * <code>DLIBaseSegment</code> class learns the layout of each field in the message
 * which then allows the user to easily access as well as update each of these fields.
 * <p>
 * <code>IMSFieldMessage</code> is used to define either normal input and output
 * messages (usually a separate subclass for each) or to define a Scratch Pad Area (SPA).
 * The SPA is used in conversational transactions to store information between steps
 * in the conversation. To define a SPA message, the <code>IMSFieldMessage</code> subclass 
 * must set the <code>isSPA</code> argument in the <code>IMSFieldMessage</code> 
 * constructor to <code>true</code>.
 * <p>
 *
 * IMPORTANT NOTE:
 * Fields of type <code>CHAR</code> or <code>VARCHAR</code> will be encoded using the 
 * platform's default character encoding.  If another encoding is desired, invoke the
 * inherited <code>setDefaultEncoding</code> method of the <code>IMSFieldMessage</code>
 * class. This method is used to specify the character encoding for all character data 
 * in a the message. 
 * 
 * A typical <code>IMSFieldMessage</code> subclass for an input message will 
 * look similar to the following:
 *
 * <code><pre>
 * public class InputMessage extends com.ibm.ims.application.IMSFieldMessage {
 *    static final DLITypeInfo[] segmentInfo = {
 *      new DLITypeInfo("Field1", DLITypeInfo.CHAR,      1, 10), 
 *      new DLITypeInfo("Field2", DLITypeInfo.INTEGER,  11,  4),
 *      new DLITypeInfo("Field3", DLITypeInfo.SMALLINT, 15,  2)
 *    };
 *
 *    public MySegment() {
 *       super(segmentInfo, 16, false);
 *
 *       // can set the default character encoding to be used here for all character data
 *       // in this segment, otherwise it will default to the system's default encoding
 *
 *       // this.setDefaultEncoding("UTF8");
 *    }
 * }   
 * </pre></code>
 *
 * Code to access the fields in <code>InputMessage</code> will look similar to the
 * following:
 *
 * <code><pre>
 * InputMessage inputMessage = new InputMessage();
 * messageQueue.getUniqueMessage(inputMessage);
 * 
 * // Return "Field1" as a String using its index                                            
 * String field1 = inputMessage.getString(1);  
 * 
 * // Return "Field2" as a String using its field name (note: int to String conversion) 
 * String field2 = inputMessage.getString("Field2");
 * </pre></code>
 *
 * @see DLIBaseSegment
 * @see DLITypeInfo
 */
public abstract class IMSFieldMessage extends com.ibm.ims.base.DLIBaseSegment {

    /** Hard-code the desired OS/390 Encoding */
    private final static String EBCDIC = "Cp1047";

    /** The maximum size of the body of a message. */
    public final static short MAX_MESSAGE_LENGTH = 32760;

    /** 
     *  The transaction code rule indicating that exactly one blank 
     *  will be present between the transaction code and the message body
     *  if the transaction code is less than 8 bytes in length.
     *  If the transaction code is exactly 8 bytes in length, the
     *  message body will immediately follow. 
     */
    public final static int TRANSCODE_VAR_8 = 1;

    /**
     *  The transaction code rule indicating that exactly one blank will 
     *  always be present between the transaction code and the message 
     *  body.  If the transaction code is exactly 8 bytes in length, 
     *  the ninth byte will be a blank. 
     */
    public final static int TRANSCODE_VAR_9 = 4;

    /**
     *  The transaction code rule indicating that any transaction code
     *  will be padded with blanks to a total of 9 bytes in length before
     *  the message body begins. 
     */
    public final static int TRANSCODE_FIXED_9 = 2;

    /**
     *  The transaction code rule indicating that any transaction code
     *  will be padded with blanks to a total of 8 bytes in length before
     *  the message body begins. 
     */
    public final static int TRANSCODE_FIXED_8 = 3;

    final static int LLZZ_LENGTH = 4;
    final static int LLZZZZ_LENGTH = 6;
    final static int MAX_TRANSCODE_LENGTH = 8;

    /** The length of the transaction code. The transaction code field makes 
     * the code below a little complicated. For a SPA message, the transaction
     * code length is always 8 characters (MAX_TRANCODE_LENGTH).  However,
     * on a get unique call, the trancode is variable length, up to eight
     * bytes, plus a trailing blank character.  When IMSMessageQueue reads
     * a message, it needs to find the blank character to determine the 
     * length of the tran code, and then adjust the ioArea length and 
     * ioAreaOffset accordingly.  
     */
    int transCodeLength = 0;
    static int transCodeRule = TRANSCODE_FIXED_8;

    /** The flag telling whether this message is used as a SPA message */
    boolean isSPAMessage = false;   

    /**
     * Creates a new <code>IMSFieldMessage</code> with the specified message length.
     * The maximum length is limited to <code>MAX_MESSAGE_LENGTH</code> (32750) bytes.
     *
     * @param typeInfo The array of <code>DLITypeInfo</code> objects defining the fields in 
     *                 the message.
     * @param length   The length for the message body of this message. 
     * @param isSPA    Flag indicating whether this message is a SPA message
     * @exception IllegalArgumentException if the length is invalid
     */
    public IMSFieldMessage(DLITypeInfo[] typeInfo, int length, boolean isSPA) throws java.lang.IllegalArgumentException {
        // Copy Array to remove potential Cross-Heap References
        super("", (DLITypeInfo[])typeInfo.clone(), length);

        this.isSPAMessage = isSPA;

        // Ensure a valid length is specified
        if (length > MAX_MESSAGE_LENGTH || (length <= 0)) {
            throw new java.lang.IllegalArgumentException(IMSErrorMessages.getIMSBundle().getString("INVALID_LENGTH"));
        }

        // Reset the ioArea data in DLIBaseSegment and allocate the array.  Note
        // that we allocate the array to be large enough to hold the maximum
        // header, even though not all of it may be used initially.  We do this
        // because the same IMSFieldMessage object gets used on a get unique call
        // (with a trancode) and on a get next call (without a trancode).  The
        // max length is length+LLZZ_LENGTH+MAX_TRANCODE_LENGTH+1 for a normal
        // message and length+LLZZZZ_LENGTH+MAX_TRANCODE_LENGTH for a SPA message. 
        // We maintain ioAreaLength as the used portion of ioArea.  Use 
        // ioArea.length to determine the length of the allocated ioArea.
        // ioAreaOffset identifies the start of the message body and is equivalent
        // to the header length.

        if (isSPA) {
            this.transCodeLength = MAX_TRANSCODE_LENGTH;
            this.ioAreaOffset = LLZZZZ_LENGTH + MAX_TRANSCODE_LENGTH;
            this.ioArea = new byte[length+LLZZZZ_LENGTH+MAX_TRANSCODE_LENGTH]; 
        } else {
            this.ioAreaOffset = LLZZ_LENGTH;
            this.ioArea = new byte[length+LLZZ_LENGTH+MAX_TRANSCODE_LENGTH+1];
        }

        this.ioAreaLength = length+this.ioAreaOffset;

        // If not a SPA message, set LL and ZZ into the header
        if (!isSPA) {
            this.ioArea[0] = (byte) (this.ioAreaLength >>> 8);      // LL
            this.ioArea[1] = (byte) (this.ioAreaLength & 0x00FF);
            this.ioArea[2] = (byte) (0x00);                         // ZZ
            this.ioArea[3] = (byte) (0x00);
        }
    }

    /**
     * Creates a new <code>IMSFieldMessage</code> with the specified message length.
     * The maximum length is limited to <code>MAX_MESSAGE_LENGTH</code> (32750) bytes.
     *
     * @param typeInfo The array of <code>DLITypeInfo</code> objects defining the fields in 
     *                 the message.
     * @param length   The length for the message body of this message. 
     * @param isSPA    Flag indicating whether this message is a SPA message
     * @param transCodeRule    the trans code rule used
     * @exception IllegalArgumentException if the length is invalid
     */
    //add date=7/26/2002
    public IMSFieldMessage(DLITypeInfo[] typeInfo, int length, boolean isSPA, int transCodeRule) 
    throws java.lang.IllegalArgumentException {

        this(typeInfo, length, isSPA);

        this.setTransCodeRule(transCodeRule);
    }

    /**
     * Creates a new <code>IMSFieldMessage</code> with the specified message length.
     * The maximum length is limited to <code>MAX_MESSAGE_LENGTH</code> (32750) bytes.
     *
     * @param typeInfo The array of <code>DLITypeInfo</code> objects defining the fields in 
     *                 the message.
     * @param length   The length for the message body of this message. 
     * @param isSPA    Flag indicating whether this message is a SPA message
     * @param ioArea   the input byte array
     * @param transCodeRule    the trans code rule used
     * @exception IllegalArgumentException if the length is invalid
     */
    //add date=7/26/2002
    public IMSFieldMessage(DLITypeInfo[] typeInfo, int length, boolean isSPA, byte[] ioArea, int transCodeRule) 
    throws java.lang.IllegalArgumentException {

        this(typeInfo, length, isSPA, transCodeRule);

        this.setBytes(ioArea);
    }

    /**
     * Creates a new <code>IMSFieldMessage</code> from an existing 
     * <code>IMSFieldMessage</code>.  This allows an application to define one message
     * that contains definitions of fields common to any input message and use that
     * message to construct similar messages that define the remaining fields.
     *
     * The following code demonstrates how this can be used to define three messages
     * that have the field 'CommandCode' in common.
     * 	<code><pre>
     *	public class LogicalBaseMessage extends IMSFieldMessage {
     *		final static DLITypeInfo[] typeInfo = {
     *			new DLITypeInfo("CommandCode", DLITypeInfo.CHAR, 1, 20),
     *		}
     *
     *		public LogicalBaseMessage() {
     *			super(typeInfo, 30, false);
     *		}
     *	}
     *	
     *	public class LogicalSublcassA extends IMSFieldMessage {
     *		final static DLITypeInfo[] typeInfo = {
     *			new DLITypeInfo("CommandCode", DLITypeInfo.CHAR, 1,  20),
     *			new DLITypeInfo("SomeFieldA",  DLITypeInfo.CHAR, 21, 10),
     *		}
     *
     *		public LogicalSubclassA(IMSFieldMessage message) {
     *			super(message, typeInfo);
     *		}
     *	}
     * 
     *	public class LogicalSubclassB extends IMSFieldMessage {
     *		final static DLITypeInfo[] typeInfo = {
     *			new DLITypeInfo("CommandCode", DLITypeInfo.CHAR, 1,  20),
     *			new DLITypeInfo("SomeFieldB",  DLITypeInfo.CHAR, 21, 5),
     *		}
     *
     *		public LogicalSubclassB(IMSFieldMessage message) {
     *			super(message, typeInfo);
     *		}
     *	}
     * </code></pre>
     *
     * Notice a few points about the preceding code:
     * 	1. The 'CommandCode' field is defined within every class.  This is really
     *     only necessary if users of LogicalSubclassA and LogicalSubclassB require
     *     access to this field.
     *	2. The length of the "logical" base class, LogicalBaseClass, must be large
     *     enough to contain any of its "logical" subclasses.  Therefore, 
     *     LogicalBaseClass is 30 bytes long because the fields of LogicalSubclassA
     *     require it.
     * 	3. Each "logical" subclass must provide a constructor to create itself from
     *     another IMSFieldMessage.
     *
     * Given this approach, an application can provide message reading logic similar
     * to the following:
     *	<code><pre>
     *	LogicalBaseClass inputMessage = new LogicalBaseClass();
     *	while(messageQueue.getUniqueMessage(inputMessage)) {
     *		String commandCode = inputMessage.getString("CommandCode").trim();
     *		if (commandCode.equals("LogicalSubclassA")) {
     *			processA(new LogicalSubclassA(inputMessage));
     *		}
     *		else if (commandCode.equals("LogicalSubclassB")) {
     *			processB(new LogicalSubclassB(inputMessage));
     *		}
     *		else {
     *			// process an error
     *		}
     *	}
     *	</code></pre>
     *
     * @param message    the <code>IMSFieldMessage</code>, or "logical" base class, that this 
     *                   message can be created from
     * @param typeInfo   the array of <code>DLITypeInfo</code> objects defining the fields in 
     *                   the message
     */
    public IMSFieldMessage(IMSFieldMessage message, DLITypeInfo[] typeInfo) {
        // Copy Array to remove potential Cross-Heap References
        super(message.getSegmentName(), (DLITypeInfo[])typeInfo.clone(), message.ioAreaLength);

        this.isSPAMessage = message.isSPAMessage;
        this.transCodeLength = message.transCodeLength;
        this.ioAreaOffset = message.ioAreaOffset;
        this.ioArea = message.ioArea;
        this.ioAreaLength = message.ioAreaLength;
        this.setDefaultEncoding(message.getDefaultEncoding());
    }

    /**
     * Clear out the message body with " ".  This will reset all CHAR data fields to
     * blanks, yet leave all non-CHAR fields with 0x40 (if EBCDIC) 0x20 (if ASCII) data.
     * This can cause potential problems for ZONEDDECIMAL and PACKEDDECIMAL since any 
     * 'get' access to that field will throw a <code>DLIException</code> given that it
     * will look for a sign trailing byte of which the " " is invalid.  To correct for
     * this problem make sure that following a clearBody, a 'set' command is issued to
     * any PACKEDDECIMAL or ZONEDDECIMAL field before any 'get' command.
     */
    public void clearBody() {
        try {
            byte blank = " ".getBytes(EBCDIC)[0];
            for (int i=this.ioAreaOffset; i<ioArea.length; i++) {
                ioArea[i] = blank;
            }
        } catch (java.io.UnsupportedEncodingException e) {
            throw new RuntimeException(e.toString());
        }
    }

    /** 
     * Package-level function to clear a portion of the ioArea, starting
     * from the beginning of the ioArea with the clearByte value.
     */
    final void clearBytes(int startingPos, int length, byte clearByte ) {
        while (0 != length--) {
            this.ioArea[startingPos+length] = clearByte;
        }
    }

    public boolean getIsSPAMessage() {
    	return(isSPAMessage);
    }
    /**
     * Clears the warning chain for this IMSFieldMessage.  After this call getWarnings
     * returns null until a new warning is reported for this IMSFieldMessage.
     */
    public void clearWarnings() {
        super.clearWarnings();
    }

    /**
     * Returns the raw byte array of the message field.
     *   This byte array includes the LL and ZZ fields,
     *   the transaction code and the message data.
     *
     * @return the byte array containing the LL and ZZ fields,
     *         the transaction code and the message data.
     */
    public byte[] getBytes() {
        return super.getBytes();
    }

    /**
     * Reads a portion of the DLIBaseSegment. It will read the specified number of
     * bytes, starting at the specified index.
     *
     * @param beginIndex   the beginning index, inclusive
     * @param length   the number of bytes to read
     *
     * @return   the byte array containing the data
     */
    final byte[] getBytes(int beginIndex, int length) {

        byte[] data = new byte[length];

        System.arraycopy(ioArea, beginIndex, data, 0, length);

        return data;
    }

    /**
     * Provides package-level access to DLIBaseSegment.ioArea.
     */
    final byte[] getIOArea() {
        return this.ioArea;
    }

    /**
     * Provides package-level access to DLIBaseSegment.ioAreaLength.
     */
    final int getIOAreaLength() {
        return this.ioAreaLength;
    }

    /**
     * Provides package-level access to DLIBaseSegment.ioAreaOffset.
     */
    final int getIOAreaOffset() {
        return this.ioAreaOffset;
    }

    /**
     * Reads the LL portion of the message to determine the length.
     */
    final short getLL( ) {

        short tempShort = (short) this.ioArea[0];
        short shortData = (short)(tempShort << 8);
        tempShort = (short) this.ioArea[1];
        tempShort = (short)(tempShort & 0x00FF);
        shortData = (short)(shortData | tempShort);

        return shortData;
    }

    /**
     * Return the length of this message. This applies to the message body only,
     * which excludes the length for the header of the message.
     * @return the length of the message body
     */
    public int getMessageLength() {
        return this.ioAreaLength - this.ioAreaOffset;
    }

    /**
     * Returns the transaction ID for the input message after being read
     * off the message queeue by <code>IMSMessageQueue</code>. 
     * @return the transaction ID or <code>null</code> if the transaction ID is not
     * present for this message.
     */
    public String getTransactionID() {
        String result = null;
        try {
            if (transCodeLength != 0) {
                if (this.isSPAMessage) {
                    result = new String(this.ioArea, LLZZZZ_LENGTH, MAX_TRANSCODE_LENGTH, EBCDIC);
                } else {
                    result = new String(ioArea, LLZZ_LENGTH, transCodeLength, EBCDIC);
                }
            }
        } catch (java.io.UnsupportedEncodingException e) {
            throw new RuntimeException(e.toString());
        }

        return result;
    }

    /**
     * Sets the transaction ID in the SPA message. 
     * 
     * This is used for deferred program switching. 
     */
    public void setTransactionID(String transactionID) throws DLIException {
        if (transactionID == null) {
            throw new DLIException(com.ibm.ims.application.IMSErrorMessages.getIMSBundle().getString("TRANSACTION_ID_CANNOT_BE_NULL"));
        }

        // Remove cross heap references
        String newTransactionID = new String(transactionID.trim());
        transactionID = null;

        if (newTransactionID.length() == 0 || newTransactionID.length() > MAX_TRANSCODE_LENGTH) {
            Object[] inserts = {"1", String.valueOf(this.MAX_TRANSCODE_LENGTH)};
            throw new DLIException(com.ibm.ims.application.IMSErrorMessages.getIMSBundle().getString("INVALID_TRANSACTION_ID_LENGTH", inserts));
        }

        byte[] tranBytes = null;

        try {
            tranBytes = new String("        ").getBytes(this.EBCDIC);
            System.arraycopy(newTransactionID.getBytes(this.EBCDIC), 0, tranBytes, 0, newTransactionID.length());

        } catch (java.io.UnsupportedEncodingException e) {
            throw new RuntimeException(e.toString());
        }

        if (this.isSPAMessage) {
            System.arraycopy(tranBytes, 0, ioArea, LLZZZZ_LENGTH, MAX_TRANSCODE_LENGTH);
        } else {
            throw new DLIException(com.ibm.ims.application.IMSErrorMessages.getIMSBundle().getString("ATTEMPT_TO_CHANGE_NONSPA_TRANSACTION_ID"));
        }
    }

    /**
     * Reads the ioArea to determine the length of the transcode.  This
     * function assumes the ioArea contains a transcode.  Its behavior
     * is undefined if the ioArea does not contain a transcode.
     */
    final int getTransCodeLength () {
        int i=0;  

        try {
            if (isSPAMessage) {
                return MAX_TRANSCODE_LENGTH;
            }

            while (i < MAX_TRANSCODE_LENGTH && this.ioArea[i+LLZZ_LENGTH] != " ".getBytes(EBCDIC)[0]) {
                i++;
            }
        } catch (java.io.UnsupportedEncodingException e) {
            throw new RuntimeException(e.toString());
        }

        return i;
    }

    /**
     * The first warning reported by calls on this IMSFieldMessage is returned.  Subsequent warnings
     * will be chained to this DLIWarning. 
     *
     * The warning chain is automatically cleared each time a new message is read from the queue. 
     *
     * Note: This warning chain only covers warnings caused by IMSFieldMessage methods.
     *
     * @return the first DLIWarning or null
     */
    public DLIWarning getWarnings() {
        return super.getWarnings();
    }

    /**
     * Provides package-level access to DLIBaseSegment.ioArea.
     */
    final void setIOArea(byte[] ioArea) {
        this.ioArea = ioArea;
    }

    /**
     * Provides package-level access to DLIBaseSegment.ioAreaLength.
     */
    final void setIOAreaLength(int ioAreaLength) {
        this.ioAreaLength = ioAreaLength;
    }

    /**
     * Provides package-level access to DLIBaseSegment.ioAreaOffset.
     */
    final void setIOAreaOffset(int ioAreaOffset) {
        this.ioAreaOffset = ioAreaOffset;
    }

    /**
     * Sets the transaction code rule for this message.
     * Valid rules are TRANSCODE_VAR_8, TRANSCODE_FIXED_9, TRANSCODE_VAR_9, and
     * TRANSCODE_FIXED_8 (default). The transaction code rule is used to determine
     * the start of the message body when processing an input message.
     *
     * @param rule - the transaction code rule for this message 
     */
    public final static void setTransCodeRule (int rule) throws java.lang.IllegalArgumentException {
        if (rule != TRANSCODE_VAR_8 && rule != TRANSCODE_FIXED_8 && rule != TRANSCODE_VAR_9 && rule != TRANSCODE_FIXED_9) {
            throw new java.lang.IllegalArgumentException(IMSErrorMessages.getIMSBundle().getString("INVALID_RULE"));
        }
        transCodeRule = rule;
    }

    /**
     * Resets any static variables to their initial state.  Called by the Resettable
     * JVM during reset.
     */
    private static boolean ibmJVMTidyUp() {

        transCodeRule = TRANSCODE_FIXED_8;

        return(true);
    }

    void setIOAreaOffset() {
        // header will be LLZZ (4 bytes) + transcode (1 to 8 bytes) + blank (0 or 1 byte).
        this.transCodeLength = this.getTransCodeLength();

        if (this.transCodeRule == this.TRANSCODE_VAR_8) {
            if (this.transCodeLength == 8) {
                this.setIOAreaOffset(this.transCodeLength + this.LLZZ_LENGTH);
            } else {
                this.setIOAreaOffset(this.transCodeLength + 1 + this.LLZZ_LENGTH);
            }
        } else if (this.transCodeRule == this.TRANSCODE_VAR_9) {
            this.setIOAreaOffset(this.transCodeLength + 1 + this.LLZZ_LENGTH);
        } else if (this.transCodeRule == this.TRANSCODE_FIXED_8) {
            this.setIOAreaOffset(this.MAX_TRANSCODE_LENGTH + this.LLZZ_LENGTH);
        } else if (this.transCodeRule == this.TRANSCODE_FIXED_9) {
            this.setIOAreaOffset(this.MAX_TRANSCODE_LENGTH + 1 + this.LLZZ_LENGTH);
        }

        this.setIOAreaLength(this.getLL());
    }

    protected void setBytes(byte[] ioArea) {
        if (!this.isSPAMessage) {
            this.setIOAreaOffset();
        }

        super.setBytes(ioArea);
    }

}
