package com.ibm.ims.db.cci;

/* (c) Copyright International Business Machines Corporation 2008. All rights reserved. */

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.resource.ResourceException;

/**
 * Use this class to access data from an IMS database with the IMS Universal Database Resource Adapter 
 * using DL/I-like queries. The <code>DLInteractionSpec</code> is used by an <code>Interaction</code> 
 * to execute the specified function on an IMS database.
 * 
 */
public class DLIInteractionSpec extends InteractionSpecImpl {

	private String pcbName;
	private String ssaList;

	private static final long serialVersionUID = -6456756057531203795L;
	private static final Logger logger = Logger.getLogger("com.ibm.ims.db.opendb.cci");

	private String functionName;
	public static final String CREATE = "CREATE";
	public static final String RETRIEVE = "RETRIEVE";
	public static final String UPDATE = "UPDATE";
	public static final String DELETE = "DELETE";

	private int functionType;
	static final int FUNC_CREATE = 701;
	static final int FUNC_RETRIEVE = 702;
	static final int FUNC_UPDATE = 703;
	static final int FUNC_DELETE = 704;

	// Required for JavaBean impl requirement of CCI
	/**
	 * The default constructor.
	 */
	public DLIInteractionSpec() {
	}

	/**
	 * Sets the segment search argument list to be used with the DL/I query when the interaction is executed.  
	 * 
	 * <p>
	 * The following code example shows how to set the SSA qualification statement string to return the 
	 * last patient admitted to all wards with more than five doctors and less than three nurses in hospital 
	 * "ALEXANDRIA".  The command *L means "last occurance".
	 * 
	 * <pre>
	 * String query = &quot;Hospital(HospName=ALEXANDRIA) Ward(Doccount&gt;5 | Nurcount&lt;3) Patient *L&quot;;
	 * iSpec.setSSAList(query);
	 * </pre>
	 * 
	 * @param ssaList
	 *            a <code>String</code> set with the SSA qualification statement
	 */
	public void setSSAList(String ssaList) {
		this.ssaList = ssaList;
	}

	/**
	 * Sets the name of the PCB the interaction will use when executed.  
	 * 
	 * @param pcbName
	 *            the PCB name
	 */
	public void setPCBName(String pcbName) {
		this.pcbName = pcbName.toUpperCase();
	}

	/**
	 * Returns the segment search argument list the interaction will use when executed.   
	 * @return the segment search argument list represented in a <code>String</code> format. 
	 */
	public String getSSAList() {
		return this.ssaList;
	}

	/**
	 * Returns the PCB name the interaction will use when executed. 
	 * 
	 * @return the PCB name
	 */
	public String getPCBName() {
		if (logger.isLoggable(Level.FINEST)) {
			logger.entering(getClass().getName(), "getPCBName()");
		}

		if (logger.isLoggable(Level.FINEST)) {
			logger.exiting(getClass().getName(), "getPCBName()", "pcbName: " + this.pcbName);
		}

		return this.pcbName;
	}
	
	/**
	 * Retrieves the function to be performed in this interaction by name.
	 * 
	 * @return the function name ("CREATE", "RETREIVE", "UPDATE", or "DELETE") as a <code>String</code> value
	 */
	public String getFunctionName() {
		return functionName;
	}

	/**
	 * Sets the function to be performed in this interaction by function name.
	 * 
	 * @param functionName Valid values are "CREATE", "RETREIVE", "UPDATE", or "DELETE".
	 * @throws ResourceException
	 */
	public void setFunctionName(String functionName) throws ResourceException {
		this.functionName = functionName.trim().toUpperCase();

		if (this.functionName.equals(DLIInteractionSpec.CREATE)) {
			this.functionType = DLIInteractionSpec.FUNC_CREATE;
		} else if (this.functionName.equals(DLIInteractionSpec.RETRIEVE)) {
			this.functionType = DLIInteractionSpec.FUNC_RETRIEVE;
		} else if (this.functionName.equals(DLIInteractionSpec.UPDATE)) {
			this.functionType = DLIInteractionSpec.FUNC_UPDATE;
		} else if (this.functionName.equals(DLIInteractionSpec.DELETE)) {
			this.functionType = DLIInteractionSpec.FUNC_DELETE;
		} else {
			Object[] inserts = {
				functionName
			};
			throw new ResourceException(CCIErrorMessages.getIMSBundle().getString("INVALID_FUNCTION_NAME", inserts));
		}
	}
	
	/**
	 * Retrieves the function to be performed in this interaction by type.
	 * 
	 * @return the <code>int</code> constant value for the function (<code>FUNC_CREATE</code>, 
	 *  <code>FUNC_RETRIEVE</code>, <code>FUNC_UPDATE</code>, or <code>FUNC_DELETE</code>)
	 */
	public int getFunctionType() {
		return this.functionType;
	}

	/**
     * Gives the DLI driver a hint as to the number of rows that should 
     * be fetched from the database when more rows are needed. The number 
     * of rows specified affects only result sets created using this interaction. 
     * If the value specified is zero, then the hint is ignored. The default 
     * value is zero.
     * 
     *  @param rows the number of rows to fetch
     */
    public void setFetchSize(int rows) {
        super.setFetchSize(rows);
    }
    
    /**
     * Retrieves the number of result set rows that is the default 
     * fetch size for <code>DLIResultSet</code> objects generated from this interaction.
     * 
     * @return the default fetch size for result sets generated from this interaction.
     */
    public int getFetchSize() {
        return super.getFetchSize();
    }
}
