package com.ibm.ims.db.cci;

import java.io.PrintWriter;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;

import javax.resource.ResourceException;

import com.ibm.ims.dli.logging.PrintWriterHandler;

/**
 * Factory class used to create non-managed connections to IMS.
 * 
 * <p>The following code fragment shows how to use <code>IMSManagedConnectionFactory</code>
 * to obtain a <code>Connection</code> object.
 * 
 * <pre>
 * Connection connection = null;
 * IMSManagedConnectionFactory mcf = new IMSManagedConnectionFactory();
 * 
 * // Set connection properties
 * mcf.setUser("myUserid");
 * mcf.setPassword("myPassword");
 * mcf.setDatastoreName("IMS1");
 * mcf.setDatastoreServer("ec0123.my.host.com");
 * mcf.setPortNumber(5555);
 * mcf.setDatabaseName("class://testcases.jdbo.BMP255DatabaseView");
 * mcf.setSSLConnection(false);
 * mcf.setDriverType(IMSManagedConnectionFactory.DRIVER_TYPE_4);
 * mcf.setLoginTimeout(10);
 * 
 * // Create CCI Connection
 * private ConnectionFactoryImpl cf = (ConnectionFactoryImpl) mcf.createConnectionFactory();
 * connection = cf.getConnection();
 * </pre>
 * 
 */
public class IMSManagedConnectionFactory extends com.ibm.ims.db.spi.ManagedConnectionFactoryImpl {

	private static final long serialVersionUID = 4888431389234407903L;
	
    /**
     * Constant indicating the driver is pure Java and implements the network protocol for the data source.
     */
    public static final String DRIVER_TYPE_4 = "4";
    public static final String DRIVER_TYPE_2 = "2";
    public static final String DRIVER_TYPE_2_CTX = "2_CTX";

	private static final Logger logger = Logger.getLogger("com.ibm.ims.db.opendb.cci");
    private PrintWriterHandler pwh;
    private PrintWriter pw;

	/**
	 * Creates a <code>ConnectionFactory</code> instance. The <code>ConnectionFactory</code> instance 
	 * is initialized with a default <code>ConnectionManager</code> object provided by the resource adapter.
	 * 
	 * @return an <code>Object</code> that implements the <code>ConnectionFactory</code> interface
	 * @throws ResourceException
	 *             if an error occurs during processing
	 */
	public Object createConnectionFactory() throws ResourceException {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "createConnectionFactory()");
		}

		Object cf = super.createConnectionFactory();

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "createConnectionFactory()");
		}

		return cf;
	}

    /**
     * Sets the location of the database metadata representing the target IMS database.  
     * The metadata URL is the fully qualified name of the Java metadata class generated by the IMS Explorer for Development.
     * 
     * The URL must begin with class://
     *
     * @param metadataURL metadata url
     * 
     * @deprecated {@link #setDatabaseName(String)}
     */
	public void setMetadataURL(String metadataURL) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setMetadataURL(String)", metadataURL);
		}

		super.setMetadataURL(metadataURL);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setMetadataURL(String)");
		}
	}
	
    /**
     * Sets the name of the target IMS database to be accessed. 
     * 
     * If the metadata repository is the IMS catalog then the database name is
     * the name of the PSB (program specification block) containing the database(s)
     * to be accessed.
     * 
     * If the metadata repository is the file system then the database name
     * is the fully qualified name of the Java metadata class generated
     * by the IMS Explorer for Development.  In this case the name must begin with class://
     * 
     * @param databaseName
     *            database name
     */
    public void setDatabaseName(String databaseName) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setDatabaseName(String)", databaseName);
        }

        super.setDatabaseName(databaseName);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setDatabaseName(String)");
        }
    }
	

    /**
     * Sets the name of the IMS datastore to access. 
     * The datastore parameter must match either the name of the datastore
     * defined to ODBM or be blank.  The datastore name is defined in the ODBM CSLDCxxx PROCLIB 
     * member using either the DATASTORE (NAME=name) or ALIAS (NAME=aliasname) parameter.  
     * If the datastore value is left blank (or not supplied), IMS Connect will connect 
     * to any available instance of ODBM as it is assumed all datastores defined to ODBM are data shared.   
     * 
     * @param datastoreName IMS datastore name
     */
	public void setDatastoreName(String datastoreName) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setDatastoreName(String)", datastoreName);
		}

		super.setDatastoreName(datastoreName);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setDatastoreName(String)");
		}
	}

    /**
     * Sets the name or IP address of the datastore server (IMS Connect).  You can provide either the 
     * host name (for example, dev123.svl.ibm.com) or the IP address  (for example, 192.166.0.2).
     * 
     * @param datastoreServer name or IP address of the datastore server (IMS Connect)
     */
	public void setDatastoreServer(String datastoreServer) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setDatastoreServer(String)", datastoreServer);
		}

		super.setDatastoreServer(datastoreServer);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setDatastoreServer(String)");
		}
	}

    /**
     * Sets the port number to be used to communicate with IMS Connect. The port number is 
     * defined using the DRDAPORT parameter on the ODACCESS statement in the integrated 
     * IMS Connect configuration PROCLIB member. 
     * <p>By default, the port number is 8888.
     * 
     * @param portNumber port number
     */
    public void setPortNumber(int portNumber) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setPortNumber(Integer)", portNumber);
        }

        super.setPortNumber(portNumber);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setPortNumber(Integer)");
        }

    }

    /**
     * Sets the loginTimeout. This value represents the number of seconds to wait for TCP/IP
     * socket creation or server response before timing out. 
     * <p>By default, the value is 0, which means the driver will wait indefinitely for a
     * socket connection creation or server response.
     * 
     * @param loginTimeout
     */
    public void setLoginTimeout(int loginTimeout) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setLoginTimeout(int loginTimeout)", loginTimeout);
        }

        super.setLoginTimeout(loginTimeout);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setLoginTimeout(int loginTimeout)");
        }

    }

    /**
     * Sets the type of driver to use to connect to the database.
     * 
     * @param driverType Only 4 is currently supported.
     */
	public void setDriverType(String driverType) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setDriverType(String)", driverType);
		}

		super.setDriverType(driverType);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setDriverType(String)");
		}

	}

    /**
     * Sets SSL encryption on or off.
     * 
     * @param sslConnection Set to <code>true</code> to turn SSL encryption on.
     */
	public void setSSLConnection(boolean sslConnection) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setSSLConnection(Boolean)", sslConnection);
		}

		super.setSSLConnection(sslConnection);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setSSLConnection(Boolean)");
		}
	}
	
    /**
     * Sets the SSL Trust Store Location
     * 
     * @param sslTrustStoreLocation The SSL Trust Store Location
     */
	public void setSSLTrustStoreLocation(String sslTrustStoreLocation) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setSSLTrustStoreLocation(String)", sslTrustStoreLocation);
		}

		super.setSSLTrustStoreLocation(sslTrustStoreLocation);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setSSLTrustStoreLocation(String)");
		}
	}
	
    /**
     * Sets the SSL Trust Store Password
     * 
     * @param sslTrustStorePassword The SSL Trust Store Password
     */
	public void setSSLTrustStorePassword(String sslTrustStorePassword) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setSSLTrustStorePassword(String)", sslTrustStorePassword);
		}

		super.setSSLTrustStorePassword(sslTrustStorePassword);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setSSLTrustStorePassword(String)");
		}
	}
	
    /**
     * Sets the SSL Trust Store Manager Algorithm
     * 
     * @param sslTrustMgrAlgorithm The SSL Trust Store Manager Algorithm
     */
	public void setSSLTrustMgrAlgorithm(String sslTrustMgrAlgorithm) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setSSLTrustMgrAlgorithm(String)", sslTrustMgrAlgorithm);
		}

		super.setSSLTrustMgrAlgorithm(sslTrustMgrAlgorithm);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setSSLTrustMgrAlgorithm(String)");
		}
	}
	
	
    /**
     * Sets the SSL Key Store Location
     * 
     * @param sslKeyStoreLocation The SSL Key Store Location
     */
	public void setSSLKeyStoreLocation(String sslKeyStoreLocation) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setSSLKeyStoreLocation(String)", sslKeyStoreLocation);
		}

		super.setSSLKeyStoreLocation(sslKeyStoreLocation);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setSSLKeyStoreLocation(String)");
		}
	}
	
    /**
     * Sets the SSL Key Store Password
     * 
     * @param sslKeyStorePassword The SSL Key Store Password
     */
	public void setSSLKeyStorePassword(String sslKeyStorePassword) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setSSLKeyStorePassword(String)", sslKeyStorePassword);
		}

		super.setSSLKeyStorePassword(sslKeyStorePassword);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setSSLKeyStorePassword(String)");
		}
	}
	
    /**
     * Sets the SSL Key Store Manager Algorithm
     * 
     * @param sslKeyMgrAlgorithm The SSL Key Store Manager Algorithm
     */
	public void setSSLKeyMgrAlgorithm(String sslKeyMgrAlgorithm) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setSSLKeyMgrAlgorithm(String)", sslKeyMgrAlgorithm);
		}

		super.setSSLKeyMgrAlgorithm(sslKeyMgrAlgorithm);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setSSLKeyMgrAlgorithm(String)");
		}
	}
	
    /**
     * Sets the Key Store Type
     * 
     * @param keyStoreType The Key Store Type. 
     *        Valid values include "JKS" or "PKCS12".
     */
	public void setKeyStoreType(String keyStoreType) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setKeyStoreType(String)", keyStoreType);
		}

		super.setKeyStoreType(keyStoreType);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setKeyStoreType(String)");
		}
	}
	
    /**
     * Sets the Secure Socket Protocol
     * 
     * @param secureSocketProtocol The Secure Socket Protocol
     *        Valid values include "SSL", "SSLv3", "TLSv1.1", "TLSv1.2",
	 *        etc.
     */
	public void setSecureSocketProtocol(String secureSocketProtocol) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setSecureSocketProtocol(String)", secureSocketProtocol);
		}

		super.setSecureSocketProtocol(secureSocketProtocol);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setSecureSocketProtocol(String)");
		}
	}

    /**
     * Sets the user name
     *
     * @param user User name.
     */
	public void setUser(String user) {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(this.getClass().getName(), "setUser(String)", user);
		}

		super.setUser(user);

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(this.getClass().getName(), "setUser(String)");
		}
	}

    /**
     * Sets the user password.
     *
     * @param password User password value
     */
    public void setPassword(String password) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setPassword(String)");
        }

        super.setPassword(password);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setPassword(String)");
        }
    }
    
    /**
     * Sets the fetch size.
     *
     * @param fetchsize Fetch size value
     */
    public void setFetchSize(int fetchsize) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setFetchSize(int)");
        }

        super.setFetchSize(fetchsize);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setFetchSize(int)");
        }
    }
    
    /**
     * Sets the max rows.
     *
     * @param maxRows Max rows value
     */
    public void setMaxRows(int maxRows) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setMaxRows(int)");
        }

        super.setMaxRows(maxRows);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setMaxRows(int)");
        }
    }

    public void setTreatInvalidDecimalAsNull(boolean treatInvalidDecimalAsNull) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setTreatInvalidDecimalAsNull(boolean)");
        }

        super.setTreatInvalidDecimalAsNull(treatInvalidDecimalAsNull);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setTreatInvalidDecimalAsNull(boolean)");
        }
    }

    public void setFlattenTables(boolean flattenTables) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setFlattenTables(boolean)");
        }

        super.setFlattenTables(flattenTables);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setFlattenTables(boolean)");
        }
    }
    
    public void setPreloadUserTypeConverters(boolean preloadUserTypeConverters) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setPreloadUserTypeConverters(boolean)");
        }

        super.setPreloadUserTypeConverters(preloadUserTypeConverters);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setPreloadUserTypeConverters(boolean)");
        }
    }
    
    public void setSignedCompare(boolean signedCompare) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setSignedCompare(boolean)");
        }

        super.setSignedCompare(signedCompare);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setSignedCompare(boolean)");
        }
    }
    
    /**
     * Sets the current schema.
     *
     * @param currentSchema Current schema value
     */
    public void setCurrentSchema(String currentSchema) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setCurrentSchema(String)");
        }

        super.setCurrentSchema(currentSchema);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setCurrentSchema(String)");
        }
    }
    
    /**
     * Sets the llField property.
     *
     * @param llField llField value
     */
    public void setLLField(boolean llField) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setLLField(boolean)");
        }

        super.setLLField(llField);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setLLField(boolean)");
        }
    }

    /**
     * Sets the JDBC connectivity which indicates whether or not the connection the 
     * application receives is <code>java.sql.Connection</code> versus a <code>javax.resource.cci.Connection</code>.
     * 
     * The default is false.
     * 
     * @param isJDBC <code>true</code> indicates <code>java.sql.Connection</code>, <code>false</code> indicates
     * <code>javax.resource.cci.Connection</code>
     */
    public void setJDBCConnection(boolean isJDBC) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setJDBCConnection(boolean)");
        }

        this.isHybridConnectivity = isJDBC;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setJDBCConnection(boolean)");
        }

    }

    /**
     * Enables connection pooling in a two-tier non-managed environment.
     * 
     * The default value is <code>false</code>.
     * 
     * @param poolConnections
     */
    public void setPoolConnections(boolean poolConnections) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "setPoolConnections(boolean poolConnections)");
        }

        super.poolConnections = poolConnections;

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setPoolConnections(boolean poolConnections)");
        }

    }
    /**
     * When a connection is made and the PSB allocated, this property will indicate that the driver should automatically 
     * issue an INIT STATUS GROUPA or INIT STATUS GROUPB if a value of 'A' or 'B' is provided. The default will not  
     * issue an INIT STATUS GROUP call.
     * 
     * @param initStatusGroup
     */
    public void setInitStatusGroup(String initStatusGroup) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "public void setInitStatusGroup(String initStatusGroup)");
        }
        super.setInitStatusGroup(initStatusGroup);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setInitStatusGroup(String initStatusGroup)");
        }
        
    }
    
	/**
	 * Sets the option to expand or not expand the ResultSet of a column with data
	 * type ARRAY.
	 * <br><br>
	 * When set to <b>true</b> the array result set expands the sub-elements of the
	 * array into 1 or more columns depending on the number of fields defined within
	 * the array.
	 * <br><br>
	 * When set to <b>false</b> (default) the array result set returns a result set
	 * of exactly 2 columns. Column 1 is the index and column 2 is a Struct object
	 * that contains accessors for fields within that array element.
	 * 
	 * @param expandArrayResultSet
	 */
    
    public void setExpandArrayResultSet(boolean expandArrayResultSet) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "public void setExpandArrayResultSet(boolean expandArrayResultSet)");
        }
        super.setExpandArrayResultSet(expandArrayResultSet);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setExpandArrayResultSet(boolean expandArrayResultSet)");
        }
    }
    
	/**
	 * Sets the option to whether to optimize the SSA or not
	 * <br><br>
	 * When set to <b>true</b> the SSA will be optimized to a fully qualified key if 
	 * subfields of that key are provided.
	 * <br><br>
	 * When set to <b>false</b> (default) the SSA will not be optimized.
	 * 
	 * @param ssaOptimization
	 */
    public void setSsaOptimization(boolean ssaOptimization) {
        if (logger.isLoggable(Level.FINER)) {
            logger.entering(this.getClass().getName(), "public void setSsaOptimization(boolean ssaOptimization)");
        }
        super.setSsaOptimization(ssaOptimization);

        if (logger.isLoggable(Level.FINER)) {
            logger.exiting(this.getClass().getName(), "setSsaOptimization(boolean ssaOptimization)");
        }
    }
    
    /**
     * Retrieves the JDBC connectivity type. 
     * @return <code>true</code> indicates that the connection the 
     * 	application receives will be a <code>java.sql.Connection</code>.  <code>false</code> indicates 
     * 	that the connection will be a <code>javax.resource.cci.Connection</code>.
     */
    public java.lang.Boolean getJDBCConnection() {
        return super.getJDBCConnection();
    }

    /**
     * Retrieves the URL of the database metadata representing the target IMS database.
     * 
     * @return the metadata URL
     * 
     * @deprecated {@link #getDatabaseName()}
     */
	public String getMetadataURL() {
		return super.getMetadataURL();
	}
	
    /**
     * Retrieves the name of the target IMS database.
     * 
     * @return the database name
     */
    public String getDatabaseName() {
        return super.getDatabaseName();
    }


    /**
     * Retrieves name of the IMS datastore.
     * 
     * @return the datastore name
     */
	public String getDatastoreName() {
		return super.getDatastoreName();
	}

    /**
     * Retrieves the name or IP address of your datastore server (IMS Connect).
     * 
     * @return the name or IP address of the datastore server (IMS Connect)
     */
	public String getDatastoreServer() {
		return super.getDatastoreServer();
	}

    /**
     * Retrieves the port number.
     * 
     * @return the port number for this connection.
     */
	public Integer getPortNumber() {
		return super.getPortNumber();
	}

    /**
     * Retrieves the database connectivity method. 
     * 
     * @return the database connectivity method
     */
	public String getDriverType() {
		return super.getDriverType();
	}

    /**
     * Retrieves the SSL connectivity.
     * 
     * @return <code>true</code> indicates SSL will be used, <code>false</code> indicates otherwise
     */
	public Boolean getSSLConnection() {
		return super.getSSLConnection();
	}
	
    /**
     * Retrieves the SSL Trust Store Location
     * 
     * @return The SSL Trust Store Location
     */
	public String getSslTrustStoreLocation() {
		return super.getSslTrustStoreLocation();
	}

    /**
     * Retrieves the SSL Trust Store Password
     * 
     * @return The SSL Trust Store Password
     */
	public String getSslTrustStorePassword() {
		return super.getSslTrustStorePassword();
	}
	
    /**
     * Retrieves the SSL Trust Store Manager Algorithm
     * 
     * @return The SSL Trust Store Manager Algorithm
     */
	public String getSslTrustMgrAlgorithm() {
		return super.getSslTrustMgrAlgorithm();
	}
	
    /**
     * Retrieves the SSL Key Store Location
     * 
     * @return The SSL Key Store Location
     */
	public String getSslKeyStoreLocation() {
		return super.getSslKeyStoreLocation();
	}

    /**
     * Retrieves the SSL Key Store Password
     * 
     * @return The SSL Key Store Password
     */
	public String getSslKeyStorePassword() {
		return super.getSslKeyStorePassword();
	}
	
    /**
     * Retrieves the SSL Key Store Manager Algorithm
     * 
     * @return The SSL Key Store Manager Algorithm
     */
	public String getSslKeyMgrAlgorithm() {
		return super.getSslKeyMgrAlgorithm();
	}
	
    /**
     * Retrieves the Key Store Type
     * 
     * @return The Key Store Type
     */
	public String getKeyStoreType() {
		return super.getKeyStoreType();
	}
	
    /**
     * Retrieves the Secure Socket Protocol
     * 
     * @return The Secure Socket Protocol
     */
	public String getSecureSocketProtocol() {
		return super.getSecureSocketProtocol();
	}
	
    /**
     * Retrieves the user name.
     *
     * @return the user name for this connection.
     */
	public String getUser() {
		return super.getUser();
	}

    /**
     * Retrieves the user password.
     *
     * @return user password value for this connection
     */
    public String getPassword() {
        return super.getPassword();
    }

    /**
     * Retrieves the fetch size.
     *
     * @return fetch size value for this connection
     */
    public Integer getFetchSize() {
        return super.getFetchSize();
    }

    /**
     * Retrieves the max rows.
     *
     * @return max rows value for this connection
     */
    public Integer getMaxRows() {
        return super.getMaxRows();
    }

    /**
     * Retrieves the current schema.
     *
     * @return current schema value for this connection
     */
    public String getCurrentSchema() {
        return super.getCurrentSchema();
    }

    /**
     * Retrieves the llField.
     *
     * @return llField value for this connection
     */
    public Boolean getLLField() {
        return super.getLLField();
    }
    
    /**
     * Retrieves the initStatusGroup.
     *
     * @return initStatusGroup value for this connection
     */
	public String getInitStatusGroup() {
		return super.getInitStatusGroup();
	}
    
	/**
	 * Returns the option to expand or not expand the ResultSet of a column with data
	 * type ARRAY.
	 * <br><br>
	 * When set to <b>true</b> the array result set expands the sub-elements of the
	 * array into 1 or more columns depending on the number of fields defined within
	 * the array.
	 * <br><br>
	 * When set to <b>false</b> (default) the array result set returns a result set
	 * of exactly 2 columns. Column 1 is the index and column 2 is a Struct object
	 * that contains accessors for fields within that array element.
	 * 
	 * @param expandArrayResultSet
	 */
	public boolean getExpandArrayResultSet() {
		return super.getExpandArrayResultSet();
	}
	
	/**
	 * Returns the option to whether to optimize the SSA or not
	 * <br><br>
	 * When set to <b>true</b> the SSA will be optimized to a fully qualified key if 
	 * subfields of that key are provided.
	 * <br><br>
	 * When set to <b>false</b> (default) the SSA will not be optimized.
	 * 
	 * @param ssaOptimization
	 */
	public boolean getSsaOptimization() {
		return super.getSsaOptimization();
	}
	
    /***
     * Sets the log writer for this <code>ManagedConnectionFactory</code> object to the given 
     * <code>java.io.PrintWriter</code> object.<p> 
     * The log writer is a character output stream to which all logging and tracing messages for this 
     * data source will be printed. This includes messages printed by the methods of this object, 
     * messages printed by methods of other objects manufactured by this object, and so on.
     *  
     * @param out the new log writer; to disable logging, set to <code>null</code>. 
     */
    public void setLogWriter(PrintWriter out) {
        if (out == null) {
            if (this.pwh != null) {
                Logger logger = Logger.getLogger("com.ibm.ims.db.opendb");
                logger.setLevel(Level.OFF);
            }
        } else {
            if (this.pw == null) {
                this.pw = out;
                this.pwh = new PrintWriterHandler(out, new SimpleFormatter());

                Logger logger = Logger.getLogger("com.ibm.ims.db.opendb");
                logger.setLevel(Level.FINEST);
                logger.addHandler(pwh);
            }
        }
    }

   /**
     *  Retrieves the log writer for this <code>ManagedConnectionFactory</code> object.
     *
     *   The log writer is a character output stream to which all logging and tracing messages
     *   for this data source will be printed. This includes messages printed by the methods of this object,
     *   messages printed by methods of other objects manufactured by this object, and so on. The log writer
     *   is initially <code>null</code>; in other words, the default is for logging to be disabled.
     *
     * @return the log writer for this data source or <code>null</code> if logging is disabled
     */
    public PrintWriter getLogWriter() {
        return this.pw;
    }
}
