package com.ibm.ims.dli;

import java.math.BigDecimal;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;

/**
 * A <code>DBStruct</code> object contains a value for each attribute of the
 * structured type that it represents. Use a DBStruct when a database
 * description (DBD) contains a field with <code>DATATYPE=STRUCT</code>.
 * <p>
 * Your application must be written with an understanding of the Struct as it is
 * defined in the target database, including the type mapping for each element
 * of the Struct.
 */
public interface DBStruct {

	/**
	 * Retrieves and converts the ordered values of the attributes of the
	 * structured type that this <code>DBStruct</code> object represents. The
	 * driver processes and converts each attribute individually, based on the
	 * defined mapping for that attribute.
	 * 
	 * @return an array containing the ordered attribute values
	 * @throws DLIException
	 *             if an error occurs converting the attributes
	 */
	public Object[] getAttributes() throws DLIException;

	/**
	 * Retrieves the type name of the structured type that this
	 * <code>DBStruct</code> object represents.
	 * 
	 * @return the fully-qualified type name in the format
	 *         <code>pcbName.segmentName.structName</code>
	 */
	public String getTypeName();

	/**
	 * Retrieves the value of the designated field in the current element of the
	 * <code>DBStruct</code> object as a <code>com.ibm.ims.dli.DBArray</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 * 
	 * @see DBArray
	 */
	public DBArray getArray(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>com.ibm.ims.dli.DBArray</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 * 
	 * @see DBArray
	 */
	public void setArray(String fieldName, DBArray value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>boolean</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public boolean getBoolean(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>boolean</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setBoolean(String fieldName, boolean value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the segment of this
	 * <code>DBStruct</code> object as a <code>byte</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public byte getByte(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the segment of this
	 * <code>DBStruct</code> object as a <code>byte</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setByte(String fieldName, byte value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the segment of this
	 * <code>DBStruct</code> object as an <code>int</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public int getInt(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as an <code>int</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setInt(String fieldName, int value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>short</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public short getShort(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the s <code>DBStruct</code>
	 * object as a <code>short</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setShort(String fieldName, short value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>long</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public long getLong(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>long</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setLong(String fieldName, long value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>float</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public float getFloat(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>float</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setFloat(String fieldName, float value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>double</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public double getDouble(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>double</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setDouble(String fieldName, double value) throws DLIException;

	/**
	 * Gets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>byte</code> array.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public byte[] getBytes(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>byte</code> array.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setBytes(String fieldName, byte[] value) throws DLIException;

	/**
	 * Gets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>java.lang.String</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public String getString(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>java.lang.String</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setString(String fieldName, String value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>java.math.BigDecimal</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public BigDecimal getBigDecimal(String fieldName) throws DLIException;

	/**
	 * Sets the value of the field in the <code>DBStruct</code> object as a
	 * <code>java.math.BigDecimal</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setBigDecimal(String fieldName, BigDecimal value) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>java.sql.Date</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public Date getDate(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>java.sql.Date</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setDate(String fieldName, Date value) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>java.sql.Time</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public Time getTime(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>java.sql.Time</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setTime(String fieldName, Time value) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>java.sql.TimeStamp</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public Timestamp getTimestamp(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object as a <code>java.sql.TimeStamp</code>.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setTimestamp(String fieldName, Timestamp value) throws DLIException;

	/**
	 * Retrieves the value of the designated field in the <code>DBStruct</code>
	 * object as an <code>Object</code> in the Java Programming Language. <br>
	 * This method will return the value of the given field as a Java object.
	 * The type of the Java object will be the default Java object type
	 * corresponding to the field's actual type.
	 * 
	 * @param fieldName
	 *            name of the field
	 *            
	 * @return value of the field
	 * @exception DLIException
	 *                if an error occurs
	 */
	public Object getObject(String fieldName) throws DLIException;

	/**
	 * Sets the value of the designated field in the <code>DBStruct</code>
	 * object with the <code>Object</code> value.
	 * 
	 * @param fieldName
	 *            name of the field
	 * @param value
	 *            value of the field
	 * 
	 * @exception DLIException
	 *                if an error occurs
	 */
	public void setObject(String fieldName, Object value) throws DLIException;

}
