package com.ibm.ims.dli;

import java.util.Hashtable;
import java.util.Vector;

import com.ibm.ims.dli.DLICall.Function;

/**
 * Use this class to set and retrieve input properties used during DL/I call.
 *
 */
public class DLICallProperty {

	/**
	 * The constants that represent the properties used during the DL/I calls.
	 * 
	 */
	public enum Property {
		AIB_OUTPUT_AREA_LENGTH, AIB_SUB_FUNCTION_CODE, AOI_TOKEN, CALL_TYPE, CLASSNAME, COMMAND, DATA_AREA, FSA, KEYWORD, LOCK_CLASS, LOG_CODE_BYTE, LOG_TEXT, OPTIONS_LIST, PCB_NAME, RESOURCE, SSALIST, SETDATA, STAT_EXTENDED, STAT_FORMAT, STAT_TYPE, TOKEN, PRTO, APPC_SEND_ERROR, APPC_DEALLOCATE_ABEND, DESTINATION, ID, BUFFER_POOL, CONTROL_BLOCKS, DUMP_CONTROL_BLOCKS, ALL, REGION,
	}

	/**
	 * The constants that represent the sub-function codes used for the INQY, INIT and
	 * GMSG DL/I calls. <br>
	 * 
	 */
	public enum Subfunction {
		BLANKS, DBQUERY, ENVIRON, FIND, LERUNOPT, PROGRAM, MSGINFO, WAITAOI, RSA12, STATUSGROUPA, STATUSGROUPB,
	}

	/**
	 * The constants that represent the Q command lock classes.
	 *
	 */
	public enum LockClass {
		A, B, C, D, E, F, G, H, I, J
	}

	/**
	 * The constants that represent the CLASSNAME property used for the AUTH
	 * call.
	 * 
	 */
	public enum Classname {
		TRAN, DATABASE, SEGMENT, FIELD, OTHER
	}

	/**
	 * The constants that represent values for the REGION property used for the
	 * SNAP call.
	 * 
	 */
	public enum SNAP_Region {
		S, Y, N
	}

	/**
	 * The constants that represent the properties used for the POS call.
	 */
	public enum POS_Option {
		V5SEGRBA, PCSEGRTS, PCSEGHWM, PCHSEGTS, PCLBSGTS
	}

	/**
	 * The constants that represent the properties used for the STAT call.
	 * 
	 */
	public enum STAT_Option {
		TYPE_DBAS, TYPE_DBES, TYPE_VBAS, TYPE_VBES, FORMAT_F, FORMAT_O, FORMAT_S, FORMAT_U, STAT_NORMAL, STAT_EXTENDED
	}

	/**
	 * The constants that represent the op code for a field search argument used
	 * during the FLD call. The op code represents the operator for a change
	 * operation.
	 */
	public enum FLD_FSA_OPCODE {

		/**
		 * Add the operand to the field value.
		 */
		ADD,

		/**
		 * Subtract the operand from the field value.
		 */
		SUBTRACT,

		/**
		 * Set the field value to the value of the operand.
		 */
		SET,

		/**
		 * Verify that the field value and the operand are equal.
		 */
		E,

		/**
		 * Verify that the field value is greater than the operand.
		 */
		G,

		/**
		 * Verify that the field value is greater than or equal to the operand.
		 */
		H,

		/**
		 * Verify that the field value is less than the operand.
		 */
		L,

		/**
		 * Verify that the field value is less than or equal to the operand.
		 */
		M,

		/**
		 * Verify that the field value is not equal to the operand.
		 */
		N
	}

	// public interface FSA {
	// public void setFieldName(String fieldName) throws DLIException;
	//
	// public void setOpCode(FLD_FSA_OPCODE opCode);
	//
	// public void setOperand(Object value);
	// }
	private Function function;

	private Hashtable<Property, Object> properties;

	private Vector<FSA> fsas = null;

	/**
	 * <code>DLICallProperty</code> constructor.
	 * 
	 * @param function
	 *            A constant that represents the DL/I call function.
	 */
	public DLICallProperty(Function function) {
		this.function = function;
		this.properties = new Hashtable<Property, Object>();
	}

	/**
	 * Retrieves the DL/I call function.
	 * 
	 * @return a constant that represents the DL/I call function.
	 */
	public Function getFunction() {
		return this.function;
	}

	/**
	 * Retrieves an object representing the DL/I call property.
	 * 
	 * @param key 
	 *        the property key
	 * @return an object representing the DL/I call property.
	 */
	public Object get(Property key) {
		return this.properties.get(key);
	}

	/**
	 * Sets the value of a DL/I call property.
	 * 
	 * @param key
	 *        the key to be placed into this property list
	 * @param value
	 *        the value corresponding to key
	 * @throws DLIException
	 */
	public void put(Property key, Object value) throws DLIException {
		this.validateDataTypeOfValue(key, value);
		this.properties.put(key, value);

	}

	private void validateDataTypeOfValue(Property key, Object value) throws DLIException {
		switch (function) {
		case AUTH:
			switch (key) {
			case CLASSNAME:
				if (!(value instanceof Classname)) {
					Object[] inserts = { key.name(), Classname.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case RESOURCE:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			default:
				// other key values will be ignored
			}
			break;
		// case CMD:
		// switch (key) {
		// case COMMAND:
		// if (!(value instanceof String)) {
		// Object[] inserts = { key.name(), String.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// }
		// break;
		case DEQ:
			switch (key) {
			case LOCK_CLASS:
				if (!(value instanceof LockClass)) {
					Object[] inserts = { key.name(), LockClass.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			}
			break;
		case FLD:
			switch (key) {
			case PCB_NAME:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case FSA:
				if (!(value instanceof FSA)) {
					Object[] inserts = { key.name(), FSA.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				if (this.fsas == null) {
					this.fsas = new Vector<FSA>();
				}
				this.fsas.add((FSA) value);
				break;
			}
			break;
		// case GCMD:
		// break;
		case GMSG:
			switch (key) {
			case AIB_SUB_FUNCTION_CODE:
				if (!(value instanceof Subfunction)) {
					Object[] inserts = { key.name(), Subfunction.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case AOI_TOKEN:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
			}
			break;
		case ICMD:
			switch (key) {
			case COMMAND:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			}
			break;
		case INIT:
			break;
		case INQY:
			switch (key) {
			case AIB_SUB_FUNCTION_CODE:
				if (!(value instanceof Subfunction)) {
					Object[] inserts = { key.name(), Subfunction.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			}
			break;
		case LOG:
			switch (key) {
			case LOG_TEXT:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case LOG_CODE_BYTE:
				if (!(value instanceof Byte)) {
					Object[] inserts = { key.name(), Byte.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			}
			break;
		// case POS:
		// switch (key) {
		// case PCB_NAME:
		// if (!(value instanceof String)) {
		// Object[] inserts = { key.name(), String.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case KEYWORD:
		// if (!(value instanceof POS_Option)) {
		// Object[] inserts = { key.name(), POS_Option.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case SSALIST:
		// if (!(value instanceof SSAList)) {
		// Object[] inserts = { key.name(), SSAList.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// }
		// break;
		case RCMD:
			switch (key) {
			case COMMAND:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			}
			break;
		case RLSE:
			break;
		case ROLL:
			break;
		case ROLS:
			switch (key) {
			case PCB_NAME:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case TOKEN:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case AIB_OUTPUT_AREA_LENGTH:
				if (!(value instanceof Integer)) {
					Object[] inserts = { key.name(), Integer.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			}
			break;
		// case SETO:
		// switch (key) {
		// case PRTO:
		// if (!(value instanceof String)) {
		// Object[] inserts = { key.name(), String.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case APPC_SEND_ERROR:
		// if (!(value instanceof String)) {
		// Object[] inserts = { key.name(), String.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case APPC_DEALLOCATE_ABEND:
		// if (!(value instanceof String)) {
		// Object[] inserts = { key.name(), String.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// }
		// break;
		case SETS:
		case SETU:
			switch (key) {
			case TOKEN:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case SETDATA:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			}
			break;
		// case SNAP:
		// switch (key) {
		// case PCB_NAME:
		// if (!(value instanceof String)) {
		// Object[] inserts = { key.name(), String.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case ID:
		// if (!(value instanceof String)) {
		// Object[] inserts = { key.name(), String.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case DESTINATION:
		// if (!(value instanceof String)) {
		// Object[] inserts = { key.name(), String.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case REGION:
		// if (!(value instanceof SNAP_Region)) {
		// Object[] inserts = { key.name(), SNAP_Region.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case ALL:
		// if (!(value instanceof Boolean)) {
		// Object[] inserts = { key.name(), Boolean.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case BUFFER_POOL:
		// if (!(value instanceof Boolean)) {
		// Object[] inserts = { key.name(), Boolean.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case CONTROL_BLOCKS:
		// if (!(value instanceof Boolean)) {
		// Object[] inserts = { key.name(), Boolean.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// case DUMP_CONTROL_BLOCKS:
		// if (!(value instanceof Boolean)) {
		// Object[] inserts = { key.name(), Boolean.class.getName(),
		// value.getClass().getName() };
		// throw new
		// DLIException(DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE",
		// inserts));
		// }
		// break;
		// }
		// break;
		case STAT:
			switch (key) {
			case PCB_NAME:
				if (!(value instanceof String)) {
					Object[] inserts = { key.name(), String.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case AIB_OUTPUT_AREA_LENGTH:
				if (!(value instanceof Integer)) {
					Object[] inserts = { key.name(), Integer.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case STAT_TYPE:
				if (!(value instanceof STAT_Option)) {
					Object[] inserts = { key.name(), STAT_Option.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case STAT_FORMAT:
				if (!(value instanceof STAT_Option)) {
					Object[] inserts = { key.name(), STAT_Option.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			case STAT_EXTENDED:
				if (!(value instanceof STAT_Option)) {
					Object[] inserts = { key.name(), STAT_Option.class.getName(), value.getClass().getName() };
					throw new DLIException(
							DLIErrorMessages.getIMSBundle().getString("INVALID_PROPERTY_DATA_TYPE", inserts));
				}
				break;
			}
			break;
		default:
		}

	}

	/**
	 * Object that represents a field search argument. Use this object to set
	 * and retrieve the field name, op code, and operand value properties.
	 * <p>
	 * For more details about the usage of field search arguments, see the topic
	 * <a href=
	 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_fldcall.htm"
	 * target="_blank">FLD call</a> in the IMS Application Programming APIs
	 * reference documentation.
	 */
	public class FSA {
		String fieldName;

		FLD_FSA_OPCODE opCode;

		Object value;

		/**
		 * Set the field name value. The field name must be 8 bytes long. If the
		 * field name you are using is less than 8 bytes, the name must be
		 * left-justified and padded on the right with blanks.
		 * 
		 * @param fieldName
		 *            the field name value
		 * @throws DLIException
		 */
		public void setFieldName(String fieldName) throws DLIException {
			this.fieldName = fieldName;
			int n = this.fieldName.length() - 8;
			if (n < 0) {
				String.format("%1$-" + 8 + "s", this.fieldName);
			} else if (n > 0) {
				throw new DLIException(fieldName + " is not valid. Field name can only be a maximum of 8 bytes long.");
			}
		}

		/**
		 * Set the op code.
		 * 
		 * @param opCode
		 *            A constant representing an operator for a change
		 *            operation.
		 */
		public void setOpCode(FLD_FSA_OPCODE opCode) {
			this.opCode = opCode;
		}

		/**
		 * Set the operand value. The operand is a value that you want to test
		 * the field value against.
		 * 
		 * @param value
		 */
		public void setOperand(Object value) {
			this.value = value;
		}

		/**
		 * Retrieves the operand value.
		 * 
		 * @return an object representing the operand value.
		 */
		public Object getValue() {
			return value;
		}

		/**
		 * Set the operand value. The operand is a value that you want to test
		 * the field value against.
		 * 
		 * @param value
		 */
		public void setValue(Object value) {
			this.value = value;
		}

		/**
		 * Retrieves the field name.
		 * 
		 * @return the field name value.
		 */
		public String getFieldName() {
			return fieldName;
		}

		/**
		 * Retrieves the op code value for the FSA.
		 * 
		 * @return a constant representing the op code.
		 */
		public FLD_FSA_OPCODE getOpCode() {
			return opCode;
		}

	}

	/**
	 * Retrieves an object representing the field search argument.
	 * 
	 * @return the <code>FSA</code> object
	 */
	public FSA getFSA() {
		return new FSA();
	}

	/*
	 * Do not add javadoc to this method. Only used internally.
	 */
	public Vector<FSA> getFSAs() {
		return this.fsas;
	}

}
