package com.ibm.ims.dli;

import java.io.UnsupportedEncodingException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.ibm.ims.dli.t2.INQYCallImpl;

/**
 * Object that represents the output from an Inquiry (INQY) call that is issued
 * with the ENVIRON subfunction.
 * <p>
 * For more details about the values returned by the <code>EnvironInfo</code>
 * methods, see the topic <a href=
 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_inqycall.htm"
 * target="_blank">INQY call</a> in the IMS Application Programming APIs
 * reference documentation.
 * 
 */
public class EnvironInfo {
	private StringBuffer info = null;
	byte[] ioArea;
	private INQYCallImpl inqy = null;

	private static final Logger logger = Logger.getLogger("com.ibm.ims.db.opendb.dli");

	/**
	 * <code>EnvironInfo</code> constructor.
	 * 
	 * @param ioArea
	 *            a byte array that represents the data output area.
	 * @throws UnsupportedEncodingException
	 */
	public EnvironInfo(byte[] ioArea, INQYCallImpl inqyCallImpl) throws UnsupportedEncodingException {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(getClass().getName(), "EnvironInfo(byte[])", ioArea);
			logger.finer("Thread ID: " + Thread.currentThread().getId());

			// Print ioArea as a hex string
			String ioAreaAsHexString = bytesToHex(ioArea);
			logger.finer("ioArea: " + ioAreaAsHexString);
		}
		
		this.inqy =  inqyCallImpl;

		this.ioArea = ioArea;
		this.info = new StringBuffer(new String(ioArea, "Cp1047"));

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(getClass().getName(), "EnvironInfo(byte[])");
		}
	}
	
	public EnvironInfo(byte[] ioArea) throws UnsupportedEncodingException {
		if (logger.isLoggable(Level.FINER)) {
			logger.entering(getClass().getName(), "EnvironInfo(byte[])", ioArea);
			logger.finer("Thread ID: " + Thread.currentThread().getId());

			// Print ioArea as a hex string
			String ioAreaAsHexString = bytesToHex(ioArea);
			logger.finer("ioArea: " + ioAreaAsHexString);
		}
		
		this.ioArea = ioArea;
		this.info = new StringBuffer(new String(ioArea, "Cp1047"));

		if (logger.isLoggable(Level.FINER)) {
			logger.exiting(getClass().getName(), "EnvironInfo(byte[])");
		}
	}

	/**
	 * Gets the reason code produced by the ENVIRON INQY DLI call.
	 * 
	 * @return the reason code of the INQY DLI call
	 */
	public int getReasonCode() {
		return this.inqy.getAIB().getReasonCode();
	}
	
	/**
	 * Gets the reason code produced by the ENVIRON INQY DLI call
	 * as a hexadecimal string.
	 * 
	 * @return the reason code of the INQY DLI call
	 */
	public String getReasonCodeHex() {
		return this.inqy.getAIB().getReasonCodeHex();
	}
	
	/**
	 * Gets the return code produced by the ENVIRON INQY DLI call.
	 * 
	 * @return the return code of the INQY DLI call
	 */
	public int getReturnCode() {
		return this.inqy.getAIB().getReturnCode();
	}
	
	/**
	 * Gets the return code produced by the ENVIRON INQY DLI call
	 * as a hexadecimal string.
	 * 
	 * @return the return code of the INQY DLI call
	 */
	public String getReturnCodeHex() {
		return this.inqy.getAIB().getReturnCodeHex();
	}
	
	/**
	 * Gets the IMS status code of the INQY call from the IOPCB.
	 * 
	 * @return the IMS status code of the INQY DLI call.
	 */
	public int getStatusCode() {
		return this.inqy.getAIB().getIOPCB().getStatusCode();
	}
	
	/**
	 * Gets the IMS status code of the INQY call from the IOPCB
	 * as a character string.
	 * 
	 * @return the IMS status code of the INQY DLI call.
	 */
	public String getStatusCodeChars() {
		return this.inqy.getAIB().getIOPCB().getStatusCodeChars();
	}

	/**
	 * Retrieves the IMS identifier from the execution parameters.
	 * 
	 * @return the IMS identifier.
	 */
	public String getIdentifier() {
		return info.substring(0, 8).trim();
	}

	/**
	 * Retrieves the release level for IMS.
	 * 
	 * For example, if IMS Version is 11 and Release is 1, then the value would
	 * be 1110.
	 * 
	 * @return the release level for IMS.
	 */
	public String getReleaseLevel() {
		StringBuffer releaseLevelString = new StringBuffer();
		int startOffset = 8;
		int len = 4;
		for (int idx = startOffset; idx < startOffset + len; idx++) {
			releaseLevelString.append(Integer.toHexString(ioArea[idx]));
		}
		String s = releaseLevelString.toString();

		return Integer.toHexString(Integer.parseInt(s, 16));
	}

	/**
	 * Retrieves the IMS control region type.
	 * 
	 * @return the IMS control region type.
	 */
	public String getControlRegionType() {
		return info.substring(12, 20).trim();
	}

	/**
	 * Retrieves the IMS application region type.
	 * 
	 * @return the IMS application region type.
	 */
	public String getApplicationRegionType() {
		return info.substring(20, 28).trim();
	}

	/**
	 * Retrieves the region identifier as a hex string.
	 * 
	 * @return the region identifier.
	 */
	public String getRegionIdentifier() {
		return Integer.toHexString(getRegionIdentifierAsInt());
	}

	/**
	 * Retrieves the region identifier as an integer.
	 * 
	 * @return the region identifier.
	 */
	public int getRegionIdentifierAsInt() {
		int startOffset = 28;
		int regionID;

		regionID = (int) ioArea[startOffset] << 24;
		regionID = regionID | (((int) ioArea[startOffset + 1] << 16) & 0x00FF0000);
		regionID = regionID | (((int) ioArea[startOffset + 2] << 8) & 0x0000FF00);
		regionID = regionID | ((int) ioArea[startOffset + 3] & 0x000000FF);

		return regionID;
	}

	/**
	 * Retrieves the application program name.
	 * 
	 * @return the name of the application program being run.
	 */
	public String getApplicationProgramName() {
		return info.substring(32, 40).trim();
	}

	/**
	 * Retrieves the PSB name.
	 * 
	 * @return the name of the PSB currently allocated.
	 */
	public String getPSBName() {
		return info.substring(40, 48).trim();
	}

	/**
	 * Retrieves the transaction name. An empty string indicates that no
	 * associated transaction exists.
	 * 
	 * @return the name of the transaction.
	 */
	public String getTransactionName() {
		return info.substring(48, 56).trim();
	}

	/**
	 * Retrieves the user identifier. An empty string indicates that the user ID
	 * is unavailable.
	 * 
	 * @return the user ID.
	 */
	public String getUserIdentifier() {
		return info.substring(56, 64).trim();
	}

	/**
	 * Retrieves the user identifier. An empty string indicates that the user ID
	 * is unavailable.
	 * 
	 * @return the user ID.
	 */
	public String getUserIDIndicator() {
		return info.substring(56, 64).trim();
	}

	/**
	 * Retrieves the group name. An empty string indicates that the group name
	 * is unavailable.
	 * 
	 * @return the group name.
	 */
	public String getGroupName() {
		return info.substring(64, 72);
	}

	/**
	 * Retrieves the status group indicator.
	 * 
	 * @return the status group indicator
	 */
	public String getStatusGroupIndicator() {
		return info.substring(72, 76);
	}

	/**
	 * Retrieves the address of the LL field, followed by the recovery token.
	 * 
	 * @return the address of recovery token.
	 */
	public String getRecoveryTokenAddress() {
		StringBuffer tokenAddress = new StringBuffer();
		int startOffset = 76;
		int length = 4;

		for (int i = startOffset; i < startOffset + length; i++) {
			String s = Integer.toHexString(Byte.parseByte(Byte.toString(ioArea[i])));
			int len = s.length();
			if (len > 1) {
				tokenAddress.append(s.substring(s.length() - 2, s.length()));
			} else {
				tokenAddress.append("0").append(s);
			}
		}

		return tokenAddress.toString();
	}

	/**
	 * Retrieves the address of the LL field, followed by the application
	 * program parameter string.
	 * 
	 * @return the address of the application.
	 */
	public String getApplicationParameterAddress() {
		StringBuffer parmAddress = new StringBuffer();
		int startOffset = 80;
		int length = 4;
		for (int i = startOffset; i < startOffset + length; i++) {
			String s = Integer.toHexString(Byte.parseByte(Byte.toString(ioArea[i])));
			int len = s.length();
			if (len > 1) {
				parmAddress.append(s.substring(s.length() - 2, s.length()));
			} else {
				parmAddress.append("0").append(s);
			}
		}

		if (parmAddress.toString().equals("00000000")) {
			return "0";
		} else {
			return parmAddress.toString();
		}
	}

	/**
	 * Retrieves the shared queue indicator.
	 * 
	 * @return the shared queue indicator
	 */
	public String getSharedQueueIndicator() {
		return info.substring(84, 88);
	}

	/**
	 * Retrieves the user ID of the dependent address space.
	 * 
	 * @return user ID of the dependent address space.
	 */
	public String getAddressSpaceUserid() {
		return info.substring(88, 96).trim();
	}

	/**
	 * Retrieves the contents of the user ID field.
	 * 
	 * @return the user ID indicator field.
	 * 
	 */
	public String getUseridIndicator() {
		return info.substring(96, 97).trim();
	}
	
	/**
	 * Indicates whether IMS has interest in the UR with RRS
	 * 
	 * @return RRS indicates IMS has expressed interest in the UR with RRS. 
	 * Blank indicates that IMS has not expressed interest in the UR with RRS
	 */
	public String getRRSIndicator() {
		return info.substring(97, 100).trim();
	}
	
	/**
	 * Indicates whether the catalog is enabled
	 * 
	 * @return CATALOG if enabled. Blank if not enabled.
	 */
	public String getCatalogIndicator() {
		return info.substring(100, 107).trim();
	}

	public static void main(String[] args) {
		byte[] ioArea = { (byte) 0x00, (byte) 0x00, (byte) 0x11, (byte) 0xb3 };
		for (int i = 0; i < ioArea.length; i++) {
			int x = Byte.parseByte(Byte.toString(ioArea[i]));
			String s = Integer.toHexString(x);
			int len = s.length();
			if (len > 1) {
				System.out.println(s.substring(s.length() - 2, s.length()));
			} else {
				System.out.println("0" + s);
			}
		}
	}

	private static String bytesToHex(byte[] in) {
		final StringBuilder builder = new StringBuilder();
		for (byte b : in) {
			builder.append(String.format("%02x", b));
		}
		return builder.toString();
	}

}
