package com.ibm.ims.dli;

import java.io.UnsupportedEncodingException;

import com.ibm.ims.dli.t2.INQYCallImpl;

/**
 * Object that represents the output from an Inquiry (INQY) call that is issued
 * with the MSGINFO subfunction.
 * <p>
 * For more details about the values returned by the <code>MessageInfo</code>
 * methods, see the topic <a href=
 * "http://www.ibm.com/support/knowledgecenter/SSEPH2_14.1.0/com.ibm.ims14.doc.apr/ims_inqycall.htm"
 * target="_blank">INQY call</a> in the IMS Application Programming APIs
 * reference documentation.
 * 
 */
public class MessageInfo {

	private String imsID;
	private StringBuffer versionNumber = new StringBuffer();
	private INQYCallImpl inqy;

	/**
	 * <code>MessageInfo</code> constructor.
	 * 
	 * @param ioArea
	 *            a byte array that represents the data output area.
	 * 
	 * @throws UnsupportedEncodingException
	 */
	public MessageInfo(byte[] ioArea) throws UnsupportedEncodingException {
		this.imsID = new String(ioArea, 4, 4, "Cp1047").trim();
		for (int i = 0; i < 4; i++) {
			versionNumber.append(ioArea[i]);
		}
	}
	
	public MessageInfo(byte[] ioArea, INQYCallImpl inqyCallImpl) throws UnsupportedEncodingException {
		
		this.inqy =  inqyCallImpl;
		this.imsID = new String(ioArea, 4, 4, "Cp1047").trim();
		for (int i = 0; i < 4; i++) {
			versionNumber.append(ioArea[i]);
		}
	}

	/**
	 * Retrieves the output response version.
	 * 
	 * @return the version number
	 */
	public String getVersionNumber() {
		return versionNumber.toString();
	}

	/**
	 * Retrieves the IMS identifier from which the input message originated.
	 * 
	 * @return IMS identifier.
	 */
	public String getIMSID() {
		return imsID;
	}
	
	/**
	 * Gets the reason code produced by the MSGINFO INQY DLI call.
	 * 
	 * @return the reason code of the INQY DLI call
	 */
	public int getReasonCode() {
		return this.inqy.getAIB().getReasonCode();
	}
	
	/**
	 * Gets the reason code produced by the MSGINFO INQY DLI call
	 * as a hexadecimal string.
	 * 
	 * @return the reason code of the INQY DLI call
	 */
	public String getReasonCodeHex() {
		return this.inqy.getAIB().getReasonCodeHex();
	}
	
	/**
	 * Gets the return code produced by the MSGINFO INQY DLI call.
	 * 
	 * @return the return code of the INQY DLI call
	 */
	public int getReturnCode() {
		return this.inqy.getAIB().getReturnCode();
	}
	
	/**
	 * Gets the return code produced by the MSGINFO INQY DLI call
	 * as a hexadecimal string.
	 * 
	 * @return the return code of the INQY DLI call
	 */
	public String getReturnCodeHex() {
		return this.inqy.getAIB().getReturnCodeHex();
	}
	
	/**
	 * Gets the IMS status code of the INQY call from the IOPCB.
	 * 
	 * @return the IMS status code of the INQY DLI call.
	 */
	public int getStatusCode() {
		return this.inqy.getAIB().getIOPCB().getStatusCode();
	}
	
	/**
	 * Gets the IMS status code of the INQY call from the IOPCB
	 * as a character string.
	 * 
	 * @return the IMS status code of the INQY DLI call.
	 */
	public String getStatusCodeChars() {
		return this.inqy.getAIB().getIOPCB().getStatusCodeChars();
	}
}
