package com.ibm.ims.dli;

import java.nio.ByteBuffer;

/**
 * The program communication block (PCB) interface is used to generate a SSAList
 * based on the PCB's view of an IMS database and issue DL/I calls to IMS to
 * retrieve, insert, update, and delete database information. The
 * <code>PCB</code> is essentially a cursor position in an IMS database. To
 * obtain a PCB handle, a <code>PSB</code> must first be created to connect to
 * the database. For a code example of how to create the <code>PSB</code> and
 * obtain the PCB handle, see {@link PSB}.
 * <p>
 * The PCB interface also returns the Application Interface Block (AIB)
 * associated with the most recent DL/I call.
 * 
 * @see AIB
 * @see PSB
 * @see SSAList
 */
public interface PCB {

	/**
	 * Returns the name of the PCB as defined by IMS
	 * 
	 * @return the name of the PCB
	 */
	public String getIMSName();

	/**
	 * Returns the alias/external name of the PCB
	 * 
	 * @return the alias/external name of the PCB
	 */
	public String getName();

	/**
	 * Constant indicating the requested catalog resource is a PSB resource
	 * 
	 * @see #getCatalogMetaDataAsXML(String, byte[])
	 */
	public static final byte[] PSB_RESOURCE = { (byte) 0xD7, (byte) 0xE2, (byte) 0xC2, (byte) 0x40, (byte) 0x40,
			(byte) 0x40, (byte) 0x40, (byte) 0x40 };

	/**
	 * Constant indicating the requested catalog resource is a DBD resource
	 * 
	 * @see #getCatalogMetaDataAsXML(String, byte[])
	 */
	public static final byte[] DBD_RESOURCE = { (byte) 0xC4, (byte) 0xC2, (byte) 0xC4, (byte) 0x40, (byte) 0x40,
			(byte) 0x40, (byte) 0x40, (byte) 0x40 };

	/************************************************/
	/* Traditional DL/I API (status code in Record) */
	/************************************************/

	// TODO - we need method to return Path because we have no idea what type of
	// Path
	// will be returned in the event of no SSAList...
	// If they don't pass in SSAList then force them to check for null on the
	// return type
	// and if null they can get the AIB from the PCB to find out what happened
	// No more Path.next

	// public Path getUnique(SSAList ssaList, boolean isHoldCall) throws
	// DLIException;
	/**
	 * Issue a GU (Get Unique) database call. If isHoldCall is <code>true</code>
	 * , a GHU (Get Hold Unique) database call is issued.
	 * 
	 * <p>
	 * The following code fragment illustrates how to use <code>getUnique</code>
	 * to retrieve patient records where the hospital name is "SANTA TERESA",
	 * the ward number is "44", and the number of doctors is greater than 2.
	 * <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification("WARD", "WARDNO, SSAList.EQUALS, "44");
	 * ssaList.appendQualification("WARD",SSAList.AND,"DOCCOUNT","SSAList.GREATER_THAN, "2");
	 * ssaList.addInitialQualification("HOSPITAL","HOSPNAME",SSAList.EQUALS,"SANTA TERESA");
	 * ssaList.markAllFieldsForRetrieval("PATIENT", true);
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, true);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param path
	 *            the <code>Path</code> to be used in the database call. It will
	 *            contain the data returned from the call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return a <code>boolean</code> indicating whether data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public boolean getUnique(Path path, SSAList ssaList, boolean isHoldCall) throws DLIException;
	// public boolean getHoldUnique(Path, SSAList);

	// public Path getNext(SSAList ssaList, boolean isHoldCall) throws
	// DLIException;
	/**
	 * Issues a GN (Get Next) database call. If isHoldCall is <code>true</code>,
	 * a GHN (Get Hold Next) database call is issued.
	 * <p>
	 * This method is used when the type of segment returned from the call is
	 * not known until the call is successful.
	 * 
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return the <code>Path</code> containing the results of the database
	 *         call, or <code>null</code> if no data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public Path getNext(boolean isHoldCall) throws DLIException;

	/**
	 * Issues a GN (Get Next) database call. Use this method when you don't have
	 * an <code>SSAList</code>. The method returns a <code>Path</code> to
	 * whatever segment the cursor is currently position on. If isHoldCall is
	 * <code>true</code>, a GHN (Get Hold Next) database call is issued.
	 * 
	 * @param path
	 *            the <code>Path</code> to be used in the database call. It will
	 *            contain the data returned from the call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return a <code>boolean</code> indicating whether data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public boolean getNext(Path path, SSAList ssaList, boolean isHoldCall) throws DLIException;
	// public boolean getHoldNext(Path, SSAList);

	// public Path getNextWithinParent(SSAList ssaList, boolean isHoldCall)
	// throws DLIException;
	/**
	 * Issues a GNP (Get Next Within Parent) database call. If isHoldCall is
	 * <code>true</code>, a GHNP (Get Hold Next Within Parent) database call
	 * will be issued.
	 * <p>
	 * This method is used when the type of segment returned from the call is
	 * not known until the call is successful.
	 * 
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return the <code>Path</code> containing the results of the database
	 *         call, or <code>null</code> if no data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public Path getNextWithinParent(boolean isHoldCall) throws DLIException;

	/**
	 * Issue a GNP (Get Next Within Parent) database call. Use this method when
	 * you don't have an <code>SSAList</code>. The method returns a
	 * <code>Path</code> to whatever segment the cursor is currently position
	 * on. If isHoldCall is <code>true</code>, a GHNP (Get Hold Next Within
	 * Parent) database call will be issued.
	 * 
	 * @param path
	 *            the <code>Path</code> to be used in the database call. It will
	 *            contain the data returned from the call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return a <code>boolean</code> indicating whether data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public boolean getNextWithinParent(Path path, SSAList ssaList, boolean isHoldCall) throws DLIException;

	/*
	 * public boolean insert(Path path, SSAList ssaList) throws DLIException;
	 * public boolean insert(Path path) throws DLIException; // Used for
	 * inserting at root public boolean replace(Path path, SSAList ssaList)
	 * throws DLIException; // SSAList only for command code N usage - TODO -
	 * check for format public boolean replace(Path path) throws DLIException;
	 * public boolean delete(SSAList ssaList) throws DLIException; // SSAList
	 * optional - TODO - check for format
	 */

	// Insert will provide defaults if any available for fields that weren't set

	/**
	 * Issues a DL/I ISRT (Insert) call. An exception is thrown if a key field
	 * is not set.
	 * <p>
	 * The following code fragment illustrates how to use <code>insert</code> to
	 * add a new PATIENT and ILLNESS segment in the database where the hospital
	 * name is "SANTA TERESA" and the ward number is "WARDNO2". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.addInitialQualification("WARD", "WARDNO", SSAList.EQUALS, "WARDNO2");
	 * Path path = ssaList.getPathForInsert("HOSPITAL");
	 * path.setString("PATIENT", "PATNUM", "PATIENTNO7");
	 * path.setString("PATIENT", "PATNAME", "ROCKY BALBOA");
	 * path.setString("ILLNAME", "APPENDICITIS");
	 * pcb.insert(path, ssaList);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param path
	 *            the <code>Path</code> to be used in the database call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call. The
	 *            segment search arguments in this <code>SSAList</code> must be
	 *            unqualified, otherwise an exception is thrown.
	 * @return the status code returned by IMS
	 * @throws DLIException
	 */
	public short insert(Path path, SSAList ssaList) throws DLIException;

	/**
	 * Issues a DL/I REPL (Replace) call. The application must obtain a
	 * <code>SSAList</code> and perform a Hold operation before issuing the
	 * replace call. The Hold operation can be a <code>getUnique</code>,
	 * <code>getNext</code>, or <code>getNextWithinParent</code> method call.
	 * 
	 * <p>
	 * The following code fragment shows how to use the <code>replace</code>
	 * method to update a patient's name to "JANET BOENINGER" in patient records
	 * where the hospital name is "ALEXANDRIA". In the example, the N command
	 * code is used to omit the HOSPITAL and WARD segments in the
	 * <code>Path</code> instance from being updated. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification(1,"HOSPNAME",SSAList.EQUALS,"ALEXANDRIA");
	 *
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, true);
	 * ssaList = pcb.getSSAList("HOSPITAL", "PATIENT");
	 * ssaList.addCommandCode("HOSPITAL", SSAList.CC_N); // exclude the HOSPITAL segment from updates
	 * ssaList.addCommandCode("WARD", SSAList.CC_N); // exclude the WARD segment from updates
	 * 
	 * path.setString("HOSPITAL", "HOSPNAME", "SANTA CRUZ MED"); // update is not performed
	 * path.setString("WARD", "WARDNAME", "COSMETIC");           // update is not performed
	 * path.setString("PATIENT", "PATNAME", "JANET BOENINGER");  // update is performed
	 * pcb.replace(path, ssaList)
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param path
	 *            the <code>Path</code> to be used in the database call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @return the status code returned by IMS
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	// public short insert(Path path) throws DLIException; // Used for inserting
	// at root
	public short replace(Path path, SSAList ssaList) throws DLIException; // SSAList
																			// only
																			// for
																			// command
																			// code
																			// N
																			// usage
																			// -
																			// TODO
																			// -
																			// check
																			// for
																			// format

	/**
	 * Issues a DL/I REPL (Replace) call. The application must obtain a
	 * <code>SSAList</code> and perform a Hold operation before issuing the
	 * replace call. The Hold operation can be a <code>getUnique</code>,
	 * <code>getNext</code>, or <code>getNextWithinParent</code> method call.
	 * <p>
	 * The following code fragment illustrates how to use replace to update a
	 * patient's name in patient records where the patient name is
	 * "ANDREA SMITH", the ward name is "SURGICAL", and the hospital name is
	 * "ALEXANDRIA". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification("HOSPITAL","HOSPNAME",SSAList.EQUALS,"ALEXANDRIA");
	 * ssaList.addInitialQualification("WARD","WARDNAME",SSAList.EQUALS,"SURGICAL");
	 * ssaList.addInitialQualification("PATIENT","PATNAME",SSAList.EQUALS,"ANDREA SMITH");
	 * 
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * if (pcb.getUnique(path, ssaList, true)) {
	 *    path.setString("PATNAME", "ANDREA TAYLOR");
	 *    pcb.replace(path);
	 * }   
	 * while (pcb.getNext(path, ssaList, true) {
	 *    path.setString("PATNAME", "ANDREA TAYLOR");
	 *    pcb.replace(path);
	 * }
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param path
	 *            the <code>Path</code> to be used in the database call.
	 * @return the status code returned by IMS
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public short replace(Path path) throws DLIException;

	/**
	 * Performs a DL/I Delete (DLET) operation to remove existing segments in
	 * the database. The application must obtain a <code>SSAList</code> and
	 * perform a Hold operation before issuing the delete call. The Hold
	 * operation can be a <code>getUnique</code>, <code>getNext</code>, or
	 * <code>getNextWithinParent</code> method call. The delete call must be
	 * provided with an unqualified <code>SSAList</code> as an argument. The
	 * delete call removes the first segment that is specified by the
	 * unqualified <code>SSAList</code> and its child segments if any. An
	 * exception is thrown if a qualified SSAList is provided as an argument.
	 * 
	 * <p>
	 * The following code fragment illustrates how to use the
	 * <code>delete</code> method to remove all ILLNESS segments and its
	 * dependent segments (TREATMNT, DOCTOR) where the patient name is
	 * "ANDREA SMITH", the ward name is "SURGICAL", the hospital name is
	 * "ALEXANDRIA", and the patient number is "PatientNo7". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "ALEXANDRIA");
	 * ssaList.addInitialQualification("WARD", "WARDNAME", SSAList.EQUALS, "SURGICAL");
	 * ssaList.addInitialQualification("PATIENT", "PATNAME", SSAList.EQUALS, "ANDREA SMITH");
	 * ssaList.markAllFieldsForRetrieval("PATIENT", true);
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * SSAList illnessSSAList = pcb.getSSAList("ILLNESS");
	 * if (pcb.getUnique(path, ssaList, true)) {
	 * 	if (path.getString("PATIENT", "PATNUM").equals("PATIENTNO7")) {
	 * 		pcb.delete(illnessSSAList);
	 * 	}
	 * }
	 * while (pcb.getNext(path, ssaList, true)) {
	 * 	if (path.getString("PATIENT", "PATNUM").equals("PATIENTNO7")) {
	 * 		pcb.delete(illnessSSAList);
	 * 	}
	 * }
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @return the status code returned by IMS
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public short delete(SSAList ssaList) throws DLIException; // SSAList
																// optional -
																// TODO - check
																// for format

	/**
	 * Performs a DL/I Delete (DLET) operation to remove existing segments in
	 * the database. The application must obtain a <code>SSAList</code> and
	 * perform a Hold operation before issuing the delete call. The Hold
	 * operation can be a <code>getUnique</code>, <code>getNext</code>, or
	 * <code>getNextWithinParent</code> method call. The delete call removes the
	 * first segment marked for retrieval and its child segments if any. If no
	 * segments are explicitly marked for retrieval, the delete call removes the
	 * lowest level segment specified by the <code>SSAList</code> and its child
	 * segments.
	 * 
	 * <p>
	 * The following code fragment illustrates how to use <code>delete</code> to
	 * selectively remove all PATIENT segments and its dependent segments
	 * (ILLNESS, TREATMNT, DOCTOR, BILLING) where the patient name is
	 * "ANDREA SMITH", the ward name is "SURGICAL", the hospital name is
	 * "ALEXANDRIA", and the patient number is "PatientNo7". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "ALEXANDRIA");
	 * ssaList.addInitialQualification("WARD", "WARDNAME", SSAList.EQUALS, "SURGICAL");
	 * ssaList.addInitialQualification("PATIENT", "PATNAME", SSAList.EQUALS, "ANDREA SMITH");
	 * ssaList.addCommandCode("PATIENT", SSAList.CC_D);
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * if (pcb.getUnique(path, ssaList, true)) {
	 * 	if (path.getString("PATIENT", "PATNUM").equals("PATIENTNO7")) {
	 * 		pcb.delete();
	 * 	}
	 * }
	 * while (pcb.getNext(path, ssaList, true)) {
	 * 	if (path.getString("PATIENT", "PATNUM").equals("PATIENTNO7")) {
	 * 		pcb.delete();
	 * 	}
	 * }
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @return the status code returned by IMS
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public short delete() throws DLIException;

	/***********************************************/
	/* Basic CRUD database operations */
	/***********************************************/

	// Create will provide defaults if any available for fields that weren't set
	/**
	 * Performs a DL/I insert operation. An exception is thrown if a key field
	 * is not set.
	 * 
	 * <p>
	 * The following code fragment illustrates how to use <code>create</code> to
	 * add a new PATIENT and ILLNESS segment in the database where the hospital
	 * name is "SANTA TERESA" and the ward number is "WARDNO2". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.addInitialQualification("WARD", "WARDNO", SSAList.EQUALS, "WARDNO2");
	 * Path path = ssaList.getPathForInsert("HOSPITAL");
	 * path.setString("PATIENT", "PATNUM", "PATIENTNO7");
	 * path.setString("PATIENT", "PATNAME", "ROCKY BALBOA");
	 * path.setString("ILLNAME", "APPENDICITIS");
	 * pcb.create(path, ssaList);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param path
	 *            the <code>Path</code> to be used in the database call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call. The
	 *            segment search arguments in this <code>SSAList</code> must be
	 *            unqualified, otherwise an exception is thrown.
	 * @return the number of rows inserted. For now, this will always return 1.
	 * @throws DLIException
	 */
	public int create(Path path, SSAList ssaList) throws DLIException; // DLIBatchException?

	/**
	 * Returns a <code>PathSet</code> containing all records that satisfy a
	 * given <code>SSAList</code> criteria. In a batch retrieve operation, the
	 * host will do all of the GHU/GHN processing and will deliver results back to
	 * the client in a single batch network operation. Note that the fetch size
	 * determines how much data is sent back on each batch network operation.
	 * <p>
	 * The following code fragment illustrates how to use
	 * <code>batchRetrieve</code> to retrieve all PATIENT segments in the
	 * Hospital database. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("PATIENT");
	 * PathSet pathSet = pcb.batchRetrieve(ssaList);
	 * while (pathSet.hasNext()) {
	 * 	Path path = pathSet.next();
	 * 	String fieldValue = path.getString("PATIENT", "PATNAME");
	 * }
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @see PathSet
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @return the <code>PathSet</code> containing the results of the database
	 *         call, or <code>null</code> if no data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public PathSet batchRetrieve(SSAList ssaList) throws DLIException;

	/**
	 * Returns a <code>PathSet</code> containing all records that satisfy a
	 * given <code>SSAList</code> criteria. In a batch retrieve operation, the
	 * host will do all of the GU/GN (or GHU/GHN) processing and will deliver 
	 * results back to the client in a single batch network operation. Note 
	 * that the fetch size determines how much data is sent back on each batch
	 * network operation.
	 * <p>
	 * The following code fragment illustrates how to use
	 * <code>batchRetrieve</code> to retrieve all PATIENT segments in the
	 * Hospital database. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("PATIENT");
	 * PathSet pathSet = pcb.batchRetrieve(ssaList, true);
	 * while (pathSet.hasNext()) {
	 * 	Path path = pathSet.next();
	 * 	String fieldValue = path.getString("PATIENT", "PATNAME");
	 * }
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @see PathSet
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @param isHoldCall
	 * 			  if true, GHU/GHN processing is used. if false, GU/GN processing is used.
	 * @return the <code>PathSet</code> containing the results of the database
	 *         call, or <code>null</code> if no data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public PathSet batchRetrieve(SSAList ssaList, boolean isHoldCall) throws DLIException;

	/**
	 * Performs a batch update operation in a single network call on multiple
	 * records matching a given <code>SSAList</code> criteria. In a batch update
	 * operation, the host will do a GU/GN loop and inside the loop perform the
	 * updates until there are no more segments to update and then return the
	 * number of records updated.
	 * 
	 * 
	 * <p>
	 * The following code fragment illustrates how to use
	 * <code>batchUpdate</code> to modify a patient's name. The
	 * <code>SSAList</code> is set to update only records where the patient name
	 * is "ANDREA SMITH", the ward name is "SURGICAL", and the hospital name is
	 * "ALEXANDRIA". The <code>SSAList.getPathForBatchUpdate</code> method is
	 * called to obtain a <code>Path</code> contaning the PATIENT segment and
	 * its child segments. Finally, <code>batchUpdate</code> is called to change
	 * the value of the patient name field to "ANDREA TAYLOR".
	 * 
	 * <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "PATIENT");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "ALEXANDRIA");
	 * ssaList.addInitialQualification("WARD", "WARDNAME", SSAList.EQUALS, "SURGICAL");
	 * ssaList.addInitialQualification("PATIENT", "PATNAME", SSAList.EQUALS, "ANDREA SMITH");
	 * Path path = ssaList.getPathForBatchUpdate("PATIENT");
	 * path.setString("PATNAME", "ANDREA TAYLOR");
	 * pcb.batchUpdate(path, ssaList);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @see SSAList#getPathForBatchUpdate(String)
	 * @param path
	 *            the <code>Path</code> to be used in the database call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @return the number of rows updated
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public int batchUpdate(Path path, SSAList ssaList) throws DLIException;

	/**
	 * Performs a batch delete operation in a single network call on multiple
	 * records matching a given <code>SSAList</code> criteria. In a batch delete
	 * operation, the host will do a GU/GN loop and inside the loop delete each
	 * record until there are no more segments matching the <code>SSAList</code>
	 * and return the number of records deleted.
	 * <p>
	 * The following code fragment illustrates how to use
	 * <code>batchDelete</code> to remove a patient's records. The
	 * <code>SSAList</code> is set to restrict the deletion operation to remove
	 * only records where the patient name is "ANDREA SMITH", the ward name is
	 * "SURGICAL", and the hospital name is "ALEXANDRIA". The records are
	 * deleted by calling <code>batchDelete</code>. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "PATIENT");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "ALEXANDRIA");
	 * ssaList.addInitialQualification("WARD", "WARDNAME", SSAList.EQUALS, "SURGICAL");
	 * ssaList.addInitialQualification("PATIENT", "PATNAME", SSAList.EQUALS, "ANDREA SMITH");
	 * pcb.batchDelete(ssaList);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @return the number of rows deleted
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public int batchDelete(SSAList ssaList) throws DLIException;

	/**
	 * This method returns an <code>SSAList</code> that consists of unqualified
	 * SSAs from the top level segment down to the bottom level segment
	 * specified.
	 * 
	 * @param topLevelSegment
	 *            the top level segment to be included in the
	 *            <code>SSAList</code>
	 * @param bottomLevelSegment
	 *            the bottom level segment to be included in the
	 *            <code>SSAList</code>
	 * @return the <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public SSAList getSSAList(String topLevelSegment, String bottomLevelSegment) throws DLIException;

	// Create a an unqualified SSA for the specified segment
	/**
	 * This method returns an <code>SSAList</code> with a single unqualified SSA
	 * for the target segment.
	 * 
	 * @param segment
	 *            the name of the segment that the <code>SSAList</code> will
	 *            represent
	 * @return the <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public SSAList getSSAList(String segment) throws DLIException;

	/**
	 * Gives the client a hint as to the number of rows that should be fetched
	 * from the database when more records are needed. The number of rows (
	 * <code>Path</code> objects) specified affects only data returned using
	 * this <code>PCB</code>. If the value specified is zero or negative, the
	 * hint is ignored.
	 * 
	 * @param fetchSize
	 *            the number of rows to fetch
	 * @see PCB#batchRetrieve(SSAList)
	 */
	public void setFetchSize(int fetchSize);

	// Properties is used on retrieve call...TODO - should not be in
	// interface...only PCBImpl class?
	// public void setProperties(Properties properties); // Properties can
	// contain things like fetch size
	// public void setProperty(String property, String value); // Properties can
	// contain things like fetch size
	// public Properties getProperties(); // Properties can contain things like
	// fetch size
	// public String getProperty(String property); // Properties can contain
	// things like fetch size

	/**
	 * Returns the Application Interface Block associated with the most recent
	 * DL/I call
	 * 
	 * @return the <code>AIB</code>
	 */
	public AIB getAIB();

	/**
	 * Closes this PCB and any resources associated with it.
	 * 
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void close() throws DLIException;

	/**
	 * This method returns a byte array containing the requested catalog
	 * resource as an XML document.
	 * 
	 * @param resourceName
	 *            the name of the PSB or DBD resource in the catalog
	 * @param resourceType
	 *            the type of resource (PCB.PSB_RESOURCE or PCB.DBD_RESOURCE)
	 * @return the resource metadata in XML
	 * @throws DLIException
	 *             if the resource was not found in the catalog or an error
	 *             occurs during processing
	 * @see #PSB_RESOURCE
	 * @see #DBD_RESOURCE
	 */
	public byte[] getCatalogMetaDataAsXML(String resourceName, byte[] resourceType) throws DLIException;

	/**
	 * This method returns a byte array containing the requested catalog
	 * resource as an XML document.
	 * 
	 * <p>
	 * The following code fragment illustrates how to retrieve the timestamp
	 * (TSVERS) value from the IMS Catalog. <blockquote>
	 * 
	 * <pre>
	 * PCB pcb = psb.getPCB("DFSCAT00");
	 * SSAList ssaList = pcb.getSSAList("HEADER", "DBD");
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, false);
	 * String timestamp = path.getString("TSVERS");
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param resourceName
	 *            the name of the PSB or DBD resource in the catalog
	 * @param resourceType
	 *            the type of resource (PCB.PSB_RESOURCE or PCB.DBD_RESOURCE)
	 * @param timestamp
	 *            the TSVERS version for the resource following the pattern
	 *            yyDDDHHmmssff
	 * @return the resource metadata in XML
	 * @throws DLIException
	 *             if the resource was not found in the catalog or an error
	 *             occurs during processing
	 * @see #PSB_RESOURCE
	 * @see #DBD_RESOURCE
	 */
	public byte[] getCatalogMetaDataAsXML(String resourceName, byte[] resourceType, String timestamp)
			throws DLIException;

	/**********************
	 * START BYTEBUFFER WORK
	 ************************/
	
	/**
	 * Issues a GU (Get Unique) database call. If isHoldCall is <code>true</code>
	 * , a GHU (Get Hold Unique) database call is issued.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * <p>
	 * The following code fragment illustrates how to use <code>getUnique</code>
	 * to retrieve patient records where the hospital name is "SANTA TERESA",
	 * the ward number is "44", and the number of doctors is greater than 2.
	 * <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification("WARD", "WARDNO, SSAList.EQUALS, "44");
	 * ssaList.appendQualification("WARD",SSAList.AND,"DOCCOUNT","SSAList.GREATER_THAN, "2");
	 * ssaList.addInitialQualification("HOSPITAL","HOSPNAME",SSAList.EQUALS,"SANTA TERESA");
	 * ssaList.markAllFieldsForRetrieval("PATIENT", true);
	 * ByteBuffer buffer = ByteBuffer.allocate(40);
	 * pcb.getUnique(buffer, ssaList, true);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data returned from the call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return a <code>boolean</code> indicating whether data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public boolean getUnique(ByteBuffer buffer, SSAList ssaList, boolean isHoldCall) throws DLIException;
	
	/**
	 * Issues a GU (Get Unique) database call. If isHoldCall is <code>true</code>,
	 * a GHU (Get Hold Unique) database call is issued.
	 * <p>
	 * This method is used when the type of segment returned from the call is
	 * not known until the call is successful.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * <p>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data returned from the call.
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return the <code>Path</code> containing the results of the database
	 *         call, or <code>null</code> if no data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public boolean getUnique(ByteBuffer buffer, boolean isHoldCall) throws DLIException;

	/**
	 * Issues a GU (Get Unique) database call. If isHoldCall is <code>true</code>,
	 * a GHU (Get Hold Unique) database call is issued.
	 * <p>
	 * This method is used when the type of segment returned from the call is
	 * not known until the call is successful.
	 * 
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return the <code>Path</code> containing the results of the database
	 *         call, or <code>null</code> if no data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public Path getUnique(boolean isHoldCall) throws DLIException;

	/**
	 * Issues a GN (Get Next) database call. If isHoldCall is <code>true</code>
	 * , a GHN (Get Hold Next) database call is issued.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * <p>
	 * The following code fragment illustrates how to use <code>getNext</code>
	 * to retrieve patient records where the hospital name is "SANTA TERESA",
	 * the ward number is "44", and the number of doctors is greater than 2.
	 * <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification("WARD", "WARDNO, SSAList.EQUALS, "44");
	 * ssaList.appendQualification("WARD",SSAList.AND,"DOCCOUNT","SSAList.GREATER_THAN, "2");
	 * ssaList.addInitialQualification("HOSPITAL","HOSPNAME",SSAList.EQUALS,"SANTA TERESA");
	 * ssaList.markAllFieldsForRetrieval("PATIENT", true);
	 * ByteBuffer buffer = ByteBuffer.allocate(40);
	 * pcb.getNext(buffer, ssaList, true);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data returned from the call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return a <code>boolean</code> indicating whether data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public boolean getNext(ByteBuffer buffer, SSAList ssaList, boolean isHoldCall) throws DLIException;
	
	/**
	 * Issues a GN (Get Next) database call. If isHoldCall is <code>true</code>,
	 * a GHN (Get Hold Next) database call is issued.
	 * <p>
	 * This method is used when the type of segment returned from the call is
	 * not known until the call is successful.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * <p>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data returned from the call.
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return the <code>Path</code> containing the results of the database
	 *         call, or <code>null</code> if no data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public boolean getNext(ByteBuffer buffer, boolean isHoldCall) throws DLIException;
	
	/**
	 * Issues a GNP (Get Next Parent) database call. If isHoldCall is <code>true</code>
	 * , a GHNP (Get Hold Next Parent) database call is issued.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * <p>
	 * The following code fragment illustrates how to use <code>getNextWithinParrent</code>
	 * to retrieve patient records where the hospital name is "SANTA TERESA",
	 * the ward number is "44", and the number of doctors is greater than 2.
	 * <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification("WARD", "WARDNO, SSAList.EQUALS, "44");
	 * ssaList.appendQualification("WARD",SSAList.AND,"DOCCOUNT","SSAList.GREATER_THAN, "2");
	 * ssaList.addInitialQualification("HOSPITAL","HOSPNAME",SSAList.EQUALS,"SANTA TERESA");
	 * ssaList.markAllFieldsForRetrieval("PATIENT", true);
	 * ByteBuffer buffer = ByteBuffer.allocate(40);
	 * pcb.getNextWithinParent(buffer, ssaList, true);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data returned from the call.
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return a <code>boolean</code> indicating whether data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public boolean getNextWithinParent(ByteBuffer buffer, SSAList ssaList, boolean isHoldCall) throws DLIException;
	
	/**
	 * Issues a GNP (Get Next Parent) database call. If isHoldCall is <code>true</code>,
	 * a GHN (Get Hold Next Parent) database call is issued.
	 * <p>
	 * This method is used when the type of segment returned from the call is
	 * not known until the call is successful.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the output object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * <p>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data returned from the call.
	 * @param isHoldCall
	 *            a <code>boolean</code> indicating whether a hold call is
	 *            required
	 * @return the <code>Path</code> containing the results of the database
	 *         call, or <code>null</code> if no data was returned
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public boolean getNextWithinParent(ByteBuffer buffer,  boolean isHoldCall) throws DLIException;
	
	/**
	 * Issues a DL/I ISRT (Insert) call. An exception is thrown if a key field
	 * is not set.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the input object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * 
	 * <p>
	 * The following code fragment illustrates how to use <code>insert</code> to
	 * add a new PATIENT and ILLNESS segment in the database where the hospital
	 * name is "SANTA TERESA" and the ward number is "WARDNO2". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.addInitialQualification("WARD", "WARDNO", SSAList.EQUALS, "WARDNO2");
	 * ByteBuffer buffer = ByteBuffer.allocate(900);
	 * buffer.put("PATIENTN07".getBytes("Cp1047")
	 * buffer.put("ROCKY BALBOA".getBytes("Cp1047")
	 * buffer.put("APPENDICITIS".getBytes("Cp1047")
	 * pcb.insert(buffer, ssaList);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data to send to IMS
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call. The
	 *            segment search arguments in this <code>SSAList</code> must be
	 *            unqualified, otherwise an exception is thrown.
	 * @return the status code returned by IMS
	 * @throws DLIException
	 */
	public short insert(ByteBuffer buffer, SSAList ssaList) throws DLIException;
	
	/**
	 * Issues a DL/I ISRT (Insert) call. An exception is thrown if a key field
	 * is not set.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the input object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * 
	 * <p>
	 * The following code fragment illustrates how to use <code>insert</code> to
	 * add a new PATIENT and ILLNESS segment in the database where the hospital
	 * name is "SANTA TERESA" and the ward number is "WARDNO2". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.addInitialQualification("WARD", "WARDNO", SSAList.EQUALS, "WARDNO2");
	 * ByteBuffer buffer = ByteBuffer.allocate(900);
	 * buffer.put("PATIENTN07".getBytes("Cp1047")
	 * buffer.put("ROCKY BALBOA".getBytes("Cp1047")
	 * buffer.put("APPENDICITIS".getBytes("Cp1047")
	 * pcb.insert(buffer, ssaList, 50);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data to send to IMS
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call. The
	 *            segment search arguments in this <code>SSAList</code> must be
	 *            unqualified, otherwise an exception is thrown.
	 * @param ioAreaUsed
	 *            the <code>int</code> to be used to specify how much of the ByteBuffer's 
	 *            capacity should be processed for insert
	 * @return the status code returned by IMS
	 * @throws DLIException
	 */
	public short insert(ByteBuffer buffer, SSAList ssaList, int ioAreaUsed) throws DLIException;
	
	/**
	 * Performs a DL/I insert operation. An exception is thrown if a key field
	 * is not set.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the input object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * 
	 * <p>
	 * The following code fragment illustrates how to use <code>insert</code> to
	 * add a new PATIENT and ILLNESS segment in the database where the hospital
	 * name is "SANTA TERESA" and the ward number is "WARDNO2". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.addInitialQualification("WARD", "WARDNO", SSAList.EQUALS, "WARDNO2");
	 * ByteBuffer buffer = ByteBuffer.allocate(900);
	 * buffer.put("PATIENTN07".getBytes("Cp1047")
	 * buffer.put("ROCKY BALBOA".getBytes("Cp1047")
	 * buffer.put("APPENDICITIS".getBytes("Cp1047")
	 * pcb.create(buffer, ssaList);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data to send to IMS
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call. The
	 *            segment search arguments in this <code>SSAList</code> must be
	 *            unqualified, otherwise an exception is thrown.
	 * @return the status code returned by IMS
	 * @throws DLIException
	 */
	public int create(ByteBuffer buffer, SSAList ssaList) throws DLIException;
	
	/**
	 * Performs a DL/I insert operation. An exception is thrown if a key field
	 * is not set.
	 * 
	 * <p>Uses a <code>ByteBuffer</code> as the input object. The
	 * <code>ByteBuffer</code> object can be either direct or non-direct. If running
	 * in a 64-bit JVM and require the use of a Direct <code>ByteBuffer</code>, a
	 * 31-bit <code>ByteBuffer</code> will need to be allocated. See
	 * {@link Application#get31BitDirectByteBuffer(Int)}
	 * 
	 * <p>
	 * The following code fragment illustrates how to use <code>insert</code> to
	 * add a new PATIENT and ILLNESS segment in the database where the hospital
	 * name is "SANTA TERESA" and the ward number is "WARDNO2". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.addInitialQualification("WARD", "WARDNO", SSAList.EQUALS, "WARDNO2");
	 * ByteBuffer buffer = ByteBuffer.allocate(900);
	 * buffer.put("PATIENTN07".getBytes("Cp1047")
	 * buffer.put("ROCKY BALBOA".getBytes("Cp1047")
	 * buffer.put("APPENDICITIS".getBytes("Cp1047")
	 * pcb.create(buffer, ssaList, 50);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data to send to IMS
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call. The
	 *            segment search arguments in this <code>SSAList</code> must be
	 *            unqualified, otherwise an exception is thrown.
	 * @param ioAreaUsed
	 *            the <code>int</code> to be used to specify how much of the ByteBuffer's 
	 *            capacity should be processed for insert
	 * @return the status code returned by IMS
	 * @throws DLIException
	 */
	public int create(ByteBuffer buffer, SSAList ssaList, int ioAreaUsed) throws DLIException;

	/**
	 * Issues a DL/I REPL (Replace) call. The application must obtain a
	 * <code>SSAList</code> and perform a Hold operation before issuing the
	 * replace call. The Hold operation can be a <code>getUnique</code>,
	 * <code>getNext</code>, or <code>getNextWithinParent</code> method call.
	 * 
	 * <p>
	 * The following code fragment shows how to use the <code>replace</code>
	 * method to update a patient's name to "JANET BOENINGER" in patient records
	 * where the hospital name is "ALEXANDRIA". In the example, the N command
	 * code is used to omit the HOSPITAL and WARD segments in the
	 * <code>Path</code> instance from being updated. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification(1,"HOSPNAME",SSAList.EQUALS,"ALEXANDRIA");
	 *
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, true);
	 * ssaList = pcb.getSSAList("HOSPITAL", "PATIENT");
	 * ssaList.addCommandCode("HOSPITAL", SSAList.CC_N); // exclude the HOSPITAL segment from updates
	 * ssaList.addCommandCode("WARD", SSAList.CC_N); // exclude the WARD segment from updates
	 * 
	 * ByteBuffer buffer = ByteBuffer.allocate(2700);
	 * buffer.put("SANTA CRUZ MED".getBytes(Cp1047)); // update is not performed (part of segment Hospital)
	 * buffer.position(900);
     * buffer.put("COSMETIC".getBytes(Cp1047));       // update is not performed (part of segment WARD)
     * buffer.postion(1800);
	 * buffer.put("JANET BOENINGER".getBytes(Cp1047));// update is performed
	 * pcb.replace(buffer, ssaList);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data to send to IMS
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 * @return the status code returned by IMS
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public short replace(ByteBuffer buffer, SSAList ssaList) throws DLIException;

	/**
	 * Issues a DL/I REPL (Replace) call. The application must obtain a
	 * <code>SSAList</code> and perform a Hold operation before issuing the
	 * replace call. The Hold operation can be a <code>getUnique</code>,
	 * <code>getNext</code>, or <code>getNextWithinParent</code> method call.
	 * 
	 * <p>
	 * The following code fragment shows how to use the <code>replace</code>
	 * method to update a patient's name to "JANET BOENINGER" in patient records
	 * where the hospital name is "ALEXANDRIA". In the example, the N command
	 * code is used to omit the HOSPITAL and WARD segments in the
	 * <code>Path</code> instance from being updated. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification(1,"HOSPNAME",SSAList.EQUALS,"ALEXANDRIA");
	 *
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, true);
	 * ssaList = pcb.getSSAList("HOSPITAL", "PATIENT");
	 * ssaList.addCommandCode("HOSPITAL", SSAList.CC_N); // exclude the HOSPITAL segment from updates
	 * ssaList.addCommandCode("WARD", SSAList.CC_N); // exclude the WARD segment from updates
	 * 
	 * ByteBuffer buffer = ByteBuffer.allocate(5000);
	 * buffer.put("SANTA CRUZ MED".getBytes(Cp1047)); // update is not performed (part of segment Hospital)
	 * buffer.position(900);
     * buffer.put("COSMETIC".getBytes(Cp1047));       // update is not performed (part of segment WARD)
     * buffer.postion(1800);
	 * buffer.put("JANET BOENINGER".getBytes(Cp1047));// update is performed
	 * pcb.replace(buffer, ssaList, 2700);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data to send to IMS
	 * @param ssaList
	 *            the <code>SSAList</code> to be used in the database call
	 *            
	 * @param ioAreaUsed
	 *            the <code>int</code> to be used to specify how much of the ByteBuffer's 
	 *            capacity should be processed for insert
	 * @return the status code returned by IMS
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public short replace(ByteBuffer buffer, SSAList ssaList, int ioAreaUsed) throws DLIException;
	
	/**
	 * Issues a DL/I REPL (Replace) call. The application must obtain a
	 * <code>SSAList</code> and perform a Hold operation before issuing the
	 * replace call. The Hold operation can be a <code>getUnique</code>,
	 * <code>getNext</code>, or <code>getNextWithinParent</code> method call.
	 * <p>
	 * The following code fragment illustrates how to use replace to update a
	 * patient's name in patient records where the patient name is
	 * "ANDREA SMITH", the ward name is "SURGICAL", and the hospital name is
	 * "ALEXANDRIA". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification("HOSPITAL","HOSPNAME",SSAList.EQUALS,"ALEXANDRIA");
	 * ssaList.addInitialQualification("WARD","WARDNAME",SSAList.EQUALS,"SURGICAL");
	 * ssaList.addInitialQualification("PATIENT","PATNAME",SSAList.EQUALS,"ANDREA SMITH");
	 * 
	 * ByteBuffer buffer = ByteBuffer.allocate(900);
	 * if (pcb.getUnique(buffer, ssaList, true)) {
	 * 	  buffer.position(10);
	 *    buffer.put("ANDREA TAYLOR".getBytes(Cp1047));
	 *    pcb.replace(buffer);
	 * }   
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data to send to IMS
	 *            
	 * @return the status code returned by IMS
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public short replace(ByteBuffer buffer) throws DLIException;
	
	/**
	 * Issues a DL/I REPL (Replace) call. The application must obtain a
	 * <code>SSAList</code> and perform a Hold operation before issuing the
	 * replace call. The Hold operation can be a <code>getUnique</code>,
	 * <code>getNext</code>, or <code>getNextWithinParent</code> method call.
	 * <p>
	 * The following code fragment illustrates how to use replace to update a
	 * patient's name in patient records where the patient name is
	 * "ANDREA SMITH", the ward name is "SURGICAL", and the hospital name is
	 * "ALEXANDRIA". <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL","PATIENT");
	 * ssaList.addInitialQualification("HOSPITAL","HOSPNAME",SSAList.EQUALS,"ALEXANDRIA");
	 * ssaList.addInitialQualification("WARD","WARDNAME",SSAList.EQUALS,"SURGICAL");
	 * ssaList.addInitialQualification("PATIENT","PATNAME",SSAList.EQUALS,"ANDREA SMITH");
	 * 
	 * ByteBuffer buffer = ByteBuffer.allocate(900);
	 * if (pcb.getUnique(buffer, ssaList, true)) {
	 * 	  buffer.position(10);
	 *    buffer.put("ANDREA TAYLOR".getBytes(Cp1047));
	 *    pcb.replace(buffer, 50);
	 * }   
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param buffer
	 *            the <code>ByteBuffer</code> to be used in the database call. It will
	 *            contain the data to send to IMS
	 * @param ioAreaUsed
	 *            the <code>int</code> to be used to specify how much of the ByteBuffer's 
	 *            capacity should be processed for insert
	 * @return the status code returned by IMS
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public short replace(ByteBuffer buffer, int ioAreaUsed) throws DLIException;

	// AIB information (these are also on Record...but sort of need to be here
	// as well since
	// two calls don't take a Record as input...dlet and delete)
	/*
	 * public short getStatusCode(); // This needs to be returned from ODBM on
	 * every call for each IOArea public int getReturnCode(); // This will be in
	 * SQLCA public String getReturnCodeHex(); public int getReasonCode(); //
	 * This will be in SQLCA public String getReasonCodeHex(); public int
	 * getErrorCodeExtension(); // This will be in SQLCA //public String
	 * getErrorCodeExtensionHex(); public byte[] getKeyFeedbackArea(); //
	 * Perhaps a Properties option indicating return KFB for each call //
	 * Alternative is to make another call to ODBM to give KFB for a particular
	 * record // which means that ODBM must cache this info for a retrieve call
	 */
}
