package com.ibm.ims.dli;

/* (c) Copyright International Business Machines Corporation 2008. All rights reserved. */
import java.util.Collection;

/**
 * The <code>SSAList</code> interface represents a list of segment search
 * arguments used to specify the segments to target in a particular database
 * call.
 * 
 * Use the <code>SSAList</code> interface to construct each segment search
 * argument in the list, and to set the command codes and lock class for the
 * segment search arguments. The segments can be referenced by name or by using
 * the 1-based offset of the SSA representing that segment within the
 * <code>SSAList</code>.
 * 
 * <ul>
 * <li>For non-batch DL/I data retrieval or update operations, use
 * {@link SSAList#getPathForRetrieveReplace()}
 * <li>For a DL/I insert call, use {@link SSAList#getPathForInsert(String)}
 * <li>For a batch update operation, use
 * {@link SSAList#getPathForBatchUpdate(String)}
 * </ul>
 * 
 * <p>
 * In addition, your application can specify which segment fields are to be
 * returned from a database retrieve call by using the
 * <code>markFieldForRetrieval</code> or the
 * <code>markAllFieldsForRetrieval</code> methods.
 * 
 * <p>
 * For code examples of how to create and use <code>SSAList</code>, see:
 * {@link PCB#getUnique(Path, SSAList, boolean)},
 * {@link PCB#batchRetrieve(SSAList)}, {@link PCB#batchUpdate(Path, SSAList)},
 * {@link PCB#batchDelete(SSAList)}, {@link PCB#create(Path, SSAList)},
 * {@link PCB#replace(Path)}, {@link PCB#delete(SSAList)} ,
 * {@link SSAList#createPreparedValue(String)},
 * {@link PCB#insert(Path, SSAList)},
 * {@link SSAList#markFieldForRetrieval(int, String, boolean)},
 * {@link SSAList#markFieldForRetrieval(String, String, boolean)},
 * {@link SSAList#markAllFieldsForRetrieval(int, boolean)},
 * {@link SSAList#markAllFieldsForRetrieval(String, boolean)}.
 * 
 * @see Path
 * @see PCB
 * @see PreparedValue
 */
public interface SSAList {

	/**
	 * Constant indicating the relational operator for equals (=) in an SSA
	 * qualification statement
	 */
	public static final short EQUALS = (short) 0xC5D8; /* EQ */

	/**
	 * Constant indicating the relational operator for greater than or equal to
	 * (>=) in an SSA qualification statement
	 */
	public static final short GREATER_OR_EQUAL = (short) 0xC7C5; /* GE */

	/**
	 * Constant indicating the relational operator for less than or equal to
	 * (<=) in an SSA qualification statement
	 */
	public static final short LESS_OR_EQUAL = (short) 0xD3C5; /* LE */

	/**
	 * Constant indicating the relational operator for greater than (>) in an
	 * SSA qualification statement
	 */
	public static final short GREATER_THAN = (short) 0xC7E3; /* GT */

	/**
	 * Constant indicating the relational operator for less than (<) in an SSA
	 * qualification statement
	 */
	public static final short LESS_THAN = (short) 0xD3E3; /* LT */

	/**
	 * Constant indicating the relational operator for not equal to (!=) in an
	 * SSA qualification statement
	 */
	public static final short NOT_EQUAL = (short) 0xD5C5; /* NE */

	// Boolean Operator Constants

	/**
	 * Constant indicating the boolean operator <code>AND</code> in an SSA
	 * qualification statement
	 */
	public static final byte AND = (byte) 0x50; /* & */

	/**
	 * Constant indicating the boolean operator <code>OR</code> in an SSA
	 * qualification statement
	 */
	public static final byte OR = (byte) 0x4E; /* + */

	/**
	 * Constant indicating the boolean operator <code>INDEPENDENT AND</code> in
	 * an SSA qualification statement
	 */
	public static final byte INDEPENDENT_AND = (byte) 0x7B; /* # */

	// Command Code Constants

	/**
	 * Constant indicating the A command code (clear position)
	 */
	public static final byte CC_A = (byte) 0xC1; /* A */

	//
	// Constant indicating the C command code (concatenated key)
	//
	// public static final byte CC_C = (byte)0xC3; /*C*/

	/**
	 * Constant indicating the D command code (path call)
	 */
	public static final byte CC_D = (byte) 0xC4; /* D */

	/**
	 * Constant indicating the F command code (first occurrence)
	 */
	public static final byte CC_F = (byte) 0xC6; /* F */

	/**
	 * Constant indicating the G command code (no randomization)
	 */
	public static final byte CC_G = (byte) 0xC7; /* G */

	/**
	 * Constant indicating the L command code (last occurrence)
	 */
	public static final byte CC_L = (byte) 0xD3; /* L */

	/**
	 * Constant indicating the N command code (path call ignore)
	 */
	public static final byte CC_N = (byte) 0xD5; /* N */

	/**
	 * Constant indicating the O command code (set parentage)
	 */
	public static final byte CC_O = (byte) 0xD6; /* O */

	/**
	 * Constant indicating the P command code (set parentage)
	 */
	public static final byte CC_P = (byte) 0xD7; /* P */

	/**
	 * Constant indicating the U command code (maintain position at this level)
	 */
	public static final byte CC_U = (byte) 0xE4; /* U */

	/**
	 * Constant indicating the V command code (maintain position at this level
	 * and all superior levels)
	 */
	public static final byte CC_V = (byte) 0xE5; /* V */

	/**
	 * Constant indicating lock class A
	 */
	public static final byte LC_A = (byte) 0xC1; /* A */

	/**
	 * Constant indicating lock class B
	 */
	public static final byte LC_B = (byte) 0xC2; /* B */

	/**
	 * Constant indicating lock class C
	 */
	public static final byte LC_C = (byte) 0xC3; /* C */

	/**
	 * Constant indicating lock class D
	 */
	public static final byte LC_D = (byte) 0xC4; /* D */

	/**
	 * Constant indicating lock class E
	 */
	public static final byte LC_E = (byte) 0xC5; /* E */

	/**
	 * Constant indicating lock class F
	 */
	public static final byte LC_F = (byte) 0xC6; /* F */

	/**
	 * Constant indicating lock class G
	 */
	public static final byte LC_G = (byte) 0xC7; /* G */

	/**
	 * Constant indicating lock class H
	 */
	public static final byte LC_H = (byte) 0xC8; /* H */

	/**
	 * Constant indicating lock class I
	 */
	public static final byte LC_I = (byte) 0xC9; /* I */

	/**
	 * Constant indicating lock class J
	 */
	public static final byte LC_J = (byte) 0xD1; /* J */

	/**
	 * Returns a <code>Path</code> that contains all segments with fields marked
	 * for retrieval. If no fields are marked for retrieval and the
	 * <code>getPathForRetrieveReplace</code> method is called, the
	 * <code>Path</code> instance will contain the fields in the lowest level
	 * segment. The <code>getPathForRetrieveReplace</code> method can be used
	 * together with the data retrieval methods in the {@link PCB} interface or
	 * with the replace methods to update data.
	 * <p>
	 * For code examples of how to use this method, refer to the related
	 * <code>PCB</code> methods in the Javadoc.
	 * 
	 * @see PCB#getUnique(Path, SSAList, boolean)
	 * @see PCB#getNextWithinParent(Path, SSAList, boolean)
	 * @see PCB#getNext(Path, SSAList, boolean)
	 * @see PCB#replace(Path)
	 * @see PCB#replace(Path, SSAList)
	 * 
	 * @return The <code>Path</code> containing the database segments and member
	 *         fields.
	 * 
	 */
	public Path getPathForRetrieveReplace();

	/**
	 * Returns a <code>Path</code> that contains one or more database segments
	 * along a hierarchich path for an insert operation. The
	 * <code>getPathForInsert</code> method is used together with the insert
	 * method in the {@link PCB} interface to insert new segments in the
	 * database.
	 * <p>
	 * For a code example of how to use this method, refer to the related
	 * <code>PCB</code> method in the Javadoc.
	 * 
	 * @see PCB#insert(Path, SSAList)
	 * 
	 * @param segmentName
	 *            the name of a segment that currently exists in this
	 *            <code>SSAList</code>. The segment search arguments in this
	 *            <code>SSAList</code> must be unqualified, otherwise an
	 *            exception is thrown
	 * @return the <code>Path</code> containing the database segment specified
	 *         as input, as well as its children segments down to the leaf-most
	 *         segment along the hierarchic path indicated by this
	 *         <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public Path getPathForInsert(String segmentName) throws DLIException;

	/**
	 * Returns a <code>Path</code> that contains a one or more database segments
	 * along a hierarchic path for a batch update operation. The
	 * <code>getPathForBatchUpdate</code> method is used together with
	 * {@link PCB#batchUpdate(Path, SSAList)} to update multiple segment
	 * instances in a database in a single network call. In a batch update
	 * operation, the host will do a GU/GN loop and inside the loop perform the
	 * updates until there are no more segments to update and then return the
	 * number of records updated.
	 * <p>
	 * The <code>segmentName</code> parameter indicates the topmost segment to
	 * be updated in the database. Its children segments are included in the
	 * returned <code>Path</code>.
	 * <p>
	 * For a code example of how to use this method, refer to the related
	 * <code>PCB</code> method in the Javadoc.
	 * 
	 * @see PCB#batchUpdate(Path,SSAList)
	 * 
	 * @param segmentName
	 *            the name of a segment that currently exists in this
	 *            <code>SSAList</code>
	 * @return the <code>Path</code> containing the database segment specified
	 *         as input, as well as its children segments down to the leaf-most
	 *         segment along the hierarchic path indicated by this
	 *         <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public Path getPathForBatchUpdate(String segmentName) throws DLIException;

	/**
	 * Add a command code to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code>
	 * @param commandCode
	 *            the command code to add
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addCommandCode(String segmentName, byte commandCode) throws DLIException;

	/**
	 * Add a command code to the <code>SSAList</code>
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the command code to
	 * @param commandCode
	 *            the command code to add
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addCommandCode(int ssaLevel, byte commandCode) throws DLIException;

	/**
	 * Adds command code C and the concatenated key to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            command code to
	 * @param concatenatedKey
	 *            the concatenated key
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addConcatenatedKey(String segmentName, byte[] concatenatedKey) throws DLIException;

	/**
	 * Adds command code C and the concatenated key to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the command code to
	 * @param concatenatedKey
	 *            the concatenated key
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addConcatenatedKey(int ssaLevel, byte[] concatenatedKey) throws DLIException;

	/**
	 * Remove a command code from the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code>
	 * @param commandCode
	 *            the command code to remove
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void removeCommandCode(String segmentName, byte commandCode) throws DLIException;

	/**
	 * Remove a command code from the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            remove the command code from
	 * @param commandCode
	 *            the command code to remove
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void removeCommandCode(int ssaLevel, byte commandCode) throws DLIException;

	/**
	 * Checks to see if a command code is present in the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code>
	 * @param commandCode
	 *            the command code to check
	 * @return <code>true</code> if the command code exists in the
	 *         <code>SSA</code> for the specified segment, <code>false</code>
	 *         otherwise
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	boolean hasCommandCode(String segmentName, byte commandCode) throws DLIException;

	/**
	 * Checks to see if a command code is present in the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code>
	 * @param commandCode
	 *            the command code to check
	 * @return <code>true</code> if the command code exists in the
	 *         <code>SSA</code> for the specified segment, <code>false</code>
	 *         otherwise
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	boolean hasCommandCode(int ssaLevel, byte commandCode) throws DLIException;

	/**
	 * Adds a lock class to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code>
	 * @param lockClass
	 *            the lock class to add
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addLockClass(String segmentName, byte lockClass) throws DLIException;

	/**
	 * Adds a lock class to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code>
	 * @param lockClass
	 *            the lock class to add
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addLockClass(int ssaLevel, byte lockClass) throws DLIException;

	/**
	 * Removes the lock class from the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void removeLockClass(String segmentName) throws DLIException;

	/**
	 * Removes the lock class from the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void removeLockClass(int ssaLevel) throws DLIException;

	/**
	 * Removes all command codes from the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void removeAllCommandCodes(String segmentName) throws DLIException;

	/**
	 * Removes all command codes from the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void removeAllCommandCodes(int ssaLevel) throws DLIException;

	/**
	 * Removes all qualification statements from the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void removeAllQualificationStatements(String segmentName) throws DLIException;

	/**
	 * Removes all qualification statements from the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code>
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void removeAllQualificationStatements(int ssaLevel) throws DLIException;

	// TODO - Do we need these 3 below?
	// public void removeAllCommandCodes();
	// public void removeAllLockClasses();
	// public void removeAllQualificationStatements();

	/**
	 * This method is used to mark a specific field for retrieval from the
	 * database. The <code>markFieldForRetrieval</code> method is used together
	 * with {@link SSAList#getPathForRetrieveReplace} and with the data
	 * retrieval methods in the {@link PCB} interface. When a retrieve call is
	 * made, the resulting <code>Path</code> object will contain all the fields
	 * that have been marked for retrieval. Following the IMS default, all of
	 * the fields in the lowest level segment specified by the
	 * <code>SSAList</code> are inititally marked for retrieval.
	 * 
	 * <p>
	 * The following code example shows how to mark and retrieve the ward number
	 * and patient name fields from the WARD and PATIENT segments. The WARD,
	 * PATIENT, and ILLNESS segments are located in decreasing hierarchic order
	 * below the HOSPITAL segment where HOSPNAME is equal to "SANTA TERESA". By
	 * default, the retrieval operation also returns all the fields of the
	 * ILLNESS segment (ILLLL and ILLNAME). <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.markFieldForRetrieval("WARD", "WARDNO", true);
	 * ssaList.markFieldForRetrieval("PATIENT", "PATNAME", true);
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, false);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * <p>
	 * When one or more of the fields in the lowest level segment specified in
	 * an <code>SSAList</code> is marked for retrieval, only the explicitly
	 * marked fields are retrieved.
	 * <p>
	 * The following code example shows how to mark and retrieve a single field
	 * ILLNAME from the ILLNESS segment where ILLNESS is the lowest level
	 * segment specified in the <code>SSAList</code> instance. The retrieval
	 * operation returns no other fields. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.markFieldForRetrieval("ILLNESS", "ILLNAME", true);
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, false);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> containing
	 *            the field
	 * @param fieldName
	 *            the name of the field to be marked for retrieval from the
	 *            database
	 * @param enableRetrieval
	 *            <code>boolean</code> indicating whether the field should be
	 *            retrieved from the database
	 * @throws DLIException
	 *             if an error occurs during processing
	 * @see SSAList#getPathForRetrieveReplace()
	 * @see PCB#getUnique(Path, SSAList, boolean)
	 * @see PCB#getNextWithinParent(Path, SSAList, boolean)
	 * @see PCB#getNext(Path, SSAList, boolean)
	 */
	public void markFieldForRetrieval(String segmentName, String fieldName, boolean enableRetrieval)
			throws DLIException;

	/**
	 * This method is used to mark a specific field for retrieval from the
	 * database. The <code>markFieldForRetrieval</code> method is used together
	 * with {@link SSAList#getPathForRetrieveReplace} and with the data
	 * retrieval methods in the {@link PCB} interface. When a retrieve call is
	 * made, the resulting <code>Path</code> object will contain all the fields
	 * that have been marked for retrieval. Following the IMS default, all of
	 * the fields in the lowest level segment specified by the
	 * <code>SSAList</code> are inititally marked for retrieval.
	 * 
	 * <p>
	 * The following code example shows how to mark and retrieve the ward number
	 * and patient name fields from the WARD and PATIENT segments. The WARD,
	 * PATIENT, and ILLNESS segments are located in decreasing hierarchic order
	 * below the HOSPITAL segment where HOSPNAME is equal to "SANTA TERESA". By
	 * default, the retrieval operation also returns all the fields of the
	 * ILLNESS segment (ILLLL and ILLNAME). <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.markFieldForRetrieval(3, "WARDNO", true);
	 * ssaList.markFieldForRetrieval(4, "PATNAME", true);
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, false);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * <p>
	 * When one or more of the fields in the lowest level segment specified in
	 * an <code>SSAList</code> is marked for retrieval, only the explicitly
	 * marked fields are retrieved.
	 * <p>
	 * The following code example shows how to mark and retrieve a single field
	 * ILLNAME from the ILLNESS segment where ILLNESS is the lowest level
	 * segment specified in the <code>SSAList</code> instance. The retrieval
	 * operation returns no other fields. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "ILLNESS");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.markFieldForRetrieval(5, "ILLNAME", true);
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, false);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> that
	 *            represents the segment containing the field
	 * @param fieldName
	 *            the name of the field to be marked for retrieval from the
	 *            database
	 * @param enableRetrieval
	 *            <code>boolean</code> indicating whether the field should be
	 *            retrieved from the database
	 * @throws DLIException
	 *             if an error occurs during processing
	 * @see SSAList#getPathForRetrieveReplace()
	 * @see PCB#getUnique(Path, SSAList, boolean)
	 * @see PCB#getNextWithinParent(Path, SSAList, boolean)
	 * @see PCB#getNext(Path, SSAList, boolean)
	 */
	public void markFieldForRetrieval(int ssaLevel, String fieldName, boolean enableRetrieval) throws DLIException;

	/**
	 * This method is used to mark all fields in the specified segment for
	 * retrieval from the database. The <code>markAllFieldsForRetrieval</code>
	 * method is used together with {@link SSAList#getPathForRetrieveReplace}
	 * and with the data retrieval methods in the {@link PCB} interface. When a
	 * retrieve call is made the resulting <code>Path</code> object will contain
	 * only the fields marked for retrieval. Following the IMS default, all of
	 * the fields in the lowest level segment specified by the
	 * <code>SSAList</code> are inititally marked for retrieval.
	 * 
	 * <p>
	 * The following code example shows how to retrieve the all fields for the
	 * PATIENT segment instance. The PATIENT, ILLNESS, and TREATMNT segments are
	 * located in decreasing hierarchic order below the HOSPITAL segment where
	 * HOSPNAME is equal to "SANTA TERESA". By default, the retrieval operation
	 * also returns all the fields of the TREATMNT segment. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "TREATMNT");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.markAllFieldsForRetrieval("PATIENT", true);
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, false);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> containing
	 *            the desired fields
	 * @param enableRetrieval
	 *            <code>boolean</code> indicating whether the fields should be
	 *            retrieved from the database
	 * @throws DLIException
	 *             if an error occurs during processing
	 * @see SSAList#getPathForRetrieveReplace()
	 * @see PCB#getUnique(Path, SSAList, boolean)
	 * @see PCB#getNextWithinParent(Path, SSAList, boolean)
	 * @see PCB#getNext(Path, SSAList, boolean)
	 */
	public void markAllFieldsForRetrieval(String segmentName, boolean enableRetrieval) throws DLIException;

	/**
	 * This method is used to mark all fields in the specified segment for
	 * retrieval from the database. The <code>markAllFieldsForRetrieval</code>
	 * method is used together with {@link SSAList#getPathForRetrieveReplace}
	 * and with the data retrieval methods in the {@link PCB} interface. When a
	 * retrieve call is made the resulting <code>Path</code> object will contain
	 * only the fields marked for retrieval. Following the IMS default, all of
	 * the fields in the lowest level segment specified by the
	 * <code>SSAList</code> are inititally marked for retrieval.
	 * 
	 * <p>
	 * The following code example shows how to retrieve the all fields for the
	 * PATIENT segment instance. The PATIENT, ILLNESS, and TREATMNT segments are
	 * located in decreasing hierarchic order below the HOSPITAL segment where
	 * HOSPNAME is equal to "SANTA TERESA". By default, the retrieval operation
	 * also returns all the fields of the TREATMNT segment. <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "TREATMNT");
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, "SANTA TERESA");
	 * ssaList.markAllFieldsForRetrieval(4, true);
	 * Path path = ssaList.getPathForRetrieveReplace();
	 * pcb.getUnique(path, ssaList, false);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> that
	 *            represents the segment containing the desired fields
	 * @param enableRetrieval
	 *            <code>boolean</code> indicating whether the fields should be
	 *            retrieved from the database
	 * @throws DLIException
	 *             if an error occurs during processing
	 * @see SSAList#getPathForRetrieveReplace()
	 * @see PCB#getUnique(Path, SSAList, boolean)
	 * @see PCB#getNextWithinParent(Path, SSAList, boolean)
	 * @see PCB#getNext(Path, SSAList, boolean)
	 */
	public void markAllFieldsForRetrieval(int ssaLevel, boolean enableRetrieval) throws DLIException;

	// TODO - do we need analogous markXXX methods for insert/update? Makes no
	// sense to send all fields over for insert/update
	// when they aren't all set. Especially with HOSPITAL database where there
	// are 900 bytes per segment yet only 5 fields
	// actually defined in segments totalling 30 or so bytes. Should only send
	// either what API was told to send or mark the
	// fields that were actually set so only those are sent.

	// TODO - have this duplicate exception or just silently replace if the name
	// parm already exists in the list?

	/**
	 * This method is used when the value of a field in a qualification
	 * statement isn't known until runtime. When the <code>SSAList</code> is
	 * being constructed, a <code>PreparedValue</code> object can be passed in
	 * as the value of the field. When the value is known, the
	 * <code>PreparedValue</code> object can be set with the value of the field
	 * and the qualification statement will be complete.
	 * 
	 * <p>
	 * <code>PreparedValues</code> are also very useful when the same
	 * <code>SSAList</code> is going to be reused multiple times with different
	 * values for the fields in the qualification statements.
	 * <p>
	 * The following code fragment shows how <code>PreparedValue</code> is used
	 * to retrieve the patient and treatment records for patient "ANDREA SMITH"
	 * from two different hospitals ("SANTA TERESA" and "ALEXANDRIA").
	 * <blockquote>
	 * 
	 * <pre>
	 * SSAList ssaList = pcb.getSSAList("HOSPITAL", "TREATMNT");
	 * PreparedValue pv = ssaList.createPreparedValue("HOSPNAME");  
	 * 
	 * // pass in the prepared value instead of an actual value as the actual value is unknown at this time
	 * ssaList.addInitialQualification("HOSPITAL", "HOSPNAME", SSAList.EQUALS, pv);  
	 * ssaList.addInitialQualification("PATIENT", "PATNAME", SSAList.EQUALS, "ANDREA SMITH");
	 * Path path = ssaList.getPathForRetrieveReplace();  
	 * ...
	 * pv.setValue("SANTA TERESA"); 
	 * pcb.getUnique(path, ssaList, true);  
	 * ...
	 * pv.setValue("ALEXANDRIA"); 
	 * pcb.getUnique(path, ssaList, true);
	 * </pre>
	 * 
	 * </blockquote>
	 * 
	 * @see SSAList#addInitialQualification(String, String, short,
	 *      PreparedValue)
	 * @see SSAList#appendQualification(int, byte, String, short, PreparedValue)
	 * @param name
	 *            the name of the <code>PreparedValue</code>, typically the name
	 *            of the field whose value will be set at a later time
	 * @return the <code>PreparedValue</code> object
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public PreparedValue createPreparedValue(String name) throws DLIException;

	/**
	 * Returns a previously created <code>PreparedValue</code> object.
	 * 
	 * @param name
	 *            the name of a previously created <code>PreparedValue</code>
	 * @return the <code>PreparedValue</code> object
	 */
	public PreparedValue getPreparedValue(String name);

	/**
	 * Returns a <code>Collection</code> of all previously created
	 * <code>PreparedValue</code> objects with this <code>SSAList</code>
	 * 
	 * @return the <code>Collection</code> of <code>PreparedValue</code> objects
	 */
	public Collection<PreparedValue> getPreparedValues();

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, boolean value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, boolean value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, byte value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, byte value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, int value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, int value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, short value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, short value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, long value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, long value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, float value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, float value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, double value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, double value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, byte[] value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, byte[] value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, String value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, String value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp,
			java.math.BigDecimal value) throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, java.math.BigDecimal value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, java.sql.Date value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, java.sql.Date value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, java.sql.Time value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, java.sql.Time value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp,
			java.sql.Timestamp value) throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, java.sql.Timestamp value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            a <code>PreparedValue</code> object representing the value of
	 *            the field
	 * 
	 * @see SSAList#createPreparedValue(String)
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(String segmentName, String fieldName, short relationalOp, PreparedValue value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            a <code>PreparedValue</code> object representing the value of
	 *            the field
	 * 
	 * @see SSAList#createPreparedValue(String)
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void addInitialQualification(int ssaLevel, String fieldName, short relationalOp, PreparedValue value)
			throws DLIException;

	/**
	 * Add the initial qualifiation statement to the <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			boolean value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified <code>SSA</code> in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp, boolean value)
			throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp, int value)
			throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp, int value)
			throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			short value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp, short value)
			throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			long value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp, long value)
			throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			float value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp, float value)
			throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			byte[] value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp, byte[] value)
			throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			String value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp, String value)
			throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			java.math.BigDecimal value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp,
			java.math.BigDecimal value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			java.sql.Date value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp,
			java.sql.Date value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			java.sql.Time value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp,
			java.sql.Time value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			java.sql.Timestamp value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            the value of the field
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp,
			java.sql.Timestamp value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param segmentName
	 *            the name of the segment in the <code>SSAList</code> to add the
	 *            qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            a <code>PreparedValue</code> object representing the value of
	 *            the field
	 * 
	 * @see SSAList#createPreparedValue(String)
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(String segmentName, byte booleanOp, String fieldName, short relationalOp,
			PreparedValue value) throws DLIException;

	/**
	 * Append a qualifiation statement to the specified SSA in the
	 * <code>SSAList</code>.
	 * 
	 * @param ssaLevel
	 *            the 1-based level of the SSA in the <code>SSAList</code> to
	 *            add the qualification to
	 * @param booleanOp
	 *            the boolean operator to use to connect this qualification to
	 *            the previous - one of the <code>SSAList</code> constants
	 * @param fieldName
	 *            the name of the field
	 * @param relationalOp
	 *            the relational operator to use - one of the
	 *            <code>SSAList</code> constants
	 * @param value
	 *            a <code>PreparedValue</code> object representing the value of
	 *            the field
	 * 
	 * @see SSAList#createPreparedValue(String)
	 * @throws DLIException
	 *             if an error occurs during processing
	 */
	public void appendQualification(int ssaLevel, byte booleanOp, String fieldName, short relationalOp,
			PreparedValue value) throws DLIException;

	/**
	 * Builds the <code>SSAList</code> in proper DL/I format.
	 * <p>
	 * <B>Note:</B> This method doesn't need to be called by client
	 * applications. It is only used by clients for debugging purposes.
	 * 
	 * @return the <code>SSAList</code> in proper DL/I format
	 * @throws DLIException
	 */
	public byte[][] buildSSAListInBytes() throws DLIException;
}
